/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: linepolypolygonbase.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:15:06 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>

#ifndef _COM_SUN_STAR_GEOMETRY_AFFINEMATRIX2D_HPP_
#include <com/sun/star/geometry/AffineMatrix2D.hpp>
#endif
#ifndef _COM_SUN_STAR_GEOMETRY_MATRIX2D_HPP_
#include <com/sun/star/geometry/Matrix2D.hpp>
#endif

#ifndef _COM_SUN_STAR_RENDERING_RENDERSTATE_HPP__
#include <com/sun/star/rendering/RenderState.hpp>
#endif
#ifndef _COM_SUN_STAR_RENDERING_VIEWSTATE_HPP__
#include <com/sun/star/rendering/ViewState.hpp>
#endif
#ifndef _COM_SUN_STAR_RENDERING_XCANVAS_HPP__
#include <com/sun/star/rendering/XCanvas.hpp>
#endif
#ifndef _COM_SUN_STAR_RENDERING_COMPOSITEOPERATION_HPP__
#include <com/sun/star/rendering/CompositeOperation.hpp>
#endif

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_RANGE_B2DRANGE_HXX
#include <basegfx/range/b2drange.hxx>
#endif
#ifndef _BGFX_RANGE_B2DRECTANGLE_HXX
#include <basegfx/range/b2drectangle.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_TOOLS_CANVASTOOLS_HXX
#include <basegfx/tools/canvastools.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYGON_HXX
#include <basegfx/polygon/b2dpolygon.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYPOLYGONTOOLS_HXX
#include <basegfx/polygon/b2dpolypolygontools.hxx>
#endif

#include <canvas/linepolypolygonbase.hxx>


using namespace ::com::sun::star;

namespace canvas
{

    namespace tools
    {

		LinePolyPolygonBase::LinePolyPolygonBase( const ::basegfx::B2DPolyPolygon& rPolyPoly ) :
            LinePolyPolygon_Base( m_aMutex ),
			maPolyPoly( rPolyPoly )
		{
		}

		LinePolyPolygonBase::~LinePolyPolygonBase()
		{
		}

		void SAL_CALL LinePolyPolygonBase::addPolyPolygon(
			const geometry::RealPoint2D& 						position, 
			const uno::Reference< rendering::XPolyPolygon2D >&	polyPolygon ) throw (lang::IllegalArgumentException,uno::RuntimeException)
		{
            // TODO(F1): Correctly fulfill the UNO API
            // specification. This will probably result in a vector of
            // poly-polygons to be stored in this object.

            const sal_Int32 nPolys( polyPolygon->getNumberOfPolygons() );

            if( !polyPolygon.is() || !nPolys )
            {
                // invalid or empty polygon - nothing to do.
                return;
            }

            ::basegfx::B2DPolyPolygon 	aSrcPoly;
            const LinePolyPolygonBase* 	pSrc( dynamic_cast< LinePolyPolygonBase* >(polyPolygon.get()) );

            // try to extract polygon data from interface. First,
            // check whether it's the same implementation object,
            // which we can tunnel then.
            if( pSrc )
            {
                aSrcPoly = pSrc->getPolyPolygon();
            }
            else
            {
                // not a known implementation object - try data source
                // interfaces
                uno::Reference< rendering::XBezierPolyPolygon2D > xBezierPoly(
                    polyPolygon,
                    uno::UNO_QUERY );

                if( xBezierPoly.is() )
                {
                    aSrcPoly = ::basegfx::unotools::polyPolygonFromBezier2DSequenceSequence(
                        xBezierPoly->getBezierSegments( 0,
                                                        nPolys,
                                                        0,
                                                        -1 ) );
                }
                else
                {
                    uno::Reference< rendering::XLinePolyPolygon2D > xLinePoly(
                        polyPolygon,
                        uno::UNO_QUERY );

                    // no implementation class and no data provider
                    // found - contract violation.
                    CHECK_AND_THROW( xLinePoly.is(),
                                     "LinePolyPolygonBase::addPolyPolygon(): Invalid input "
                                     "poly-polygon, cannot retrieve vertex data" );

                    aSrcPoly = ::basegfx::unotools::polyPolygonFromPoint2DSequenceSequence(
                        xLinePoly->getPoints( 0,
                                              nPolys,
                                              0,
                                              -1 ) );
                }
            }

            const ::basegfx::B2DRange	aBounds( ::basegfx::tools::getRange( aSrcPoly ) );
            const ::basegfx::B2DVector 	aOffset( ::basegfx::unotools::b2DPointFromRealPoint2D( position ) -
                                                 aBounds.getMinimum() );

            if( !aOffset.equalZero() )
            {
                ::basegfx::B2DHomMatrix aTranslate;
                aTranslate.translate( aOffset.getX(), aOffset.getY() );

                aSrcPoly.transform( aTranslate );
            }

            maPolyPoly.append( aSrcPoly );
		}

		sal_Int32 SAL_CALL LinePolyPolygonBase::getNumberOfPolygons() throw (uno::RuntimeException)
		{
			return maPolyPoly.count();
		}

		sal_Int32 SAL_CALL LinePolyPolygonBase::getNumberOfPolygonPoints( 
			sal_Int32 polygon ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            checkIndex( polygon );

            return maPolyPoly.getB2DPolygon(polygon).count();
		}

		rendering::FillRule SAL_CALL LinePolyPolygonBase::getFillRule() throw (uno::RuntimeException)
		{
			// TODO(F1)
			return rendering::FillRule_NON_ZERO;
		}

		void SAL_CALL LinePolyPolygonBase::setFillRule( 
			rendering::FillRule fillRule ) throw (uno::RuntimeException)
		{
			// TODO(F1)
		}

		sal_Bool SAL_CALL LinePolyPolygonBase::isClosed( 
			sal_Int32 index ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            checkIndex( index );

			return maPolyPoly.getB2DPolygon(index).isClosed();
		}

		void SAL_CALL LinePolyPolygonBase::setClosed(
			sal_Int32 index, 
			sal_Bool closedState ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
			if( index == -1L )
            {
                // set all
				maPolyPoly.setClosed( closedState );
            }
			else
			{
                checkIndex( index );

				// fetch referenced polygon, change state
				::basegfx::B2DPolygon aTmp( maPolyPoly.getB2DPolygon(index) );
				aTmp.setClosed( closedState );

				// set back to container
				maPolyPoly.setB2DPolygon( index, aTmp );
			}
		}

		uno::Sequence< uno::Sequence< geometry::RealPoint2D > > SAL_CALL LinePolyPolygonBase::getPoints(
			sal_Int32 nPolygonIndex, 
			sal_Int32 nNumberOfPolygons, 
			sal_Int32 nPointIndex, 
			sal_Int32 nNumberOfPoints ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            return ::basegfx::unotools::pointSequenceSequenceFromB2DPolyPolygon( 
                getSubsetPolyPolygon( nPolygonIndex,
                                      nNumberOfPolygons,
                                      nPointIndex,
                                      nNumberOfPoints ) );
		}

		void SAL_CALL LinePolyPolygonBase::setPoints( 
			const uno::Sequence< uno::Sequence< geometry::RealPoint2D > >& points, 
			sal_Int32 nPolygonIndex ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            const ::basegfx::B2DPolyPolygon& rNewPolyPoly( 
                ::basegfx::unotools::polyPolygonFromPoint2DSequenceSequence( points ) );

            if( nPolygonIndex == -1 )
            {
                maPolyPoly = rNewPolyPoly;
            }
            else
            {
                checkIndex( nPolygonIndex );

                maPolyPoly.insert( nPolygonIndex, rNewPolyPoly );
            }
		}

		geometry::RealPoint2D SAL_CALL LinePolyPolygonBase::getPoint( 
			sal_Int32 nPolygonIndex, 
			sal_Int32 nPointIndex ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            checkIndex( nPolygonIndex );

            const ::basegfx::B2DPolygon& rPoly( maPolyPoly.getB2DPolygon( nPolygonIndex ) );

            if( nPointIndex < 0 || nPointIndex >= static_cast<sal_Int32>(rPoly.count()) )
                throw lang::IndexOutOfBoundsException();

            return ::basegfx::unotools::point2DFromB2DPoint( rPoly.getB2DPoint( nPointIndex ) );
		}

		void SAL_CALL LinePolyPolygonBase::setPoint( 
			const geometry::RealPoint2D& point, 
			sal_Int32 nPolygonIndex, 
			sal_Int32 nPointIndex ) throw (lang::IndexOutOfBoundsException,uno::RuntimeException)
		{
            checkIndex( nPolygonIndex );

            ::basegfx::B2DPolygon aPoly( maPolyPoly.getB2DPolygon( nPolygonIndex ) );

            if( nPointIndex < 0 || nPointIndex >= static_cast<sal_Int32>(aPoly.count()) )
                throw lang::IndexOutOfBoundsException();

            aPoly.setB2DPoint( nPointIndex,
                               ::basegfx::unotools::b2DPointFromRealPoint2D( point ) );
            maPolyPoly.setB2DPolygon( nPolygonIndex, aPoly );
		}

        uno::Sequence< uno::Sequence< geometry::RealBezierSegment2D > > SAL_CALL LinePolyPolygonBase::getBezierSegments( 
            sal_Int32 nPolygonIndex,
            sal_Int32 nNumberOfPolygons, 
            sal_Int32 nPointIndex, 
            sal_Int32 nNumberOfPoints ) throw (lang::IndexOutOfBoundsException, uno::RuntimeException)
        {
            return ::basegfx::unotools::bezierSequenceSequenceFromB2DPolyPolygon( 
                getSubsetPolyPolygon( nPolygonIndex,
                                      nNumberOfPolygons,
                                      nPointIndex,
                                      nNumberOfPoints ) );
        }

        void SAL_CALL LinePolyPolygonBase::setBezierSegments( 
            const uno::Sequence< uno::Sequence< geometry::RealBezierSegment2D > >&	points, 
            sal_Int32 																nPolygonIndex ) throw (lang::IndexOutOfBoundsException, 
                                                                                                           uno::RuntimeException)
        {
            const ::basegfx::B2DPolyPolygon& rNewPolyPoly( 
                ::basegfx::unotools::polyPolygonFromBezier2DSequenceSequence( points ) );

            if( nPolygonIndex == -1 )
            {
                maPolyPoly = rNewPolyPoly;
            }
            else
            {
                checkIndex( nPolygonIndex );

                maPolyPoly.insert( nPolygonIndex, rNewPolyPoly );
            }
        }

        geometry::RealBezierSegment2D SAL_CALL LinePolyPolygonBase::getBezierSegment( sal_Int32 nPolygonIndex, 
                                                                                      sal_Int32 nPointIndex ) throw (lang::IndexOutOfBoundsException, 
                                                                                                                     uno::RuntimeException)
        {
            checkIndex( nPolygonIndex );

            const ::basegfx::B2DPolygon& rPoly( maPolyPoly.getB2DPolygon( nPolygonIndex ) );

            if( nPointIndex < 0 || nPointIndex >= static_cast<sal_Int32>(rPoly.count()) )
                throw lang::IndexOutOfBoundsException();

            const ::basegfx::B2DPoint& rPt( rPoly.getB2DPoint( nPointIndex ) );
            const ::basegfx::B2DPoint& rCtrl0( rPoly.getControlPointA( nPointIndex ) );
            const ::basegfx::B2DPoint& rCtrl1( rPoly.getControlPointB( nPointIndex ) );

            return geometry::RealBezierSegment2D( rPt.getX(),
                                                  rPt.getY(),
                                                  rCtrl0.getX(),
                                                  rCtrl0.getY(),
                                                  rCtrl1.getX(),
                                                  rCtrl1.getY() );
        }

        void SAL_CALL LinePolyPolygonBase::setBezierSegment( const geometry::RealBezierSegment2D&	segment, 
                                                             sal_Int32 								nPolygonIndex, 
                                                             sal_Int32								nPointIndex ) throw (lang::IndexOutOfBoundsException, 
                                                                                                                         uno::RuntimeException)
        {
            checkIndex( nPolygonIndex );

            ::basegfx::B2DPolygon aPoly( maPolyPoly.getB2DPolygon( nPolygonIndex ) );

            if( nPointIndex < 0 || nPointIndex >= static_cast<sal_Int32>(aPoly.count()) )
                throw lang::IndexOutOfBoundsException();

            aPoly.setB2DPoint( nPointIndex,
                               ::basegfx::B2DPoint( segment.Px,
                                                    segment.Py ) );
            aPoly.setControlPointA( nPointIndex,
                                    ::basegfx::B2DPoint( segment.C1x,
                                                         segment.C1y ) );
            aPoly.setControlPointB( nPointIndex,
                                    ::basegfx::B2DPoint( segment.C2x,
                                                         segment.C2y ) );

            maPolyPoly.setB2DPolygon( nPolygonIndex, aPoly );
        }

		::basegfx::B2DPolyPolygon LinePolyPolygonBase::getSubsetPolyPolygon(
			sal_Int32 nPolygonIndex, 
			sal_Int32 nNumberOfPolygons, 
			sal_Int32 nPointIndex, 
			sal_Int32 nNumberOfPoints ) const
		{
            checkIndex( nPolygonIndex );

            const sal_Int32 nPolyCount( maPolyPoly.count() );

            // check for "full polygon" case
            if( !nPolygonIndex && 
                !nPointIndex &&
                nNumberOfPolygons == nPolyCount &&
                nNumberOfPoints == -1 )
            {
                return maPolyPoly;
            }

            ::basegfx::B2DPolyPolygon aSubsetPoly;

            // create temporary polygon (as an extract from maPoly,
            // which contains the requested subset)
            for( sal_Int32 i=nPolygonIndex; i<nNumberOfPolygons; ++i )
            {
                checkIndex(i);

                const ::basegfx::B2DPolygon& rCurrPoly( maPolyPoly.getB2DPolygon(i) );

                sal_Int32 nFirstPoint(0);
                sal_Int32 nLastPoint(nPolyCount-1);

                if( nPointIndex && i==nPolygonIndex )
                {
                    // very first polygon - respect nPointIndex, if
                    // not zero

                    // empty polygon - impossible to specify _any_
                    // legal value except 0 here!
                    if( !nPolyCount && nPointIndex )
                        throw lang::IndexOutOfBoundsException();

                    nFirstPoint = nPointIndex;
                }
                
                if( i==nNumberOfPolygons-1 && nNumberOfPoints != -1 )
                {
                    // very last polygon - respect nNumberOfPoints

                    // empty polygon - impossible to specify _any_
                    // legal value except -1 here!
                    if( !nPolyCount )
                        throw lang::IndexOutOfBoundsException();

                    nLastPoint = nFirstPoint+nNumberOfPoints;
                }

                if( !nPolyCount )
                {
                    // empty polygon - index checks already performed
                    // above, now simply append empty polygon
                    aSubsetPoly.append( rCurrPoly );
                }
                else
                {
                    if( nFirstPoint < 0 || nFirstPoint >= nPolyCount )
                        throw lang::IndexOutOfBoundsException();

                    if( nLastPoint < 0 || nLastPoint >= nPolyCount )
                        throw lang::IndexOutOfBoundsException();

                    ::basegfx::B2DPolygon aTmp;
                    for( sal_Int32 j=nFirstPoint; j<nLastPoint; ++j )
                        aTmp.append( rCurrPoly.getB2DPoint(j) );

                    aSubsetPoly.append( aTmp );
                }
            }
            
            return aSubsetPoly;
        }

		::basegfx::B2DPolyPolygon LinePolyPolygonBase::getPolyPolygon() const
		{
			return maPolyPoly;
		}

	} // namespace tools

} // namespace canvas
