/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: filtercachedata.cxx,v $
 *
 *  $Revision: 1.25 $
 *
 *  last change: $Author: obo $ $Date: 2006/09/16 13:46:21 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_framework.hxx"


//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_CLASSES_FILTERCACHEDATA_HXX_
#include <classes/filtercachedata.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_WRITEGUARD_HXX_
#include <threadhelp/writeguard.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_READGUARD_HXX_
#include <threadhelp/readguard.hxx>
#endif

#ifndef __FRAMEWORK_MACROS_DEBUG_HXX_
#include <macros/debug.hxx>
#endif

#ifndef __FRAMEWORK_FILTERFLAGS_H_
#include <filterflags.h>
#endif

#ifndef __FRAMEWORK_SERVICES_H_
#include <services.h>
#endif

#ifndef __FRAMEWORK_GENERAL_H_
#include <general.h>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	other includes
//_________________________________________________________________________________________________________________

#ifndef _UTL_CONFIGMGR_HXX_
#include <unotools/configmgr.hxx>
#endif

#ifndef _UTL_CONFIGITEM_HXX_
#include <unotools/configitem.hxx>
#endif

#ifndef _UTL_CONFIGPATHES_HXX_
#include <unotools/configpathes.hxx>
#endif

#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _RTL_URI_HXX_
#include <rtl/uri.hxx>
#endif

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//  non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//  non exported definitions
//_________________________________________________________________________________________________________________

template< class HashType > void SetNodeHash< HashType >::appendChange( const ::rtl::OUString& sName  ,
                                                                             EModifyState     eState )
{
    OUStringList::iterator pAdded   = lAddedItems.find  (sName);
    OUStringList::iterator pChanged = lChangedItems.find(sName);
    OUStringList::iterator pRemoved = lRemovedItems.find(sName);

    sal_Bool bAdded   = (pAdded   != lAddedItems.end()  );
    sal_Bool bChanged = (pChanged != lChangedItems.end());
    sal_Bool bRemoved = (pRemoved != lRemovedItems.end());
    sal_Bool bUnknown = (!bAdded && !bChanged && !bRemoved);

    #ifdef ENABLE_FILTERDBG
        sal_Int32 nCheck = 0;
        if (bAdded)
            ++nCheck;
        if (bChanged)
            ++nCheck;
        if (bRemoved)
            ++nCheck;
        LOG_COND_FILTERDBG(nCheck > 1, "SetNodeHash< xxx >::appendChange()", "changes list seams to be corrupted")
    #endif // ENABLE_FILTERDBG

    switch(eState)
    {
        //-------------------------------------------------------------------------------
        case E_ADDED :
        {
            //  a)
            //      bRemoved + ADD => bChanged
            //      ... because it was not removed inside cfg layer!
            //      It was only marked as "please remove it" inside this structures.
            //      Now it will be added again before cfg layer was realy updated.
            //      And then it's not an ADDED it's CHANGED.
            if (bRemoved)
            {
                lRemovedItems.erase(pRemoved);
                lChangedItems.push_back(sName);
                return;
            }

            //  c)
            //      ? + ADD => bAdded
            //      ... OK
            if (bUnknown)
            {
                lAddedItems.push_back(sName);
                return;
            }

            #ifdef ENABLE_FILTERDBG
            ::rtl::OUStringBuffer sBuffer(256);
            sBuffer.appendAscii("invalid state for E_ADDED detected");
            sBuffer.appendAscii("\nbAdded   = "            );
            sBuffer.appendAscii(bAdded   ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbChanged = "            );
            sBuffer.appendAscii(bChanged ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbRemoved = "            );
            sBuffer.appendAscii(bRemoved ? "TRUE" : "FALSE");
            LOG_FILTERDBG("SetNodeHash< xxx >::appendChange()", U2B(sBuffer.makeStringAndClear()))
            #endif // ENABLE_FILTERDBG
        }
        break;

        //-------------------------------------------------------------------------------
        case E_CHANGED :
        {
            //  a)
            //      bAdded + CHANGE => bChanged
            //      ... OK
            if (bAdded)
            {
                lAddedItems.erase(pAdded);
                lChangedItems.push_back(sName);
                return;
            }

            //  b)
            //      ? + CHANGE => bChanged
            //      ... OK
            if (bUnknown)
            {
                lChangedItems.push_back(sName);
                return;
            }

            #ifdef ENABLE_FILTERDBG
            ::rtl::OUStringBuffer sBuffer(256);
            sBuffer.appendAscii("invalid state for E_CHANGED detected");
            sBuffer.appendAscii("\nbAdded   = "            );
            sBuffer.appendAscii(bAdded   ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbChanged = "            );
            sBuffer.appendAscii(bChanged ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbRemoved = "            );
            sBuffer.appendAscii(bRemoved ? "TRUE" : "FALSE");
            LOG_FILTERDBG("SetNodeHash< xxx >::appendChange()", U2B(sBuffer.makeStringAndClear()))
            #endif // ENABLE_FILTERDBG
        }
        break;

        //-------------------------------------------------------------------------------
        case E_REMOVED :
        {
            //  a)
            //      bChanged + REMOVE => bRemoved
            //      ... OK
            if (bChanged)
            {
                lChangedItems.erase(pChanged);
                lRemovedItems.push_back(sName);
                return;
            }

            //  b)
            //      bAdded + REMOVE => bRemoved
            //      ... OK
            if (bAdded)
            {
                lAddedItems.erase(pAdded);
                lRemovedItems.push_back(sName);
                return;
            }

            //  c)
            //      ? + REMOVE => bRemoved
            //      ... OK
            if (bUnknown)
            {
                lRemovedItems.push_back(sName);
                return;
            }

            #ifdef ENABLE_FILTERDBG
            ::rtl::OUStringBuffer sBuffer(256);
            sBuffer.appendAscii("invalid state for E_REMOVED detected");
            sBuffer.appendAscii("\nbAdded   = "            );
            sBuffer.appendAscii(bAdded   ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbChanged = "            );
            sBuffer.appendAscii(bChanged ? "TRUE" : "FALSE");
            sBuffer.appendAscii("\nbRemoved = "            );
            sBuffer.appendAscii(bRemoved ? "TRUE" : "FALSE");
            LOG_FILTERDBG("SetNodeHash< xxx >::appendChange()", U2B(sBuffer.makeStringAndClear()))
            #endif // ENABLE_FILTERDBG
        }
        break;
    }
}

/*-****************************************************************************************************//**
    @short      standard constructor
    @descr      This will open an configuration access to given path ... but no values
                are readed! Because user will do that by explicit calling of read-method. He give
                us a pointer to his container, and we fill it.
                Writing works in the same way. We use given data container to do that.

    @seealso    baseclass ConfigItem
    @seealso    method read()
    @seealso    method write()

    @param      "sPath", path to configuration file and root node in this file
    @return     -

    @onerror    -
*//*-*****************************************************************************************************/
FilterCFGAccess::FilterCFGAccess( const ::rtl::OUString& sPath   ,
                                        sal_Int32        nVersion,
                                        sal_Int16        nMode   )
    : ::utl::ConfigItem ( sPath, nMode )
    , m_nVersion        ( nVersion     )
{
    // Check incoming parameter first.
    LOG_ASSERT2( implcp_ctor( sPath, nVersion, nMode ), "FilterCFGAccess::ctor", "Invalid parameter detected!" )

    // Use given path to set internal "working mode".
    // It's neccessary to handle read/write operations in a different way.
    // Some items are not handled for additional configuration file!
    if( sPath == PACKAGENAME_TYPEDETECTION_STANDARD )
    {
        m_ePackage = E_STANDARD;
    }
    else
    if( sPath == PACKAGENAME_TYPEDETECTION_ADDITIONAL )
    {
        m_ePackage = E_ADDITIONAL;
    }

    css::uno::Any aProperty = ::utl::ConfigManager::GetConfigManager()->GetDirectConfigProperty( ::utl::ConfigManager::PRODUCTNAME );
    sal_Bool      bState    = (aProperty >>= m_sProductName);
    if(
        ( bState                     == sal_False )    ||
        ( m_sProductName.getLength() <  1         )
      )
    {
        LOG_ERROR( "FilterCFGAccess::ctor()", "Couldn't get ProductName from configuration! Use \"StarOffice\" as fallback ..." )
        m_sProductName = PRODUCTNAME_FALLBACK;
    }

    aProperty = ::utl::ConfigManager::GetConfigManager()->GetDirectConfigProperty( ::utl::ConfigManager::PRODUCTXMLFILEFORMATVERSION );
    bState    = (aProperty >>= m_sFormatVersion);
    if(
        ( bState                       == sal_False )    ||
        ( m_sFormatVersion.getLength() <  1         )
      )
    {
        LOG_ERROR( "FilterCFGAccess::ctor()", "Couldn't get FormatVersion from configuration! Use \"6.0/7\" as fallback ..." )
        m_sFormatVersion = FORMATVERSION_FALLBACK;
    }

    // Initialize right sub key count for current set file version.
    // It's neccessary to read right structure from configuration and get enough memory for that operations!
    impl_initKeyCounts();
}

/*-****************************************************************************************************//**
    @short      standard destructor
    @descr      We do nothing here!

    @seealso    -

    @param      -
    @return     -

    @onerror    -
*//*-*****************************************************************************************************/
FilterCFGAccess::~FilterCFGAccess()
{
}

/*-****************************************************************************************************//**
    @short      read configuration data into given data container
    @descr      We use given data container to fill it with readed values from configuration.
                Reading values depend from opened path! Because - our filter configuration is
                tiled into two parts ... standard and additional filter. Format of both files
                is always the same ... but some items are not set in additional file.
                e.g. default detector and loader
                This informations are set in standard file only!

    @seealso    -

    @param      "rData", pointer to data container for filling
    @return     Filled data container.

    @onerror    We do nothing! (But we show some assertions.)
*//*-*****************************************************************************************************/
void FilterCFGAccess::read( DataContainer&          rData ,
                            DataContainer::ECFGType eType )
{
    // Safe impossible cases!
    LOG_ASSERT2( implcp_read( rData ), "FilterCFGAccess::read()", "Invalid parameter detected!" )

    // Clear container first!
    rData.free();

    // Get current set locale from configuration.
    // We read all localized values for all locales from our configuration ...
    // but sometimes we must give our owners one value for current set locale!
    // This value is neccessary for later "impl_load...()" calls too!!!
    css::uno::Any   aProperty = ::utl::ConfigManager::GetConfigManager()->GetDirectConfigProperty( ::utl::ConfigManager::LOCALE );
    sal_Bool        bState    = (aProperty >>= rData.m_sLocale)                                                                    ;
    if(
        ( bState                      == sal_False )    ||
        ( rData.m_sLocale.getLength() <  1         )
      )
    {
        LOG_ERROR( "FilterCFGAccess::read()", "Couldn't get Locale of configuration! Localized values will not handled correctly. But I try it by using \"en-US\" as fallback ..." )
        rData.m_sLocale = LOCALE_FALLBACK;
    }

    // Read values from configuration. Look for right package and version handling too!
    switch( m_ePackage )
    {
        case E_STANDARD     :   {
                                    impl_loadTypes      ( rData );
                                    impl_loadFilters    ( rData );
                                    impl_loadDetectors  ( rData );
                                    impl_loadLoaders    ( rData );
                                    impl_loadDefaults   ( rData );
                                    if( m_nVersion >= 5 )
                                    {
                                        impl_loadContentHandlers( rData );
                                    }
                                }
                                break;
        case E_ADDITIONAL   :   {
                                    impl_loadTypes      ( rData );
                                    impl_loadFilters    ( rData );
                                }
                                break;
    }
}

/*-****************************************************************************************************//**
    @short      write configuration data from given data container to configuration
    @descr      We use given data container to write his values to configuration.
                Writing values depend from opened path! Because - our filter configuration is
                tiled into two parts ... standard and additional filter. Format of both files
                is always the same ... but some items are not set in additional file.
                e.g. default detector and loader
                This informations are set in standard file only!

    @seealso    -

    @param      "rData", pointer to data container
    @return     -

    @onerror    We do nothing! (But we show some assertions.)
*//*-*****************************************************************************************************/
void FilterCFGAccess::write( DataContainer&          rData ,
                             DataContainer::ECFGType eType )
{
    // look for version and type handling!
    // ignore superflous calls ...
    // Don't forget to reset modified flag ...
    // otherwise we will flush our data again and again and ...
    if (
        (rData.m_bTypesModified                                  ) &&
        ((eType & DataContainer::E_TYPE) == DataContainer::E_TYPE)
       )
    {
        impl_saveTypes(rData);
        rData.m_bTypesModified = sal_False;
    }

    if (
        (rData.m_bFiltersModified                                    ) &&
        ((eType & DataContainer::E_FILTER) == DataContainer::E_FILTER)
       )
    {
        impl_saveFilters(rData);
        rData.m_bFiltersModified = sal_False;
    }

    if (
        (rData.m_bDetectorsModified                                                ) &&
        ((eType & DataContainer::E_DETECTSERVICE) == DataContainer::E_DETECTSERVICE)
       )
    {
        impl_saveDetectors(rData);
        rData.m_bDetectorsModified = sal_False;
    }
}

/*-************************************************************************************************************//**
    @short      encode/decode set names
    @descr      Our configuration support special characters of set names by a special encoding/decoding.
                These two methods do that for our values.

                encode:     <name>                     =>      <templatetype>["<name>"]
                decode:     <templatetype>["<name>"]   =>      <name>

                - with <templatetype> = {Type|Filter|DetectService|FrameLoader|ContentHandler}
                  depends from given eType!
                - with <name> valid html-encoded! "&" => "&amp;" ...!

    @seealso    -

    @param      "sName", source for convertion
    @param      "eType", specify template
    @return     Converted value or untouched one, if there is nothing to do.

    @onerror    -
*//*-*************************************************************************************************************/
/*
::rtl::OUString FilterCFGAccess::encodeSetName( const ::rtl::OUString& sName, ETemplateType eType )
{
    ::rtl::OUStringBuffer  sSource     ( sName )          ;
    ::rtl::OUStringBuffer  sDestination( 256   )          ;
    sal_Int32              nCount      = sName.getLength();
    sal_Int32              nPosition   = 0                ;

    // add <templatename>
    switch( eType )
    {
        case E_TEMPLATE_TYPE             :   sDestination.append( TEMPLATE_TYPE              );
                                    break;
        case E_TEMPLATE_FILTER           :   sDestination.append( TEMPLATE_FILTER            );
                                    break;
        case E_TEMPLATE_DETECTSERVICE    :   sDestination.append( TEMPLATE_DETECTSERVICE     );
                                    break;
        case E_TEMPLATE_FRAMELOADER      :   sDestination.append( TEMPLATE_FRAMELOADER       );
                                    break;
        case E_TEMPLATE_CONTENTHANDLER   :   sDestination.append( TEMPLATE_CONTENTHANDLER    );
                                    break;
    }

    // add ["
    sDestination.append( CFG_ENCODING_OPEN );

    // add name (html encoded!)
    for( nPosition=0; nPosition<nCount; ++nPosition )
	{
        sal_Unicode cSign = sSource.charAt(nPosition);
        switch( cSign )
		{
            // code &, ", ', <, > ...
            case '&' :  sDestination.appendAscii( "&amp;"   );
						break;
            case '<' :  sDestination.appendAscii( "&lt;"    );
						break;
            case '>' :  sDestination.appendAscii( "&gt;"    );
						break;
            case '\'':  sDestination.appendAscii( "&rsquo;" );
						break;
            case '\"':  sDestination.appendAscii( "&quot;"  );
						break;
			// copy all other letters
            default :   sDestination.append( cSign );
						break;
		}
	}

    // add "]
    sDestination.append( CFG_ENCODING_CLOSE );

    // return encoded name
    return sDestination.makeStringAndClear();
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::decodeSetName( const ::rtl::OUString& sName, ETemplateType eType )
{
    LOG_ERROR( "FilterCFGAccess::decodeSetName()", "Not implemented yet!" )
    return sName;
}
*/

/*-************************************************************************************************************//**
    @short      build own formated string of type/filter properties with right encoding of forbidden signs
    @descr      To make our configuration file smaller and guarant a faster acces - we don't handle all properties
                seperatly. We write an own formated string for some of them.
                These two methods build this string for types/filters ... look at "parse..." methods too.
                They analyze this string at reading and fill right type/filter structures.

    @seealso    method decodeTypeData()
    @seealso    method decodeFilterData()

    @param      "aType"  , source for convertion
    @param      "aFilter", source for convertion
    @return     Own formated string of item properties.

    @onerror    -
*//*-*************************************************************************************************************/
::rtl::OUString FilterCFGAccess::encodeTypeData( const FileType& aType )
{
    ::rtl::OUStringBuffer sData( 1000 );
    ::rtl::OUString       sValue       ;

    // Preferred
    if( aType.bPreferred == sal_True )
    {
        sData.appendAscii( "1" );
    }
    else
    {
        sData.appendAscii( "0" );
    }
    sData.append( PROPERTY_SEPERATOR                    );

    // MediaType
    sValue = ::rtl::Uri::encode( aType.sMediaType, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                );
    sData.append( PROPERTY_SEPERATOR                    );

    // ClipboardFormat
    sValue = ::rtl::Uri::encode( aType.sClipboardFormat, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                );
    sData.append( PROPERTY_SEPERATOR                    );

    // URLPattern
    sData.append( encodeStringList( aType.lURLPattern ) );
    sData.append( PROPERTY_SEPERATOR                    );

    // Extensions
    sData.append( encodeStringList( aType.lExtensions ) );
    sData.append( PROPERTY_SEPERATOR                    );

    // DocumentIconID
    sData.append( (sal_Int32)aType.nDocumentIconID      );
    sData.append( PROPERTY_SEPERATOR                    );

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
void FilterCFGAccess::decodeTypeData( const ::rtl::OUString& sData, FileType& aType )
{
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;
    sal_Int32       nToken  = 0;

    do
    {
        sToken = sData.getToken( 0, PROPERTY_SEPERATOR, nIndex );
        switch( nToken )
        {
            // Preferred
            case 0:     {
                            aType.bPreferred = sal_False;
                            if( sToken.toInt32() == 1 )
                            {
                                aType.bPreferred = sal_True;
                            }
                        }
                        break;
            // MediaType
            case 1:     aType.sMediaType       = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // ClipboardFormat
            case 2:     aType.sClipboardFormat = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // URLPattern
            case 3:     aType.lURLPattern      = decodeStringList( sToken );
                        break;
            // Extensions
            case 4:     aType.lExtensions      = decodeStringList( sToken );
                        break;
            // DocumentIconID
            case 5:     aType.nDocumentIconID  = sToken.toInt32();
                        break;
        }
        ++nToken;
    }
    while( nIndex >= 0 );
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::encodeFilterData( const Filter& aFilter )
{
    ::rtl::OUStringBuffer sData ( 1000 );
    ::rtl::OUString       sValue        ;

    // Order
    sData.append( (sal_Int32)aFilter.nOrder                );
    sData.append( PROPERTY_SEPERATOR                       );

    // Type
    sValue = ::rtl::Uri::encode( aFilter.sType, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // DocumentService
    sValue = ::rtl::Uri::encode( aFilter.sDocumentService, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // FilterService
    sValue = ::rtl::Uri::encode( aFilter.sFilterService, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // Flags
    sData.append( (sal_Int32)aFilter.nFlags                );
    sData.append( PROPERTY_SEPERATOR                       );

    // UserData
    sData.append( encodeStringList( aFilter.lUserData )    );
    sData.append( PROPERTY_SEPERATOR                       );

    // FileFormatVersion
    sData.append( (sal_Int32)aFilter.nFileFormatVersion    );
    sData.append( PROPERTY_SEPERATOR                       );

    // TemplateName
    sValue = ::rtl::Uri::encode( aFilter.sTemplateName, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );

    // [opional!] UIComponent
    if(aFilter.sUIComponent.getLength()>0)
    {
        sData.append( PROPERTY_SEPERATOR                   );
        sValue = ::rtl::Uri::encode( aFilter.sUIComponent, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
        sData.append( sValue                               );
    }

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
void FilterCFGAccess::decodeFilterData( const ::rtl::OUString& sData, Filter& aFilter )
{
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;
    sal_Int32       nToken  = 0;

    do
    {
        sToken = sData.getToken( 0, PROPERTY_SEPERATOR, nIndex );
        switch( nToken )
        {
            // Order
            case 0:     aFilter.nOrder              = sToken.toInt32();
                        break;
            // Type
            case 1:     aFilter.sType               = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // DocumentService
            case 2:     aFilter.sDocumentService    = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // FilterService
            case 3:     aFilter.sFilterService      = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // Flags
            case 4:     aFilter.nFlags              = sToken.toInt32();
                        break;
            // UserData
            case 5:     aFilter.lUserData           = decodeStringList( sToken );
                        break;
            // FileFormatVersion
            case 6:     aFilter.nFileFormatVersion  = sToken.toInt32();
                        break;
            // TemplateName
            case 7:     aFilter.sTemplateName       = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // [optional!] UIComponent
            case 8:     aFilter.sUIComponent        = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
        }
        ++nToken;
    }
    while( nIndex >= 0 );

    // Attention: 7 tokens are neccessary - Nr. 8 is optional - otherwise the xml config file seams to be corrupt.
    // But we check for <8 instead of <7 here because "++nToken" was called before!
    LOG_ASSERT2(nToken<8, "FilterCFGAccess::decodeFilterData()", "Not enough filter properties detected. May the config file is corrupt!")
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::encodeStringList( const OUStringList& lList )
{
    ::rtl::OUStringBuffer sData( 1000 );
    ::rtl::OUString       sValue       ;

    OUStringList::const_iterator pItem = lList.begin();
    while( pItem != lList.end() )
    {
        sValue = ::rtl::Uri::encode( *pItem, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
        sData.append( sValue );
        ++pItem;
        if( pItem != lList.end() )
        {
            sData.append( LIST_SEPERATOR );
        }
    }

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
OUStringList FilterCFGAccess::decodeStringList( const ::rtl::OUString& sValue )
{
    OUStringList    lList      ;
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;

    do
    {
        sToken = sValue.getToken( 0, LIST_SEPERATOR, nIndex );
        lList.push_back( ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 ) );
    }
    while( nIndex >= 0 );

    return lList;
}

/*-************************************************************************************************************//**
    @short      set right key count of set nodes depending from config file version
    @descr      Our file "TypeDetection.xml" exist in different versions! So we can support reading/writing of older
                ones. It's neccessary for our own tool "xml2xcd". Some set nodes changed here key count often.
                So we must react in a right way. We set our internal count variables and use it at read/write access
                on our configuration services.

    @attention  If count is set wrong - nothing will work. Warn programmer or user!
                ... but some entries doesn't depend from this version handling ... They are fix ...

    @seealso    method impl_load...()
    @seealso    method impl_save...()

    @param      -
    @return     -

    @onerror    Warn programmer ... it must work!
*//*-*************************************************************************************************************/
void FilterCFGAccess::impl_initKeyCounts()
{
    // set fix values
    m_nKeyCountDetectors        = 1;
    m_nKeyCountLoaders          = 2;
    m_nKeyCountContentHandlers  = 1;

    // set variable values ...
    // ... for types
    if( m_nVersion <= 2 )
        m_nKeyCountTypes = 7 ;
    else
        m_nKeyCountTypes = 2 ;

    // ... for filters
    if( m_nVersion == 1 )
        m_nKeyCountFilters = 9 ;
    else
    if( m_nVersion == 2 )
        m_nKeyCountFilters = 10;
    else
        m_nKeyCountFilters = 3 ;
}

/*-************************************************************************************************************//**
    @short      methods to load all configuration values
    @descr      Call this methods to read configuration values from configuration services into given data container.

    @seealso    method impl_save...()

    @param      "rData", reference to data container for filling
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void FilterCFGAccess::impl_loadTypes( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure version 1/2:       /Types
    //                                  /<entry_01>
    //                                      /Preferred              [boolean    ]
    //                                      /UIName                 [string     ] localized!
    //                                      /MediaType              [string     ]
    //                                      /ClipboardFormat        [string     ]
    //                                      /URLPattern             [stringlist ]
    //                                      /Extensions             [stringlist ]
    //                                      /DocumentIconID         [integer    ]
    //                                  /<entry_02>
    //                                      /...
    // structure version 3:         /Types
    //                                  /<entry_01>
    //                                      /UIName                 [string     ] localized!
    //                                      /Data                   [string     ] own format ... {Preferred,MediaType,ClipboardFormat,URLPattern,Extensions,DocumentIconID}
	//
	// a)	Get names of all current existing type nodes in configuration.
	// b)	Reserve memory for list of all type pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
    //-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadTypes" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_TYPES, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_TYPES, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                            nNodeCount      = lNodeNames.getLength()       ;
    css::uno::Sequence< ::rtl::OUString > lPropertyNames  ( nNodeCount*m_nKeyCountTypes );
    sal_uInt32                            nNodeName       = 0                            ;
    sal_uInt32                            nProperty       = 0                            ;
    ::rtl::OUString                       sPath                                          ;

    // c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_TYPES          ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        if( m_nVersion<=2  )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_PREFERRED        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME           ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_MEDIATYPE        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_CLIPBOARDFORMAT  ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_URLPATTERN       ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_EXTENSIONS       ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DOCUMENTICONID   ;
            ++nProperty;
        }
        else if( m_nVersion>=3  )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME           ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DATA             ;
            ++nProperty;
        }
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "FilterCFGAccess::impl_loadTypes()", "Miss some configuration values of type set!" )

	// e)
    nProperty = 0;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        FileType                              aType      ;
        css::uno::Sequence< ::rtl::OUString > lTempList  ;
        ::rtl::OUString                       sTempString;

        if( m_nVersion<=2 )
        {
            lPropertyValues[nProperty]  >>= aType.bPreferred;
            ++nProperty;
            DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aType.lUINames );
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.sMediaType;
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.sClipboardFormat;
            ++nProperty;
            lPropertyValues[nProperty]  >>= lTempList;
            aType.lURLPattern = Converter::convert_seqOUString2OUStringList(lTempList);
            ++nProperty;
            lPropertyValues[nProperty]  >>= lTempList;
            aType.lExtensions = Converter::convert_seqOUString2OUStringList(lTempList);
            DataContainer::correctExtensions( aType.lExtensions );
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.nDocumentIconID;
            ++nProperty;
        }
        else if( m_nVersion>=3 )
        {
            DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aType.lUINames );
            ++nProperty;
            lPropertyValues[nProperty]  >>= sTempString;
            FilterCFGAccess::decodeTypeData( sTempString, aType );
            ++nProperty;
        }

        if( m_nVersion<=5 )
            aType.sName = lNodeNames[nNodeName];
        else
            aType.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        FilterCFGAccess::setProductName(aType.lUINames);
        rData.addType( aType, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadFilters( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure version 1/2:       /Filters
    //                                  /<filtername_01>
    //                                      /Installed          [boolean    ]
    //                                      /Order              [integer    ] => only for version=2!
    //                                      /Type               [string     ]
    //                                      /UIName             [string     ]
    //                                      /DocumentService    [string     ]
    //                                      /FilterService      [string     ]
    //                                      /Flags              [integer    ]
    //                                      /UserData           [stringlist ]
    //                                      /FileFormatVersion  [integer    ]
    //                                      /TemplateName       [string     ]
    //                                  /<filtername_02>
    //                                      /...
    // structure version 3:         /Filters
    //                                  /<filtername_01> = old format encoded!
    //                                      /Installed          [boolean    ]
    //                                      /UIName             [string     ]
    //                                      /Data               [string     ] own format {Order,Type,DocumentService,FilterService,Flags,UserData, FileFormatVersion,TemplateName}
    //                                  /<filtername_02>
    //                                      /...
	//
	// a)	Get names of all current existing filter nodes in configuration.
	// b)	Reserve memory for list of all filter pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadFilters" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_FILTERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_FILTERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()         ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountFilters );
    sal_uInt32                              nNodeName       = 0                              ;
    sal_uInt32                              nProperty       = 0                              ;
    ::rtl::OUString                         sPath                                            ;

    // c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_FILTERS        ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        if( m_nVersion<=2 )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_INSTALLED            ;
            ++nProperty;
            if( m_nVersion==2 )
            {
                lPropertyNames[nProperty] = sPath + SUBKEY_ORDER            ;
                ++nProperty;
            }
            lPropertyNames[nProperty] = sPath + SUBKEY_TYPE                 ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME               ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DOCUMENTSERVICE      ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FILTERSERVICE        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FLAGS                ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_USERDATA             ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FILEFORMATVERSION    ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_TEMPLATENAME         ;
            ++nProperty;
        }
        else if( m_nVersion>=3 )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_INSTALLED            ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME               ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DATA                 ;
            ++nProperty;
        }
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "FilterCFGAccess::impl_loadFilters()", "Miss some configuration values of filter set!" )

	// e)
    nProperty = 0;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        Filter                                 aFilter    ;
        css::uno::Sequence< ::rtl::OUString >  lTempList  ;
        sal_Bool                               bInstalled ;
        ::rtl::OUString                        sTempString;

		sal_Bool bCheck = ( lPropertyValues[nProperty] >>= bInstalled);
        LOG_ASSERT2( bCheck==sal_False, "FilterCFGAccess::impl_loadFilters()", "Invalid property type for installed flag detected!" )
		++nProperty;

		// Ignore not installed filter!
        // Attention: Wrong packed "ANY"s are ignored too!
		if(
            ( bInstalled == sal_False ) ||
            ( bCheck     == sal_False )
          )
		{
			// If this filter was ignored .. we must step to next one in our lPropertyNames and lPropertyValues lists!
            // Because ... every name in list 1 match with a value in list 2.
            nProperty += m_nKeyCountFilters;
            // -1 ... because "Installed" already readed!
            --nProperty;
		}
		else
		{
            // Attention: If you change anything in this section ...
			// change "nProperty += ..." too! (see before!)
            // To find out, if somewhere forget this ... check right count in debug version!
            #ifdef ENABLE_ASSERTIONS
            sal_uInt32 nCheckPropCount = nProperty;
            #endif

            if( m_nVersion<=2 )
            {
                if( m_nVersion==2 )
                {
                    lPropertyValues[nProperty]  >>= aFilter.nOrder          ;
                    ++nProperty;
                }
                lPropertyValues[nProperty]  >>= aFilter.sType               ;
                ++nProperty;
                DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aFilter.lUINames );
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sDocumentService    ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sFilterService      ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.nFlags              ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= lTempList                   ;
                aFilter.lUserData = Converter::convert_seqOUString2OUStringList(lTempList);
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.nFileFormatVersion  ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sTemplateName       ;
                ++nProperty;
            }
            else if( m_nVersion>=3 )
            {
                DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aFilter.lUINames );
                ++nProperty;
                lPropertyValues[nProperty]  >>= sTempString;
                FilterCFGAccess::decodeFilterData( sTempString, aFilter );
                ++nProperty;
            }

            LOG_ASSERT2( (sal_uInt32)(nProperty-nCheckPropCount)!=(sal_uInt32)(m_nKeyCountFilters-1), "FilterCFGAccess::impl_loadFilters()", "Mismatch between key count of filters and count of ignored filter properties detected. Follow readed filters would be wrong!" )

            if( m_nVersion<=5 )
                aFilter.sName = lNodeNames[nNodeName];
            else
                aFilter.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
            FilterCFGAccess::setProductName(aFilter.lUINames);
            rData.addFilter( aFilter, sal_False );
		}
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadDetectors( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/DetectServices
	//						/<entry_01>
	//							/Types				[stringlist]
	//						/<entry_02>
	//							/...
	//
	// a)	Get names of all current existing detector nodes in configuration.
	// b)	Reserve memory for list of all detector pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadDetectors" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_DETECTSERVICES, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_DETECTSERVICES, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()           ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountDetectors );
    sal_uInt32                              nNodeName       = 0                                ;
    sal_uInt32                              nProperty       = 0                                ;
    ::rtl::OUString                         sPath                                              ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_DETECTSERVICES ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "DetectorCache::impl_loadDetectors()", "Miss some configuration values of detector set!" )

	// e)
    nProperty = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        Detector                              aDetector;
        css::uno::Sequence< ::rtl::OUString > lTempList;

        lPropertyValues[nProperty]	>>=	lTempList;
        aDetector.lTypes = Converter::convert_seqOUString2OUStringList(lTempList);
		++nProperty;
        if( m_nVersion<=5 )
            aDetector.sName = lNodeNames[nNodeName];
        else
            aDetector.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addDetector( aDetector, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadLoaders( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /FrameLoaders
	//						/<entry_01>
	//							/UIName		[string		]
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Get names of all current existing loader nodes in configuration.
	// b)	Reserve memory for list of all loader pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadLoaders" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_FRAMELOADERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_FRAMELOADERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()         ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountLoaders );
    sal_uInt32                              nNodeName       = 0                              ;
    sal_uInt32                              nProperty       = 0                              ;
    ::rtl::OUString                         sPath                                            ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_FRAMELOADERS   ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_UINAME;
        ++nProperty;
        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES ;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "LoaderCache::impl_loadLoaders()", "Miss some configuration values of loader set!" )

	// e)
    nProperty  = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        Loader                                aLoader  ;
        css::uno::Sequence< ::rtl::OUString > lTempList;
        ::rtl::OUString                       sUIName  ;

        DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aLoader.lUINames );
		++nProperty;
		lPropertyValues[nProperty]	>>=	lTempList;
        aLoader.lTypes = Converter::convert_seqOUString2OUStringList(lTempList);
		++nProperty;
        if( m_nVersion<=5 )
            aLoader.sName = lNodeNames[nNodeName];
        else
            aLoader.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
         FilterCFGAccess::setProductName(aLoader.lUINames);
        rData.addLoader( aLoader, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadContentHandlers( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /ContentHandlers
	//						/<entry_01>
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
    // a)   Get names of all current existing handler nodes in configuration.
    // b)   Reserve memory for list of all handler pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadContentHandlers" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_CONTENTHANDLERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_CONTENTHANDLERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()                 ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountContentHandlers );
    sal_uInt32                              nNodeName       = 0                                      ;
    sal_uInt32                              nProperty       = 0                                      ;
    ::rtl::OUString                         sPath                                                    ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_CONTENTHANDLERS;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "HandlerCache::impl_loadContentHandlers()", "Miss some configuration values of content handler set!" )

	// e)
    nProperty = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        ContentHandler                        aHandler ;
        css::uno::Sequence< ::rtl::OUString > lTempList;

		lPropertyValues[nProperty]	>>=	lTempList;
        aHandler.lTypes = Converter::convert_seqOUString2OUStringList(lTempList);
		++nProperty;
        if( m_nVersion<=5 )
            aHandler.sName = lNodeNames[nNodeName];
        else
            aHandler.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addContentHandler( aHandler, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadDefaults( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle follow structure!
	// structure:		/Defaults
	//						/DetectService
	//							/<name>
	//						/FrameLoader
	//							/<name>
	//
	// a)	Build list with pathes to configuration nodes.
	// b)	Get values for this list.
	// c)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadDefaults" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lPropertyNames( 1 );
    ::rtl::OUString                       sPath              ;

    sPath  = SUBLIST_DEFAULTS  ;
    sPath += CFG_PATH_SEPERATOR;

    lPropertyNames[0] = sPath + SUBKEY_GENERICLOADER  ;

	// b)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "LoaderCache::impl_loadDefaults()", "Miss some configuration values of default set!" )

	// c)
    lPropertyValues[0]  >>= rData.m_aGenericLoader.sName  ;

	// Follow properties are fix!
    DataContainer::setLocalelizedString( rData.m_aGenericLoader.lUINames, LOCALE_FALLBACK, UINAME_GENERICLOADER );
    rData.m_aGenericLoader.lTypes.push_back  ( TYPELIST_GENERICLOADER   );
}

void FilterCFGAccess::impl_removeNodes(        OUStringList&      rChangesList  ,
                                         const ::rtl::OUString&   sTemplateType ,
                                         const ::rtl::OUString&   sSetName      )
{
    css::uno::Sequence< ::rtl::OUString > lRemovedElements = Converter::convert_OUStringList2seqOUString(rChangesList);
    ClearNodeElements( sSetName, lRemovedElements );
    rChangesList.free();
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_saveTypes( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure: ... see impl_loadTypes() for further informations ...
	//
    // a)   Step over all cache entries which are set to MODIFIED!
	// b)	Build list of all properties of a type node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_saveTypes" );

    // Collect names of all removed elements and use list
    // to remove it realy.
    if( rData.m_aTypeCache.lRemovedItems.size() > 0 )
    {
        impl_removeNodes( rData.m_aTypeCache.lRemovedItems, TEMPLATENAME_TYPE, SUBLIST_TYPES );
    }

    // Collect names of all added AND changed elements (there is no difference in handling!)
    // ... build a list of properties for every set node
    // ... add values to this list
    // ... and write changes to configuration.
    // Use one list - don't handle every set node seperat!
    if  (
            ( rData.m_aTypeCache.lAddedItems.size  () > 0 ) ||
            ( rData.m_aTypeCache.lChangedItems.size() > 0 )
        )
    {
        ::rtl::OUString sBasePath       ;
        ::rtl::OUString sFullPath       ;
        sal_Int32       nStep       = 0 ;

        sBasePath  = SUBLIST_TYPES       ;   //  "Types"
        sBasePath += CFG_PATH_SEPERATOR  ;   //  "Types/"

        // Reserve enough memory for complete list.
        // = count of added AND changed elements * count of properties for every set node
        sal_Int32 nSize;
        nSize  = rData.m_aTypeCache.lAddedItems.size()  ;
        nSize += rData.m_aTypeCache.lChangedItems.size();
        nSize *= m_nKeyCountTypes                                      ;
        css::uno::Sequence< css::beans::PropertyValue > lChangedElements( nSize );

        // Add "added" types to list.
        for( OUStringList::const_iterator pAddedType=rData.m_aTypeCache.lAddedItems.begin(); pAddedType!=rData.m_aTypeCache.lAddedItems.end(); ++pAddedType )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pAddedType; // without encoding
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pAddedType, TEMPLATENAME_TYPE ); // encoded
            sFullPath += CFG_PATH_SEPERATOR;

            if( m_nVersion<=2 )
            {
                // "Types/<typename>/Preferred"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_PREFERRED                                                               ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].bPreferred                                                 ;
                ++nStep;
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pAddedType].lUINames ) ;
                ++nStep;
                // "Types/<typename>/MediaType"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_MEDIATYPE                                                               ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].sMediaType                                                 ;
                ++nStep;
                // "Types/<typename>/ClipboardFormat"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_CLIPBOARDFORMAT                                                         ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].sClipboardFormat                                           ;
                ++nStep;
                // "Types/<typename>/URLPattern"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_URLPATTERN                                                              ;
                lChangedElements[nStep].Value    <<= Converter::convert_OUStringList2seqOUString(rData.m_aTypeCache[*pAddedType].lURLPattern)   ;
                ++nStep;
                // "Types/<typename>/Extensions"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_EXTENSIONS                                                              ;
                lChangedElements[nStep].Value    <<= Converter::convert_OUStringList2seqOUString(rData.m_aTypeCache[*pAddedType].lExtensions)   ;
                ++nStep;
                // "Types/<typename>/DocumentIconID"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DOCUMENTICONID                                                          ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].nDocumentIconID                                            ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pAddedType].lUINames ) ;
                ++nStep;
                // "Types/<typename>/Data"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DATA                                                                    ;
                lChangedElements[nStep].Value    <<= FilterCFGAccess::encodeTypeData( rData.m_aTypeCache[*pAddedType] )                         ;
                ++nStep;
            }
        }

        // Attention: Don't change "nStep" here!

        // Add "changed" types to list.
        for( OUStringList::const_iterator pChangedType=rData.m_aTypeCache.lChangedItems.begin(); pChangedType!=rData.m_aTypeCache.lChangedItems.end(); ++pChangedType )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pChangedType;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pChangedType, TEMPLATENAME_TYPE );
            sFullPath += CFG_PATH_SEPERATOR;

            if( m_nVersion<=2 )
            {
                // "Types/<typename>/Preferred"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_PREFERRED                                                                   ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].bPreferred                                                   ;
                ++nStep;
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pChangedType].lUINames )   ;
                ++nStep;
                // "Types/<typename>/MediaType"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_MEDIATYPE                                                                   ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].sMediaType                                                   ;
                ++nStep;
                // "Types/<typename>/ClipboardFormat"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_CLIPBOARDFORMAT                                                             ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].sClipboardFormat                                             ;
                ++nStep;
                // "Types/<typename>/URLPattern"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_URLPATTERN                                                                  ;
                lChangedElements[nStep].Value    <<= Converter::convert_OUStringList2seqOUString(rData.m_aTypeCache[*pChangedType].lURLPattern)     ;
                ++nStep;
                // "Types/<typename>/Extensions"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_EXTENSIONS                                                                  ;
                lChangedElements[nStep].Value    <<= Converter::convert_OUStringList2seqOUString(rData.m_aTypeCache[*pChangedType].lExtensions)     ;
                ++nStep;
                // "Types/<typename>/DocumentIconID"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DOCUMENTICONID                                                              ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].nDocumentIconID                                              ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pChangedType].lUINames )   ;
                ++nStep;
                // "Types/<typename>/Data"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DATA                                                                        ;
                lChangedElements[nStep].Value    <<= FilterCFGAccess::encodeTypeData( rData.m_aTypeCache[*pChangedType] )                           ;
                ++nStep;
            }
        }

        // Commit complete list of changes.
        SetSetProperties( SUBLIST_TYPES, lChangedElements );

        // Don't forget to free "changes lists". Otherwise these changes will bewritten every time ...
        rData.m_aTypeCache.lAddedItems.free();
        rData.m_aTypeCache.lChangedItems.free();
	}
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveFilters( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure: ... see impl_loadFilters() for further informations ...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a filter node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_saveFilters" );

    // Collect names of all removed elements and use list
    // to remove it realy.
    if( rData.m_aFilterCache.lRemovedItems.size() > 0 )
    {
        impl_removeNodes( rData.m_aFilterCache.lRemovedItems, TEMPLATENAME_FILTER, SUBLIST_FILTERS );
    }

    // Collect names of all added AND changed elements (there is no difference in handling!)
    // ... build a list of properties for every set node
    // ... add values to this list
    // ... and write changes to configuration.
    // Use one list - don't handle every set node seperat!
    if  (
            ( rData.m_aFilterCache.lAddedItems.size  () > 0 ) ||
            ( rData.m_aFilterCache.lChangedItems.size() > 0 )
        )
    {
        ::rtl::OUString sBasePath       ;
        ::rtl::OUString sFullPath       ;
        sal_Int32       nStep       = 0 ;

        sBasePath  = SUBLIST_FILTERS    ;   //  "Filters"
        sBasePath += CFG_PATH_SEPERATOR ;   //  "Filters/"

        // Reserve enough memory for complete list.
        // = count of added AND changed elements * count of properties for every set node
        sal_Int32 nSize;
        nSize  = rData.m_aFilterCache.lAddedItems.size()  ;
        nSize += rData.m_aFilterCache.lChangedItems.size();
        nSize *= m_nKeyCountFilters                       ;
        css::uno::Sequence< css::beans::PropertyValue > lChangedElements( nSize );

        // Add "added" filters to list.
        for( OUStringList::const_iterator pAddedFilter=rData.m_aFilterCache.lAddedItems.begin(); pAddedFilter!=rData.m_aFilterCache.lAddedItems.end(); ++pAddedFilter )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pAddedFilter;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pAddedFilter, TEMPLATENAME_FILTER );
            sFullPath += CFG_PATH_SEPERATOR;

            OUStringHash lUINames = rData.m_aFilterCache[*pAddedFilter].lUINames;
            resetProductName( lUINames );

            if( m_nVersion<=2 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                               ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                   ;
                ++nStep;
                // "Filters/<filtername>/Order"
                if( m_nVersion==2 )
                {
                    lChangedElements[nStep].Name      = sFullPath + SUBKEY_ORDER                                                               ;
                    lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nOrder                                             ;
                    ++nStep;
                }
                // "Filters/<filtername>/Type"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TYPE                                                                    ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sType                                                  ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, lUINames );
                ++nStep;
                // "Filters/<filtername>/DocumentService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DOCUMENTSERVICE                                                         ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sDocumentService                                       ;
                ++nStep;
                // "Filters/<filtername>/FilterService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILTERSERVICE                                                           ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sFilterService                                         ;
                ++nStep;
                // "Filters/<filtername>/Flags"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FLAGS                                                                   ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nFlags                                                 ;
                ++nStep;
                // "Filters/<filtername>/UserData"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_USERDATA                                                                ;
                lChangedElements[nStep].Value   <<= Converter::convert_OUStringList2seqOUString(rData.m_aFilterCache[*pAddedFilter].lUserData) ;
                ++nStep;
                // "Filters/<filtername>/FileFormatVersion"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILEFORMATVERSION                                                       ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nFileFormatVersion                                     ;
                ++nStep;
                // "Filters/<filtername>/TemplateName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TEMPLATENAME                                                            ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sTemplateName                                          ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                               ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                   ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, lUINames );
                ++nStep;
                // "Filters/<filtername>/Data"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DATA                                                                    ;
                lChangedElements[nStep].Value   <<= FilterCFGAccess::encodeFilterData( rData.m_aFilterCache[*pAddedFilter] )                   ;
                ++nStep;
            }
        }

        // Attention: Don't change "nStep" here!

        // Add "changed" filters to list.
        for( OUStringList::const_iterator pChangedFilter=rData.m_aFilterCache.lChangedItems.begin(); pChangedFilter!=rData.m_aFilterCache.lChangedItems.end(); ++pChangedFilter )
        {
            sFullPath  = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pChangedFilter;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pChangedFilter, TEMPLATENAME_FILTER );
            sFullPath += CFG_PATH_SEPERATOR;

            OUStringHash lUINames = rData.m_aFilterCache[*pChangedFilter].lUINames;
            resetProductName( lUINames );

            if( m_nVersion<=2 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                                   ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                       ;
                ++nStep;
                // "Filters/<filtername>/Order"
                if( m_nVersion==2 )
                {
                    lChangedElements[nStep].Name      = sFullPath + SUBKEY_ORDER                                                                   ;
                    lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nOrder                                               ;
                    ++nStep;
                }
                // "Filters/<filtername>/Type"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TYPE                                                                        ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sType                                                    ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, lUINames )  ;
                ++nStep;
                // "Filters/<filtername>/DocumentService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DOCUMENTSERVICE                                                             ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sDocumentService                                         ;
                ++nStep;
                // "Filters/<filtername>/FilterService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILTERSERVICE                                                               ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sFilterService                                           ;
                ++nStep;
                // "Filters/<filtername>/Flags"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FLAGS                                                                       ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nFlags                                                   ;
                ++nStep;
                // "Filters/<filtername>/UserData"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_USERDATA                                                                    ;
                lChangedElements[nStep].Value   <<= Converter::convert_OUStringList2seqOUString(rData.m_aFilterCache[*pChangedFilter].lUserData)   ;
                ++nStep;
                // "Filters/<filtername>/FileFormatVersion"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILEFORMATVERSION                                                           ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nFileFormatVersion                                       ;
                ++nStep;
                // "Filters/<filtername>/TemplateName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TEMPLATENAME                                                                ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sTemplateName                                            ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                                   ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                       ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( ConfigItem::GetMode(), rData.m_sLocale, lChangedElements[nStep].Value, lUINames )  ;
                ++nStep;
                // "Filters/<filtername>/Data"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DATA                                                                        ;
                lChangedElements[nStep].Value   <<= FilterCFGAccess::encodeFilterData( rData.m_aFilterCache[*pChangedFilter] )                     ;
                ++nStep;
            }
        }

        // Commit complete list of changes.
        SetSetProperties( SUBLIST_FILTERS, lChangedElements );

        // Don't forget to free "changes lists". Otherwise these changes will bewritten every time ...
        rData.m_aFilterCache.lAddedItems.free();
        rData.m_aFilterCache.lChangedItems.free();
	}
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveDetectors( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/DetectServices
	//						/<entry_01>
	//							/Types				[stringlist]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a detector node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_saveDetectors" );

    // Collect names of all removed elements and use list
    // to remove it realy.
    if (rData.m_aDetectorCache.lRemovedItems.size() > 0)
        impl_removeNodes(rData.m_aDetectorCache.lRemovedItems, TEMPLATENAME_DETECTSERVICE, SUBLIST_DETECTSERVICES );

    // Collect names of all added AND changed elements (there is no difference in handling!)
    // ... build a list of properties for every set node
    // ... add values to this list
    // ... and write changes to configuration.
    // Use one list - don't handle every set node seperat!
    if (
        (rData.m_aDetectorCache.lAddedItems.size  () > 0) ||
        (rData.m_aDetectorCache.lChangedItems.size() > 0)
       )
    {
        ::rtl::OUString sBasePath       ;
        ::rtl::OUString sFullPath       ;
        sal_Int32       nStep       = 0 ;

        sBasePath  = SUBLIST_DETECTSERVICES ;   //  "Detectors"
        sBasePath += CFG_PATH_SEPERATOR     ;   //  "Detectors/"

        // Reserve enough memory for complete list.
        // = count of added AND changed elements * count of properties for every set node
        sal_Int32 nSize;
        nSize  = rData.m_aDetectorCache.lAddedItems.size()  ;
        nSize += rData.m_aDetectorCache.lChangedItems.size();
        nSize *= m_nKeyCountDetectors                       ;
        css::uno::Sequence< css::beans::PropertyValue > lChangedElements(nSize);

        // Add "added" detectors to list.
        for (OUStringList::const_iterator pAddedDetector  = rData.m_aDetectorCache.lAddedItems.begin();
                                          pAddedDetector != rData.m_aDetectorCache.lAddedItems.end()  ;
                                        ++pAddedDetector                                              )
        {
            sFullPath = sBasePath;
            if (m_nVersion <= 5)
                sFullPath += *pAddedDetector;
            else
                sFullPath += ::utl::wrapConfigurationElementName(*pAddedDetector, TEMPLATENAME_DETECTSERVICE);
            sFullPath += CFG_PATH_SEPERATOR;

            // "Detectors/<detector name>/Types"
            lChangedElements[nStep].Name    = sFullPath + SUBKEY_TYPES;
            lChangedElements[nStep].Value <<= Converter::convert_OUStringList2seqOUString(rData.m_aDetectorCache[*pAddedDetector].lTypes);
            ++nStep;
        }

        // Attention: Don't change "nStep" here!

        // Add "changed" detectors to list.
        for (OUStringList::const_iterator pChangedDetector  = rData.m_aDetectorCache.lChangedItems.begin();
                                          pChangedDetector != rData.m_aDetectorCache.lChangedItems.end()  ;
                                        ++pChangedDetector                                                )
        {
            sFullPath = sBasePath;
            if (m_nVersion <= 5)
                sFullPath += *pChangedDetector;
            else
                sFullPath += ::utl::wrapConfigurationElementName(*pChangedDetector, TEMPLATENAME_DETECTSERVICE);
            sFullPath += CFG_PATH_SEPERATOR;

            // "Detectors/<detectorname>/Types"
            lChangedElements[nStep].Name    = sFullPath + SUBKEY_TYPES;
            lChangedElements[nStep].Value <<= Converter::convert_OUStringList2seqOUString(rData.m_aDetectorCache[*pChangedDetector].lTypes);
            ++nStep;
        }

        // Commit complete list of changes.
        SetSetProperties(SUBLIST_DETECTSERVICES, lChangedElements);

        // Don't forget to free "changes lists". Otherwise these changes will be written again and again and ...
        rData.m_aDetectorCache.lAddedItems.free();
        rData.m_aDetectorCache.lChangedItems.free();
    }
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveLoaders( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/FrameLoader
	//						/<entry_01>
	//							/UIName		[string		]
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a loader node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    LOG_WARNING( "FilterCFGAccess::impl_saveLoaders()", "Not implemented yet!" )
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveContentHandlers( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /ContentHandler
	//						/<entry_01>
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
    // b)   Build list of all properties of a handler node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    LOG_WARNING( "FilterCFGAccess::impl_saveContentHandlers()", "Not implemented yet!" )
}

DataContainer::DataContainer()
    : ThreadHelpBase      (         )
    , m_bTypesModified    (sal_False)
    , m_bFiltersModified  (sal_False)
    , m_bDetectorsModified(sal_False)
    , m_bLoadersModified  (sal_False)
    , m_bHandlersModified (sal_False)
{
}

sal_Bool DataContainer::isModified()
{
    // SAFE {
    ReadGuard aReadLock(m_aLock);
    return (m_bTypesModified     ||
            m_bFiltersModified   ||
            m_bDetectorsModified ||
            m_bLoadersModified   ||
            m_bHandlersModified  );
}

void DataContainer::startListener()
{
/*    // SAFE {
    WriteGuard aWriteLock(m_aLock);
    if (m_pDetectorListener == NULL)
    {
        m_pDetectorListener = new FilterCFGListener(::utl::getProcessServiceFactory(), FilterCFGListener.E_DETECTOR, this);
        m_pDetectorListener->startListening();
    }
    aWriteLock.unlock();
    // } SAFE*/
}

void DataContainer::stopListener()
{
/*    // SAFE {
    WriteGuard aWriteLock(m_aLock);
    if (m_pDetectorListener != NULL)
    {
        m_pDetectorListener->stopListening();
        delete m_pDetectorListener;
        m_pDetectorListener = NULL;
    }
    aWriteLock.unlock();
    // } SAFE*/
}

LockHelper& DataContainer::getSyncronizer()
{
    return m_aLock;
}

//_______________________________________________

/** check minimum required cache states.
    If these requirements are not OK ... the filter configuration won't
    be ready for normal office working!

    Note: We don't need a list of detectors everytime!
          Flat detection based on cached config values
          must be enough in that case then.
 */
sal_Bool DataContainer::isValidOrRepairable() const
{
    LOG_COND_FILTERDBG( m_aTypeCache.size()               < 1, "DataContainer::isValidOrRepairable()", "No types found!"                            )
    LOG_COND_FILTERDBG( m_aPreferredTypesCache.size()     < 1, "DataContainer::isValidOrRepairable()", "No preferred types found!"                  )
    LOG_COND_FILTERDBG( m_aFilterCache.size()             < 1, "DataContainer::isValidOrRepairable()", "No filters found!"                          )
    LOG_COND_FILTERDBG( m_aFastFilterCache.size()         < 1, "DataContainer::isValidOrRepairable()", "Optimization for filter search won't work!" )
    LOG_COND_FILTERDBG( m_aGenericLoader.sName.getLength()< 1, "DataContainer::isValidOrRepairable()", "No generic loader service found!"           )
    LOG_COND_FILTERDBG( m_sLocale.getLength()             < 1, "DataContainer::isValidOrRepairable()", "Localization based funcionality won't work!")

    return(
            (m_aTypeCache.size()               >0) &&
            (m_aPreferredTypesCache.size()     >0) &&
            (m_aFilterCache.size()             >0) &&
            (m_aFastFilterCache.size()         >0) &&
            (m_aGenericLoader.sName.getLength()>0) &&
            (m_sLocale.getLength()             >0)
          );
}

//_______________________________________________

/** @short  check list of types and try to repeair it, if some items seams to be invalid.

    @descr  In case repairing of such invalid entries won't be possible ... it returns false.
            But then this cache won't be useable further more!

    @return True, if current type list don't include invalid items or repairing of such items
            was successfully.

    @attention  Don't remove as invalid detected items directly inside the loop!
                Very often such items are part of different lists/maps of this data container,
                And removing of it must be done in all these structures.
                That's why we call specialized methods of class DataContainer.
                And in this case it's not enough to remove the current iterator entry ...

                On the other side we can use the same loop to correct invalid but repairable
                entries. In case we would use e.g. ::std::remove_if(...) we must steop over all
                container items twice - first time to remove invalid entries - second one to repair
                some other ones.
 */

sal_Bool DataContainer::validateAndRepairTypes()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepairTypes" );

    // ----------------------------------------------------------------------------------
    // check for types, which are not currently used or invalid
    // remove it!
    OUStringList lRemoveTypes;
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "{ check types" );
    for (FileTypeHash::iterator pTypeIt=m_aTypeCache.begin(); pTypeIt!=m_aTypeCache.end(); ++pTypeIt)
    {
        FileType* pType = &(pTypeIt->second);
        // check if this type is used by any other TypeDetection component (e.g. filters, frame loader ...)
        // Don't look for any default loader or detect service. They can't help us in every case!
        sal_Bool bReferencedByFilter   = (m_aFastFilterCache.find(pType->sName)        !=m_aFastFilterCache.end()        );
        sal_Bool bReferencedByDetector = (m_aFastDetectorCache.find(pType->sName)      !=m_aFastDetectorCache.end()      );
        sal_Bool bReferencedByLoader   = (m_aFastLoaderCache.find(pType->sName)        !=m_aFastLoaderCache.end()        );
        sal_Bool bReferencedByHandler  = (m_aFastContentHandlerCache.find(pType->sName)!=m_aFastContentHandlerCache.end());

        if (!bReferencedByFilter && !bReferencedByDetector && !bReferencedByLoader && !bReferencedByHandler)
        {
            lRemoveTypes.push_back(pType->sName);
            continue;
        }

        // check for missing extensions/URLPattern in combination with a missing detect service
        // Without an extension a detection will fail with a guarantee of 99% if no deep detection service
        // is available. Because nobody give us a mime type normaly, which can be used for a flat detection too.
        if (
            (!bReferencedByDetector    ) &&
            (pType->lExtensions.size()<1) &&
            (pType->lURLPattern.size()<1)
           )
        {
            lRemoveTypes.push_back(pType->sName);
            continue;
        }
    }

    // remove invalid types realy ...
    for (OUStringList::const_iterator pTypeNameIt=lRemoveTypes.begin(); pTypeNameIt!=lRemoveTypes.end(); ++pTypeNameIt)
    {
        LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pTypeNameIt), "[type] is not valid and will be removed")
/*FIXME
    Don't remove it realy here! Not yet!
    Because filters are installable by our setup .. but not there typs; may we
    remove types here for a standard office installation, which can't be re-added
    if user makes a full setup later. Start thinking about a better design now ...

        removeType(*pTypeNameIt, sal_True);
        if (m_aTypeCache.find(*pTypeNameIt) != m_aTypeCache.end())
        {
            LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pTypeNameIt), "[type] could not be removed successfully")
            return sal_False;
        }
*/
    }
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "} check types" );

    return sal_True;
}

//_______________________________________________

/** @short  check list of filters and try to repeair it, if some items seams to be invalid.

    @descr  In case repairing of such invalid entries won't be possible ... it returns false.
            But then this cache won't be useable further more!

    @return True, if current filter list don't include invalid items or repairing of such items
            was successfully.

    @attention  Don't remove as invalid detected items directly inside the loop!
                Very often such items are part of different lists/maps of this data container,
                And removing of it must be done in all these structures.
                That's why we call specialized methods of class DataContainer.
                And in this case it's not enough to remove the current iterator entry ...

                On the other side we can use the same loop to correct invalid but repairable
                entries. In case we would use e.g. ::std::remove_if(...) we must steop over all
                container items twice - first time to remove invalid entries - second one to repair
                some other ones.
 */

sal_Bool DataContainer::validateAndRepairFilter()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepairFilter" );

    #ifdef ENABLE_FILTERDBG
    sal_Bool bTextDefault    = sal_False;
    sal_Bool bGlobalDefault  = sal_False;
    sal_Bool bWebDefault     = sal_False;
    sal_Bool bDrawDefault    = sal_False;
    sal_Bool bImpressDefault = sal_False;
    sal_Bool bMathDefault    = sal_False;
    sal_Bool bChartDefault   = sal_False;
    sal_Bool bCalcDefault    = sal_False;
    #endif

    // ----------------------------------------------------------------------------------
    // check for filters, which are not currently used or invalid
    // remove it!
    OUStringList lRemoveFilters;
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "{ check filters" );
    for (FilterHash::iterator pFilterIt=m_aFilterCache.begin(); pFilterIt!=m_aFilterCache.end(); ++pFilterIt)
    {
        Filter* pFilter = &(pFilterIt->second);
        // check if this filter is registered for an existing type
        if (m_aTypeCache.find(pFilter->sType) == m_aTypeCache.end())
        {
            lRemoveFilters.push_back(pFilter->sName);
            continue;
        }

        // check for completly wrong flags
        // Such filters will not be useable in general.
        if (pFilter->nFlags<1)
        {
            lRemoveFilters.push_back(pFilter->sName);
            continue;
        }

        // repair a wrong order number
        // It can't faile ... don't remove this filter or break this loop.
        if (pFilter->nOrder<0)
        {
            LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(pFilter->sName), "[filter] has wrong order number ... set it to 0")
            pFilter->nOrder = 0;
        }

        // Log information about unknown filter flags ...
        // May user set a wrong combination. It will not produce problems.
        // But it can be the result of a damaged configuration file!

        #ifdef ENABLE_FILTERDBG
        sal_Bool              bLog   = sal_False;
        ::rtl::OUStringBuffer sBuffer(256);
        if (!FlagCheck::isValid(pFilter->nFlags))
        {
            sBuffer.append     (pFilter->sName         );
            sBuffer.appendAscii(" uses invalid flags\n");
            bLog = sal_True;
        }
        if (FlagCheck::useDeprecated(pFilter->nFlags))
        {
            sBuffer.append     (pFilter->sName            );
            sBuffer.appendAscii(" uses deprecated flags\n");
            bLog = sal_True;
        }
        if (FlagCheck::isMaskSet(pFilter->nFlags,FILTERFLAG_DEFAULT))
        {
            sal_Bool bSecondRegistration = sal_False;
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.text.TextDocument"))
            {
                bSecondRegistration = bTextDefault;
                bTextDefault        = sal_True    ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.text.WebDocument"))
            {
                bSecondRegistration = bWebDefault;
                bWebDefault         = sal_True   ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.text.GlobalDocument"))
            {
                bSecondRegistration = bGlobalDefault;
                bGlobalDefault      = sal_True      ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.sheet.SpreadsheetDocument"))
            {
                bSecondRegistration = bCalcDefault;
                bCalcDefault        = sal_True    ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.drawing.DrawingDocument"))
            {
                bSecondRegistration = bDrawDefault;
                bDrawDefault        = sal_True    ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.presentation.PresentationDocument"))
            {
                bSecondRegistration = bImpressDefault;
                bImpressDefault     = sal_True       ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.formula.FormulaProperties"))
            {
                bSecondRegistration = bMathDefault;
                bMathDefault        = sal_True    ;
            }
            else
            if (pFilter->sDocumentService.equalsAscii("com.sun.star.chart.ChartDocument"))
            {
                bSecondRegistration = bChartDefault;
                bChartDefault       = sal_True     ;
            }

            if (bSecondRegistration)
            {
                sBuffer.append     (pFilter->sName                                        );
                sBuffer.appendAscii(" is registered as n.th default filter for module: \"");
                sBuffer.append     (pFilter->sDocumentService                             );
                sBuffer.appendAscii("\"\n"                                                );
                bLog = sal_True;
            }
        }
        LOG_COND_FILTERDBG_1_PARAM(bLog, "DataContainer::validateAndRepair()", U2B(sBuffer.makeStringAndClear()), "[filter] seams to have set some problematic flags.\nPlease check it ... it will be ignored here!");
        #endif // ENABLE_FILTERDBG

        // Log information about missing UIName ... but don't react in any way.
        // It's not a reason to break the office. Normaly any UI code should ignore this filter
        // then. But nevertheless it will be useable at the API!
        LOG_COND_FILTERDBG_1_PARAM(pFilter->lUINames.size()<1, "DataContainer::validateAndRepair()", U2B(pFilter->sName), "[filter] has no UIName.\nPlease check it ... it will be ignored here!");
    }

    // remove invalid filter realy ...
    for (OUStringList::const_iterator pFilterName=lRemoveFilters.begin(); pFilterName!=lRemoveFilters.end(); ++pFilterName)
    {
        LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pFilterName), "[filter] is not valid and will be removed")
        removeFilter(*pFilterName, sal_True);
        if (m_aFilterCache.find(*pFilterName) != m_aFilterCache.end())
        {
            LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pFilterName), "[filter] could not be removed successfully")
            return sal_False;
        }
    }
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "} check filters" );

    return sal_True;
}

//_______________________________________________

/** @short  check list of detect services and try to repeair it, if some items seams to be invalid.

    @descr  In case repairing of such invalid entries won't be possible ... it returns false.
            But then this cache won't be useable further more!

    @return True, if current detect service list don't include invalid items or repairing of such items
            was successfully.

    @attention  Don't remove as invalid detected items directly inside the loop!
                Very often such items are part of different lists/maps of this data container,
                And removing of it must be done in all these structures.
                That's why we call specialized methods of class DataContainer.
                And in this case it's not enough to remove the current iterator entry ...

                On the other side we can use the same loop to correct invalid but repairable
                entries. In case we would use e.g. ::std::remove_if(...) we must steop over all
                container items twice - first time to remove invalid entries - second one to repair
                some other ones.
 */

sal_Bool DataContainer::validateAndRepairDetectors()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepairDetectors" );

    // ----------------------------------------------------------------------------------
    // check for detectors, which are not currently used or invalid
    // remove it!
    OUStringList lRemoveDetectors;
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "{ check detectors" );
    for (DetectorHash::iterator pDetectorIt=m_aDetectorCache.begin(); pDetectorIt!=m_aDetectorCache.end(); ++pDetectorIt)
    {
        Detector* pDetector = &(pDetectorIt->second);
        // check if this detector is registered for an existing type
        // Note: Such detector can be registered for more then one type.
        // Remove it only, if no type registrations is a valid one.
        // But try to repair it's type list by removing wrong entries.
        // May the detector won't be removed ... but then it's registration should be valid.
        sal_Bool     bAnyRegistered = sal_False;
        OUStringList lRemoveTypeNames;
        for (OUStringList::const_iterator pTypeIt=pDetector->lTypes.begin(); pTypeIt!=pDetector->lTypes.end(); ++pTypeIt)
        {
            if (m_aTypeCache.find(*pTypeIt)!=m_aTypeCache.end())
            {
                bAnyRegistered = sal_True;
                continue;
            }
            lRemoveTypeNames.push_back(*pTypeIt);
        }

        // if detector is registered for no valid type ...
        // remove it and continue with next detect service.
        if (!bAnyRegistered)
        {
            lRemoveDetectors.push_back(pDetector->sName);
            continue;
        }

        // otherwhise try to repair this detector.
        // Only some of it's type registrations are wrong.
        // So the detector itself can be untouched ..
        // but these invalid type entries must be removed.
        for (OUStringList::const_iterator pTypeNameIt = lRemoveTypeNames.begin(); pTypeNameIt != lRemoveTypeNames.end(); ++pTypeNameIt)
        {
            #ifdef ENABLE_FILTERDBG
            ::rtl::OUStringBuffer sBuffer(256);
            sBuffer.append     (pDetector->sName);
            sBuffer.appendAscii(".Type="        );
            sBuffer.append     (*pTypeNameIt    );
            #endif // ENABLE_FILTERDBG
            LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(sBuffer.makeStringAndClear()), "[type entry for detector] is not valid and will be removed")

            LOG_WARNING("DataContainer::validateAndRepair()", "removing of invalid detect services not implemented yet! Please do it manually and check filterdbg.log.")
            /*FIXME
                This type entry must be removed from pDetector->lTypes list and further
                the binding of the detector to this type must be removed from the special hash
                m_aFastDetectorHash ...
            if (pDetector->lTypes.find(*pTypeNameIt) != pDetector->lTypes.end())
            {
                LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(sBuffer.makeStringAndClear()), "[type entry for detector] could not be removed successfully")
                return sal_False;
            }
            */
        }
    }

    // remove invalid detector realy ...
    for (OUStringList::const_iterator pDetectorName=lRemoveDetectors.begin(); pDetectorName!=lRemoveDetectors.end(); ++pDetectorName)
    {
        LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pDetectorName), "[detector] is not references by others and will be removed")
        removeDetector(*pDetectorName, sal_True);
        if (m_aDetectorCache.find(*pDetectorName) != m_aDetectorCache.end())
        {
            LOG_FILTERDBG_1_PARAM("DataContainer::validateAndRepair()", U2B(*pDetectorName), "[detector] could not be removed successfully")
            return sal_False;
        }
    }
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "} check detectors" );

    return sal_True;
}

//_______________________________________________

/** @short  check list of loaders and try to repeair it, if some items seams to be invalid.

    @descr  In case repairing of such invalid entries won't be possible ... it returns false.
            But then this cache won't be useable further more!

    @return True, if current loader list don't include invalid items or repairing of such items
            was successfully.

    @attention  Don't remove as invalid detected items directly inside the loop!
                Very often such items are part of different lists/maps of this data container,
                And removing of it must be done in all these structures.
                That's why we call specialized methods of class DataContainer.
                And in this case it's not enough to remove the current iterator entry ...

                On the other side we can use the same loop to correct invalid but repairable
                entries. In case we would use e.g. ::std::remove_if(...) we must steop over all
                container items twice - first time to remove invalid entries - second one to repair
                some other ones.
 */

sal_Bool DataContainer::validateAndRepairLoader()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepairLoader" );
    return sal_True;
}

//_______________________________________________

/** @short  check list of handlers and try to repeair it, if some items seams to be invalid.

    @descr  In case repairing of such invalid entries won't be possible ... it returns false.
            But then this cache won't be useable further more!

    @return True, if current handler list don't include invalid items or repairing of such items
            was successfully.

    @attention  Don't remove as invalid detected items directly inside the loop!
                Very often such items are part of different lists/maps of this data container,
                And removing of it must be done in all these structures.
                That's why we call specialized methods of class DataContainer.
                And in this case it's not enough to remove the current iterator entry ...

                On the other side we can use the same loop to correct invalid but repairable
                entries. In case we would use e.g. ::std::remove_if(...) we must steop over all
                container items twice - first time to remove invalid entries - second one to repair
                some other ones.
 */

sal_Bool DataContainer::validateAndRepairHandler()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepairHandler" );
    return sal_True;
}

/**
 */

sal_Bool DataContainer::validateAndRepair()
{
    RTL_LOGFILE_CONTEXT( aLog, "framework (as96863) ::DataContainer::validateAndRepair" );

    // ----------------------------------------------------------------------------------
    // check minimum set of informations exist
    // If the are not valid ... there will be no chance to repair anything!
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "{ check minimum requirements" );
    if (!isValidOrRepairable())
    {
        RTL_LOGFILE_CONTEXT_TRACE( aLog, "} check minimum requirements" );
        LOG_FILTERDBG("DataContainer::validateAndRepair()", "can't repair damaged cache :-(")
        return sal_False;
    }
    RTL_LOGFILE_CONTEXT_TRACE( aLog, "} check minimum requirements" );

    // ----------------------------------------------------------------------------------
    // check all other items of this cache using specialized functions ...
    return (
            validateAndRepairTypes()     &&
            validateAndRepairFilter()    &&
            validateAndRepairDetectors() &&
            validateAndRepairLoader()    &&
            validateAndRepairHandler()
           );
}

//*****************************************************************************************************************
//  static helper
//      (1) replace %PRODUCTNAME% with right value from configuration
//      (2) by the way: if product name is "openoffice.org" it tries to find "6.0" and replace it with "1.0"
//*****************************************************************************************************************
void FilterCFGAccess::setProductName( OUStringHash& lUINames )
{
    ::rtl::OUString* pValue = NULL;
    for( OUStringHash::iterator pItem=lUINames.begin(); pItem!=lUINames.end(); ++pItem )
    {
        pValue = &(pItem->second);
        // replace %productname%
        sal_Int32 nIndex = pValue->indexOf( PRODUCTNAME_VARIABLE );
        while( nIndex != -1 )
        {
            *pValue = pValue->replaceAt( nIndex, PRODUCTNAME_VARLENGTH, m_sProductName );
            nIndex  = pValue->indexOf( PRODUCTNAME_VARIABLE, nIndex );
        }
        // replace %formatversion%
        nIndex = pValue->indexOf( FORMATVERSION_VARIABLE );
        while( nIndex != -1 )
        {
            *pValue = pValue->replaceAt( nIndex, FORMATVERSION_VARLENGTH, m_sFormatVersion );
            nIndex  = pValue->indexOf( FORMATVERSION_VARIABLE, nIndex );
        }
    }
}

//*****************************************************************************************************************
//  static helper
//*****************************************************************************************************************
void FilterCFGAccess::resetProductName( OUStringHash& lUINames )
{
    /**
     * temp. disabled till another solution exist for that.
     * We change filter ui names at loading time and save this changes ones
     * back to the configuration. Next startup willn't force such replacements ...
     */

    /**
    ::rtl::OUString* pValue  = NULL;
    sal_Int32        nLength = m_sProductName.getLength();
    for( OUStringHash::iterator pItem=lUINames.begin(); pItem!=lUINames.end(); ++pItem )
    {
        pValue = &(pItem->second);
        sal_Int32 nIndex = pValue->indexOf( m_sProductName );
        if( nIndex != -1 )
        {
            *pValue = pValue->replaceAt( nIndex, nLength, PRODUCTNAME_VARIABLE );
            if(m_bActivateOpenofficePatch)
            {
                nIndex = pValue->indexOf( PRODUCTPATCH_NEWVERSION );
                if( nIndex != -1 )
                {
                    *pValue = pValue->replaceAt( nIndex, PRODUCTPATCH_VARLENGTH, PRODUCTPATCH_ORGVERSION );
                }
            }
        }
    }
    */
}

/*-************************************************************************************************************//**
    @short      filter all "*." from an extension
    @descr      We search for extensions without wildcards at runtime!
                Our configuration contains sometimes extensions WITH wildcards ..
                We should correct it at runtime.
                Algorithm:
                    - Step over all strings in list - search for "*.<extension>" and convert it to "<extension>".
                    - If first signs are not "*." ignore item and do nothing!
                    - But make all extensions lowercase too!

    @seealso    search methods of class FilterCFGAccess

    @param      "lExtensions", list of extensions to correct
    @return     -

    @onerror    No error should occure.
*//*-*************************************************************************************************************/
void DataContainer::correctExtensions( OUStringList& lExtensions )
{
	sal_Int32 nPosition;
    for( OUStringList::iterator pExtension=lExtensions.begin(); pExtension!=lExtensions.end(); ++pExtension )
	{
		nPosition = pExtension->indexOf( DECLARE_ASCII("*.") );
		if( nPosition != -1 )
		{
			nPosition += 2;
			*pExtension = pExtension->copy( nPosition, pExtension->getLength()-nPosition );
		}
        *pExtension = pExtension->toAsciiLowerCase();
	}
}

/*-************************************************************************************************************//**
    @short      extract localized strings from given configuration item or reverse
    @descr      It exist to methods to get a localized value from our configuration.
                a) cfg returns one value for current set locale
                b) cfg returns a list of values for all possible locales
                These method detect it and fills our internal "localelized" hash automaticly or
                convert internal structures to external ones.

    @seealso    baseclass ConfigItem
    @seealso    method impl_setLocalelizedString()
    @seealso    method impl_getLocalelizedString()

    @param      "pLocale"   , is used by a) only to insert value at right hash position
    @param      "aCFGValue" , contains a) [OUString] ... or b) [Sequence< PropertyValue >]
    @param      "lLocales"  , internal description of localized values!
    @return     List of sorted locales with values.

    @onerror    No error should occure.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::extractLocalizedStrings( const ::rtl::OUString& sCurrentLocale, const css::uno::Any& aCFGValue, OUStringHash& lLocales )
{
	// Algorithm:
	//	a	)	free current hash (we return an empty hash if method failed!)
	//	b	)	detect type of Any-value
	//	c.1	)	if( type = string )
	//				insert it for right locale "pCurrentLocale" in hash
	//	c.2)	or if( type = seq< prop > )
	//				copy sequence to hash (ignore "pCurrentLocale"!)

	// a)
	lLocales.free();
	// b)
    css::uno::Type aType = aCFGValue.getValueType();
	// c.1)
    if( aType == ::getCppuType( (const ::rtl::OUString*)NULL ) )
	{
        ::rtl::OUString sValue;
        if (aCFGValue >>= sValue)
            setLocalelizedString( lLocales, sCurrentLocale, sValue );
	}
	else
	// c.2)
    if( aType == ::getCppuType( (const css::uno::Sequence< css::beans::PropertyValue >*)NULL ) )
	{
        css::uno::Sequence< css::beans::PropertyValue > lLocaleValues;
        if (aCFGValue >>= lLocaleValues)
            lLocales = Converter::convert_seqProp2OUStringHash(lLocaleValues);
	}
	#ifdef ENABLE_ASSERTIONS
	else
	{
        LOG_ERROR( "FilterCFGAccess::impl_extractLocalizedStrings()", "Unknown cfg-type detected. Ignore item ...\nIt seams that the ALL_LOCALES mode of ConfigItem does not work!" )
	}
    #endif
}

//*****************************************************************************************************************
void DataContainer::packLocalizedStrings( sal_Int16 nMode, const ::rtl::OUString& sCurrentLocale, css::uno::Any& aCFGValue, const OUStringHash& lLocales )
{
	// Our cache works in "ALL LOCALES" mode only!
	// We can ignore "pCurrentLocale" here ...
	// but if someone change that ... he should have a look at this method too!
    sal_Bool bAllLocaleMode = ((nMode & CONFIG_MODE_ALL_LOCALES) == CONFIG_MODE_ALL_LOCALES);
    if (bAllLocaleMode)
        aCFGValue <<= Converter::convert_OUStringHash2seqProp(lLocales);
    else
    {
        OUStringHash::const_iterator pIt = lLocales.find(sCurrentLocale);
        aCFGValue <<= pIt->second;
    }
}

/*-************************************************************************************************************//**
    @short      return right value from list of all localized ones
    @descr      We hold a list of all e.g. UINames for every set item.
                If user whish to get current UIName for current set locale ... you can call
                this method to get this information.

    @attention  We have optimized our TypeDetection.xml ... So we don't save all localized values, if
                there nothing exist, which should be loclialized (e.g. "Writer 6.0")!
                In this case we must implement a fallback if no value exist for current set locale.
                Our fallback is "en-US" everytime - because only this locale exit in xml file!

    @seealso    method impl_extractLocalizedStrings()

    @param      "lLocales"  , list of all localized values
    @param      "sLocale"   , current set locale
    @param      "sValue"    , new localized value to set in list for current locale
    @return     Matching UIName for current locale from list.

    @onerror    No error should occure.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
::rtl::OUString DataContainer::getLocalelizedString( const OUStringHash& lLocales, const ::rtl::OUString& sLocale )
{
    ::rtl::OUString         sValue                             ;
    OUStringHash::const_iterator pItem   = lLocales.find( sLocale ) ;
	if( pItem != lLocales.end() )
	{
		sValue = pItem->second;
	}
    else
    {
        pItem = lLocales.find( LOCALE_FALLBACK );
        if( pItem != lLocales.end() )
        {
    		sValue = pItem->second;
        }
    }
    LOG_ASSERT2( sValue.getLength()<1, "DataContainer::getLocalizedString()", "Couldn't find any value for this locale.\nIt seams that ALL_LOCALES mode of ConfigItem does not work!" )
	return sValue;
}

//*****************************************************************************************************************
void DataContainer::setLocalelizedString( OUStringHash& lLocales, const ::rtl::OUString& sLocale, const ::rtl::OUString& sValue )
{
    // Follow case can occure:
    //  a) given locale is the fallback one                                                                         => we can set this value directly
    //  b) given locale is different from fallback & but fallback not exist in list                                 => set new value for given locale ... but don't touch add it as fallback
    //                                                                                                                 May be we override it at pushing item into our internal structures.
    //                                                                                                                 e.g.: i  ) creation of empty filter item
    //                                                                                                                       ii ) copy properties set by user to this item
    //                                                                                                                       iii) set item at internal structures
    //                                                                                                                       At ii) no fallback value exist ... but if we set it we must merge data in step iii)!
    //  c) given locale is different from fallback & fallback exist & new value is different from fallback value    => set value for given locale only
    //  d) given locale is different from fallback & fallback exist & new value is same like fallback value         => do nothing! use fallback

    // a)
    if( sLocale == LOCALE_FALLBACK )
    {
        lLocales[sLocale] = sValue;
    }
    else
    {
        OUStringHash::const_iterator pFallback = lLocales.find( LOCALE_FALLBACK );
        // b)
        if( pFallback == lLocales.end() )
        {
            lLocales[sLocale] = sValue;
        }
        else
        {
            // c)
            if( pFallback->second != sValue )
            {
                lLocales[sLocale] = sValue;
            }
            // d!)
        }
    }
}

/*-************************************************************************************************************//**
    @short      clear complete cache REALY
    @descr      STL don't free memory if you call erase() or clear()! The only way to do it realy, is to
                swap an empty container into this one :-( We use special free methods of our container structures
                to do that.

    @seealso    stl

    @param      -
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void DataContainer::free()
{
    m_aTypeCache.free                 ();
    m_aFilterCache.free               ();
    m_aDetectorCache.free             ();
    m_aLoaderCache.free               ();
    m_aContentHandlerCache.free       ();

    m_aFastFilterCache.free           ();
    m_aFastDetectorCache.free         ();
    m_aFastLoaderCache.free           ();
    m_aFastContentHandlerCache.free   ();

    m_aPreferredTypesCache.free       ();

    m_aGenericLoader.free             ();

    m_sLocale = ::rtl::OUString();
}

//_______________________________________________

sal_Bool DataContainer::existsType( const ::rtl::OUString& sName )
{
    ReadGuard aReadLock(m_aLock);
    return (m_aTypeCache.find(sName) != m_aTypeCache.end());
}

//_______________________________________________

sal_Bool DataContainer::existsFilter( const ::rtl::OUString& sName )
{
    ReadGuard aReadLock(m_aLock);
    return (m_aFilterCache.find(sName) != m_aFilterCache.end());
}

//_______________________________________________

sal_Bool DataContainer::existsDetector( const ::rtl::OUString& sName )
{
    ReadGuard aReadLock(m_aLock);
    return ( m_aDetectorCache.find(sName) != m_aDetectorCache.end() );
}

//_______________________________________________

sal_Bool DataContainer::existsLoader( const ::rtl::OUString& sName )
{
    ReadGuard aReadLock(m_aLock);
    return (
            (m_aLoaderCache.find(sName) != m_aLoaderCache.end()) ||
            (m_aGenericLoader.sName.equals(sName)              )
           );
}

//_______________________________________________

sal_Bool DataContainer::existsContentHandler( const ::rtl::OUString& sName )
{
    ReadGuard aReadLock(m_aLock);
    return (m_aContentHandlerCache.find(sName) != m_aContentHandlerCache.end());
}

/*-************************************************************************************************************//**
    @short      register new config item in data container
    @descr      This should be used to add a new item to this data container.

    @attention  We differ between impl and normal methods. Normal versions set modified flag additional to impl ones.
                This enable flushing of changed data to configuration. Impl operations could be used to fill
                cache too.

    @seealso    method replace...()
    @seealso    method remove...()

    @param      "aType, aFilter ...", information about new item to set it on internal structures
    @param      "bSetModified"      , regulate registration of new item as realy new element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::addType( const FileType& aType, sal_Bool bSetModified )
{
    // Insert entry in normal type cache.
    m_aTypeCache[ aType.sName ] = aType;
	// If this is a preferred type ...
	// add it to special hash too!
	// Register him for all supported extensions.
	if( aType.bPreferred == sal_True )
	{
        for (   OUStringList::const_iterator pExtension=m_aTypeCache[ aType.sName ].lExtensions.begin()  ;
                pExtension!=m_aTypeCache[ aType.sName ].lExtensions.end()                           ;
				++pExtension
			)
		{
            m_aPreferredTypesCache[ *pExtension ] = aType.sName;
		}
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( aType.sName, E_ADDED );
        m_bTypesModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addFilter( const Filter& aFilter, sal_Bool bSetModified )
{
	// Do it in our normal filter cache ...
    m_aFilterCache[ aFilter.sName ] = aFilter;
	// ... but don't forget it in our faster performance cache!
    // Look for right updating of this structure by using
    // the preferred flag so it should have follow structure:
    //  [0] = the preferred filter
    //  [1] = first other filter
    //  ...   (unsorted!)
    //  [n] = last other filter
    // In case of a new preferred one with an already registered one follow will happen:
    //  [0] = the new preferred filter
    //  [1] = the old preferred filter without this flag now!
    //  [2] = first other filter
    //  ...   (unsorted!)
    //  [n] = last other filter
    // In case of a new filter without the preferred flag:
    //  [0]   = the current preferred filter
    //  [1]   = first other filter
    //  ...     (unsorted!)
    //  [n]   = last other filter
    //  [n+1] = the new filter
    // Note: We check for list size too ... because;
    //  it doesn't matter if the preferred flag exist on this filter or not ...
    //  list was empty before! Then appending as last one will set him as first one
    //  automaticly too. So we save time by supressing possible expensive
    //  reorganisation of structure.
    OUStringList& rList = m_aFastFilterCache[ aFilter.sType ];
    if(
        ( rList.size() > 0                                          )   &&
        ( FlagCheck::isMaskSet(aFilter.nFlags, FILTERFLAG_PREFERED) )
      )
    {
        m_aFilterCache[ *(rList.begin()) ].nFlags &= ~FILTERFLAG_PREFERED;
        rList.push_front( aFilter.sName );
    }
    else
    {
        rList.push_back( aFilter.sName );
    }
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( aFilter.sName, E_ADDED );
        m_bFiltersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addDetector( const Detector& aDetector, sal_Bool bSetModified )
{
	// Do it for our normal cache first ...
    m_aDetectorCache[ aDetector.sName ] = aDetector;
	// But don*t forget to add every detector item to our faster performance cache for type-detector assigning too!
	sal_uInt32 nLength = (sal_uInt32)(aDetector.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastDetectorCache[ aDetector.lTypes[nType] ].push_back( aDetector.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( aDetector.sName, E_ADDED );
        m_bDetectorsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addLoader( const Loader& aLoader, sal_Bool bSetModified )
{
	// Put entry in stl-container.
    m_aLoaderCache[ aLoader.sName ] = aLoader;
	// Add every loader item to our faster performance cache for type-loader assigning too!
	sal_uInt32 nLength = (sal_uInt32)(aLoader.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastLoaderCache[ aLoader.lTypes[nType] ].push_back( aLoader.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( aLoader.sName, E_ADDED );
        m_bLoadersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addContentHandler( const ContentHandler& aHandler, sal_Bool bSetModified )
{
	// Put entry in stl-container.
    m_aContentHandlerCache[ aHandler.sName ] = aHandler;
    // Add every handler item to our faster performance cache for type-Handler assignment too!
    sal_uInt32 nLength = (sal_uInt32)(aHandler.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastContentHandlerCache[ aHandler.lTypes[nType] ].push_back( aHandler.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( aHandler.sName, E_ADDED );
        m_bHandlersModified = sal_True;
    }
}

/*-************************************************************************************************************//**
    @short      replace existing config item in data container
    @descr      This should be used to change a item of this data container.

    @attention  Call combination of remove/add by using flag "sal_False" ...
                because a "sal_True" will trigger two flush operations!
                One change call should be enough ...

    @seealso    method add...()
    @seealso    method remove...()

    @param      "aType, aFilter ...", information about changed item to set it on internal structures
    @param      "bSetModified"      , regulate registration of changed item as "realy" changed element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::replaceType( const FileType& aType, sal_Bool bSetModified )
{
    removeType( aType.sName, sal_False );
    addType   ( aType      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( aType.sName, E_CHANGED );
        m_bTypesModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceFilter( const Filter& aFilter, sal_Bool bSetModified )
{
    removeFilter( aFilter.sName, sal_False );
    addFilter   ( aFilter      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( aFilter.sName, E_CHANGED );
        m_bFiltersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceDetector( const Detector& aDetector, sal_Bool bSetModified )
{
    removeDetector( aDetector.sName, sal_False );
    addDetector   ( aDetector      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( aDetector.sName, E_CHANGED );
        m_bDetectorsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceLoader( const Loader& aLoader, sal_Bool bSetModified )
{
    removeLoader( aLoader.sName, sal_False );
    addLoader   ( aLoader      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( aLoader.sName, E_CHANGED );
        m_bLoadersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceContentHandler( const ContentHandler& aHandler, sal_Bool bSetModified )
{
    removeContentHandler( aHandler.sName, sal_False );
    addContentHandler   ( aHandler      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( aHandler.sName, E_CHANGED );
        m_bHandlersModified = sal_True;
    }
}

/*-************************************************************************************************************//**
    @short      remove existing config item from data container
    @descr      This should be used to remove an item of this data container.

    @attention  We differ between impl and normal methods. Normal versions set modified flag additional to impl ones.
                This enable flushing of changed data to configuration.

    @seealso    method add...()
    @seealso    method replace...()

    @param      "aType, aFilter ...", information about changed item to set it on internal structures
    @param      "bSetModified"      , regulate registration of removed item as "realy" deleted element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::removeType( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
    // Remove it from our normal cache.
    m_aTypeCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( sName, E_REMOVED );
        m_bTypesModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeFilter( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which filter has registered!
    // It's neccessary to find these filter in our performance cache which is reverse to our normal filter cache!!!
    // Otherwise we have no chance to find it there ...
    ::rtl::OUString sType = m_aFilterCache[ sName ].sType  ;
    OUStringList&   rList = m_aFastFilterCache[ sType ]    ;
    rList.erase( ::std::find( rList.begin(), rList.end(), sName ) );
	// then in our normal cache.
    m_aFilterCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( sName, E_REMOVED );
        m_bFiltersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeDetector( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which detector has registered!
    // It's neccessary to find these detector in our performance cache which is reverse to our normal detector cache!!!
    // Otherwise we have no chance to find it there ...
    for( OUStringList::const_iterator pType=m_aDetectorCache[ sName ].lTypes.begin(); pType!=m_aDetectorCache[ sName ].lTypes.end(); ++pType )
    {
        OUStringList rDetectors = m_aFastDetectorCache[ *pType ];
        rDetectors.erase( ::std::find( rDetectors.begin(), rDetectors.end(), sName ) );
    }
	// then in our normal cache.
    m_aDetectorCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( sName, E_REMOVED );
        m_bDetectorsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeLoader( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which loader has registered!
    // It's neccessary to find these loader in our performance cache which is reverse to our normal loader cache!!!
    // Otherwise we have no chance to find it there ...
    for( OUStringList::const_iterator pType=m_aLoaderCache[ sName ].lTypes.begin(); pType!=m_aLoaderCache[ sName ].lTypes.end(); ++pType )
    {
        OUStringList rLoaders = m_aFastLoaderCache[ *pType ];
        rLoaders.erase( ::std::find( rLoaders.begin(), rLoaders.end(), sName ) );
    }
	// then in our normal cache.
    m_aLoaderCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( sName, E_REMOVED );
        m_bLoadersModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeContentHandler( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which handler has registered!
    // It's neccessary to find these handler in our performance cache which is reverse to our normal handler cache!!!
    // Otherwise we have no chance to find it there ...
    for( OUStringList::const_iterator pType=m_aContentHandlerCache[ sName ].lTypes.begin(); pType!=m_aContentHandlerCache[ sName ].lTypes.end(); ++pType )
    {
        OUStringList rHandlers = m_aFastContentHandlerCache[ *pType ];
        rHandlers.erase( ::std::find( rHandlers.begin(), rHandlers.end(), sName ) );
    }
	// then in our normal cache.
    m_aContentHandlerCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( sName, E_REMOVED );
        m_bHandlersModified = sal_True;
    }
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:11
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertFileTypeToPropertySequence( const FileType&                                        aSource        ,
                                                             css::uno::Sequence< css::beans::PropertyValue >& lDestination   ,
                                                       const ::rtl::OUString&                                 sCurrentLocale )
{
    lDestination.realloc(9);
    sal_Int32 n = 0;

    lDestination[n].Name    = PROPERTY_NAME;
    lDestination[n].Value <<= aSource.sName;
    ++n;
    lDestination[n].Name    = PROPERTY_PREFERRED;
    lDestination[n].Value <<= aSource.bPreferred;
    ++n;
    lDestination[n].Name    = PROPERTY_UINAME;
    lDestination[n].Value <<= DataContainer::getLocalelizedString(aSource.lUINames, sCurrentLocale);
    ++n;
    lDestination[n].Name    = PROPERTY_UINAMES;
    lDestination[n].Value <<= Converter::convert_OUStringHash2seqProp(aSource.lUINames);
    ++n;
    lDestination[n].Name    = PROPERTY_MEDIATYPE;
    lDestination[n].Value <<= aSource.sMediaType;
    ++n;
    lDestination[n].Name    = PROPERTY_CLIPBOARDFORMAT;
    lDestination[n].Value <<= aSource.sClipboardFormat;
    ++n;
    lDestination[n].Name    = PROPERTY_URLPATTERN;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lURLPattern);
    ++n;
    lDestination[n].Name    = PROPERTY_EXTENSIONS;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lExtensions);
    ++n;
    lDestination[n].Name    = PROPERTY_DOCUMENTICONID;
    lDestination[n].Value <<= aSource.nDocumentIconID;
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:16
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertFilterToPropertySequence( const Filter&                                          aSource        ,
                                                           css::uno::Sequence< css::beans::PropertyValue >& lDestination   ,
                                                     const ::rtl::OUString&                                 sCurrentLocale )
{
    lDestination.realloc(12);
    sal_Int32 n = 0;

    lDestination[n].Name    = PROPERTY_NAME;
    lDestination[n].Value <<= aSource.sName;
    ++n;
    lDestination[n].Name    = PROPERTY_TYPE;
    lDestination[n].Value <<= aSource.sType;
    ++n;
    lDestination[n].Name    = PROPERTY_UINAME;
    lDestination[n].Value <<= DataContainer::getLocalelizedString(aSource.lUINames, sCurrentLocale);
    ++n;
    lDestination[n].Name    = PROPERTY_UINAMES;
    lDestination[n].Value <<= Converter::convert_OUStringHash2seqProp(aSource.lUINames);
    ++n;
    lDestination[n].Name    = PROPERTY_DOCUMENTSERVICE;
    lDestination[n].Value <<= aSource.sDocumentService;
    ++n;
    lDestination[n].Name    = PROPERTY_FILTERSERVICE;
    lDestination[n].Value <<= aSource.sFilterService;
    ++n;
    lDestination[n].Name    = PROPERTY_FLAGS;
    lDestination[n].Value <<= aSource.nFlags;
    ++n;
    lDestination[n].Name    = PROPERTY_USERDATA;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lUserData);
    ++n;
    lDestination[n].Name    = PROPERTY_FILEFORMATVERSION;
    lDestination[n].Value <<= aSource.nFileFormatVersion;
    ++n;
    lDestination[n].Name    = PROPERTY_TEMPLATENAME;
    lDestination[n].Value <<= aSource.sTemplateName;
    ++n;
    lDestination[n].Name    = PROPERTY_ORDER;
    lDestination[n].Value <<= aSource.nOrder;
    ++n;
    lDestination[n].Name    = PROPERTY_UICOMPONENT;
    lDestination[n].Value <<= aSource.sUIComponent;
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:17
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertLoaderToPropertySequence( const Loader&                                          aSource        ,
                                                           css::uno::Sequence< css::beans::PropertyValue >& lDestination   ,
                                                     const ::rtl::OUString&                                 sCurrentLocale )
{
    lDestination.realloc(4);
    sal_Int32 n = 0;

    lDestination[n].Name    = PROPERTY_NAME;
    lDestination[n].Value <<= aSource.sName;
    ++n;
    lDestination[n].Name    = PROPERTY_TYPES;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lTypes);
    ++n;
    lDestination[n].Name    = PROPERTY_UINAME;
    lDestination[n].Value <<= DataContainer::getLocalelizedString(aSource.lUINames, sCurrentLocale);
    ++n;
    lDestination[n].Name    = PROPERTY_UINAMES;
    lDestination[n].Value <<= Converter::convert_OUStringHash2seqProp(aSource.lUINames);
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:18
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertDetectorToPropertySequence( const Detector&                                        aSource      ,
                                                             css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
    lDestination.realloc(2);
    sal_Int32 n = 0;

    lDestination[n].Name    = PROPERTY_NAME;
    lDestination[n].Value <<= aSource.sName;
    ++n;
    lDestination[n].Name    = PROPERTY_TYPES;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lTypes);
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:18
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertContentHandlerToPropertySequence( const ContentHandler&                                  aSource      ,
                                                                   css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
    lDestination.realloc(2);
    sal_Int32 n = 0;

    lDestination[n].Name    = PROPERTY_NAME;
    lDestination[n].Value <<= aSource.sName;
    ++n;
    lDestination[n].Name    = PROPERTY_TYPES;
    lDestination[n].Value <<= Converter::convert_OUStringList2seqOUString(aSource.lTypes);
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:23
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertPropertySequenceToFilter( const css::uno::Sequence< css::beans::PropertyValue >& lSource        ,
                                                           Filter&                                          aDestination   ,
                                                     const ::rtl::OUString&                                 sCurrentLocale )
{
	// First clear all entries of destination struct!
    aDestination.free();
	// Then copy existing properties from source to destination.
	// Non existing properties will have a default value in destination then!
	sal_Int32 nCount = lSource.getLength();
    for (sal_Int32 p = 0; p < nCount; ++p)
	{
        if( lSource[p].Name == PROPERTY_NAME )
            lSource[p].Value >>= aDestination.sName;
        else
        if( lSource[p].Name == PROPERTY_TYPE )
            lSource[p].Value >>= aDestination.sType;
        else
        if( lSource[p].Name == PROPERTY_DOCUMENTSERVICE )
            lSource[p].Value >>= aDestination.sDocumentService;
        else
        if( lSource[p].Name == PROPERTY_FILTERSERVICE )
            lSource[p].Value >>= aDestination.sFilterService;
        else
        if( lSource[p].Name == PROPERTY_UICOMPONENT )
            lSource[p].Value >>= aDestination.sUIComponent;
        else
        if( lSource[p].Name == PROPERTY_FLAGS )
            lSource[p].Value >>= aDestination.nFlags;
        else
        if( lSource[p].Name == PROPERTY_FILEFORMATVERSION )
            lSource[p].Value >>= aDestination.nFileFormatVersion;
        else
        if( lSource[p].Name == PROPERTY_TEMPLATENAME )
            lSource[p].Value >>= aDestination.sTemplateName;
        else
        if( lSource[p].Name == PROPERTY_ORDER )
            lSource[p].Value >>= aDestination.nOrder;
		else
        if( lSource[p].Name == PROPERTY_UINAME )
		{
            ::rtl::OUString sUIName;
            lSource[p].Value >>= sUIName;
            DataContainer::setLocalelizedString(aDestination.lUINames, sCurrentLocale, sUIName);
		}
        else
        if( lSource[p].Name == PROPERTY_UINAMES )
        {
            css::uno::Sequence< css::beans::PropertyValue > lUINames;
            lSource[p].Value >>= lUINames;
            aDestination.lUINames = Converter::convert_seqProp2OUStringHash(lUINames);
        }
		else
        if( lSource[p].Name == PROPERTY_USERDATA )
		{
            css::uno::Sequence< ::rtl::OUString > lUserData;
            lSource[p].Value >>= lUserData;
            aDestination.lUserData = Converter::convert_seqOUString2OUStringList(lUserData);
		}
	}
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:25
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertPropertySequenceToFileType( const css::uno::Sequence< css::beans::PropertyValue >& lSource        ,
                                                             FileType&                                        aDestination   ,
                                                       const ::rtl::OUString&                                 sCurrentLocale )
{
    // First clear all entries of destination struct!
    aDestination.free();
    // Then copy existing properties from source to destination.
    // Non existing properties will have a default value in destination then!
    sal_Int32 nCount = lSource.getLength();
    for (sal_Int32 p = 0; p < nCount; ++p)
    {
        if( lSource[p].Name == PROPERTY_NAME )
            lSource[p].Value >>= aDestination.sName;
        else
        if( lSource[p].Name == PROPERTY_PREFERRED )
            lSource[p].Value >>= aDestination.bPreferred;
        else
        if( lSource[p].Name == PROPERTY_MEDIATYPE )
            lSource[p].Value >>= aDestination.sMediaType;
        else
        if( lSource[p].Name == PROPERTY_CLIPBOARDFORMAT )
            lSource[p].Value >>= aDestination.sClipboardFormat;
        else
        if( lSource[p].Name == PROPERTY_DOCUMENTICONID )
            lSource[p].Value >>= aDestination.nDocumentIconID;
        else
        if( lSource[p].Name == PROPERTY_UINAME )
        {
            ::rtl::OUString sUIName;
            lSource[p].Value >>= sUIName;
            DataContainer::setLocalelizedString(aDestination.lUINames, sCurrentLocale, sUIName);
        }
        else
        if( lSource[p].Name == PROPERTY_UINAMES )
        {
            css::uno::Sequence< css::beans::PropertyValue > lUINames;
            lSource[p].Value >>= lUINames;
            aDestination.lUINames = Converter::convert_seqProp2OUStringHash(lUINames);
        }
        else
        if( lSource[p].Name == PROPERTY_URLPATTERN )
        {
            css::uno::Sequence< ::rtl::OUString > lURLPattern;
            lSource[p].Value >>= lURLPattern;
            aDestination.lURLPattern = Converter::convert_seqOUString2OUStringList(lURLPattern);
        }
        else
        if( lSource[p].Name == PROPERTY_EXTENSIONS )
        {
            css::uno::Sequence< ::rtl::OUString > lExtensions;
            lSource[p].Value >>= lExtensions;
            aDestination.lExtensions = Converter::convert_seqOUString2OUStringList(lExtensions);
        }
    }
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:26
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertPropertySequenceToDetector( const css::uno::Sequence< css::beans::PropertyValue >& lSource      ,
                                                             Detector&                                        aDestination )
{
    // First clear all entries of destination struct!
    aDestination.free();
    // Then copy existing properties from source to destination.
    // Non existing properties will have a default value in destination then!
    sal_Int32 nCount = lSource.getLength();
    for (sal_Int32 p = 0; p < nCount; ++p)
    {
        if( lSource[p].Name == PROPERTY_NAME )
            lSource[p].Value >>= aDestination.sName;
        else
        if( lSource[p].Name == PROPERTY_TYPES )
        {
            css::uno::Sequence< ::rtl::OUString > lTypes;
            lSource[p].Value >>= lTypes;
            aDestination.lTypes = Converter::convert_seqOUString2OUStringList(lTypes);
        }
    }
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:27
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertPropertySequenceToLoader( const css::uno::Sequence< css::beans::PropertyValue >& lSource        ,
                                                           Loader&                                          aDestination   ,
                                                     const ::rtl::OUString&                                 sCurrentLocale )
{
    // First clear all entries of destination struct!
    aDestination.free();
    // Then copy existing properties from source to destination.
    // Non existing properties will have a default value in destination then!
    sal_Int32 nCount = lSource.getLength();
    for (sal_Int32 p = 0; p < nCount; ++p)
    {
        if( lSource[p].Name == PROPERTY_NAME )
            lSource[p].Value >>= aDestination.sName;
        else
        if( lSource[p].Name == PROPERTY_TYPES )
        {
            css::uno::Sequence< ::rtl::OUString > lTypes;
            lSource[p].Value >>= lTypes;
            aDestination.lTypes = Converter::convert_seqOUString2OUStringList(lTypes);
        }
        else
        if( lSource[p].Name == PROPERTY_UINAME )
        {
            ::rtl::OUString sUIName;
            lSource[p].Value >>= sUIName;
            DataContainer::setLocalelizedString(aDestination.lUINames, sCurrentLocale, sUIName);
        }
        else
        if( lSource[p].Name == PROPERTY_UINAMES )
        {
            css::uno::Sequence< css::beans::PropertyValue > lUINames;
            lSource[p].Value >>= lUINames;
            aDestination.lUINames = Converter::convert_seqProp2OUStringHash(lUINames);
        }
    }
}

/*-------------------------------------------------------------------------------------------------
    06.03.2003 08:29
-------------------------------------------------------------------------------------------------*/
void DataContainer::convertPropertySequenceToContentHandler( const css::uno::Sequence< css::beans::PropertyValue >& lSource        ,
                                                                   ContentHandler&                                  aDestination   ,
                                                             const ::rtl::OUString&                                 sCurrentLocale )
{
    // First clear all entries of destination struct!
    aDestination.free();
    // Then copy existing properties from source to destination.
    // Non existing properties will have a default value in destination then!
    sal_Int32 nCount = lSource.getLength();
    for (sal_Int32 p = 0; p < nCount; ++p)
    {
        if( lSource[p].Name == PROPERTY_NAME )
            lSource[p].Value >>= aDestination.sName;
        else
        if( lSource[p].Name == PROPERTY_TYPES )
        {
            css::uno::Sequence< ::rtl::OUString > lTypes;
            lSource[p].Value >>= lTypes;
            aDestination.lTypes = Converter::convert_seqOUString2OUStringList(lTypes);
        }
    }
}

//*****************************************************************************************************************
// We accept right package names and valid versions only. Otherwise we would work on ANYTHING ... but not
// on right data set!
sal_Bool FilterCFGAccess::implcp_ctor( const ::rtl::OUString& sPath   ,
                                             sal_Int32        nVersion,
                                             sal_Int16        nMode   )
{
    return(
            ( &sPath   == NULL                        ) ||
            ( nVersion >  DEFAULT_FILTERCACHE_VERSION ) ||
            ( nVersion <  1                           ) ||
            (
                ( sPath != PACKAGENAME_TYPEDETECTION_STANDARD   )   &&
                ( sPath != PACKAGENAME_TYPEDETECTION_ADDITIONAL )
            )
          );
}

//*****************************************************************************************************************
// Check valid c++ reference only!
sal_Bool FilterCFGAccess::implcp_read( const DataContainer& rData )
{
    return(
            ( &rData == NULL )
          );
}

//*****************************************************************************************************************
// Check valid c++ reference only!
sal_Bool FilterCFGAccess::implcp_write( const DataContainer& rData )
{
    return(
            ( &rData == NULL )
          );
}
/*
//_______________________________________________

DEFINE_XINTERFACE_1(FilterCFGListener                            ,
                    OWeakObject                                  ,
                    DIRECT_INTERFACE(css::util::XChangesListener))

//_______________________________________________

FilterCFGListener::FilterCFGListener( const css::uno::Reference< css::lang::XMultiServiceFactory >& xSMGR  ,
                                            ECFGType                                                eType  ,
                                            DataContainer*                                          pData  )
    : ThreadHelpBase()
    , OWeakObject   ()
{
    m_pData      = pData    ;
    m_eType      = eType    ;
    m_bListening = sal_False;

    css::uno::Reference< css::lang::XMultiServiceFactory > xConfigProvider(xSMGR->createInstance(SERVICENAME_CFGPROVIDER), css::uno::UNO_QUERY);
    if (xConfigProvider.is())
    {
        ::rtl::OUString sPath = DECLARE_ASCII("/org.openoffice.Office.TypeDetection/");
        switch(m_eType)
        {
            case E_TYPE           : sPath += SUBLIST_TYPES          ; break;
            case E_FILTER         : sPath += SUBLIST_FILTERS        ; break;
            case E_LOADER         : sPath += SUBLIST_FRAMELOADERS   ; break;
            case E_DETECTOR       : sPath += SUBLIST_DETECTSERVICES ; break;
            case E_CONTENTHANDLER : sPath += SUBLIST_CONTENTHANDLERS; break;
        }

        css::beans::PropertyValue aParam;
        aParam.Name    = DECLARE_ASCII("nodepath");
        aParam.Value <<= sPath;

        css::uno::Sequence< css::uno::Any > lParams(1);
        lParams[0] <<= aParam;

        m_xCFG = xConfigProvider->createInstanceWithArguments(SERVICENAME_CFGREADACCESS, lParams);
    }
}

//_______________________________________________

void FilterCFGListener::startListening()
{
    // SAFE {
    WriteGuard aWriteLock(m_aLock);
    if (!m_bListening)
    {
        css::uno::Reference< css::util::XChangesNotifier > xNotifier(m_xCFG, css::uno::UNO_QUERY);
        if (xNotifier.is())
        {
            css::uno::Reference< css::util::XChangesListener > xListener(static_cast< ::cppu::OWeakObject* >(this), css::uno::UNO_QUERY);
            xNotifier->addChangesListener(xListener);
            m_bListening = sal_True;
        }
    }
    aWriteLock.unlock();
    // } SAFE
}

//_______________________________________________

void FilterCFGListener::stopListening()
{
    // SAFE {
    WriteGuard aWriteLock(m_aLock);
    if (m_bListening)
    {
        css::uno::Reference< css::util::XChangesNotifier > xNotifier(m_xCFG, css::uno::UNO_QUERY);
        if (xNotifier.is())
        {
            css::uno::Reference< css::util::XChangesListener > xListener(static_cast< ::cppu::OWeakObject* >(this), css::uno::UNO_QUERY);
            xNotifier->removeChangesListener(xListener);
            m_bListening = sal_False;
        }
    }
    aWriteLock.unlock();
    // } SAFE
}

//_______________________________________________

void SAL_CALL FilterCFGListener::changesOccurred( const css::util::ChangesEvent& rEvent )
    throw(css::uno::RuntimeException)
{
    // SAFE {
    ReadGuard aReadLock(m_aLock);
    if (!m_bListening)
    {
        LOG_WARNING("FilterCFGListener::changesOccurred()", "not registered as listener ... but got notify!?")
        return;
    }
    if (!m_pData)
    {
        LOG_WARNING("FilterCFGListener::changesOccurred()", "no valid data container")
        return;
    }
    aReadLock.unlock();
    // } SAFE

    ::rtl::OUStringBuffer sBuffer(1000);
    sBuffer.appendAscii("******************************************************\n");

    if (!rEvent.Base.hasValue())
        sBuffer.appendAscii("no base");
    else
        sBuffer.append(rEvent.Base.getValueTypeName());
    sBuffer.appendAscii("\n");

    const css::util::ElementChange* pElementChanges = rEvent.Changes.getConstArray();
    BaseHash< EModifyState > lItems;
    for (sal_Int32 i = 0; i < rEvent.Changes.getLength(); ++i)
    {
        ::rtl::OUString sPath;
        pElementChanges[i].Accessor >>= sPath;

        // DEBUG ...
        if (sPath.equalsAscii("Filter['Filters']/Filter['my_testfilter_1']"))
            sPath = DECLARE_ASCII("Filter['my_testfilter_1']");
        // DEBUG ...

        sBuffer.append(sPath);

        ::rtl::OUString sName = ::utl::extractFirstFromConfigurationPath(sPath);
        sBuffer.appendAscii("item name = \"");
        sBuffer.append     (sName           );
        sBuffer.appendAscii("\"\n"          );

        sal_Bool bRemoved = (!pElementChanges[i].Element.hasValue() &&  pElementChanges[i].ReplacedElement.hasValue());
        sal_Bool bAdded   = ( pElementChanges[i].Element.hasValue() && !pElementChanges[i].ReplacedElement.hasValue());
        sal_Bool bChanged = ( pElementChanges[i].Element.hasValue() &&  pElementChanges[i].ReplacedElement.hasValue());
        sal_Bool bUnknown = (!pElementChanges[i].Element.hasValue() && !pElementChanges[i].ReplacedElement.hasValue());

        css::uno::Reference< css::container::XHierarchicalNameAccess > xHAccess;
        if (!(rEvent.Base >>= xHAccess) || !xHAccess.is())
        {
            sBuffer.appendAscii("invalid accessor\n");
            continue;
        }

        if (bUnknown)
        {
            bUnknown = sal_False;
            if (xHAccess->hasByHierarchicalName(sPath))
                bAdded = sal_True;
            else
                bRemoved = sal_True;
        }

        EModifyState eState;
        if (bUnknown)
        {
            sBuffer.appendAscii("[UNKNOWN]\n");
            eState = E_UNTOUCHED;
        }
        if (bAdded)
        {
            sBuffer.appendAscii("[ADDED]\n"  );
            eState = E_ADDED;
        }
        if (bChanged)
        {
            sBuffer.appendAscii("[CHANGED]\n");
            eState = E_CHANGED;
        }
        if (bRemoved)
        {
            sBuffer.appendAscii("[REMOVED]\n");
            eState = E_REMOVED;
        }

        if (lItems.find(sName) == lItems.end())
            lItems[sName] = eState;
    }

    for (BaseHash< EModifyState >::const_iterator pIt =lItems.begin();
                                                  pIt!=lItems.end()  ;
                                                ++pIt                )
    {
        ::rtl::OUString sName = pIt->first;
        sBuffer.append(sName);
        switch(pIt->second)
        {
            case E_UNTOUCHED : sBuffer.appendAscii(" = untouched\n"); break;
            case E_ADDED     : sBuffer.appendAscii(" = added\n"    ); break;
            case E_CHANGED   : sBuffer.appendAscii(" = changed\n"  ); break;
            case E_REMOVED   : sBuffer.appendAscii(" = removed\n"  ); break;
        }

        // SAFE {
        WriteGuard aWriteLock(m_aLock);
        css::uno::Reference< css::container::XNameAccess > xAccess(m_xCFG, css::uno::UNO_QUERY);
        if (!xAccess.is())
            continue;

        sal_Bool bExistInCFG = xAccess->hasByName(sName);
        css::uno::Reference< css::uno::XInterface > xItem;
        if (bExistInCFG)
            xAccess->getByName(sName) >>= xItem;
        sal_Bool bModifyCache = sal_False;
        switch(m_eType)
        {
            case E_DETECTOR :
            {
                if (!bExistInCFG && m_pData->existsDetector(sName))
                    m_pData->removeDetector(sName, bModifyCache);
                else
                {
                    Detector aDetector = impl_readDetector(xItem)
                    aDetector.sName = sName;
                    if (bExistInCFG && !m_pData->existsDetector(sName))
                        m_pData->addDetector(aDetector, bModifyCache);
                    else
                        m_pData->replaceDetector(aDetector, bModifyCache);
                }
            }
            break;

            default :
            {
                LOG_WARNING("FilterCFGListener::changesOccured()", "implemented only for detect services!")
            }
            break;
        }
        aWriteLock.unlock();
        // } SAFE
    }

    sBuffer.appendAscii("\n");
    WRITE_LOGFILE("filterchanges.log", U2B(sBuffer.makeStringAndClear()))
}

//_______________________________________________

void SAL_CALL FilterCFGListener::disposing( const css::lang::EventObject& aEvent ) throw(css::uno::RuntimeException)
{
    // SAFE {
    WriteGuard aWriteLock(m_aLock);
    if (aEvent.Source.is() && aEvent.Source == m_xCFG)
    {
        m_xCFG       = css::uno::Reference< css::uno::XInterface >();
        m_bListening = sal_False;
    }
    aWriteLock.unlock();
    // } SAFE
}

//_______________________________________________

FileType FilterCFGListener::impl_readType( const css::uno::Reference< css::uno::XInterface >& xNode )
{
    FileType aItem;
    return aItem;
}

//_______________________________________________

Filter FilterCFGListener::impl_readFilter( const css::uno::Reference< css::uno::XInterface >& xNode )
{
    Filter aItem;
    return aItem;
}

//_______________________________________________

Detector FilterCFGListener::impl_readDetector( const css::uno::Reference< css::uno::XInterface >& xNode )
{
    Detector aItem;
    css::uno::Reference< css::container::XNameAccess > xAccess(xNode, css::uno::UNO_QUERY);
    css::uno::Sequence< ::rtl::OUString > lTypes;
    xAccess->getByName(DECLARE_ASCII("Types")) >>= lSeqTypes;
    aItem.lTypes = Converter::convert_seqOUString2OUStringList(lSeqTypes);
    return aItem;
}

//_______________________________________________

Loader FilterCFGListener::impl_readLoader( const css::uno::Reference< css::uno::XInterface >& xNode )
{
    Loader aItem;
    return aItem;
}

//_______________________________________________

ContentHandler FilterCFGListener::impl_readHandler( const css::uno::Reference< css::uno::XInterface >& xNode )
{
    ContentHandler aItem;
    return aItem;
}
*/
} // namespace framework
