/*
 * Copyright 1993, 1995 Christopher Seiwald.
 *
 * This file is part of Jam - see jam.c for Copyright information.
 */

/*
 * execunix.c - execute a shell script on UNIX/WinNT/OS2/AmigaOS
 *
 * If $(JAMSHELL) is defined, uses that to formulate execvp()/spawnvp().
 * The default is:
 *
 *	/bin/sh -c %		[ on UNIX/AmigaOS ]
 *	cmd.exe /c %		[ on OS2/WinNT ]
 *
 * Each word must be an individual element in a jam variable value.
 *
 * In $(JAMSHELL), % expands to the command string and ! expands to 
 * the slot number (starting at 1) for multiprocess (-j) invocations.
 * If $(JAMSHELL) doesn't include a %, it is tacked on as the last
 * argument.
 *
 * Don't just set JAMSHELL to /bin/sh or cmd.exe - it won't work!
 *
 * External routines:
 *	execcmd() - launch an async command execution
 * 	execwait() - wait and drive at most one execution completion
 *
 * Internal routines:
 *	onintr() - bump intr to note command interruption
 *
 * 04/08/94 (seiwald) - Coherent/386 support added.
 * 05/04/94 (seiwald) - async multiprocess interface
 * 01/22/95 (seiwald) - $(JAMSHELL) support
 * 06/02/97 (gsar)    - full async multiprocess support for Win32
 * 01/20/00 (seiwald) - Upgraded from K&R to ANSI C
 * 11/04/02 (seiwald) - const-ing for string literals
 * 12/27/02 (seiwald) - grist .bat file with pid for system uniqueness
 */

# include "jam.h"
# include "lists.h"
# include "execcmd.h"
# include <errno.h>
# include <stdlib.h>

# ifdef USE_EXECUNIX
# ifdef OS_OS2
# define USE_EXECNT
# include <process.h>
# endif

# ifdef OS_NT 
# define USE_EXECNT
# include <process.h>
# define WIN32_LEAN_AND_MEAN
# include <windows.h>		/* do the ugly deed */
# define USE_MYWAIT
# if !defined( __BORLANDC__ ) 
# define wait my_wait
static int my_wait( int *status );
# endif
# endif

# ifndef USE_EXECNT
# include <unistd.h>
# endif

static int intr = 0;
static int cmdsrunning = 0;
static void (*istat)( int );

static struct
{
        int	pid; /* on win32, a real process handle */
        void	(*func)( void *closure, int status );
	void 	*closure;
} cmdtab[ MAXJOBS ] = {{0}};

/*
 * onintr() - bump intr to note command interruption
 */

void
onintr( int disp )
{
	intr++;
	printf( "...interrupted\n" );
}

/*
 * execcmd() - launch an async command execution
 */

void
execcmd( 
	char *string,
	void (*func)( void *closure, int status ),
	void *closure,
	LIST *shell )
{
	int pid;
	int slot;
	const char *argv[ MAXARGC + 1 ];	/* +1 for NULL */
#ifdef OS_NT
        STARTUPINFO startupInfo;
        PROCESS_INFORMATION processInfo;
        BOOL cpOK;
        char cmdline[ MAXLINE + 100 ]; /* Random padding */
#endif


	char *p;

	/* Find a slot in the running commands table for this one. */

	for( slot = 0; slot < MAXJOBS; slot++ )
	    if( !cmdtab[ slot ].pid )
		break;

	if( slot == MAXJOBS )
	{
	    printf( "no slots for child!\n" );
	    exit( EXITBAD );
	}

	/* Trim leading, ending white space */

	while( isspace( *string ) )
		++string;

	p = strchr( string, '\n' );

	while( p && isspace( *p ) )
		++p;

	/* Forumulate argv */
	/* If shell was defined, be prepared for % and ! subs. */
	/* Otherwise, use stock /bin/sh (on unix) or cmd.exe (on NT). */

	if( shell )
	{
	    int i;
	    char jobno[4];
	    int gotpercent = 0;

	    sprintf( jobno, "%d", slot + 1 );

	    for( i = 0; shell && i < MAXARGC; i++, shell = list_next( shell ) )
	    {
		switch( shell->string[0] )
		{
		case '%':	argv[i] = string; gotpercent++; break;
		case '!':	argv[i] = jobno; break;
		default:	argv[i] = shell->string;
		}
		if( DEBUG_EXECCMD )
		    printf( "argv[%d] = '%s'\n", i, argv[i] );
	    }

	    if( !gotpercent )
		argv[i++] = string;

	    argv[i] = 0;
	}
	else
	{
# ifdef USE_EXECNT
            argv[0] = getenv("ComSpec");
            argv[1] = "/Q/C";		/* anything more is non-portable */

# else
	    argv[0] = "/bin/sh";
	    argv[1] = "-c";
# endif
	    argv[2] = string;
	    argv[3] = 0;
	}


	/* Catch interrupts whenever commands are running. */

	if( !cmdsrunning++ )
	    istat = signal( SIGINT, onintr );

# ifdef OS_NT
        ZeroMemory( &startupInfo, sizeof( startupInfo ) );
        startupInfo.cb = sizeof( STARTUPINFO );

        if( strcmp( argv[0], "NOSHELL" ) == 0 ) {
             /* Ignores ! and % shell substitution variables, which is OK */
             cpOK = CreateProcess( NULL, string, NULL, NULL, FALSE, 0,
                                   NULL, NULL, &startupInfo, &processInfo );
        }
        else {
             int i = 0;

             cmdline[0] = '\0';

             /* O(n^2) but assumes the first few argv's are usually very short */
             /* also issues with buffer overrun */

             strcat( cmdline, argv[i++] );
             while(argv[i]) {
                  strcat( cmdline, " " );
                  strcat( cmdline, argv[i] );
                  i++; 
             }

             cpOK = CreateProcess( NULL, cmdline, NULL, NULL, FALSE, 0,
                                   NULL, NULL, &startupInfo, &processInfo );
        }
        if(!cpOK) {
             perror( "CreateProcess" );
             exit( EXITBAD );
        }
        
        pid = (int)processInfo.hProcess;
# else /* OS2, Unix */

	/* In case the shell is set to "NOSHELL" we run the command directly */
        /* and need to split up the command line into pieces */
        if( strcmp( argv[0], "NOSHELL" ) == 0 ) {
             int i = 0;
             char* arg = string;
             p = strchr( string, ' ' );
             while( p ) {
                  *p++ = '\0';
                  argv[i++] = arg;
                  arg = p;
                  p = strchr( p, ' ' );
             }
             argv[i++] = arg;
             argv[i] = 0;          

	     /* Strip trailing newline */
	     p = strchr( arg, '\n' );
	     *p = '\0';
        }

	/* Start the command */

# ifdef USE_EXECNT
        if( ( pid = spawnvp( P_NOWAIT, argv[0], argv ) ) == -1 )
        {
            perror( "spawn" );
            exit( EXITBAD );
        }
# else
# ifdef NO_VFORK
	if ((pid = fork()) == 0)
   	{
	    execvp( argv[0], (char **) argv );
	    _exit(127);
	}
# else
	if ((pid = vfork()) == 0)
   	{
	    execvp( argv[0], (char **) argv );
	    _exit(127);
	}
# endif /* NO_VFORK */

	if( pid == -1 )
	{
	    perror( "vfork" );
	    exit( EXITBAD );
	}
# endif /* USE_EXECNT */
# endif /* OS_NT */

	/* Save the operation for execwait() to find. */

	cmdtab[ slot ].pid = pid;
	cmdtab[ slot ].func = func;
	cmdtab[ slot ].closure = closure;

	/* Wait until we're under the limit of concurrent commands. */
	/* Don't trust globs.jobs alone. */

	while( cmdsrunning >= MAXJOBS || cmdsrunning >= globs.jobs )
	    if( !execwait() )
		break;
}

/*
 * execwait() - wait and drive at most one execution completion
 */

int
execwait()
{
	int i;
	int status, w;
	int rstat;

	/* Handle naive make1() which doesn't know if cmds are running. */

	if( !cmdsrunning )
	    return 0;

	/* Pick up process pid and status */
    
	while( ( w = wait( &status ) ) == -1 && errno == EINTR )
		;

	if( w == -1 )
	{
	    printf( "child process(es) lost!\n" );
	    perror("wait");
	    exit( EXITBAD );
	}

	/* Find the process in the cmdtab. */

	for( i = 0; i < MAXJOBS; i++ )
	    if( w == cmdtab[ i ].pid )
		break;

	if( i == MAXJOBS )
	{
	    printf( "waif child found!\n" );
	    exit( EXITBAD );
	}

	/* Drive the completion */

	if( !--cmdsrunning )
	    signal( SIGINT, istat );

	if( intr )
	    rstat = EXEC_CMD_INTR;
	else if( w == -1 || status != 0 )
	    rstat = EXEC_CMD_FAIL;
	else
	    rstat = EXEC_CMD_OK;

	cmdtab[ i ].pid = 0;

	(*cmdtab[ i ].func)( cmdtab[ i ].closure, rstat );

	return 1;
}

# ifdef USE_MYWAIT

static int
my_wait( int *status )
{
	int i, num_active = 0;
	DWORD exitcode, waitcode;
	static HANDLE *active_handles = 0;

	if (!active_handles)
	    active_handles = (HANDLE *)malloc(globs.jobs * sizeof(HANDLE) );

	/* first see if any non-waited-for processes are dead,
	 * and return if so.
	 */
	for ( i = 0; i < globs.jobs; i++ ) {
	    if ( cmdtab[i].pid ) {
		if ( GetExitCodeProcess((HANDLE)cmdtab[i].pid, &exitcode) ) {
		    if ( exitcode == STILL_ACTIVE )
			active_handles[num_active++] = (HANDLE)cmdtab[i].pid;
		    else {
			CloseHandle((HANDLE)cmdtab[i].pid);
			*status = (int)((exitcode & 0xff) << 8);
			return cmdtab[i].pid;
		    }
		}
		else
		    goto FAILED;
	    }
	}

	/* if a child exists, wait for it to die */
	if ( !num_active ) {
	    errno = ECHILD;
	    return -1;
	}
	waitcode = WaitForMultipleObjects( num_active,
					   active_handles,
					   FALSE,
					   INFINITE );
	if ( waitcode != WAIT_FAILED ) {
	    if ( waitcode >= WAIT_ABANDONED_0
		&& waitcode < WAIT_ABANDONED_0 + num_active )
		i = waitcode - WAIT_ABANDONED_0;
	    else
		i = waitcode - WAIT_OBJECT_0;
	    if ( GetExitCodeProcess(active_handles[i], &exitcode) ) {
		CloseHandle(active_handles[i]);
		*status = (int)((exitcode & 0xff) << 8);
		return (int)active_handles[i];
	    }
	}

FAILED:
	errno = GetLastError();
	return -1;
    
}

# endif /* USE_MYWAIT */

# endif /* USE_EXECUNIX */
