/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: BitBuffer.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 16:43:19 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
/*
  BitBuffer's client has to stick to a usage protocol:
  
  - start with new or cleared BitBuffer
  - do appends for a while
  - finish w/ close()
  
  */

package com.sun.xmlsearch.util;
import java.io.*;

public final class BitBuffer {
  private static final int InitSize   = 256;
  private static final int NBits      =  32;
  private static final int BitsInByte =   8;
  private static final int BytesInInt =   4;
  
  private int   _avail = NBits;
  private int   _word = 0;
  private int   _free = 0;
  private int   _size = InitSize;
  private int[] _array = new int[InitSize];

  public void close() {
    if (_avail < NBits)
      store(_word << _avail);
    else
      _avail = 0;
  }
  
  public void write(DataOutput out) throws IOException {
    for (int i = 0; i < _free - 1; i++)
      out.writeInt(_array[i]);
    final int word = _array[_free - 1];
    int bytes = BytesInInt - _avail/BitsInByte;
    int shift = NBits;
    while (bytes-- > 0)
      out.writeByte((word >>> (shift -= BitsInByte)) & 0xFF);
  }

  public void clear() {
    _word   = 0;
    _avail  = NBits;
    _free   = 0;
  }
  
  public int byteCount() {
    return _free*BytesInInt - _avail/BitsInByte;
  }
  
  public int bitCount() {
    return _free*NBits - _avail;
  }

  public void setFrom(BitBuffer rhs) {
    _word  = rhs._word;
    _avail = rhs._avail;
    if ((_free = rhs._free) > _size)
      _array = new int[_size = rhs._free];
    System.arraycopy(rhs._array, 0, _array, 0, _free);
  }
  
  private void growArray(int newSize) {
    int[] newArray = new int[_size = newSize];
    System.arraycopy(_array, 0, newArray, 0, _free);
    _array = newArray;
  }

  private void store(int value) {
    if (_free == _size)
      growArray(_size * 2);
    _array[_free++] = value;
  }

  public void append(int bit) {
    _word = (_word << 1) | bit;
    if (--_avail == 0) {
      store(_word);
      _word = 0;
      _avail = NBits;
    }
  }
  
  public void append(int source, int kBits) {
    if (kBits < _avail) {
      _word = (_word << kBits) | source;
      _avail -= kBits;
    } else if (kBits > _avail) {
      int leftover = kBits - _avail;
      store((_word << _avail) | (source >>> leftover));
      _word = source;
      _avail = NBits - leftover;
    } else {
      store((_word << kBits) | source);
      _word = 0;
      _avail = NBits;
    }
  }
  
  public void concatenate(BitBuffer bb) {
    if (_size - _free < bb._free)
      growArray(_free + bb._free + 1);

    if (_avail == 0) {
      System.arraycopy(bb._array, 0, _array, _free, bb._free);
      _avail = bb._avail;
      _free += bb._free;
    } else {
      int tp = _free - 1;	// target
      int sp = 0;		// source
      do {
	_array[tp] |= bb._array[sp] >>> (NBits - _avail);
	_array[++tp] = bb._array[sp++] << _avail;
      }
      while (sp < bb._free);
      _free += bb._free;
      if ((_avail += bb._avail) >= NBits) {
	_avail -= NBits;
	_free--;
      }
    }
  }
}
