/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: macro.cxx,v $
 *
 *  $Revision: 1.13 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 16:02:43 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "macro.hxx"

#ifndef _OSL_PROFILE_H_
#include <osl/profile.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_BOOTSTRAP_HXX_
#include <rtl/bootstrap.hxx>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif
#ifndef _RTL_STRING_HXX_
#include <rtl/string.hxx>
#endif
#ifndef INCLUDED_RTL_INSTANCE_HXX
#include <rtl/instance.hxx>
#endif

#ifndef INCLUDED_SAL_INTERNAL_ALLOCATOR_HXX
#include "internal/allocator.hxx"
#endif

#ifndef INCLUDED_FUNCTIONAL
#include <functional>
#define INCLUDED_FUNCTIONAL
#endif

#include <hash_set>

using rtl::OUString;

//----------------------------------------------------------------------------
// global set to detect recursion during macro expansion
//----------------------------------------------------------------------------

typedef std::hash_set<
    OUString,
	rtl::OUStringHash ,
	std::equal_to<OUString>,
	sal::Allocator<OUString>
> OUString_hash_set;

namespace { struct lookups: public rtl::Static< OUString_hash_set, lookups > {}; }

//----------------------------------------------------------------------------

struct ArgParserState
{
	rtl::OUStringBuffer _result; // accumulate the result here
	rtl::OUStringBuffer _term;   // the active chunk of the to parsing string
	rtl::OUString _first_part;   // first part of an ini-reference
	rtl::OUString _second_part;  // second part of an ini-reference

	sal_Bool _lookUp; // determines if we have to lookup _term during expand
	rtlBootstrapHandle _handle;

	// expand _term and add it to _result
	void expandTerm(sal_Bool nextLookUp);
};

//----------------------------------------------------------------------------

void ArgParserState::expandTerm(sal_Bool nextLookUp) 
{
	if(_first_part.getLength()) // look up ini?
	{
		// get file name
		OUString file    = expandMacros(_handle, _first_part);

		// get section name
		OUString section = expandMacros(_handle, _second_part);

		// get key name
		OUString key     = expandMacros(_handle, _term.makeStringAndClear());
		
		// clear strings
		_first_part = _term.makeStringAndClear();
		_second_part = _first_part;

		// do we have a section?
		OUString value;
		if (section.getLength())
		{
			// expand from profile
			oslProfile hProfile;

			hProfile = osl_openProfile (file.pData, osl_Profile_DEFAULT);
			if (hProfile)
			{
				sal_Char pBuffer[1024];

				if (osl_readProfileString (
					hProfile,
					rtl::OString (
						section.getStr(), section.getLength(),
						RTL_TEXTENCODING_ASCII_US),
					rtl::OString (
						key.getStr(), key.getLength(),
						RTL_TEXTENCODING_ASCII_US),
					pBuffer, sizeof(pBuffer), ""))
				{
					value = OUString (
						pBuffer, rtl_str_getLength (pBuffer),
						RTL_TEXTENCODING_ASCII_US);
				}

				osl_closeProfile (hProfile);
			}
		}
		else
		{
			// expand from ini:
            ::rtl::OUStringBuffer buf;
            // to reuse the __lookups set and to avoid collisions with
            // variable lookups, a '=' is prepended:
            buf.append( static_cast<sal_Unicode>('=') );
            buf.append( file );
            // separating url, from key:
            buf.append( static_cast<sal_Unicode>('=') );
            buf.append( key );
            OUString name( buf.makeStringAndClear() );
    		OUString_hash_set &rLookups = lookups::get();
            ::std::pair<OUString_hash_set::iterator, bool> insertion(
                rLookups.insert(name) );
            if (insertion.second)
            {
                rtl::Bootstrap bootstrap(file);
                bootstrap.getFrom(key, value);
                rLookups.erase(name);
            } // else: break recursion of file lookup
        }

		_result.append(value);
	}
	else if(_lookUp) // variable look up?
	{
		OUString name = _term.makeStringAndClear();
		OUString value;

		OUString_hash_set &rLookups = lookups::get();
		OUString_hash_set::const_iterator it = rLookups.find(name);
		if (it != rLookups.end())
		{
			// found, so it must be a recursion
			value = OUString (
				RTL_CONSTASCII_USTRINGPARAM("***RECURSION DETECTED***"));
		}
		else
		{
			rLookups.insert(name);

			rtl_bootstrap_get_from_handle (
				_handle, name.pData, &value.pData, NULL);

			rLookups.erase(name);
		}

		_result.append(value);
	}
	else // no look up at all
		_result.append(_term.makeStringAndClear());

	_lookUp = nextLookUp;
}

//----------------------------------------------------------------------------

rtl::OUString SAL_CALL expandMacros (
	rtlBootstrapHandle handle, const rtl::OUString & argstr)
{
	ArgParserState argParserState;

	sal_Int32 length = argstr.getLength();
	const sal_Unicode * ustr = argstr.getStr();

	argParserState._lookUp = sal_False;
	argParserState._handle = handle;
	sal_Bool quote = sal_False;

	sal_Int32 i = 0;
	sal_Int32 deepness;

	while(i < length) 
	{
		if(quote) // quote next char?
		{
			if(ustr[i] == 'u') // a unicode character?
			{
				sal_Unicode code = 0;

				for(sal_Int32 j = 3; j >= 0 && i < length; -- j)
				{
					++ i;

					if(ustr[i] >= 'a' && ustr[i] <= 'z')
						code += (ustr[i] - 'a' + 10) << (j * 4);

					else if(ustr[i] >= 'A' && ustr[i] <= 'Z')
						code += (ustr[i] - 'A' + 10) << (j * 4);

					else if(ustr[i] >= '0' && ustr[i] <= '9')
						code += (ustr[i] - '0') << (j * 4);
				}

				argParserState._term.append(&code, 1);
			}
			else
				argParserState._term.append(ustr + i, 1);

			quote = sal_False;
		}
		else 
		{
			switch(ustr[i])
			{
			case '$': // expand via look up
 				argParserState.expandTerm(sal_True);
				break;
					
			case '\\': // quote
				quote = sal_True;
				break;
				
			case '{': // brackets
				deepness = 1;
				++ i;
				while(i < length && deepness > 0)
				{
					if(ustr[i] == ':' && deepness == 1)
					{
						if(!argParserState._first_part.getLength())
							argParserState._first_part = argParserState._term.makeStringAndClear();

						else if(!argParserState._second_part.getLength())
							argParserState._second_part = argParserState._term.makeStringAndClear();
					}
					else if(ustr[i] == '}')
					{
						-- deepness;
						if(deepness)
							argParserState._term.append(ustr + i, 1);
					}
					else if(ustr[i] == '{')
					{
						++ deepness;
						argParserState._term.append(ustr + i, 1);
					}
					else
						argParserState._term.append(ustr + i, 1);

					if(deepness)
						++ i;
				}
				argParserState.expandTerm(sal_False);
				
				break;
				
			case '-': // delimeter
			case ';':
			case '/':
			case ' ':
				argParserState.expandTerm(sal_False);
					
				argParserState._term.append(ustr + i, 1);
				break;

			default: // just add
				argParserState._term.append(ustr + i, 1);
			}
		}

		++ i;
	}

	argParserState.expandTerm(sal_False);

	return argParserState._result.makeStringAndClear();
}

