/*************************************************************************
 *
 *  $RCSfile: CDROutputStream.java,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 15:27:53 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

package com.sun.star.lib.uno.protocols.iiop;


import java.io.ByteArrayOutputStream;
import java.io.IOException;


public class CDROutputStream {
	public static boolean DEBUG = false;

    static final int DEFAULT_BUFFER_SIZE = 1024;
    protected boolean littleEndian;
    protected int size;
    protected byte[] buf;

    public CDROutputStream() {
		this(false);
    }

    public CDROutputStream(boolean littleEndian) {
		this(littleEndian, DEFAULT_BUFFER_SIZE);
    }

    public CDROutputStream(boolean littleEndian, int size) {
		this.littleEndian = littleEndian;
		this.buf = new byte[size];
		this.size = 0;
    }

    private final void alignAndReserve(int align, int n) {
		int cnt;

		cnt = ((size - 1) + align) & (~(align - 1));

		size = cnt;
		if (cnt + n < buf.length) {
			return;
		}

		grow(align, n);
    }

    //
    // Default implementation of grow.  Subclassers may override this.
    // Always grow the single buffer. This needs to delegate
    // fragmentation policy for IIOP 1.1.
    //
    protected void grow(int align, int n) {
		int new_size = (int)Math.pow(2, Math.ceil(Math.log((buf == null ? 0 : buf.length) + n) / Math.log(2)));

		byte[] old = buf;
		buf = new byte[new_size];
		if(old != null)
			System.arraycopy(old, 0, buf, 0, old.length);
    }

    public final void write_boolean(boolean x)
    {
		alignAndReserve(1, 1);

		buf[size++] = x? (byte)1:(byte)0;
    }

    public final void write_ascii(char x)
    {
		alignAndReserve(1, 1);

		buf[size++] = (byte)((x >>> 0) & 0xFF);
    }

    public final void write_unicode(char value)
    {
		write_short((short)value);
    }

    public final void write_octet(byte x)
    {
		alignAndReserve(1, 1);

		if(DEBUG) System.err.println("#### CDROutputStream.write_octet:" + size + " " + x);

		buf[size++] = x;
    }

    public final void write_short(short x)
    {
		alignAndReserve(2, 2);

		if (littleEndian) {
			buf[size++] = (byte)((x >>> 0) & 0xFF);
			buf[size++] = (byte)((x >>> 8) & 0xFF);
		} else {
			buf[size++] = (byte)((x >>> 8) & 0xFF);
			buf[size++] = (byte)((x >>> 0) & 0xFF);
		}
    }

    public final void write_long(int x)
    {
		if(DEBUG) System.err.println("#### CDROutputStream.write_long:" + size + " " + x);

		alignAndReserve(4, 4);

		if (littleEndian) {
			buf[size++] = (byte)((x >>> 0) & 0xFF);
			buf[size++] = (byte)((x >>> 8) & 0xFF);
			buf[size++] = (byte)((x >>> 16) & 0xFF);
			buf[size++] = (byte)((x >>> 24) & 0xFF);
		} else {
			buf[size++] = (byte)((x >>> 24) & 0xFF);
			buf[size++] = (byte)((x >>> 16) & 0xFF);
			buf[size++] = (byte)((x >>> 8) & 0xFF);
			buf[size++] = (byte)((x >>> 0) & 0xFF);
		}
    }

    public final void write_longlong(long x)
    {
		alignAndReserve(8, 8);

		if (littleEndian) {
			buf[size++] = (byte)((x >>> 0) & 0xFF);
			buf[size++] = (byte)((x >>> 8) & 0xFF);
			buf[size++] = (byte)((x >>> 16) & 0xFF);
			buf[size++] = (byte)((x >>> 24) & 0xFF);
			buf[size++] = (byte)((x >>> 32) & 0xFF);
			buf[size++] = (byte)((x >>> 40) & 0xFF);
			buf[size++] = (byte)((x >>> 48) & 0xFF);
			buf[size++] = (byte)((x >>> 56) & 0xFF);
		} else {
			buf[size++] = (byte)((x >>> 56) & 0xFF);
			buf[size++] = (byte)((x >>> 48) & 0xFF);
			buf[size++] = (byte)((x >>> 40) & 0xFF);
			buf[size++] = (byte)((x >>> 32) & 0xFF);
			buf[size++] = (byte)((x >>> 24) & 0xFF);
			buf[size++] = (byte)((x >>> 16) & 0xFF);
			buf[size++] = (byte)((x >>> 8) & 0xFF);
			buf[size++] = (byte)((x >>> 0) & 0xFF);
		}
    }

    public final void write_float(float x)
    {
		write_long(Float.floatToIntBits(x));
    }

    public final void write_double(double x)
    {
		write_longlong(Double.doubleToLongBits(x));
    }

    public final void write_asciistring(String x) {
		if (DEBUG) System.err.println("CDROutputStream.write_asciistring:" + x);

		int len = x == null ? 0 : x.length();
		write_long(len + 1);
		for (int i=0; i < len; i++) {
			write_ascii(x.charAt(i));
		}

		//
		// Send the string null termination byte.
		//
		write_ascii((char)0);
    }

    public final void write_unicodestring(String value)
    {
		if(DEBUG) System.err.println("#### CDROutputStream.write_unicodestring:"+ size + " " + value);

		int len = value == null ? 0 : value.length();
		write_long(len + 1);
		for (int i=0; i < len; i++) {
			write_unicode(value.charAt(i));
		}

		//
		// Send the string null termination byte.
		//
		write_unicode((char)0);
    }

    public final void write_octet_array(byte b[] )
    {
		if(DEBUG) System.err.println("#### CDROutputStream.write_octet_array:" + size + " " + b.length);

        write_long( b.length );
		alignAndReserve(1, b.length);
		System.arraycopy(b, 0, buf, size, b.length);
		size += b.length;
    }

    public final void write_boolean_array(boolean[]value ) 
	{
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_boolean(value[i+0]);
		}
    }

    public final void write_ascii_array(char[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_ascii(value[i+0]);
		}
    }

    public final void write_unicode_array(char[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_unicode(value[i+0]);
		}
    }

    public final void write_short_array(short[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_short(value[i+0]);
		}
    }

    public final void write_long_array(int[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_long(value[i+0]);
		}
    }

    public final void write_longlong_array(long[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_longlong(value[i+0]);
		}
    }

    public final void write_float_array(float[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_float(value[i+0]);
		}
    }

    public final void write_double_array(double[]value ) {
        write_long( value.length );
		for(int i=0; i < value.length; i++) {
			write_double(value[i+0]);
		}
    }

    //--------------------------------------------------------------------//
    // CDROutputStream state management.
    //

    public byte []reset() {
		byte bytes[] = buf;

		this.littleEndian = littleEndian;
		this.buf = new byte[DEFAULT_BUFFER_SIZE];
		this.size = 0;

		return bytes;
    }

    /**
     * Write the contents of the CDROutputStream to the specified output stream.
     * @param s The output stream to write to.
     */
    public void writeTo(java.io.OutputStream s) throws java.io.IOException {
		s.write(buf, 0, size);
    }

    public final int getSize() {
		return size;
    }

    public final byte[] getByteArray() {
		return buf;
    }

}

