# Copyright (c) 2004 Divmod.
# See LICENSE for details.

"""Builtin template loaders that supply a Page or renderer with a
document to render.

Nevow provides the following DocFactory loaders by default:

  - B{stan} - turn a stan tag tree into a DocFactory
  - B{xmlfile} - load a well formed XML document from file
  - B{htmlfile} - load a HTML file from disk
  - B{xmlstr} - load a well formed XML document from a string
  - B{htmlstr} - load a HTML document from a string

Unless absolutely necessary you should use either the stan loader or
one of the xml loaders. The html loaders should only be used for badly
formed HTML documents, i.e. if your HTML developer hasn't heard of
XHTML yet. Even then, you should probably try to educate the HTML
developer first ;-).
"""

import os.path

from nevow import context
from nevow import inevow
from nevow import tags


class DocFactory:
    """Base class for document factories. I am responsible for loading a document
    template for the renderer to use.
    """

    __implements__ = inevow.IDocFactory,
    
    pattern = None
    precompiledDoc = None

    def __init__(self, pattern=None):
        if pattern:
            self.pattern = pattern

    def precompile(self):
        ctx = context.WovenContext(precompile=True)
        from nevow import flat
        doc = flat.precompile(self.getDoc(), ctx)
        if self.pattern:
            tag = tags.invisible[doc]
            doc = [tag.onePattern(self.pattern)]
        return doc

    def load(self):
        """Load and return the document for the renderer"""
        if self.precompiledDoc is None:
            self.precompiledDoc = self.precompile()
        return self.precompiledDoc


# Caches for the document factories
_stanCache = {}
_htmlstrCache = {}
_htmlfileCache = {}
_xmlstrCache = {}
_xmlfileCache = {}


class stan(DocFactory):
    """A stan tags document factory"""

    def __init__(self, doc, **kw):
        self.doc = doc
        DocFactory.__init__(self, **kw)

    def getDoc(self):
        return self.doc


class htmlstr(DocFactory):
    """A document factory for HTML contained in a string"""

    beExtremelyLenient = True

    def __init__(self, template, beExtremelyLenient=None, **kw):
        self.template = template
        if beExtremelyLenient:
            self.beExtremelyLenient = beExtremelyLenient
        DocFactory.__init__(self, **kw)

    def getDoc(self):
        from twisted.web import microdom
        return microdom.parseString(
            self.template,
            beExtremelyLenient=self.beExtremelyLenient
            )

    def precompile(self):
        doc = _htmlstrCache.get((self.template, self.pattern), None)
        if doc:
            return doc
        doc = DocFactory.precompile(self)
        _htmlstrCache[(self.template, self.pattern)] = doc
        return doc


class htmlfile(DocFactory):
    """A document factory for an HTML disk template"""

    template = None
    templateDir = ''
    beExtremelyLenient = True
    _mtime = None

    def __init__(self, template, templateDir=None, beExtremelyLenient=None, **kw):
        self.template = template
        if templateDir is not None:
            self.templateDir = templateDir
        if beExtremelyLenient is not None:
            self.beExtremelyLenient = beExtremelyLenient
        self._filename = os.path.join(self.templateDir, self.template)
        DocFactory.__init__(self, **kw)

    def getDoc(self):
        from twisted.web import microdom
        return microdom.parse(
            self._filename,
            beExtremelyLenient=self.beExtremelyLenient
            )

    def precompile(self):
        mtime, doc = _htmlfileCache.get((self._filename, self.pattern), (None,None))
        self._mtime = os.path.getmtime(self._filename)
        if mtime == self._mtime:
            return doc
        doc = DocFactory.precompile(self)
        _htmlfileCache[(self._filename, self.pattern)] = self._mtime, doc
        return doc
        
    def load(self):
        if self._mtime != os.path.getmtime(self._filename):
            self.precompiledDoc = None
        return DocFactory.load(self)


class xmlstr(DocFactory):

    doc = None

    def __init__(self, template, ignoreDocType=False, ignoreComment=False, **kw):
        self.template = template
        self.ignoreDocType = ignoreDocType
        self.ignoreComment = ignoreComment
        DocFactory.__init__(self, **kw)

    def getDoc(self):
        from nevow.flat import flatsax
        return flatsax.parseString(self.template, self.ignoreDocType, self.ignoreComment)

    def precompile(self):
        doc = _xmlstrCache.get((self.template, self.pattern), None)
        if doc:
            return doc
        doc = DocFactory.precompile(self)
        _xmlstrCache[(self.template, self.pattern)] = doc
        return doc
        

class xmlfile(DocFactory):

    template = None
    templateDir = ''

    _mtime = None

    def __init__(self, template, templateDir=None, ignoreDocType=False, ignoreComment=False, **kw):
        self.template = template
        if templateDir:
            self.templateDir = templateDir
        self._filename = os.path.join(self.templateDir, self.template)
        self.ignoreDocType = ignoreDocType
        self.ignoreComment = ignoreComment
        DocFactory.__init__(self, **kw)

    def getDoc(self):
        from nevow.flat import flatsax
        return flatsax.parse(file(self._filename), self.ignoreDocType, self.ignoreComment)
    
    def precompile(self):
        mtime, doc = _xmlfileCache.get((self._filename, self.pattern), (None,None))
        self._mtime = os.path.getmtime(self._filename)
        if mtime == self._mtime:
            return doc
        doc = DocFactory.precompile(self)
        _xmlfileCache[(self._filename, self.pattern)] = self._mtime, doc
        return doc
        
    def load(self):
        if self._mtime != os.path.getmtime(self._filename):
            self.precompiledDoc = None
        return DocFactory.load(self)
