from cStringIO import StringIO
import time

from twisted.python import htmlizer
from twisted.web import microdom
from twisted.web import static

from nevow import inevow
from nevow import loaders
from nevow import rend
from nevow.tags import *
from nevow import url

from formless import annotate
from formless import iformless
from formless import webform

from pastebin import pasting

ANONYMOUS = 'anonymous'


##
# Text colourisers (aka syntax highlighting)
##

def _python_colouriser(text):
    out = StringIO()
    htmlizer.filter(StringIO(text), out)
    return out.getvalue()

_colourisers = {
    'python': _python_colouriser
    }


##
# Formless
##

class IAddPasting(annotate.TypedInterface):
    def addPasting(
        self,
        request=annotate.Request(),
        author=annotate.String(strip=True),
        text=annotate.Text(strip=True, required=True)):
        pass
    addPasting = annotate.autocallable(addPasting)


class IEditPasting(annotate.TypedInterface):
    def editPasting(
        self,
        request=annotate.Request(),
        postedBy=annotate.String(immutable=1),
        author=annotate.String(strip=True),
        text=annotate.Text(strip=True, required=True)):
        pass
    editPasting = annotate.autocallable(editPasting)


##
# "Standard" renderers
##

def render_time(theTime):
    def _(context, data):
        return time.strftime('%Y-%m-%d %H:%M:%S %Z', theTime)
    return _

def render_pastingText(text):
    def _(context, data):
        colouriser = _colourisers.get('python')
        if colouriser:
            return xml(colouriser(text))
        return pre[xml(text)]
    return _

def render_pasting(version):
    def _(context, data):
        context.fillSlots('author', version.getAuthor() or ANONYMOUS)
        time = context.fillSlots('time', render_time(version.getTime()))
        text = context.fillSlots('text', render_pastingText(version.getText()))
        return context.tag
    return _


class BasePage(rend.Page):

    docFactory = loaders.htmlfile(templateDir='templates', template='site.html')

    child_css = static.File('static/css')
    child_images = static.File('static/images')

    def data_pastings(self, context, data):
        return self.pastebin.getListOfPastings(20)

    def render_pasting(self, context, data):
        oid, author, time = data
        link = context.fillSlots('link', a(href=url.root.child(str(oid)))['%s (%s)' % (oid, author or ANONYMOUS)])
        return context.tag

    def render_content(self, context, data):
        tag = context.tag.clear()
        tag[loaders.htmlfile(templateDir='templates', template=self.contentTemplateFile)]
        return tag
    

class RootPage(BasePage):

    __implements__ = BasePage.__implements__, IAddPasting

    def __init__(self, pastebin):
        BasePage.__init__(self)
        self.pastebin = pastebin

    def locateChild(self, request, segments):
        try:
            return Pasting(self.pastebin, int(segments[0])), segments[1:]
        except ValueError:
            pass
        return BasePage.locateChild(self, request, segments)

    def render_content(self, context, data):
        tag = context.tag.clear()
        return tag[webform.renderForms()]

    def addPasting(self, request, author, text):
        oid = self.pastebin.addPasting(author, text)
        request.setComponent(iformless.IRedirectAfterPost, '/'+str(oid))
        

class Pasting(BasePage):

    __implements__ = BasePage.__implements__, IEditPasting
    contentTemplateFile = 'pasting.html'

    def __init__(self, pastebin, pastingOid, version=-1):
        BasePage.__init__(self)
        self.pastebin = pastebin
        self.pastingOid = pastingOid
        self.version = version
        self.pasting = self.pastebin.getPasting(self.pastingOid)

    def locateChild(self, request, segments):
        try:
            return Pasting(self.pastebin, self.pastingOid, int(segments[0])), segments[1:]
        except:
            pass
        return BasePage.locateChild(self, request, segments)

    def data_history(self, context, data):
        return self.pasting.getHistory()

    def render_aPasting(self, context, data):
        return render_pasting(self.pasting.getVersion(self.version))

    def render_form(self, context, data):
        if self.version != -1:
            return ''
        version = self.pasting.getVersion(self.version)
        formDefaults = context.locate(iformless.IFormDefaults)
        formDefaults.setDefault('editPasting.text', version.getText())
        formDefaults.setDefault('editPasting.postedBy', version.getAuthor())
        return webform.renderForms()

    def render_version(self, context, data):
        version, author, theTime = data
        if self.version != -1:
            u = url.here.sibling
        else:
            u = url.here.child
        context.fillSlots('link', a(href=[u(version)])[
            render_time(theTime), ' (',author or ANONYMOUS,')'
            ])
        return context.tag

    def editPasting(self, request, postedBy, author, text):
        self.pastebin.updatePasting(self.pastingOid, author, text)
        request.setComponent(iformless.IRedirectAfterPost, '/%s'%self.pastingOid)


class Version(BasePage):

    contentTemplateFile = "pasting.html"

    child_ = rend.FourOhFour()

    def __init__(self, pastebin, pasting, version):
        BasePage.__init__(self)
        self.pastebin = pastebin
        self.pasting = pasting
        self.version = version
        
    def data_history(self, context, data):
        return self.pasting.getHistory()

    def render_aPasting(self, context, data):
        return render_pasting(self.pasting.getVersion(self.version))

    def render_version(self, context, data):
        version, author, theTime = data
        context.fillSlots('link', a(href=[url.here.sibling(str(version))])[
            render_time(theTime), ' (',author,')'
            ])
        return context.tag
    
