/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n.h>

#include <gio/gio.h>
#include <glib/gstdio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libwnck/libwnck.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnome/gnome-desktop-item.h>

#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutter/x11/clutter-x11.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <GL/gl.h>
#include <GL/glx.h>

#include <libnotify/notify.h>
#include <launcher/launcher.h>

#include "nl-app.h"
#include "nl-app-loader.h"
#include "nl-favorite-item.h"
#include "nl-folders-source.h"
#include "nl-menu-items.h"
#include "nl-plugin-manager.h"
#include "nl-uri-loader.h"
#include "nl-volumes-source.h"
#include "nl-window.h"

#define APP_NAME "Netbook Launcher"
#define EXIT_BUSY 16

/* Forwards */
static void              check_required_files_and_folders (void);
static gboolean          check_old_config_before_running  (void);
static void              default_signal_handler           (gint sig);
static void              migrate_favorites                (void);
static DBusGProxy      * connect_to_session               (void);
static DBusGProxy      * setup_boot_curtain               (void);
static gboolean          notify_boot_curtain              (DBusGProxy *proxy);

/* I know, globals */
gboolean      windowed       = FALSE;
gboolean      show_favorites = FALSE;
gint          window_width   = 1024;
gint          window_height  = 576;
gboolean      norestart      = FALSE;
static gchar *add_favorite   = NULL;

static gboolean check_run = FALSE;

static GOptionEntry entries[] =
{
  {
    "windowed",
    'w', 0,
    G_OPTION_ARG_NONE,
    &windowed,
    "Launch in windowed mode (for testing - 1024x600)",
    NULL
  },
  {
    "width",
    'd', 0,
    G_OPTION_ARG_INT,
    &window_width,
    "width",
    NULL
  },
  {
    "height",
    'h', 0,
    G_OPTION_ARG_INT,
    &window_height,
    "height",
    NULL
  },
  {
    "add-favorite",
    'a', 0,
    G_OPTION_ARG_STRING,
    &add_favorite,
    "Path of favorite to add"
  },
  {
    "show-favorites",
    's', 0,
    G_OPTION_ARG_NONE,
    &show_favorites,
    "Show favorites menu"
  },
  {
    "no-restart",
    'r', 0,
    G_OPTION_ARG_NONE,
    &norestart,
    "Do not restart on VT changes",
    NULL
  },
  {
    "check-run",
    'c', 0,
    G_OPTION_ARG_NONE,
    &check_run,
    "This option is available to facilitate the upgrade from ume-nl to "
    "netbook-launcher. If enabled, it will check the existance of the "
    "ume-nl.desktop autostart suppression file in ~/.config/autostart "
    "prior to starting, and replace it with one for netbook-launcher, so "
    "launchers that upgraded while in Classic mode do not have a nl popup.",
    NULL
  },
  {
    NULL
  }
};

static void
on_session_quit (void)
{
  g_debug ("Quitting nicely");
  gtk_widget_destroy (nl_window_get_default ());
  gtk_main_quit ();
}

/* take the optimistic choice: only return FALSE when we are sure it's FALSE */
static gboolean
rendering_available (void)
{
  char *displayName = NULL;
  gchar *renderer = NULL;
  Display *dpy;
  int scrnum = 0;
  Bool allowDirect = True;
  Window win;
  XSetWindowAttributes attr;
  unsigned long mask;
  Window root;
  GLXContext ctx = NULL;
  XVisualInfo *visinfo;
  int width = 100, height = 100;
  int attribSingle[] = {
     GLX_RGBA,
     GLX_RED_SIZE, 1,
     GLX_GREEN_SIZE, 1,
     GLX_BLUE_SIZE, 1,
     None };
  int attribDouble[] = {
     GLX_RGBA,
     GLX_RED_SIZE, 1,
     GLX_GREEN_SIZE, 1,
     GLX_BLUE_SIZE, 1,
     GLX_DOUBLEBUFFER,
     None };

  dpy = XOpenDisplay(displayName); 
  if (!dpy)
   {
    fprintf(stderr, "Error: unable to open display %s\n", XDisplayName(displayName));
    return TRUE;
   }
  root = RootWindow(dpy, scrnum);
  visinfo = glXChooseVisual(dpy, scrnum, attribSingle);
  if (!visinfo)
     visinfo = glXChooseVisual(dpy, scrnum, attribDouble);
  if (visinfo)
     ctx = glXCreateContext( dpy, visinfo, NULL, allowDirect );
  if (!visinfo)
   {
      fprintf(stderr, "Error: couldn't find RGB GLX visual or fbconfig\n");
      return TRUE;
    }
   if (!ctx)
    {
      fprintf(stderr, "Error: glXCreateContext failed\n");
      XFree(visinfo);
      return TRUE;
    }

   attr.background_pixel = 0;
   attr.border_pixel = 0;
   attr.colormap = XCreateColormap(dpy, root, visinfo->visual, AllocNone);
   attr.event_mask = StructureNotifyMask | ExposureMask;
   mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;

   win = XCreateWindow(dpy, root, 0, 0, width, height,
                       0, visinfo->depth, InputOutput,
                       visinfo->visual, mask, &attr);

   if (glXMakeCurrent(dpy, win, ctx))
    {
      const char *glRenderer = (const char *) glGetString(GL_RENDERER);
      renderer = g_ascii_strdown (glRenderer, -1);
      g_debug ("OpenGL renderer string: %s\n", glRenderer);
    }

    if (renderer && strstr(renderer, "software"))
       return FALSE;
    return TRUE;
}

static gint
fallback_to_efl (void)
{
  GError *error = NULL;
  
  if (!g_spawn_command_line_async ("/usr/bin/netbook-launcher-efl", &error))
    {
      g_print ("Error in netbook-launcher-efl: %s\n", error->message);
      g_error_free (error);
      return EXIT_FAILURE;
    }
  return 0;
}

gint
main (gint argc, gchar *argv[])
{
  GOptionContext        *context;
  GError                *error = NULL;
  NlApp                 *app;
  GtkWidget             *window;
  LauncherFavorites     *favorites;
  GObject               *folders_source;
  GObject               *volumes_source;
  GObject               *app_loader;
  GObject               *uri_loader;
  NlMenuItems           *menu_items;
  NlPluginManager       *plugin_manager;
  NlFavoriteItem        *favorite_item;
  DBusGProxy            *proxy;
  DBusGProxy            *curtain_proxy;

  NlConfig *cfg = NULL;

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  if (!rendering_available ())
    {
      g_print ("No rendering avaible for netbook-launcher, try to run netbook-launcher-efl\n");
      
      return fallback_to_efl ();
    }

  if (!notify_init ("Netbook Launcher"))
    return EXIT_FAILURE;

  if (!g_thread_supported ())
    g_thread_init (NULL);
  g_set_application_name (APP_NAME);
  gtk_clutter_init (&argc, &argv);

  context = g_option_context_new ("- " APP_NAME);
  g_option_context_add_main_entries (context, entries, APP_NAME);
  g_option_context_parse (context, &argc, &argv, &error);
  g_option_context_free (context);

  if (error)
    {
      g_print ("Unable to run " APP_NAME ": %s", error->message);
      g_error_free (error);
      return EXIT_FAILURE;
    }

  /* Check the existance of the ume-nl.desktop suppression file
   * before starting */
  if (check_run)
    {
      if (check_old_config_before_running ())
        return EXIT_SUCCESS;
    }

  curtain_proxy = setup_boot_curtain ();

  check_required_files_and_folders ();

  favorites      = launcher_favorites_get_default ();
  migrate_favorites ();
  
  gnome_program_init ("netbook-launcher", "0.1.2", LIBGNOMEUI_MODULE,
                      argc, argv, GNOME_PARAM_NONE, NULL);
  proxy = connect_to_session ();  

  /* DO NOT REMOVE, that is, unless
   * https://bugs.freedesktop.org/show_bug.cgi?id=17327 is fixed properly on
   * Intel systems
   */
  clutter_set_font_flags (0);

  /* If we're in low graphics mode, disable motion events */
  if (windowed)
    nl_config_set_windowed (window_width, window_height);

  cfg = nl_config_get_default ();
  if (cfg->low_graphics)
    clutter_set_motion_events_enabled (FALSE);

  /* Make sure we're the only netbook-launcher instance running */
  app = nl_app_get_default ();
  if (unique_app_is_running (UNIQUE_APP (app)))
    {
      UniqueResponse     response;
      UniqueMessageData *data;
      
      gboolean bad_response = FALSE;

      /* Send the right message */
      if (add_favorite)
        {
          data = unique_message_data_new ();
          unique_message_data_set_text (data, add_favorite, -1);
          response = unique_app_send_message (UNIQUE_APP (app),
                                              COMMAND_ADD_FAVORITE,
                                              data);
          if (response != UNIQUE_RESPONSE_OK)
              bad_response = TRUE;
          
          unique_message_data_free (data);
          g_free (add_favorite);
        }
      
      if (show_favorites)
        {
        
          response = unique_app_send_message (UNIQUE_APP (app),
                                              COMMAND_SHOW_FAVORITES,
                                              NULL);
          if (response != UNIQUE_RESPONSE_OK)
              bad_response = TRUE;
        }

      g_object_unref (app);
      g_object_unref (cfg);
      notify_uninit ();
      
      if (bad_response)
          return EXIT_BUSY; /* busy */
      
      return EXIT_SUCCESS;
    }
  else
    nl_app_init_real (app);

  /* Create the nl window */
  window = nl_window_get_default ();

  if (!nl_window_is_valid (window))
    {
      g_print ("Error creating clutter launcher.  Trying to run netbook-launcher-efl\n");
      return fallback_to_efl ();
    }

  plugin_manager = nl_plugin_manager_new (NL_SHELL (window));
  folders_source = nl_folders_source_new (NL_SHELL (window));
  volumes_source = nl_volumes_source_new (NL_SHELL (window));
  menu_items     = nl_menu_items_new (NL_SHELL (window));
  app_loader     = nl_app_loader_new (NL_SHELL (window));
  favorite_item  = nl_favorite_item_new (NL_SHELL (window));

  /* Make sure this is last, as it's kind of a 'catch-all' loader */
  uri_loader     = nl_uri_loader_new (NL_SHELL (window));

  gtk_widget_show (window);

  signal (SIGHUP, default_signal_handler);
  signal (SIGINT, default_signal_handler);
  signal (SIGTERM, default_signal_handler);
  signal (SIGUSR1, default_signal_handler);
  signal (SIGUSR2, default_signal_handler);

  if (curtain_proxy)
    g_idle_add ((GSourceFunc)notify_boot_curtain, curtain_proxy);

  gtk_main ();
  
  if (GTK_IS_WIDGET (window))
    gtk_widget_destroy (GTK_WIDGET (window));

  g_object_unref (app_loader);
  g_object_unref (favorite_item);
  g_object_unref (menu_items);
  g_object_unref (folders_source);
  g_object_unref (volumes_source);
  g_object_unref (uri_loader);
  g_object_unref (favorites);

  g_object_unref (plugin_manager);

  g_object_unref (app);
  g_object_unref (cfg);

  notify_uninit ();

  if (proxy)
    {
      DBusGConnection *conn;

      conn = (DBusGConnection *)g_object_get_data (G_OBJECT (proxy),
                                                   "connection");
      if (conn)
        dbus_g_connection_unref (conn);

      g_object_unref (proxy);
    }
  if (curtain_proxy)
    {
      DBusGConnection *conn;

      conn = (DBusGConnection *)g_object_get_data (G_OBJECT (curtain_proxy),
                                                   "connection");
      if (conn)
        dbus_g_connection_unref (conn);

      g_object_unref (curtain_proxy);
    }

  return EXIT_SUCCESS;
}

static gboolean
notify_boot_curtain (DBusGProxy *proxy)
{
  if (proxy)
    dbus_g_proxy_call_no_reply (proxy,
                                "SignalLoaded",
                                G_TYPE_STRING, "netbook-launcher",
                                G_TYPE_INVALID);

  return FALSE;
}

static DBusGProxy *
setup_boot_curtain (void)
{
  DBusGConnection *connection;
  DBusGProxy      *proxy;
  GError          *error = NULL;

  connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);

  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free(error);
      return NULL;
    }
  /* Get the current session object */
  proxy = dbus_g_proxy_new_for_name (connection,
                                     "com.ubuntu.BootCurtain",
                                     "/com/ubuntu/BootCurtain",
                                     "com.ubuntu.BootCurtain");

  if (!proxy)
    {
      g_warning ("Unable to connect to BootCurtain");
      dbus_g_connection_unref (connection);
      return NULL;
    }

  dbus_g_proxy_call_no_reply (proxy,
                              "AddWaitSignal",
                               G_TYPE_STRING, "netbook-launcher",
                               G_TYPE_INVALID);

  g_object_set_data (G_OBJECT (proxy), "connection", connection);

  return proxy;
}

static DBusGProxy *
connect_to_session (void)
{
  DBusGConnection *connection;
  DBusGProxy      *proxy;
  GError          *error = NULL;

  connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);

  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free(error);
      return NULL;
    }
  /* Get the current session object */
  proxy = dbus_g_proxy_new_for_name (connection,
                                     "org.gnome.SessionManager",
                                     "/org/gnome/SessionManager",
                                     "org.gnome.SessionManager");

  if (!proxy)
    {
      g_warning ("Unable to get the SessionManager.");
      dbus_g_connection_unref (connection);
      return NULL;
    }

  dbus_g_proxy_add_signal (proxy, "SessionOver",
                           G_TYPE_INVALID, G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "SessionOver",
                               G_CALLBACK (on_session_quit), NULL,
                               NULL);

  g_object_set_data (G_OBJECT (proxy), "connection", connection);

  return proxy;
}

static void
default_signal_handler (gint sig)
{
  g_debug ("Signal caught: %d", sig);
  on_session_quit ();
}

/*
 * Creates any required folders and files for the nl
 */
static void
check_required_files_and_folders (void)
{
  gchar *local_apps;

  local_apps = g_build_filename (g_get_home_dir (),
                                 ".local", "share", "applications", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      /* As the local application dir doesn't exist, we assume this is first
       * run
       */
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local applications directory at %s", local_apps);
    }
  g_free (local_apps);

  local_apps = g_build_filename (g_get_user_config_dir (),
                                 "netbook-launcher", "cache", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      /* As the local application dir doesn't exist, we assume this is first
       * run
       */
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local applications directory at %s", local_apps);
    }
  g_free (local_apps);

  local_apps = g_build_filename (g_get_home_dir (),
                                 ".config", "netbook-launcher", "icons", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local icons directory at %s", local_apps);
    }
  g_free (local_apps);
}

/*
 * Check if there is a ~/.config/autostart/ume-nl.desktop autostart
 * suppression file present before starting (as we may be in classic mode).
 *
 */
static gboolean
check_old_config_before_running ()
{
#define AUTOSTART_ITEM "X-GNOME-Autostart-enabled"
  gchar *ume_desktop;

  ume_desktop = g_build_filename (g_get_home_dir (),
                                  ".config",
                                  "autostart",
                                  "ume-nl.desktop",
                                  NULL);

  if (g_file_test (ume_desktop, G_FILE_TEST_EXISTS))
    {
      GError           *error = NULL;
      GnomeDesktopItem *item = NULL;
      gchar            *netbook_desktop;
      gboolean          autostart_enabled = TRUE;

      item = gnome_desktop_item_new_from_file (ume_desktop,
             GNOME_DESKTOP_ITEM_LOAD_ONLY_IF_EXISTS,
             &error);


      if (error || !item)
        {
          g_warning ("Unable to read ume-nl.desktop at %s: %s",
                     ume_desktop, error ? error->message: " ");

          g_error_free (error);
          g_free (ume_desktop);
          return FALSE;
        }

      /* Check if the autostart file was to autostart or to suppress autostart */
      if (gnome_desktop_item_attr_exists (item, AUTOSTART_ITEM))
        {
          autostart_enabled = gnome_desktop_item_get_boolean (item, AUTOSTART_ITEM);
        }

      /* Update the exec string */
      gnome_desktop_item_set_string (item,
                                     GNOME_DESKTOP_ITEM_EXEC,
                                     "netbook-launcher");
      /* Save in new location */
      netbook_desktop = g_strdup_printf (
                          "file://%s/.config/autostart/netbook-launcher.desktop",
                          g_get_home_dir ());
      gnome_desktop_item_save (item, netbook_desktop, TRUE, &error);

      if (error)
        {
          g_warning ("Unable to replace ume-launcher.desktop with netbook-launcher.desktop: %s", error->message);

          g_error_free (error);
          g_free (ume_desktop);
          g_free (netbook_desktop);
          gnome_desktop_item_unref (item);
          return FALSE;
        }
      gnome_desktop_item_unref (item);

      /* Remove of ume_desktop file */
      g_unlink (ume_desktop);

      g_free (ume_desktop);
      g_free (netbook_desktop);

      return !autostart_enabled;
    }

  g_free (ume_desktop);
  return FALSE;
}
#if 0
static void
restart_nl ()
{
  gchar *exec;

  exec = g_strdup_printf ("netbook-launcher --no-restart");
  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    exec, NULL);
  g_free (exec);
  gtk_main_quit ();
}

static void
on_system_resume (NlConfig *config)
{
  if (!norestart)
    restart_nl ();
}
#endif
#if 0
/* What happens when the stage changes size
 * NOTE: Only send --no-restart if it's currently enabled, otherwise the
 * nl will not react when RESIZE>SUSPEND>RESUME
 */
static void
restart_nl_for_resize ()
{
  NlCatbar *bar;
  gchar *exec;

  bar = (NlCatbar *)nl_catbar_get_default ();
  exec = g_strdup_printf ("netbook-launcher --last-category=%d",
                          nl_catbar_get_active_category (bar));
  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    exec, NULL);
  g_free (exec);
  gtk_main_quit ();
}

static void
size_changed (GdkScreen    *screen,
              ClutterActor *app)
{
  static gboolean already_called = FALSE;

  if (CSW () == gdk_screen_get_width (screen)
      && CSH () == gdk_screen_get_height (screen))
    return;

  if (already_called)
    return;

  already_called = TRUE;
  restart_nl_for_resize ();
}

#endif

static void
foreach_func (const gchar *key, const gchar *value, LauncherFavorites *favs)
{
  gchar *uid;

  uid = g_strdup_printf ("app-%s-%d", key, (gint)time (NULL));

  g_debug ("Migrating favorite: %s", value);
  
  /* Set the properties of the favorite */
  launcher_favorites_set_string (favs, uid, "type", "application");
  launcher_favorites_set_string (favs, uid, "desktop_file", value);
  launcher_favorites_add_favorite (favs, uid);

  g_free (uid);
}

static void
migrate_favorites (void)
{
#define MIGRATE_KEY "/apps/netbook-launcher/migrated_favorites"
  GConfClient *client;
  LauncherFavorites *favs;
  gchar       *temp;
  GDir        *dir;
  const gchar *name;
  GHashTable  *default_favs;
  gint         n_user_desktop_files = 0;

  client = gconf_client_get_default ();
  if (gconf_client_get_bool (client, MIGRATE_KEY, NULL))
    {
      g_object_unref (client);
      return;
    }

  // List of gconf-defaults favorites
  default_favs = g_hash_table_new_full (g_str_hash, g_str_equal,
                                        g_free, g_free);

  g_hash_table_insert (default_favs,
                       g_strdup ("cheese-fav.desktop"),
                       g_strdup ("/usr/share/applications/cheese.desktop"));
  g_hash_table_insert (default_favs,
                       g_strdup ("evolution-fav.desktop"),
                       g_strdup ("/usr/share/applications/evolution.desktop"));
  g_hash_table_insert (default_favs,
                       g_strdup ("firefox-fav.desktop"),
                       g_strdup ("/usr/share/applications/firefox.desktop"));
  g_hash_table_insert (default_favs,
                       g_strdup ("pidgin-fav.desktop"),
                       g_strdup ("/usr/share/applications/pidgin.desktop"));
  g_hash_table_insert (default_favs,
                       g_strdup ("yelp-fav.desktop"),
                       g_strdup ("/usr/share/applications/yelp.desktop"));

  /* Grab a list of user-created desktop files */
  temp = g_build_filename (g_get_home_dir (), 
                           ".local", "share", "applications",
                           NULL);
  dir = g_dir_open (temp, 0, NULL);
  if (!dir)
    {
      g_free (temp);
      g_hash_table_destroy (default_favs);
      goto migrate_favorites_out;
    }
  while ((name = g_dir_read_name (dir)))
    {
      GKeyFile *file;
      gchar    *filename;
      gchar    *cats;

      if (!g_str_has_suffix (name, ".desktop"))
        continue;

      filename = g_build_filename (temp, name, NULL);
       
      g_hash_table_insert (default_favs,
                           g_strdup (name),
                           filename);

      file = g_key_file_new ();
      g_key_file_load_from_file (file, filename, 0, NULL);

      cats = g_key_file_get_string (file,
                                    G_KEY_FILE_DESKTOP_GROUP,
                                    G_KEY_FILE_DESKTOP_KEY_CATEGORIES,
                                    NULL);
      if (!cats
          || !g_strstr_len (cats, -1, "Favorites")
          || g_key_file_get_boolean (file,
                                     G_KEY_FILE_DESKTOP_GROUP,
                                     G_KEY_FILE_DESKTOP_KEY_NO_DISPLAY,
                                     NULL))
        {
          g_hash_table_remove (default_favs, name);
        }
      else
        {
          gchar  *buf;
          gsize   len;
          GError *error = NULL;

          /* Set nodisplay to true */
          g_key_file_set_boolean (file,
                                  G_KEY_FILE_DESKTOP_GROUP,
                                  G_KEY_FILE_DESKTOP_KEY_NO_DISPLAY,
                                  TRUE);
          buf = g_key_file_to_data (file, &len, &error);
          if (error)
            {
              g_warning ("Unable to hide favorite %s: %s",
                         filename,
                         error->message);
              g_error_free (error);
            }
          else
            {
              if (!g_file_set_contents (filename, buf, len, &error))
                {
                  g_warning ("Unable to hide favorite %s: %s",
                             filename,
                             error ? error->message : "Unknown");
                  if (error)
                    g_error_free (error);
                }
            }

          g_free (buf);
        }

      g_free (cats);
      g_key_file_free (file);
      n_user_desktop_files++;
    }
  g_free (temp);
  g_dir_close (dir);

  /* If this is a clean install, or the user hasn't changed anything, let's
   * skip over this
   */
  if (!n_user_desktop_files)
    {
      g_hash_table_destroy (default_favs);
      goto migrate_favorites_out;
    }

  favs = launcher_favorites_get_default ();
  g_hash_table_foreach (default_favs, (GHFunc)foreach_func, favs);

  g_hash_table_destroy (default_favs);
  g_object_unref (favs);

migrate_favorites_out:
  // Make sure we don't do this again
  gconf_client_set_bool (client, MIGRATE_KEY, TRUE, NULL);
  g_object_unref (client);
}
