//
// Gendarme.Rules.Performance.AvoidUncalledPrivateCodeRule
//
// Authors:
//	Nidhi Rawal <sonu2404@gmail.com>
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// Copyright (c) <2007> Nidhi Rawal
// Copyright (C) 2007-2008 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;
using System.Collections.Generic;

using Mono.Cecil;
using Mono.Cecil.Cil;
using Gendarme.Framework;
using Gendarme.Framework.Rocks;
using Gendarme.Framework.Helpers;

namespace Gendarme.Rules.Performance {
	
	[Problem ("This private or internal (assembly-level) member does not have callers in the assembly, is not invoked by the common language runtime, and is not invoked by a delegate.")]
	[Solution ("Remove the non-callable code or add the code that calls it.")]
	public class AvoidUncalledPrivateCodeRule: Rule, IMethodRule {

		static string [] ComRegistration = new string[] {
			"System.Runtime.InteropServices.ComRegisterFunctionAttribute",
			"System.Runtime.InteropServices.ComUnregisterFunctionAttribute"
		};

		static private bool Applicable (MethodDefinition method)
		{
			// rule doesn't apply to static ctor (called by the runtime)
			if (method.IsStatic && method.IsConstructor)
				return false;

			// don't consider the compiler generated add / remove on events
			if (((method.IsAddOn || method.IsRemoveOn) && method.IsSynchronized))
				return false;

			// rule doesn't apply if the method is the assembly entry point or Main
			if (method.IsEntryPoint () || method.IsMain ())
				return false;

			// rule doesn't apply if the method is generated by the compiler or by a tool
			if (method.IsGeneratedCode ())
				return false;

			// does not apply if the method is used to register/unregister COM objects
			if (method.CustomAttributes.ContainsAnyType (ComRegistration))
				return false;

			return true;
		}

		public RuleResult CheckMethod (MethodDefinition method)
		{
			// check if the the rule applies to this method
			if (!Applicable (method))
				return RuleResult.DoesNotApply;

			// we can't be sure if this code won't be reached indirectly
			if (method.IsVirtual && !method.IsFinal)
				return RuleResult.Success;

			// if the method is visible outside the assembly
			if (method.IsVisible ())
				return RuleResult.Success;

			// check if the method is private 
			if (method.IsPrivate) {
				if (!CheckPrivateMethod (method)) {
					Runner.Report (method, Severity.High, Confidence.Normal, "The private method code is not used in its declaring type.");
					return RuleResult.Failure;
				}
				return RuleResult.Success;
			}

			if (method.IsPublic && CheckPublicMethod (method))
				return RuleResult.Success;

			if (method.IsAssembly && CheckInternalMethod (method))
				return RuleResult.Success;

			// internal methods and visible methods (public or protected) inside a non-visible type
			// needs to be checked if something in the assembly is using this method
			bool need_to_check_assembly = (method.IsAssembly || 
				((method.IsPublic || method.IsFamily) && !method.DeclaringType.IsVisible ()));

			if (!need_to_check_assembly || CheckAssemblyForMethodUsage (method.DeclaringType.Module.Assembly, method))
				return RuleResult.Success;

			// method is unused and unneeded
			Runner.Report (method, Severity.High, Confidence.Normal, "The method is not visible outside its declaring assembly, nor used within.");
			return RuleResult.Failure;
		}

		private static bool CheckPrivateMethod (MethodDefinition method)
		{
			// it's ok for have unused private ctor (and common before static class were introduced in 2.0)
			// this also covers private serialization constructors
			if (method.IsConstructor)
				return true;

			// it's ok (used or not) if it's required to implement explicitely an interface
			if (method.Overrides.Count > 0)
				return true;

			TypeDefinition type = (method.DeclaringType as TypeDefinition);

			// then we must check if this type use the private method
			if (CheckTypeForMethodUsage (type, method))
				return true;

			// then we must check if this type's nested types (if any) use the private method
			foreach (TypeDefinition nested in type.NestedTypes) {
				if (CheckTypeForMethodUsage (nested, method))
					return true;
			}

			// report if the private method is uncalled
			return false;
		}

		// note: we need to be consistant with some stuff we propose in other rules
		private static bool CheckPublicMethod (MethodDefinition method)
		{
			// handle things like operators - but not properties
			if (method.IsSpecialName && !method.IsProperty ())
				return true;
			
			// handle non-virtual Equals, e.g. Equals(type)
			if ((method.Name == "Equals") && (method.Parameters.Count == 1) &&
				(method.Parameters [0].ParameterType == method.DeclaringType))
				return true;

			// check if this method is needed to satisfy an interface
			TypeDefinition type = (method.DeclaringType as TypeDefinition);
			foreach (TypeReference tr in type.Interfaces) {
				TypeDefinition intf = tr.Resolve ();
				if (intf != null) {
					foreach (MethodReference member in intf.Methods) {
						if (method.Name == member.Name)
							return true;
					}
				}
			}
			return false;
		}

		private static bool CheckInternalMethod (MethodReference method)
		{
			// internal ctor for serialization are ok
			return MethodSignatures.SerializationConstructor.Matches (method);
		}

		private static bool CheckAssemblyForMethodUsage (AssemblyDefinition ad, MethodReference method)
		{
			// scan each module
			foreach (ModuleDefinition module in ad.Modules) {
				// scan each type
				foreach (TypeDefinition type in module.Types) {
					if (CheckTypeForMethodUsage (type, method))
						return true;
				}
			}
			return false;
		}

		static Dictionary<TypeDefinition, HashSet<uint>> cache = new Dictionary<TypeDefinition, HashSet<uint>> ();

		private static uint GetToken (MethodReference method)
		{
			return method.GetOriginalMethod ().MetadataToken.ToUInt ();
		}

		private static bool CheckTypeForMethodUsage (TypeDefinition type, MethodReference method)
		{
			if (type.GenericParameters.Count > 0)
				type = type.GetOriginalType ().Resolve ();

			HashSet<uint> methods = GetCache (type);
			if (methods.Contains (GetToken (method)))
				return true;

			foreach (MethodReference mr in method.Resolve ().Overrides) {
				if (methods.Contains (GetToken (mr)))
					return true;
			}
			return false;
		}

		private static HashSet<uint> GetCache (TypeDefinition type)
		{
			HashSet<uint> methods;
			if (!cache.TryGetValue (type, out methods)) {
				methods = new HashSet<uint> ();
				cache.Add (type, methods);
				foreach (MethodDefinition md in type.AllMethods ()) {
					if (!md.HasBody)
						continue;

					BuildMethodUsage (methods, md);
				}
			}
			return methods;
		}

		private static void BuildMethodUsage (HashSet<uint> methods, MethodDefinition method)
		{
			foreach (Instruction ins in method.Body.Instructions) {
				MethodReference mr = (ins.Operand as MethodReference);
				if (mr == null)
					continue;

				TypeDefinition type = mr.DeclaringType.Resolve ();
				if ((type != null) && (type.GenericParameters.Count > 0)) {
					methods.Add (GetToken (type.GetMethod (mr.Name)));
				}
				methods.Add (GetToken (mr));
			}
		}
	}
}
