/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#include <config.h>

#include <glib.h>
#include <gio/gio.h>
#include <gio/gdesktopappinfo.h>
#include <zeitgeist.h>

#include "gapplaunchhandlerzeitgeist.h"

static ZeitgeistLog *log = NULL;
static gpointer      log_w = NULL;

struct _GAppLaunchHandlerZeitgeist {
  GObject parent;

};

static void launch_handler_iface_init (GDesktopAppInfoLaunchHandlerIface *iface);
static void g_app_launch_handler_zeitgeist_finalize (GObject *object);
static void ensure_log (void);

#define _G_IMPLEMENT_INTERFACE_DYNAMIC(TYPE_IFACE, iface_init)       { \
  const GInterfaceInfo g_implement_interface_info = { \
    (GInterfaceInitFunc) iface_init, NULL, NULL \
  }; \
  g_type_module_add_interface (type_module, g_define_type_id, TYPE_IFACE, &g_implement_interface_info); \
}

G_DEFINE_DYNAMIC_TYPE_EXTENDED (GAppLaunchHandlerZeitgeist, g_app_launch_handler_zeitgeist, G_TYPE_OBJECT, 0,
                                _G_IMPLEMENT_INTERFACE_DYNAMIC (G_TYPE_DESKTOP_APP_INFO_LAUNCH_HANDLER,
                                                                launch_handler_iface_init))

/* We lazily create the ZeitgeistLog lazily because it creates a dbus
 * connection, which if set up during the GIOModule registration phase
 * (ie inside g_app_launch_handler_zeitgeist_register()) might mess up the
 * gio module initialization process.
 *
 * Spurious deadlocks and segfaults have been observed without
 * lazy setup of the log.
 */
static void
ensure_log (void)
{
  if (log_w == NULL)
    {
      log = zeitgeist_log_new ();
      log_w = log;
      g_object_add_weak_pointer (G_OBJECT (log), &log_w);
    }
}

static void
g_app_launch_handler_zeitgeist_finalize (GObject *object)
{
  if (G_OBJECT_CLASS (g_app_launch_handler_zeitgeist_parent_class)->finalize)
    (*G_OBJECT_CLASS (g_app_launch_handler_zeitgeist_parent_class)->finalize) (object);
}

static GObject *
g_app_launch_handler_zeitgeist_constructor (GType                  type,
                                            guint                  n_construct_properties,
                                            GObjectConstructParam *construct_properties)
{
  GObject *object;
  GAppLaunchHandlerZeitgeistClass *klass;
  GObjectClass *parent_class;

  object = NULL;

  /* Invoke parent constructor. */
  klass = G_APP_LAUNCH_HANDLER_ZEITGEIST_CLASS (g_type_class_peek (G_TYPE_APP_LAUNCH_HANDLER_ZEITGEIST));
  parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
  object = parent_class->constructor (type,
                                      n_construct_properties,
                                      construct_properties);

  return object;
}

static void
g_app_launch_handler_zeitgeist_init (GAppLaunchHandlerZeitgeist *lookup)
{
}

static void
g_app_launch_handler_zeitgeist_class_finalize (GAppLaunchHandlerZeitgeistClass *klass)
{
}


static void
g_app_launch_handler_zeitgeist_class_init (GAppLaunchHandlerZeitgeistClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->constructor = g_app_launch_handler_zeitgeist_constructor;
  gobject_class->finalize = g_app_launch_handler_zeitgeist_finalize;
}

/*
static void
on_subject_info_ready (GFile *file,
                       GAsyncResult *res,
                       ZeitgeistEvent *ev)
{
  ZeitgeistSubject *su;
  GError           *error;
  GFileInfo        *info;
  GFile            *parent_file;
  gchar            *uri, *parent_uri;
  const gchar      *display_name, *filesystem, *mimetype;

  uri = g_file_get_uri (file);
  
  error = NULL;
  info = g_file_query_info_finish (file, res, &error);
  if (error != NULL)
    {
      // On anything but a file-not-found we complain in the log
      if (error->code != G_IO_ERROR_NOT_FOUND)
        g_warning ("Error querying file info for '%s': %s",
                   uri, error->message);

      g_free (uri);
      g_object_unref (ev);
      g_error_free (error);
      return;
    }

  display_name = g_file_info_get_attribute_string (info,
                                                   G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME);
  filesystem = g_file_info_get_attribute_string (info,
                                                 G_FILE_ATTRIBUTE_ID_FILESYSTEM);
  mimetype = g_file_info_get_attribute_string (info,
                                               G_FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE);

  parent_file = g_file_get_parent (file);
  parent_uri = g_file_get_uri (parent_file);
  
  // If we couldn't query the filesystem for the subject assume it was a
  // residing online somwhere
  filesystem = filesystem != NULL ? filesystem : "net";

  su = zeitgeist_subject_new_full (uri,
                                   zeitgeist_interpretation_for_mimetype (mimetype),
                                   zeitgeist_manifestation_for_uri (uri),
                                   mimetype,
                                   parent_uri,
                                   display_name,
                                   filesystem);
  zeitgeist_event_add_subject (ev, su);

  // FIXME: Right now we actually insert one event for each subject.
  //        We should rather collect all subject data (async) and
  //        insert one event with all the subjects
  zeitgeist_log_insert_events_no_reply (log, ev, NULL);

  g_free (uri);
  g_free (parent_uri);
  g_object_unref (parent_file);
  g_object_unref (info);
}
*/

/* Create an application://app.desktop URI. Caller must g_free() the result
 * if it's != NULL */
static gchar*
get_application_uri (GDesktopAppInfo              *appinfo)
{
  gchar *app_id = NULL;

  app_id = g_strdup (g_app_info_get_id (G_APP_INFO (appinfo)));

  if (app_id != NULL)
  {
    app_id = g_strconcat ("application://", app_id, NULL);
  }
  else if (g_desktop_app_info_get_filename (G_DESKTOP_APP_INFO (appinfo)))
  {
    const gchar *path = g_desktop_app_info_get_filename (
                                                G_DESKTOP_APP_INFO (appinfo));
    gchar *basename = g_path_get_basename (path);
    app_id = g_strconcat ("application://", basename, NULL);
    g_free (basename);
  }

  return app_id;
}

/* Idle handler for logging the app launch event. We do this in an
 * idle call to make sure it's done in the mainloop in order to be
 * thread safe */
static gboolean
log_event (ZeitgeistEvent *event)
{
  ensure_log ();

  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), FALSE);
  g_return_val_if_fail (ZEITGEIST_IS_LOG (log), FALSE);

  /* Send the event to Zeitgeist */
  zeitgeist_log_insert_events_no_reply (log, event, NULL);
  
  return FALSE;
}

static void
on_launched (GDesktopAppInfoLaunchHandler *launch_handler,
             GDesktopAppInfo              *app_info,
             GList                        *uris,
             GAppLaunchContext            *launch_ctx,
             gint                          pid)
{
  ZeitgeistEvent   *ev;
  ZeitgeistSubject *su;
  GFile            *file;
  gchar            *subject_uri, *actor, *prgname;
  
  g_return_if_fail (G_IS_DESKTOP_APP_INFO (app_info));

  /* Don't log stuff about apps not shown in user menus */
  if (! g_app_info_should_show (G_APP_INFO (app_info)))
    return;

  ev = zeitgeist_event_new ();
  su = zeitgeist_subject_new ();

  subject_uri = get_application_uri (app_info);

  if (subject_uri == NULL)
  {
    /* We where unable to figure out the launched app... */
    g_object_unref (ev);
    g_object_unref (su);
    return;
  }

  /* Set event metadata */
  prgname = g_get_prgname ();
  zeitgeist_event_set_interpretation (ev, ZEITGEIST_ZG_ACCESS_EVENT);
  zeitgeist_event_set_manifestation (ev, ZEITGEIST_ZG_USER_ACTIVITY);
  actor = prgname ? g_strdup_printf ("application://%s.desktop", prgname) :
                    g_strdup ("");
  zeitgeist_event_set_actor (ev, actor);
  g_free (actor);
  
  zeitgeist_event_add_subject (ev, su);
  
  zeitgeist_subject_set_uri (su, subject_uri);
  zeitgeist_subject_set_interpretation (su, ZEITGEIST_NFO_SOFTWARE);
  zeitgeist_subject_set_manifestation (su, ZEITGEIST_NFO_SOFTWARE_ITEM);
  zeitgeist_subject_set_mimetype (su, "application/x-desktop");
  zeitgeist_subject_set_text (su, g_app_info_get_display_name (
                                                       G_APP_INFO (app_info)));
  g_free (subject_uri);
  // FIXME: subject origin and storage?

  /* Do all DBus interactions in an idle handler to make sure
   * we are thread safe. The floating ref on ev swallowed by idle
   * handler log_event() */
  g_idle_add ((GSourceFunc) log_event, ev);
}

static void
launch_handler_iface_init (GDesktopAppInfoLaunchHandlerIface *iface)
{
  iface->on_launched = on_launched;
}

void
g_app_launch_handler_zeitgeist_register (GIOModule *module)
{
  g_app_launch_handler_zeitgeist_register_type (G_TYPE_MODULE (module));
  g_io_extension_point_implement (G_DESKTOP_APP_INFO_LAUNCH_HANDLER_EXTENSION_POINT_NAME,
                                  G_TYPE_APP_LAUNCH_HANDLER_ZEITGEIST,
                                  "zeitgeist",
                                  10);
}

void
g_app_launch_handler_zeitgeist_unregister (GIOModule *module)
{
  if (log_w != NULL)
    g_object_unref (log);
}
