/**************************************************************************
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 *
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this driver.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
/*
 */

#include "drmP.h"
#include "psb_drv.h"
#include "psb_drm.h"
#include "psb_reg.h"


typedef struct psb_cmdbuf_info {
	drm_buffer_object_t *cmd_buf;
	unsigned long cmd_offset;
	unsigned long cmdbuf_offset;
	u32 *cmd_page;
	unsigned cmd_page_offset;
	struct drm_bo_kmap_obj cmd_kmap;
	int cmd_is_iomem;
} psb_cmdbuf_info_t;


static int psb_2d_wait_available(drm_psb_private_t * dev_priv, unsigned size)
{
	u32 avail = PSB_RSGX32(PSB_CR_2D_SOCIF);

	if (avail < size) {
		unsigned long _end = jiffies + DRM_HZ;

		do {
			/* Need a sleep here to save CPU */
			msleep(1);
			avail = PSB_RSGX32(PSB_CR_2D_SOCIF);
		} while (avail < size && !time_after_eq(jiffies, _end));

		if (avail < size) {
			DRM_ERROR("Detected 2D engine hang.\n");
			return -EBUSY;
		}
	}
	return 0;
}

int psb_2d_submit(drm_psb_private_t * dev_priv, u32 * cmdbuf, unsigned size)
{
	int ret = 0;
	int i;
	unsigned submit_size;

	while (size > 0) {
		submit_size = (size < 0x60) ? size : 0x60;
		size -= submit_size;
		ret = psb_2d_wait_available(dev_priv, submit_size);
		if (ret)
			return ret;

		submit_size <<= 2;

		for (i = 0; i < submit_size; i += 4) {
			PSB_WSGX32(*cmdbuf++, PSB_SGX_2D_SLAVE_PORT + i);
		}
		(void)PSB_RSGX32(PSB_SGX_2D_SLAVE_PORT + i - 4);
	}
	return 0;
}

int psb_blit_sequence(drm_psb_private_t * dev_priv)
{
	u32 buffer[8];
	u32 *bufp = buffer;

	*bufp++ = PSB_2D_FENCE_BH;

	*bufp++ = PSB_2D_DST_SURF_BH |
	    PSB_2D_DST_8888ARGB | (4 << PSB_2D_DST_STRIDE_SHIFT);
	*bufp++ = dev_priv->comm_mmu_offset - dev_priv->mmu_2d_offset;

	*bufp++ = PSB_2D_BLIT_BH |
	    PSB_2D_ROT_NONE |
	    PSB_2D_COPYORDER_TL2BR |
	    PSB_2D_DSTCK_DISABLE |
	    PSB_2D_SRCCK_DISABLE | PSB_2D_USE_FILL | PSB_2D_ROP3_PATCOPY;

	*bufp++ = dev_priv->sequence << PSB_2D_FILLCOLOUR_SHIFT;
	*bufp++ = (0 << PSB_2D_DST_XSTART_SHIFT) |
	    (0 << PSB_2D_DST_YSTART_SHIFT);
	*bufp++ = (1 << PSB_2D_DST_XSIZE_SHIFT) | (1 << PSB_2D_DST_YSIZE_SHIFT);

	*bufp++ = PSB_2D_FLUSH_BH;

	return psb_2d_submit(dev_priv, buffer, bufp - buffer);
}

int psb_emit_2d_copy_blit(drm_device_t * dev,
			  u32 src_offset,
			  u32 dst_offset, __u32 pages, int direction)
{
	uint32_t cur_pages;
	drm_psb_private_t *dev_priv = dev->dev_private;
	u32 buf[10];
	u32 *bufp;
	u32 xstart;
	u32 ystart;
	u32 blit_cmd;
	u32 pg_add;
	int ret;

	if (!dev_priv)
		return 0;

	if (direction) {
		pg_add = (pages - 1) << PAGE_SHIFT;
		src_offset += pg_add;
		dst_offset += pg_add;
	}

	blit_cmd = PSB_2D_BLIT_BH |
	    PSB_2D_ROT_NONE |
	    PSB_2D_DSTCK_DISABLE |
	    PSB_2D_SRCCK_DISABLE |
	    PSB_2D_USE_PAT |
	    PSB_2D_ROP3_SRCCOPY |
	    (direction ? PSB_2D_COPYORDER_BR2TL : PSB_2D_COPYORDER_TL2BR);
	xstart = (direction) ? ((PAGE_SIZE - 1) >> 2) : 0;

	while (pages > 0) {
		cur_pages = pages;
		if (cur_pages > 2048)
			cur_pages = 2048;
		pages -= cur_pages;
		ystart = (direction) ? cur_pages - 1 : 0;

		bufp = buf;
		*bufp++ = PSB_2D_FENCE_BH;

		*bufp++ = PSB_2D_DST_SURF_BH | PSB_2D_DST_8888ARGB |
		    (PAGE_SIZE << PSB_2D_DST_STRIDE_SHIFT);
		*bufp++ = dst_offset;
		*bufp++ = PSB_2D_SRC_SURF_BH | PSB_2D_SRC_8888ARGB |
		    (PAGE_SIZE << PSB_2D_SRC_STRIDE_SHIFT);
		*bufp++ = src_offset;
		*bufp++ =
		    PSB_2D_SRC_OFF_BH | (xstart << PSB_2D_SRCOFF_XSTART_SHIFT) |
		    (ystart << PSB_2D_SRCOFF_YSTART_SHIFT);
		*bufp++ = blit_cmd;
		*bufp++ = (xstart << PSB_2D_DST_XSTART_SHIFT) |
		    (ystart << PSB_2D_DST_YSTART_SHIFT);
		*bufp++ = ((PAGE_SIZE >> 2) << PSB_2D_DST_XSIZE_SHIFT) |
		    (cur_pages << PSB_2D_DST_YSIZE_SHIFT);

		ret = psb_2d_submit(dev_priv, buf, bufp - buf);
		if (ret)
			return ret;
		pg_add = (cur_pages << PAGE_SHIFT) * ((direction) ? -1 : 1);
		src_offset += pg_add;
		dst_offset += pg_add;
	}
	return 0;
}

void psb_init_2d(drm_psb_private_t * dev_priv)
{
	PSB_WSGX32(_PSB_CS_RESET_TWOD_RESET |
		   _PSB_CS_RESET_BIF_RESET
		   , PSB_CR_SOFT_RESET);
	PSB_RSGX32(PSB_CR_SOFT_RESET);
	msleep(1);
	PSB_WSGX32(0x00, PSB_CR_SOFT_RESET);
	wmb();
	PSB_WSGX32(PSB_RSGX32(PSB_CR_BIF_CTRL) | _PSB_CB_CTRL_CLEAR_FAULT,
		   PSB_CR_BIF_CTRL);
	wmb();
	PSB_WSGX32(PSB_RSGX32(PSB_CR_BIF_CTRL) & ~_PSB_CB_CTRL_CLEAR_FAULT,
		   PSB_CR_BIF_CTRL);
	PSB_RSGX32(PSB_CR_BIF_CTRL);

	dev_priv->mmu_2d_offset = dev_priv->pg->gatt_start;
	PSB_WSGX32(dev_priv->mmu_2d_offset, PSB_CR_BIF_TWOD_REQ_BASE);
	(void)PSB_RSGX32(PSB_CR_BIF_TWOD_REQ_BASE);
}

int drm_psb_idle(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = dev->dev_private;
	unsigned long _end = jiffies + DRM_HZ;
	int busy;
	int ret;

	if ((PSB_RSGX32(PSB_CR_2D_SOCIF) == _PSB_C2_SOCIF_EMPTY) &&
	    ((PSB_RSGX32(PSB_CR_2D_BLIT_STATUS) & _PSB_C2B_STATUS_BUSY) == 0))
		return 0;

	if (dev_priv->engine_lockup_2d)
		return -EBUSY;

	drm_idlelock_take(&dev->lock);
	do {
		busy = (PSB_RSGX32(PSB_CR_2D_SOCIF) != _PSB_C2_SOCIF_EMPTY);
	} while (busy && !time_after_eq(jiffies, _end));
	if (busy)
		busy = (PSB_RSGX32(PSB_CR_2D_SOCIF) != _PSB_C2_SOCIF_EMPTY);
	if (busy) {
		ret = -EBUSY;
		goto out;
	}

	do {
		busy =
		    ((PSB_RSGX32(PSB_CR_2D_BLIT_STATUS) & _PSB_C2B_STATUS_BUSY)
		     != 0);
	} while (busy && !time_after_eq(jiffies, _end));

	if (busy)
		busy =
		    ((PSB_RSGX32(PSB_CR_2D_BLIT_STATUS) & _PSB_C2B_STATUS_BUSY)
		     != 0);
	ret = (busy) ? -EBUSY : 0;
      out:
	drm_idlelock_release(&dev->lock);

	if (ret == -EBUSY) {
		dev_priv->engine_lockup_2d = 1;
		DRM_ERROR("Detected SGX 2D Engine lockup 0x%08x 0x%08x\n",
			  PSB_RSGX32(PSB_CR_BIF_INT_STAT),
			  PSB_RSGX32(PSB_CR_BIF_FAULT));
	}

	return ret;
}

static void psb_dereference_buffers_locked(drm_buffer_object_t ** buffers,
					   unsigned num_buffers)
{
	while (num_buffers--)
		drm_bo_usage_deref_locked(*buffers++);

}

static int psb_validate_buffer_list(drm_file_t * priv,
				    unsigned fence_class,
				    unsigned long data,
				    drm_buffer_object_t ** buffers,
				    unsigned *num_buffers)
{
	drm_bo_arg_t arg;
	drm_bo_arg_reply_t rep;
	drm_bo_arg_request_t *req = &arg.d.req;
	unsigned long next;
	int ret = 0;
	unsigned buf_count = 0;
	drm_device_t *dev = priv->head->dev;

	do {
		if (buf_count >= *num_buffers) {
			DRM_ERROR("Buffer count exceeded %d\n.", *num_buffers);
			ret = -EINVAL;
			goto out_err;
		}

		buffers[buf_count] = NULL;

		if (DRM_COPY_FROM_USER(&arg, (void __user *)data, sizeof(arg))) {
			ret = -EFAULT;
			goto out_err;
		}

		if (arg.handled) {
			data = arg.next;
			buffers[buf_count] =
			    drm_lookup_buffer_object(priv, req->handle, 1);
			buf_count++;
			continue;
		}

		rep.ret = 0;
		if (req->op != drm_bo_validate) {
			DRM_ERROR
			    ("Buffer object operation wasn't \"validate\".\n");
			rep.ret = -EINVAL;
			goto out_err;
		}

		rep.ret = drm_bo_handle_validate(priv, req->handle, req->mask,
						 req->arg_handle, req->hint,
						 fence_class,
						 &rep, &buffers[buf_count]);
		if (rep.ret)
			goto out_err;

		next = arg.next;
		arg.handled = 1;
		arg.d.rep = rep;

		if (DRM_COPY_TO_USER((void __user *)data, &arg, sizeof(arg))) {
			ret = -EFAULT;
			goto out_err;
		}

		data = next;
		buf_count++;

	} while (data);

	*num_buffers = buf_count;
	return 0;
      out_err:
	mutex_lock(&dev->struct_mutex);
	psb_dereference_buffers_locked(buffers, buf_count);
	mutex_unlock(&dev->struct_mutex);
	*num_buffers = 0;
	return (ret) ? ret : rep.ret;
}

static int
psb_submit_2d_cmdbuf(drm_device_t * dev,
		     drm_buffer_object_t * cmd_buffer,
		     unsigned long cmd_offset, unsigned long cmd_size)
{
	unsigned long cmd_end = cmd_offset + (cmd_size << 2);
	drm_psb_private_t *dev_priv = dev->dev_private;
	unsigned long cmd_page_offset = cmd_offset - (cmd_offset & PAGE_MASK);
	unsigned long cmd_next;
	struct drm_bo_kmap_obj cmd_kmap;
	u32 *cmd_page;
	unsigned cmds;
	int is_iomem;
	int ret;

	if (cmd_size == 0)
		return 0;

	do {
		cmd_next = drm_bo_offset_end(cmd_offset, cmd_end);
		ret = drm_bo_kmap(cmd_buffer, cmd_offset >> PAGE_SHIFT,
				  1, &cmd_kmap);

		if (ret)
			return ret;
		cmd_page = drm_bmo_virtual(&cmd_kmap, &is_iomem);
		cmd_page_offset = (cmd_offset & ~PAGE_MASK) >> 2;
		cmds = (cmd_next - cmd_offset) >> 2;

		ret = psb_2d_submit(dev_priv, cmd_page + cmd_page_offset, cmds);
		drm_bo_kunmap(&cmd_kmap);
		if (ret)
			return ret;
	} while (cmd_offset = cmd_next, cmd_offset != cmd_end);
	return 0;
}

static int psb_apply_reloc(const drm_psb_reloc_t * reloc,
			   drm_buffer_object_t ** buffers,
			   int num_buffers, psb_cmdbuf_info_t * cmd_info)
{
	unsigned long new_cmd_offset;
	u32 val;
	unsigned index;
	int ret;

	if (reloc->buffer >= num_buffers) {
		DRM_ERROR("Illegal relocation buffer %d\n", reloc->buffer);
		return -EINVAL;
	}

	/*
	 * Fixme: Check buffer size.
	 */

	/*
	 * Make sure we are mapping the correct command buffer page.
	 */

	new_cmd_offset = (reloc->where << 2) + cmd_info->cmdbuf_offset;
	if (!cmd_info->cmd_page ||
	    !drm_bo_same_page(cmd_info->cmd_offset, new_cmd_offset)) {
		drm_bo_kunmap(&cmd_info->cmd_kmap);
		cmd_info->cmd_offset = new_cmd_offset;
		ret =
		    drm_bo_kmap(cmd_info->cmd_buf, new_cmd_offset >> PAGE_SHIFT,
				1, &cmd_info->cmd_kmap);
		if (ret) {
			DRM_ERROR("Could not map command buffer to "
				  "apply relocations\n");
			return ret;
		}
		cmd_info->cmd_page = drm_bmo_virtual(&cmd_info->cmd_kmap,
						     &cmd_info->cmd_is_iomem);
		cmd_info->cmd_page_offset =
		    ((cmd_info->cmd_offset & PAGE_MASK) >> 2);
	}

	switch (reloc->reloc_op) {
	case psb_reloc_offset:
		val = buffers[reloc->buffer]->offset;
		break;
	default:
		DRM_ERROR("Unimplemented relocation.\n");
		return -EINVAL;
	}

	val = (((val >> reloc->align_shift) << reloc->shift) & reloc->mask) + reloc->delta;
	index = reloc->where - cmd_info->cmd_page_offset;

	/*
	 * Avoid reading from a potentially non-prefetchable
	 * command buffer if at all possible.
	 */

	if (reloc->mask == 0xFFFFFFFF) {
		cmd_info->cmd_page[index] = val;
	} else {
		cmd_info->cmd_page[index] =
		    (cmd_info->cmd_page[index] & ~reloc->mask) |
		    (val & reloc->mask);
	}
	return 0;
}

static int psb_fixup_relocs(drm_file_t * priv,
			    drm_psb_cmdbuf_arg_t * arg,
			    drm_buffer_object_t ** buffers,
			    unsigned num_buffers,
			    drm_buffer_object_t ** cmd_buffer_ret)
{
	drm_device_t *dev = priv->head->dev;
	drm_buffer_object_t *reloc_buffer = NULL;
	unsigned long reloc_offset;
	unsigned long reloc_page_offset;
	unsigned long reloc_end;
	unsigned long next_offset;
	psb_cmdbuf_info_t cmd_info;
	drm_psb_reloc_t *reloc;
	struct drm_bo_kmap_obj reloc_kmap;
	int reloc_is_iomem;
	void *reloc_page;
	int ret = 0;

	memset(&cmd_info, 0, sizeof(cmd_info));
	memset(&reloc_kmap, 0, sizeof(reloc_kmap));

	cmd_info.cmd_buf =
	    drm_lookup_buffer_object(priv, arg->cmdbuf_handle, 1);

	if (!cmd_info.cmd_buf)
		goto out;

	if (arg->num_relocs == 0)
		goto out;

	cmd_info.cmdbuf_offset = arg->cmdbuf_offset;

	if (arg->reloc_handle == arg->cmdbuf_handle)
		reloc_buffer = cmd_info.cmd_buf;
	else {
		reloc_buffer =
		    drm_lookup_buffer_object(priv, arg->reloc_handle, 1);
		if (!reloc_buffer)
			goto out;
	}

	reloc_offset = arg->reloc_offset;
	reloc_end = reloc_offset + arg->num_relocs * sizeof(drm_psb_reloc_t);
	do {
		next_offset = drm_bo_offset_end(reloc_offset, reloc_end);
		ret = drm_bo_kmap(reloc_buffer, reloc_offset >> PAGE_SHIFT,
				  1, &reloc_kmap);
		if (ret) {
			DRM_ERROR("Could not map relocation buffer.\n");
			goto out;
		}
		reloc_page = drm_bmo_virtual(&reloc_kmap, &reloc_is_iomem);

		reloc_page_offset = reloc_offset & ~PAGE_MASK;

		reloc = (drm_psb_reloc_t *) ((unsigned long)reloc_page +
					     reloc_page_offset);
		do {
			ret = psb_apply_reloc(reloc, buffers,
					      num_buffers, &cmd_info);
			if (ret)
				goto out;

			reloc++;
			reloc_offset += sizeof(drm_psb_reloc_t);
		} while (reloc_offset < next_offset);
		drm_bo_kunmap(&reloc_kmap);

	} while (reloc_offset = next_offset, reloc_offset != reloc_end);

      out:
	drm_bo_kunmap(&reloc_kmap);
	drm_bo_kunmap(&cmd_info.cmd_kmap);

	mutex_lock(&dev->struct_mutex);
	if (reloc_buffer && reloc_buffer != cmd_info.cmd_buf)
		drm_bo_usage_deref_locked(reloc_buffer);
	if (ret && cmd_info.cmd_buf) {
		drm_bo_usage_deref_locked(cmd_info.cmd_buf);
		cmd_info.cmd_buf = NULL;
	}
	mutex_unlock(&dev->struct_mutex);

	*cmd_buffer_ret = cmd_info.cmd_buf;

	return ret;
}

int psb_cmdbuf_ioctl(DRM_IOCTL_ARGS)
{
	DRM_DEVICE;
	drm_psb_cmdbuf_arg_t arg;
	drm_psb_private_t *dev_priv = dev->dev_private;
	int ret = 0;
	unsigned num_buffers;
	drm_fence_object_t *fence;
	drm_buffer_object_t *cmd_buffer;

	DRM_COPY_FROM_USER_IOCTL(arg, (void __user *)data, sizeof(arg));

	LOCK_TEST_WITH_RETURN(dev, filp);
	num_buffers = PSB_NUM_VALIDATE_BUFFERS;

	ret = psb_validate_buffer_list(priv, arg.engine,
				       arg.buffer_list,
				       dev_priv->buffers, &num_buffers);

	if (ret)
		return ret;

	ret = psb_fixup_relocs(priv, &arg, dev_priv->buffers, num_buffers,
			       &cmd_buffer);

	if (ret)
		goto out_err0;

	/* switch (arg.engine) { ..... */

	ret = psb_submit_2d_cmdbuf(dev, cmd_buffer, arg.cmdbuf_offset,
				   arg.cmdbuf_size);

	if (ret)
		goto out_err0;

	ret = drm_fence_buffer_objects(dev, NULL, 0, NULL, &fence);
	if (ret)
		goto out_err0;

	if (!(arg.fence_flags & DRM_FENCE_FLAG_NO_USER)) {
		ret = drm_fence_add_user_object(priv, fence,
						arg.fence_flags &
						DRM_FENCE_FLAG_SHAREABLE);
		if (!ret) {
			atomic_inc(&fence->usage);
			arg.fence_handle = fence->base.hash.key;
		}
	}
	drm_fence_usage_deref_unlocked(dev, fence);

      out_err0:
	mutex_lock(&dev->struct_mutex);
	if (cmd_buffer)
		drm_bo_usage_deref_locked(cmd_buffer);
	psb_dereference_buffers_locked(dev_priv->buffers, num_buffers);
	mutex_unlock(&dev->struct_mutex);

	return ret;
}
