/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include <linux/config.h>

#ifdef CONFIG_USB_DEBUG
#define DEBUG 1
#endif

#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>
#include <linux/time.h>

#include <sound/driver.h>
#include <sound/core.h>
#include <sound/control.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>

#include "audio.h"
#include "config.h"
#include "driver.h"
#include "pcm.h"


/* hardware definition (capture) */
static snd_pcm_hardware_t snd_podxtpro_capture_hw = {
	.info = (SNDRV_PCM_INFO_MMAP |
					 SNDRV_PCM_INFO_INTERLEAVED |
					 SNDRV_PCM_INFO_BLOCK_TRANSFER |
					 SNDRV_PCM_INFO_MMAP_VALID |
					 SNDRV_PCM_INFO_SYNC_START),
	.formats =          SNDRV_PCM_FMTBIT_S24_3LE,
	.rates =            SNDRV_PCM_RATE_KNOT,
	.rate_min =         39063,
	.rate_max =         39063,
	.channels_min =     2,
	.channels_max =     2,
	.buffer_bytes_max = 60000,
	.period_bytes_min = PODXTPRO_ISO_PACKET_SIZE_LONG * PODXTPRO_BYTES_PER_FRAME,  /* at least the longest URB must fit into one period */
	.period_bytes_max = 8192,
	.periods_min =      1,
	.periods_max =      1024,
};


/*
	Find a free URB and submit it.
*/
static int submit_audio_in_urb(snd_pcm_substream_t *substream)
{
	int index;
	unsigned long flags;
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	int i, urb_size;
	struct urb *urb_in;

	spin_lock_irqsave(&chip->lock_audio_in, flags);
	index = find_first_zero_bit(&chip->active_urb_in, PODXTPRO_ISO_BUFFERS);

	if(index < 0 || index >= PODXTPRO_ISO_BUFFERS) {
		spin_unlock_irqrestore(&chip->lock_audio_in, flags);
		dev_err(s2m(substream), "No free URB found\n");
		return -EINVAL;
	}

	urb_in = chip->urb_audio_in[index];
	urb_size = 0;

	for(i = 0; i < PODXTPRO_ISO_PACKETS; ++i) {
		struct usb_iso_packet_descriptor *fin = &urb_in->iso_frame_desc[i];
		fin->offset = urb_size;
		fin->length = PODXTPRO_ISO_PACKET_SIZE_MAX;
		urb_size += PODXTPRO_ISO_PACKET_SIZE_MAX;
	}

	urb_in->transfer_buffer = chip->buffer_in + index * PODXTPRO_ISO_PACKETS * PODXTPRO_ISO_PACKET_SIZE_MAX;
	urb_in->transfer_buffer_length = urb_size;
	urb_in->context = substream;

	if(usb_submit_urb(urb_in, GFP_ATOMIC) == 0)
		set_bit(index, &chip->active_urb_in);
	else
		dev_err(s2m(substream), "URB in #%d submission failed\n", index);

	spin_unlock_irqrestore(&chip->lock_audio_in, flags);
	return 0;
}

/*
	Submit all currently available capture URBs.
*/
static int submit_audio_in_all_urbs(snd_pcm_substream_t *substream)
{
	int ret, i;

	for(i = 0; i < PODXTPRO_ISO_BUFFERS; ++i)
		if((ret = submit_audio_in_urb(substream)) < 0)
			return ret;

	return 0;
}

/*
	Unlink all currently active capture URBs.
*/
static void unlink_audio_in_urbs(snd_podxtpro_pcm_t *chip)
{
	unsigned int i;

	for(i = PODXTPRO_ISO_BUFFERS; i--;) {
		if(test_bit(i, &chip->active_urb_in)) {
			if(!test_and_set_bit(i, &chip->unlink_urb_in)) {
				struct urb *u = chip->urb_audio_in[i];
#ifdef URB_ASYNC_UNLINK
				u->transfer_flags |= URB_ASYNC_UNLINK;
#endif
				usb_unlink_urb(u);
			}
		}
	}
}

/*
	Wait until unlinking of all currently active capture URBs has been finished.
*/
static void wait_clear_audio_in_urbs(snd_podxtpro_pcm_t *chip)
{
	int timeout = HZ;
	unsigned int i;
	int alive;

	do {
		alive = 0;
		for (i = PODXTPRO_ISO_BUFFERS; i--;) {
			if (test_bit(i, &chip->active_urb_in))
				alive++;
		}
		if (! alive)
			break;
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1);
	} while (--timeout > 0);
	if (alive)
		snd_printk(KERN_ERR "timeout: still %d active urbs..\n", alive);

	chip->active_urb_in = 0;
	chip->unlink_urb_in = 0;
}

/*
	Unlink all currently active capture URBs, and wait for finishing.
*/
void unlink_wait_clear_audio_in_urbs(snd_podxtpro_pcm_t *chip)
{
	unlink_audio_in_urbs(chip);
	wait_clear_audio_in_urbs(chip);
}

/*
	Callback for completed capture URB.
*/
static void audio_in_callback(struct urb *urb, struct pt_regs *regs)
{
	int i, index, length = 0, shutdown = 0;
	int frames;
	unsigned long flags;

	snd_pcm_substream_t *substream = (snd_pcm_substream_t *)urb->context;
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	/* find index of URB */
	for(index = 0; index < PODXTPRO_ISO_BUFFERS; ++index)
		if(urb == chip->urb_audio_in[index])
			break;

	spin_lock_irqsave(&chip->lock_audio_in, flags);

	for(i = 0; i < PODXTPRO_ISO_PACKETS; ++i) {
		char *fbuf;
		int fsize;
		struct usb_iso_packet_descriptor *fin = &urb->iso_frame_desc[i];

		if(fin->status == -18) {
			shutdown = 1;
			break;
		}

		fbuf = urb->transfer_buffer + fin->offset;
		fsize = fin->actual_length;
		length += fsize;

		if(fsize > 0) {
			frames = fsize / PODXTPRO_BYTES_PER_FRAME;

			if(chip->pos_in_done + frames > runtime->buffer_size) {
				/*
					The transferred area goes over buffer boundary,
					copy two separate chunks.
				*/
				int len;
				len = runtime->buffer_size - chip->pos_in_done;

				if(len > 0) {
					memcpy(runtime->dma_area + chip->pos_in_done * PODXTPRO_BYTES_PER_FRAME, fbuf, len * PODXTPRO_BYTES_PER_FRAME);
					memcpy(runtime->dma_area, fbuf + len * PODXTPRO_BYTES_PER_FRAME, (frames - len) * PODXTPRO_BYTES_PER_FRAME);
				}
				else
					dev_err(s2m(substream), "driver bug: len = %d\n", len);  /* this is somewhat paranoid */
			}
			else {
				/* copy single chunk */
				memcpy(runtime->dma_area + chip->pos_in_done * PODXTPRO_BYTES_PER_FRAME, fbuf, fsize * PODXTPRO_BYTES_PER_FRAME);
			}

			if((chip->pos_in_done += frames) >= runtime->buffer_size)
				chip->pos_in_done -= runtime->buffer_size;
		}
	}

	clear_bit(index, &chip->active_urb_in);

	if(test_bit(index, &chip->unlink_urb_in))  /* better use a chip->flag! */
		shutdown = 1;

	spin_unlock_irqrestore(&chip->lock_audio_in, flags);

	if(!shutdown) {
		submit_audio_in_urb(substream);

		if((chip->bytes_in += length) >= chip->period_in) {
			chip->bytes_in -= chip->period_in;
			snd_pcm_period_elapsed(substream);
		}
	}
}

/* open capture callback */
static int snd_podxtpro_capture_open(snd_pcm_substream_t *substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	runtime->hw = snd_podxtpro_capture_hw;
	return 0;
}

/* close capture callback */
static int snd_podxtpro_capture_close(snd_pcm_substream_t *substream)
{
	return 0;
}

/* hw_params capture callback */
static int snd_podxtpro_capture_hw_params(snd_pcm_substream_t *substream, snd_pcm_hw_params_t *hw_params)
{
	int ret;
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);

	if((ret = snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params))) < 0)
		return ret;

	chip->period_in = params_period_bytes(hw_params);
	chip->buffer_in = kmalloc(PODXTPRO_ISO_BUFFERS * PODXTPRO_ISO_PACKETS * PODXTPRO_ISO_PACKET_SIZE_MAX, GFP_KERNEL);

	if(!chip->buffer_in) {
		dev_err(s2m(substream), "cannot malloc buffer_in\n");
		return -ENOMEM;
	}

	return 0;
}

/* hw_free capture callback */
static int snd_podxtpro_capture_hw_free(snd_pcm_substream_t *substream)
{
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	unlink_wait_clear_audio_in_urbs(chip);

	if(chip->buffer_in) {
		kfree(chip->buffer_in);
		chip->buffer_in = 0;
	}

	return snd_pcm_lib_free_pages(substream);
}

/* prepare capture callback */
static int snd_podxtpro_capture_prepare(snd_pcm_substream_t *substream)
{
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	chip->pos_in_done = 0;
	chip->bytes_in = 0;
	return 0;
}

/* trigger callback */
int snd_podxtpro_capture_trigger(snd_pcm_substream_t *substream, int cmd)
{
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	int err;

	switch(cmd) {
	case SNDRV_PCM_TRIGGER_START:
		if(!chip->running_capture) {
			err = submit_audio_in_all_urbs(substream);
			if(err < 0) return err;
			chip->running_capture = 1;
		}

		break;

	case SNDRV_PCM_TRIGGER_STOP:
		if(chip->running_capture) {
			unlink_audio_in_urbs(chip);
			chip->running_capture = 0;
		}

		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* capture pointer callback */
static snd_pcm_uframes_t
snd_podxtpro_capture_pointer(snd_pcm_substream_t *substream)
{
	snd_podxtpro_pcm_t *chip = snd_pcm_substream_chip(substream);
	return chip->pos_in_done;
}

/* capture operators */
snd_pcm_ops_t snd_podxtpro_capture_ops = {
	.open =        snd_podxtpro_capture_open,
	.close =       snd_podxtpro_capture_close,
	.ioctl =       snd_pcm_lib_ioctl,
	.hw_params =   snd_podxtpro_capture_hw_params,
	.hw_free =     snd_podxtpro_capture_hw_free,
	.prepare =     snd_podxtpro_capture_prepare,
	.trigger =     snd_podxtpro_trigger,
	.pointer =     snd_podxtpro_capture_pointer,
};

int create_audio_in_urbs(snd_podxtpro_pcm_t *chip)
{
	int i;
	usb_podxtpro_t *podxtpro = chip->podxtpro;

	/* create audio URBs and fill in constant values: */
	for(i = 0; i < PODXTPRO_ISO_BUFFERS; ++i)
		{
			struct urb *urb;

			/* URB for audio in: */
			urb = chip->urb_audio_in[i] = usb_alloc_urb(PODXTPRO_ISO_PACKETS, GFP_KERNEL);

			if(urb == NULL) {
				dev_err(podxtpro->msgdev, "Out of memory\n");
				return -ENOMEM;
			}

			urb->dev = podxtpro->usbdev;
			urb->pipe = usb_rcvisocpipe(podxtpro->usbdev, PODXTPRO_EP_READ_AUDIO & USB_ENDPOINT_NUMBER_MASK);
			urb->transfer_flags = URB_ISO_ASAP;
			urb->start_frame = -1;
			urb->number_of_packets = PODXTPRO_ISO_PACKETS;
			urb->interval = PODXTPRO_ISO_INTERVAL;
			urb->error_count = 0;
			urb->complete = audio_in_callback;
		}

	return 0;
}
