/*
  
  Marvell Libertas wireless driver

  Copyright (c) 2005 Luc Saillard <luc@saillard.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/ethtool.h>
#include <linux/mii.h>

#include "mrv8k.h"


static int mrv8k_wx_get_name(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

        if (!(priv->status & STATUS_ASSOCIATED))
                strcpy(wrqu->name, "unassociated");
        else
                snprintf(wrqu->name, IFNAMSIZ, "IEEE 802.11b");

        return 0;
}

static int mrv8k_wx_set_freq(struct net_device *dev,
			     struct iw_request_info *info,
			     union  iwreq_data *wrqu,
			     char  *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	struct iw_freq *fwrq = &wrqu->freq;
	int channel;

	if (fwrq->e == 1)
		channel = ieee80211_freq_to_channel(priv->ieee, fwrq->m);
	else
		channel = wrqu->freq.m;

	if (channel)
		priv->config |= CFG_STATIC_CHANNEL;
	else
		priv->config &= ~CFG_STATIC_CHANNEL;

	priv->channel = channel;
	return mrv8k_set_bind_channel(priv, channel);
}

static int mrv8k_wx_get_freq(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

        wrqu->freq.e = 0;

        /* If we are associated, trying to associate, or have a statically
         * configured CHANNEL then return that; otherwise return ANY */
        if (  (priv->config & CFG_STATIC_CHANNEL)
	   || (priv->status & STATUS_ASSOCIATED))
                wrqu->freq.m = priv->channel;
        else
                wrqu->freq.m = 0;

        return 0;
}

static int mrv8k_wx_set_mode(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	int err;

	switch (wrqu->mode) {
        case IW_MODE_ADHOC:
                err = mrv8k_switch_mode(priv, IW_MODE_ADHOC);
                break;
        case IW_MODE_INFRA:
        case IW_MODE_AUTO: 
                err = mrv8k_switch_mode(priv, IW_MODE_INFRA);
                break;
        default:
		return -EINVAL;
        }

	return err;
}

static int mrv8k_wx_get_mode(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	wrqu->mode = priv->ieee->iw_mode;
	return 0;
}

static int mrv8k_wx_set_wap(struct net_device *dev,
                            struct iw_request_info *info,
                            union  iwreq_data *wrqu,
			    char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	
	if (wrqu->ap_addr.sa_family != ARPHRD_ETHER)
		return -EINVAL;

	if (is_valid_ether_addr(wrqu->ap_addr.sa_data)) {
		priv->config |= CFG_STATIC_BSSID;
		memcpy(priv->bssid, wrqu->ap_addr.sa_data, ETH_ALEN);
	} else {
		priv->config &= ~CFG_STATIC_BSSID;
		memset(priv->bssid, 0, ETH_ALEN);
	}

	mrv8k_hw_send_bind_ap(priv, priv->bssid);

	return 0;
}


static int mrv8k_wx_get_wap(struct net_device *dev,
                            struct iw_request_info *info,
                            union  iwreq_data *wrqu,
			    char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	wrqu->ap_addr.sa_family = ARPHRD_ETHER;

	if (priv->config & CFG_STATIC_BSSID)
		memcpy(wrqu->ap_addr.sa_data, priv->bssid, ETH_ALEN);
	else
		memset(wrqu->ap_addr.sa_data, 0x00, ETH_ALEN);
	return 0;
}

static int mrv8k_wx_set_scan(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	int err;

	MRV8K_DEBUG_WX("Initiating scan...\n");


        err = mrv8k_start_scan(priv);


	return err;
}


static int mrv8k_wx_get_scan(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;

	return -EINVAL;
}


static int mrv8k_wx_set_essid(struct net_device *dev,
                              struct iw_request_info *info,
                              union  iwreq_data *wrqu,
			      char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	char *essid = "";	/* ANY */
	int length = 0;

	if (wrqu->essid.flags && wrqu->essid.length) {
		length = wrqu->essid.length - 1;
		essid = extra;
	}
	if (length == 0) {
		MRV8K_DEBUG_WX("Setting ESSID to ANY\n");
		priv->config &= ~CFG_STATIC_ESSID;
		return 0;
	}
	if (wrqu->essid.length > IW_ESSID_MAX_SIZE) {
		return -E2BIG;
	}

	if (priv->essid_len == length) {
		if (memcmp(priv->essid, essid, length) == 0) {
			MRV8K_DEBUG_WX("not changing ESSID (it's the same)\n");
			return 0;
		}
	}

	priv->config |= CFG_STATIC_ESSID;

	priv->essid_len = length;
	memcpy(priv->essid, essid, priv->essid_len);

	if (priv->status & (STATUS_ASSOCIATED | STATUS_ASSOCIATING)) {
		MRV8K_DEBUG_WX("Disassociating due to a ESSID change (todo)\n");
	}

	return 0;
}


static int mrv8k_wx_get_essid(struct net_device *dev,
                              struct iw_request_info *info,
                              union  iwreq_data *wrqu,
			      char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	MRV8K_DEBUG_WX("%s\n", __FUNCTION__);
	if (priv->config & CFG_STATIC_ESSID) {

		MRV8K_DEBUG_WX("Getting essid: ????\n");
		memcpy(extra, priv->essid, priv->essid_len);
		wrqu->essid.length = priv->essid_len;
		wrqu->essid.flags = 1;	/* active */

	} else {
		MRV8K_DEBUG_WX("Getting essid: ANY\n");
		wrqu->essid.length = 0;
		wrqu->essid.flags = 0;	/* active */
	}

	return 0;
}



static int mrv8k_wx_set_nick(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	MRV8K_DEBUG_WX("%s() length=%d\n", __FUNCTION__, wrqu->data.length);
	if (wrqu->data.length >= IW_ESSID_MAX_SIZE)
		return -E2BIG;

	memset(priv->nick, 0, IW_ESSID_MAX_SIZE);
	memcpy(priv->nick, extra, wrqu->data.length);
	return 0;
}


static int mrv8k_wx_get_nick(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	MRV8K_DEBUG_WX("%s()\n", __FUNCTION__);
	
	wrqu->data.length = strlen(priv->nick) + 1;
	memcpy(extra, priv->nick, wrqu->data.length);

	return 0;
}

static int mrv8k_is_rate_valid(int rate)
{
	int i;

	static const unsigned char supported_rates_cck[] = {
		IEEE80211_CCK_RATE_1MB,
		IEEE80211_CCK_RATE_2MB,
		IEEE80211_CCK_RATE_5MB,
		IEEE80211_CCK_RATE_11MB,
	};
	static const unsigned char supported_rates_ofdm[] = {
		IEEE80211_OFDM_RATE_6MB,
		IEEE80211_OFDM_RATE_9MB,
		IEEE80211_OFDM_RATE_12MB,
		IEEE80211_OFDM_RATE_18MB,
		IEEE80211_OFDM_RATE_24MB,
		IEEE80211_OFDM_RATE_36MB,
		IEEE80211_OFDM_RATE_48MB,
		IEEE80211_OFDM_RATE_54MB,
	};

	/* TODO: remove channel not authorized by geo */

	for (i=0; i<ARRAY_SIZE(supported_rates_cck); i++) {
		if (supported_rates_cck[i] == rate)
			return 0;
	}

	for (i=0; i<ARRAY_SIZE(supported_rates_ofdm); i++) {
		if (supported_rates_ofdm[i] == rate)
			return 0;
	}

	return -EINVAL;
}

static int mrv8k_wx_set_rate(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
#if 0
	int rate;
        if (wrqu->bitrate.value == -1)
		rate = 0;
	else {
		rate = wrqu->bitrate.value / 500000;
		if (mrv8k_is_rate_valid(rate) < 0)
			return -EINVAL;
	}

        priv->bitrate = rate;

        if (wrqu->bitrate.fixed)
                core->config.user.flags |= CONFIG_FIX_BITRATE;
        else
                core->config.user.flags &= ~CONFIG_FIX_BITRATE;

        core->config.user_commit |= COMMIT_BITRATE;


#endif

	return -EINVAL;
}


static int mrv8k_wx_get_rate(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	wrqu->bitrate.value = 0;

	return 0;
}



static int mrv8k_wx_set_rts(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	int value;

        if (wrqu->rts.fixed == 0)
                return -EINVAL;

        if (wrqu->rts.disabled)
                value = MRV8K_RTS_THRESHOLD_DEFAULT;
        else {
                if (wrqu->rts.value < MRV8K_RTS_THRESHOLD_MIN)
			return -EINVAL;
                if (wrqu->rts.value > MRV8K_RTS_THRESHOLD_MAX)
			return -EINVAL;
                value = wrqu->rts.value;
        }
	priv->rts_threshold = value;

	mrv8k_set_rts_threshold(priv, priv->rts_threshold);

	return 0;
}


static int mrv8k_wx_get_rts(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	wrqu->rts.value = priv->rts_threshold;
	wrqu->rts.fixed = 1;

	return 0;
}




static int mrv8k_wx_set_frag(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	int err;

	if (priv->ieee->iw_mode != IW_MODE_ADHOC)
		return -EINVAL;

	if (wrqu->txpower.disabled == 1 || wrqu->txpower.fixed == 0)
                priv->radio_on = 0;
        else {
                priv->radio_on = 0;
                if (wrqu->txpower.value < MRV8K_TX_POWER_MIN)
                        return -EINVAL;
                if (wrqu->txpower.value > MRV8K_TX_POWER_MAX)
                        return -EINVAL;
		/* TODO: Need to convert dbm in range 0-0xffff */
                priv->tx_antenna = wrqu->txpower.value;
        }

	err = mrv8k_set_rts_threshold(priv, priv->rts_threshold);

	return err;
}


static int mrv8k_wx_get_frag(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	wrqu->frag.value = priv->rts_threshold;
	wrqu->frag.fixed = 0;
	wrqu->frag.disabled = 0;

	return 0;
}



static int mrv8k_wx_set_txpow(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	return -EINVAL;
}


static int mrv8k_wx_get_txpow(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	return -EINVAL;
}


static int mrv8k_wx_set_retry(struct net_device *dev,
                             struct iw_request_info *info,
                             union  iwreq_data *wrqu,
			     char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	return -EINVAL;
}


static int mrv8k_wx_get_retry(struct net_device *dev,
                              struct iw_request_info *info,
                              union  iwreq_data *wrqu,
			      char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	return -EINVAL;
}



static int mrv8k_wx_set_encode(struct net_device *dev,
                               struct iw_request_info *info,
                               union  iwreq_data *wrqu,
			       char   *key)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	return ieee80211_wx_set_encode(priv->ieee, info, wrqu, key);
}


static int mrv8k_wx_get_encode(struct net_device *dev,
                               struct iw_request_info *info,
                               union  iwreq_data *wrqu,
			       char   *key)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	return ieee80211_wx_get_encode(priv->ieee, info, wrqu, key);
}

static int mrv8k_wx_set_preamble(struct net_device *dev,
				 struct iw_request_info *info,
				 union  iwreq_data *wrqu,
				 char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	int mode, err;

	mode = *(int *)extra;

	if (mode == MRV8K_PREAMBLE_AUTO)
		priv->preamble = MRV8K_PREAMBLE_AUTO;
	else if (mode == MRV8K_PREAMBLE_SHORT)
		priv->preamble = MRV8K_PREAMBLE_SHORT;
	else if (mode == MRV8K_PREAMBLE_LONG)
		priv->preamble = MRV8K_PREAMBLE_LONG;
	else
		return -EINVAL;

	err = mrv8k_hw_send_radio_setting(priv, priv->radio_on, priv->preamble);

	return err;
}


static int mrv8k_wx_get_preamble(struct net_device *dev,
				 struct iw_request_info *info,
				 union  iwreq_data *wrqu,
				 char   *extra)
{
        struct mrv8k_private *priv = ieee80211_priv(dev);
	
        if (priv->preamble == MRV8K_PREAMBLE_AUTO)
                snprintf(wrqu->name, IFNAMSIZ, "auto (%d)", MRV8K_PREAMBLE_AUTO);
	else if (priv->preamble == MRV8K_PREAMBLE_SHORT)
                snprintf(wrqu->name, IFNAMSIZ, "short (%d)", MRV8K_PREAMBLE_SHORT);
	else if (priv->preamble == MRV8K_PREAMBLE_LONG)
                snprintf(wrqu->name, IFNAMSIZ, "long (%d)", MRV8K_PREAMBLE_LONG);

	return 0;
}








static iw_handler mrv8k_wx_handlers[] =
{
        NULL,			/* SIOCSIWCOMMIT */
        mrv8k_wx_get_name,	/* SIOCGIWNAME */
        NULL,			/* SIOCSIWNWID */
        NULL,			/* SIOCGIWNWID */
        mrv8k_wx_set_freq,	/* SIOCSIWFREQ */
        mrv8k_wx_get_freq,	/* SIOCGIWFREQ */
        mrv8k_wx_set_mode,	/* SIOCSIWMODE */
        mrv8k_wx_get_mode,	/* SIOCGIWMODE */
        NULL,			/* SIOCSIWSENS */
        NULL,			/* SIOCGIWSENS */
        NULL,			/* SIOCSIWRANGE */
        NULL,			/* SIOCGIWRANGE */
        NULL,			/* SIOCSIWPRIV */
        NULL,			/* SIOCGIWPRIV */
        NULL,			/* SIOCSIWSTATS */
        NULL,			/* SIOCGIWSTATS */
        NULL,			/* SIOCSIWSPY */
        NULL,			/* SIOCGIWSPY */
        NULL,			/* SIOCGIWTHRSPY */
        NULL,			/* SIOCWIWTHRSPY */
        mrv8k_wx_set_wap,	/* SIOCSIWAP */
        mrv8k_wx_get_wap,	/* SIOCGIWAP */
        NULL,			/* -- hole -- */
        NULL,			/* SIOCGIWAPLIST -- deprecated */
        mrv8k_wx_set_scan,      /* SIOCSIWSCAN */
        mrv8k_wx_get_scan,      /* SIOCGIWSCAN */
        mrv8k_wx_set_essid,     /* SIOCSIWESSID */
        mrv8k_wx_get_essid,     /* SIOCGIWESSID */
        mrv8k_wx_set_nick,      /* SIOCSIWNICKN */
        mrv8k_wx_get_nick,      /* SIOCGIWNICKN */
        NULL,                     /* -- hole -- */
        NULL,                     /* -- hole -- */
        mrv8k_wx_set_rate,      /* SIOCSIWRATE */
        mrv8k_wx_get_rate,      /* SIOCGIWRATE */
        mrv8k_wx_set_rts,       /* SIOCSIWRTS */
        mrv8k_wx_get_rts,       /* SIOCGIWRTS */
        mrv8k_wx_set_frag,      /* SIOCSIWFRAG */
        mrv8k_wx_get_frag,      /* SIOCGIWFRAG */
        mrv8k_wx_set_txpow,     /* SIOCSIWTXPOW */
        mrv8k_wx_get_txpow,     /* SIOCGIWTXPOW */
        mrv8k_wx_set_retry,     /* SIOCSIWRETRY */
        mrv8k_wx_get_retry,     /* SIOCGIWRETRY */
        mrv8k_wx_set_encode,    /* SIOCSIWENCODE */
        mrv8k_wx_get_encode,    /* SIOCGIWENCODE */
        NULL,			/* SIOCSIWPOWER */
        NULL,			/* SIOCGIWPOWER */
};

#define MRV8K_PRIV_SET_PREAMBLE   SIOCIWFIRSTPRIV
#define MRV8K_PRIV_GET_PREAMBLE   SIOCIWFIRSTPRIV+1

static const iw_handler mrv8k_private_handler[] = {
	mrv8k_wx_set_preamble,
	mrv8k_wx_get_preamble
};

static const struct iw_priv_args mrv8k_private_args[] = {
	{
		MRV8K_PRIV_SET_PREAMBLE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_preamble"
	},
	{
		MRV8K_PRIV_GET_PREAMBLE,
		0, IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | IFNAMSIZ, "get_preamble"
	},

};

static struct iw_handler_def mrv8k_wx_handler_def =
{
        .num_standard = ARRAY_SIZE(mrv8k_wx_handlers),
        .standard = mrv8k_wx_handlers,

        .num_private = ARRAY_SIZE(mrv8k_private_handler),
        .private = mrv8k_private_handler,

        .num_private_args = ARRAY_SIZE(mrv8k_private_args),
        .private_args = mrv8k_private_args,
};



