/*
  
  Marvell Libertas wireless driver

  Copyright (c) 2005 Luc Saillard <luc@saillard.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/



#include <linux/ctype.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include "mrv8k.h"

static void mrv8k_irq_tasklet_events(struct mrv8k_private *priv);
static void mrv8k_decode_packet_configuration(struct mrv8k_private *priv, struct mrv8k_event_msg *msg);
static void mrv8k_decode_packet_statistics(struct mrv8k_private *priv, struct mrv8k_event_msg *msg);
static void mrv8k_irq_event_tx_complete(struct mrv8k_private *priv);
static void mrv8k_irq_event_rx_transfer(struct mrv8k_private *priv);
static void mrv8k_decode_packet_data(struct mrv8k_private *priv,
   				     struct mrv8k_rx_data *data,
				     struct ieee80211_rx_stats *stats);

static char *snprint_line(char *buf, size_t count,
                          const u8 *data, u32 len, u32 ofs)
{
        int out, i, j, l;
        char c;

        out = snprintf(buf, count, "%08X", ofs);

        for (l = 0, i = 0; i < 2; i++) {
                out += snprintf(buf + out, count - out, " ");
                for (j = 0; j < 8 && l < len; j++, l++)
                        out += snprintf(buf + out, count - out, "%02X ",
                                        data[(i * 8 + j)]);
                for (; j < 8; j++)
                        out += snprintf(buf + out, count - out, "   ");
        }

        out += snprintf(buf + out, count - out, " ");
        for (l = 0, i = 0; i < 2; i++) {
                out += snprintf(buf + out, count - out, " ");
                for (j = 0; j < 8 && l < len; j++, l++) {
                        c = data[(i * 8 + j)];
                        if (!isascii(c) || !isprint(c))
                                c = '.';

                        out += snprintf(buf + out, count - out, "%c", c);
                }

                for (; j < 8; j++)
                        out += snprintf(buf + out, count - out, " ");
        }

        return buf;
}

static void printk_buf(const void *data, u32 len)
{
        char line[81];
        u32 ofs = 0;

        while (len) {
                printk(KERN_DEBUG "%s\n",
                       snprint_line(line, sizeof(line), data,
                                    min(len, 16U), ofs));
                ofs += 16;
		data += 16;
                len -= min(len, 16U);
        }
}


/*
 */
void mrv8k_clear_interruptA(struct mrv8k_private *priv)
{
	mrv8k_write_nic(priv, MRV8K_REG_INTA_MASK, 0);
	mrv8k_read_nic(priv, MRV8K_REG_C14);
}

/*
 */
void mrv8k_enable_interrupt(struct mrv8k_private *priv)
{
	mrv8k_write_nic(priv, MRV8K_REG_INTA_MASK, MRV8K_INTA_MASK);
	mrv8k_read_nic(priv, MRV8K_REG_C14);
	mrv8k_write_nic(priv, MRV8K_REG_INTB_MASK, MRV8K_INTB_MASK);
	mrv8k_read_nic(priv, MRV8K_REG_C14);
}

/**
 *
 *
 */
irqreturn_t mrv8k_interrupt(int irq, void *data, struct pt_regs *regs)
{
	struct mrv8k_private *priv = data;
	unsigned int status, status_mask;

	MRV8K_DEBUG_TRACE("%s: %s\n", __FUNCTION__, priv->net_dev->name);

	if (readl(priv->dma_cookie) != MRV8K_MAGIC_COOKIE)
		return IRQ_NONE;

	status = mrv8k_read_nic(priv, MRV8K_REG_INT_STATUS);;
	status_mask = (~status) & MRV8K_INTA_MASK;	/* Ack irq */
	mrv8k_write_nic(priv, MRV8K_REG_INT_STATUS, status_mask);
	mrv8k_read_nic(priv, MRV8K_REG_C14);

	if (status == 0xFFFFFFFF)
		return IRQ_NONE;
	if (status == 0)
		return IRQ_NONE;

	status &= MRV8K_INTA_MASK;
	if (status == 0)
		return IRQ_NONE;

	mrv8k_clear_interruptA(priv);
	priv->irq_status |= status;
	tasklet_schedule(&priv->irq_tasklet);
	return IRQ_HANDLED;
}


/**
 *
 *
 */
void mrv8k_irq_tasklet(struct mrv8k_private *priv)
{
	int count=10;
	MRV8K_DEBUG_TRACE("%s: %s\n", __FUNCTION__, priv->net_dev->name);

	while (priv->irq_status && count)
	{
		if (priv->irq_status & MRV8K_INTA_RX_TRANSFER) {

			MRV8K_DEBUG_TRACE("Interrupt Rx Transfer received\n");
			priv->irq_status &= ~MRV8K_INTA_RX_TRANSFER;
			mrv8k_irq_event_rx_transfer(priv);
		}
		if (priv->irq_status & MRV8K_INTA_TX_COMPLETE) {

			MRV8K_DEBUG_TRACE("Interrupt Tx Transfer completed\n");
			priv->irq_status &= ~MRV8K_INTA_TX_COMPLETE;
			mrv8k_irq_event_tx_complete(priv);
		}
		if (priv->irq_status & MRV8K_INTA_EVENT_INTERRUPT) {

			MRV8K_DEBUG_TRACE("Interrupt Event fired\n");
			priv->irq_status &= ~MRV8K_INTA_EVENT_INTERRUPT;
			mrv8k_irq_tasklet_events(priv);

		}
		if (priv->irq_status & MRV8K_INTA_BIT3) {

			MRV8K_DEBUG_TRACE("Interrupt unknown interrupt\n");
			priv->irq_status &= ~MRV8K_INTA_BIT3;

		}
		count--;
	}

	mrv8k_enable_interrupt(priv);
}

static void mrv8k_irq_tasklet_events(struct mrv8k_private *priv)
{
	struct mrv8k_cmd_packet *packet;
	struct mrv8k_event_msg *msg;
	unsigned long flags;
	int frame_type, frame_len;

	MRV8K_DEBUG_TRACE("%s\n", __FUNCTION__);

	packet = priv->current_cmd;
	msg = (struct mrv8k_event_msg *)packet->cmd;

	frame_type = le16_to_cpu(msg->type);
	frame_len = le16_to_cpu(msg->len);

	switch (frame_type) {

		case MRV8K_EVENT_CONFIG:
			MRV8K_DEBUG_TRACE("Received an event message CONFIG len=%4.4x\n", frame_len);
			mrv8k_decode_packet_configuration(priv, msg);
			break;

		case MRV8K_ACK_STATISTICS:
			MRV8K_DEBUG_TRACE("Received an event message STATISTICS len=%4.4x\n", frame_len);
			mrv8k_decode_packet_statistics(priv, msg);
			break;

		default:
			MRV8K_DEBUG_TRACE("Received an event message type=%4.4x len=%4.4x\n", frame_type, frame_len);
			mrv8k_print_command(packet->cmd);
			break;

	}

	spin_lock_irqsave(&priv->cmd_send_lock, flags);
	list_add(&priv->current_cmd->list, &priv->cmd_free_list);
	priv->current_cmd = NULL;
	__mrv8k_hw_flush_command_queue(priv);
	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);
}

/**
 *
 */
static void mrv8k_decode_packet_configuration(struct mrv8k_private *priv, struct mrv8k_event_msg *msg)
{
  struct mrv8k_event_config *config;
  static u16 models[6] = { 0x10, 0x20, 0x30, 0x31, 0x32, 0x40 };
  unsigned int addr1, addr2, addr3;
#if CONFIG_MRV8K_DEBUG
  unsigned int i, geo_code, model_code, firmware_version, revision;
  u16 m;

  MRV8K_BUG_ON_CHECK_STRUCT(mrv8k_event_config, msg->len);
#endif
  config = (struct mrv8k_event_config *)msg;
  memcpy(priv->net_dev->dev_addr, config->macaddr, ETH_ALEN);
  addr1 = le32_to_cpu(config->addr1);
  addr2 = le32_to_cpu(config->addr2);
  addr3 = le32_to_cpu(config->addr3);

  priv->register_ring_tx_buffer_addr = addr1 & 0xffff;
  priv->register_rx_buffer0 = addr2 & 0xffff;
  priv->register_rx_buffer1 = addr3 & 0xffff;

  model_code = 0x10;
  geo_code = 0;

  for (i=0; i<6; i++) {
     m = config->model_version;
     if (m == models[i]) {
	geo_code = i;
	model_code = m;
	break;
     }
  }
  priv->geo_code = geo_code;
  priv->model_code = model_code;

#if CONFIG_MRV8K_DEBUG
  firmware_version = le32_to_cpu(config->firmware_version);
  revision = le16_to_cpu(config->revision);
  MRV8K_DEBUG_CMD("Packet configuration:\n");
  printk_buf(msg, sizeof(struct mrv8k_event_config));
  MRV8K_DEBUG_CMD(" firmware_version = %d/%8.8x\n", firmware_version, firmware_version);
  MRV8K_DEBUG_CMD(" macaddr = %x:%x:%x:%x:%x:%x\n", config->macaddr[0], config->macaddr[1],
      config->macaddr[2], config->macaddr[3], config->macaddr[4], config->macaddr[5]);
  MRV8K_DEBUG_CMD(" revision = %4.4x\n", revision);
  MRV8K_DEBUG_CMD(" model = %4.4x, modeltype = %4.4x\n", model_code, geo_code);
  MRV8K_DEBUG_CMD(" addr1 = %8.8x/ tx_buffer  = %8.8x\n", addr1, priv->register_ring_tx_buffer_addr);
  MRV8K_DEBUG_CMD(" addr2 = %8.8x/ rx_buffer0 = %8.8x\n", addr2, priv->register_rx_buffer0);
  MRV8K_DEBUG_CMD(" addr3 = %8.8x/ rx_buffer1 = %8.8x\n", addr3, priv->register_rx_buffer1);
#endif

  wake_up_interruptible(&priv->config_wait_queue);
}

static int tx_rates_table[] = {
   10000, 20000, 55000, 110000, 220000,
   60000, 90000, 120000, 180000, 240000, 360000, 480000, 540000, 720000
};

static void mrv8k_irq_event_tx_complete(struct mrv8k_private *priv)
{
	struct hw_tx_ctrl_buffer *ctrl;
	struct mrv8k_hw_frame *frame;
	int fc;

	MRV8K_DEBUG_TRACE("%s\n", __FUNCTION__);

	while (priv->tx_last_ack_index != priv->tx_index)  {

		ctrl = priv->tx_ctrl_buffers[priv->tx_last_ack_index];
		if (le32_to_cpu(ctrl->magic) & 0x80000000)
			break;	/* This packet is not ack, so no more packet can be ack */

		if (le32_to_cpu(ctrl->magic) & 0x80) {

			priv->ieee->stats.tx_packets++;
			frame = priv->tx_data_buffers[priv->tx_last_ack_index];
			fc = le16_to_cpu(frame->header.frame_ctl);
			if (WLAN_FC_GET_TYPE(fc) != IEEE80211_FTYPE_MGMT) {
				if (ctrl->speedlinkcanal >= 14)
					priv->tx_rates = 10000;
				else
					priv->tx_rates = tx_rates_table[ctrl->speedlinkcanal];
			}
		}

		atomic_sub(le16_to_cpu(ctrl->datalen), &priv->tx_pending_bytes);
		atomic_dec(&priv->tx_pending_buffers);
		ctrl->datalen = 0;
		ctrl->magic = 0;

		if (priv->tx_last_ack_index >= (MRV8K_TX_BUFFERS-1))
			priv->tx_last_ack_index = 0;
		else
			priv->tx_last_ack_index++;

	}

	mrv8k_send_frames(priv);

}

/**
 * Small function to return the index number of the rx_packets
 * Return -1, when the physical address is not found
 */
static int find_index_for_phys_addr(const struct mrv8k_private *priv, u32 addr)
{
	int i;

	for (i=0; i<MRV8K_RX_BUFFERS; i++) {
		if (priv->rx_packets[i].ctrl_dma == addr)
			return i;
  	}

	return -1;
}

static char *frame_stype[16] = {
   "ASSOC_REQ",
   "ASSOC_RESP",
   "REASSOC_REQ",
   "REASSOC_RESP",
   "PROBE_REQ",
   "PROBE_RESP",
   "invalid type 6",
   "invalid type 7",
   "BEACON",
   "ATIM",
   "DISASSOC",
   "AUTH",
   "DEAUTH",
   "ACTION",
   "invalid type 14",
   "invalid type 15"
};

static struct mfie_string {
   int id;
   const char *text;
} mfie_strings[] = {
    { MFIE_TYPE_SSID, "TYPE_SSID" },
    { MFIE_TYPE_RATES, "TYPE_RATES" }, 
    { MFIE_TYPE_FH_SET, "TYPE_FH_SET" },
    { MFIE_TYPE_DS_SET, "TYPE_DS_SET" },
    { MFIE_TYPE_CF_SET, "TYPE_CF_SET" },
    { MFIE_TYPE_TIM, "TYPE_TIM" },
    { MFIE_TYPE_IBSS_SET, "TYPE_IBSS_SET" }, 
    { MFIE_TYPE_COUNTRY, "TYPE_COUNTRY" },
    { MFIE_TYPE_HOP_PARAMS, "TYPE_HOP_PARAMS" },
    { MFIE_TYPE_HOP_TABLE, "TYPE_HOP_TABLE" },
    { MFIE_TYPE_REQUEST, "TYPE_REQUEST" },
    { MFIE_TYPE_CHALLENGE, "TYPE_CHALLENGE" },
    { MFIE_TYPE_POWER_CONSTRAINT, "TYPE_POWER_CONSTRAINT" },
    { MFIE_TYPE_POWER_CAPABILITY, "TYPE_POWER_CAPABILITY" },
    { MFIE_TYPE_TPC_REQUEST, "TYPE_TPC_REQUEST" },
    { MFIE_TYPE_TPC_REPORT, "TYPE_TPC_REPORT" },
    { MFIE_TYPE_SUPP_CHANNELS, "TYPE_SUPP_CHANNELS" },
    { MFIE_TYPE_CSA, "TYPE_CSA" },
    { MFIE_TYPE_MEASURE_REQUEST, "TYPE_MEASURE_REQUEST" },
    { MFIE_TYPE_MEASURE_REPORT, "TYPE_MEASURE_REPORT" },
    { MFIE_TYPE_QUIET, "TYPE_QUIET" },
    { MFIE_TYPE_IBSS_DFS, "TYPE_IBSS_DFS" },
    { MFIE_TYPE_ERP_INFO, "TYPE_ERP_INFO" },
    { MFIE_TYPE_RSN, "TYPE_RSN" },
    { MFIE_TYPE_RATES_EX, "TYPE_RATES_EX" },
    { MFIE_TYPE_GENERIC, "TYPE_GENERIC" },                
    { MFIE_TYPE_QOS_PARAMETER, "TYPE_QOS_PARAMETER" },
    { -1, NULL}
};

static const unsigned char *get_ie_type_name(int id)
{
	struct mfie_string *mfie;

	mfie = mfie_strings;
	while (mfie->id >=0 ) {
		if (mfie->id == id) {
		   return mfie->text;
		}
		mfie++;
	}
	return NULL;
}

static void mrv8k_print_IEs(const struct ieee80211_info_element *ie, int left)
{
	const char *ie_name;

	while (left >= sizeof(struct ieee80211_info_element)) {

	   	/* Sanity check */
		if (sizeof(struct ieee80211_info_element) + ie->len > left) {
			MRV8K_DEBUG_RX("info_element parse failed: "
			    	       "size of info_element (%d) is greater "
				       "than remaining bytes (%d)\n",
				       ie->len + 2, left);
			return;
		}


		ie_name = get_ie_type_name(ie->id);
		if (ie_name)
		  MRV8K_DEBUG_RX("  MFIE %s\n", ie_name);
		else
		  MRV8K_DEBUG_RX("  unknown MFIE %d\n", ie->id);

		left -= ie->len + 2;
		ie = (void *)ie + ie->len + 2;
	}

}


static void mrv8k_print_mgt_frame(struct mrv8k_private *priv,
				  const struct mrv8k_rx_data *rx)
{
	const struct ieee80211_hdr_4addr *ieee80211hdr = &rx->u.ieee80211hdr;
	int fc, len, stype; 

	len = le16_to_cpu(rx->datalen);
	fc = le16_to_cpu(ieee80211hdr->frame_ctl);


	stype = WLAN_FC_GET_STYPE(fc);
	MRV8K_DEBUG_RX("Received a ieee802.11 MGMT Frame, subtype %s,"
		       " duration=%4.4x\n",
		       frame_stype[(stype>>4)&0xf],
		       le16_to_cpu(ieee80211hdr->duration_id));
	/* Is is a error frame ? */
	if ( stype == IEEE80211_STYPE_ERROR) {
	   const struct rx_error_frame *error = &rx->u.rx_error;
	   MRV8K_DEBUG_RX("You have received an error: %s", error->string_error);
	   /* Let's ignore this bad message */
	   return;
	}

	if (stype == IEEE80211_STYPE_BEACON) {
		const struct ieee80211_probe_response *beacon = &rx->u.beacon;
		const struct ieee80211_info_element *ie = beacon->info_element;
		int left = len - sizeof(struct ieee80211_probe_response);

		MRV8K_DEBUG_RX("da="MAC_FMT" sa="MAC_FMT" bssid="MAC_FMT" seq_ctl=%4.4x\n",
		 		MAC_ARG(beacon->header.addr1),
				MAC_ARG(beacon->header.addr2),
				MAC_ARG(beacon->header.addr3),
				le16_to_cpu(beacon->header.seq_ctl));
		MRV8K_DEBUG_RX("timestamp=%8.8x%8.8x interval=%4.4x capability=%4.4x left=%d\n",
				le32_to_cpu(beacon->time_stamp[1]),
				le32_to_cpu(beacon->time_stamp[0]),
				le16_to_cpu(beacon->beacon_interval),
				le16_to_cpu(beacon->capability),
				left);
		mrv8k_print_IEs(ie, left);
	}


}

static void mrv8k_print_rx(struct mrv8k_private *priv,
    			   const struct mrv8k_rx_packet *rx,
			   const struct ieee80211_rx_stats *stats)
{

#if CONFIG_MRV8K_DEBUG
	int current_rx_buffer;
	int fc, datalen, write_index;
	const struct hw_rx_ctrl_buffer *ctrl = rx->ctrl;
	const struct mrv8k_rx_data *data = rx->data;
	const struct ieee80211_hdr_4addr *ieee80211hdr = &data->u.ieee80211hdr;

	current_rx_buffer = mrv8k_read_mem(priv, priv->register_rx_buffer0);

	write_index = find_index_for_phys_addr(priv, current_rx_buffer);
	if (write_index<0)
		MRV8K_ERROR("WARNING write index in rx buffer is not "
			    "valid (%8.8x)\n", current_rx_buffer);
	else
		MRV8K_DEBUG_RX("write index=%d, read_index=%d\n",
				write_index, priv->rx_index);
	
	MRV8K_DEBUG_RX("ctrl: datalen=%d rssi=%d\n", stats->len, stats->rssi);
	if (MRV8K_DEBUG_LEVEL_RX_DATA & MRV8K_DEBUG_LEVEL)
		printk_buf(ctrl, sizeof(struct hw_rx_ctrl_buffer));

	datalen = le16_to_cpu(data->datalen);
	MRV8K_DEBUG_RX("data: len=%d\n", datalen);
	if (MRV8K_DEBUG_LEVEL_RX_DATA & MRV8K_DEBUG_LEVEL)
	  printk_buf(data, stats->len);

	fc = le16_to_cpu(ieee80211hdr->frame_ctl);

	switch (fc & IEEE80211_FCTL_FTYPE) {

		case IEEE80211_FTYPE_DATA:
			MRV8K_DEBUG_RX("Received a ieee802.11 DATA Frame\n");
			break;

		case IEEE80211_FTYPE_MGMT:
			mrv8k_print_mgt_frame(priv, data);
			break;

		case IEEE80211_FTYPE_CTL:
			MRV8K_DEBUG_RX("Received a ieee802.11 CTL Frame\n");
			break;
		default:
			MRV8K_DEBUG_RX("Received a unknown frame type\n");


	 }


#endif
}

static void mrv8k_irq_event_rx_transfer(struct mrv8k_private *priv)
{
	struct hw_rx_ctrl_buffer *ctrl;
	struct mrv8k_rx_packet *packet;
	unsigned int len;
	int rx_processed;
	u32 current_rx_buffer;
	struct ieee80211_rx_stats stats;


	MRV8K_DEBUG_TRACE("%s\n", __FUNCTION__);

	if (mrv8k_read_mem(priv, priv->register_ring_tx_buffer_addr) == 0xffffffff) {
		MRV8K_ERROR("register tx buffer is error state\n");
		return;
	}

	mrv8k_read_mem(priv, priv->register_rx_buffer1);
	current_rx_buffer = mrv8k_read_mem(priv, priv->register_rx_buffer0);

	rx_processed = 0;
	while (rx_processed < 20)
	{
		packet = &priv->rx_packets[priv->rx_index];
		ctrl = packet->ctrl;

		if ((ctrl->status & 0x80) == 0)
			break;

		len = le16_to_cpu(ctrl->datalen);

		stats.mac_time = jiffies;
		stats.rssi = ctrl->rssi + MRV8K_RSSI_TO_DBM;
		stats.len = len-2;
		stats.mask = IEEE80211_STATMASK_RSSI;
		stats.freq = IEEE80211_24GHZ_BAND;

		pci_dma_sync_single_for_cpu(priv->pci_dev,
				packet->data_dma,
				sizeof(struct mrv8k_rx_data),
				PCI_DMA_FROMDEVICE);

		mrv8k_print_rx(priv, packet, &stats);
		mrv8k_decode_packet_data(priv, packet->data, &stats);

		priv->rx_index++;
		if (priv->rx_index >= MRV8K_RX_BUFFERS)
			priv->rx_index=0;

		ctrl->status = 0;

		rx_processed++;
	}

	mrv8k_write_mem(priv, priv->register_rx_buffer1, cpu_to_le32(0x12345678));

	if (rx_processed >= 20) {
		priv->irq_status |= MRV8K_INTA_RX_TRANSFER;
		/* Need to reschedule the tasklet */
	}

}

/*
 *
 *
 */
static void mrv8k_decode_packet_statistics(struct mrv8k_private *priv, struct mrv8k_event_msg *msg)
{
	struct host_command_get_statistics *stats;
	struct ieee80211_stats *istats;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_get_statistics, msg->len);

	stats = (struct host_command_get_statistics *)msg;
	istats = &priv->ieee->ieee_stats;

	mrv8k_print_command((const struct mrv8k_cmd *)msg);

	istats->tx_single_retry_frames = stats->tx_single_retry_frames;
	istats->tx_multiple_retry_frames = stats->tx_multiple_retry_frames;
	istats->tx_retry_limit_exceeded = stats->tx_failed;
	istats->rx_fcs_errors = stats->rx_fcs_errors;
}

static void mrv8k_decode_packet_data(struct mrv8k_private *priv,
   				     struct mrv8k_rx_data *data,
				     struct ieee80211_rx_stats *stats)
{
	struct ieee80211_hdr_4addr *ieee80211hdr = &data->u.ieee80211hdr;
	int fc, len, stype; 

	len = le16_to_cpu(data->datalen);
	fc = le16_to_cpu(ieee80211hdr->frame_ctl);

	switch (fc & IEEE80211_FCTL_FTYPE) {

		case IEEE80211_FTYPE_DATA:
			break;
		case IEEE80211_FTYPE_MGMT:
			stype = WLAN_FC_GET_STYPE(fc);

			/* Is is a error frame ? */
			if ( stype == 0xf0) {
				/* Let's ignore this bad message */
				return;
			}

			ieee80211_rx_mgt(priv->ieee, ieee80211hdr, stats);
			break;
		case IEEE80211_FTYPE_CTL:
			MRV8K_DEBUG_RX("Received a ieee802.11 CTL Frame\n");
			break;
		default:
			MRV8K_DEBUG_RX("Received a unknown frame type\n");
			/* Unknown frame type */
			break;
	}


	return; 
}


/*
 * vim:sw=8:sts=8:ts=8:cino=
 */
