/*
  
  Marvell Libertas wireless driver

  Copyright (c) 2005 Luc Saillard <luc@saillard.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/ethtool.h>
#include <linux/dma-mapping.h>

#include "mrv8k.h"



typedef enum {
	WL3563 = 0,
	WL138G,
	W8335,
	WLGENERIC
} board_t;


#if 0
/* indexed by board_t, above */
static struct {
	const char *name;
} board_info[] __devinitdata = {
	{ "PLANET WL-3563" },
	{ "ASUS 802.11b/g Wireless LAN Card"},
	{ "IEEE 802.11g Wireless Cardbus/PCI Adapter HW51"},
	{ "Marvell Libertas 802.11b/g Wireless SoftAP" },
};
#endif


#define PCI_DEVICE_ID_MARVELL_W8K (0x1FA6)
static struct pci_device_id mrv8k_pci_tbl[] = {
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x1FA4, 0x11AB, 0, 0, WL3563 },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x1FA5, 0x16AB, 0, 0, WL3563 },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x1FA6, 0x16AB, 0, 0, WL3563 },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x1FA7, 0x16AB, 0, 0, WL3563 },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x138f, 0x1043, 0, 0, WL138G },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x128f, 0x1043, 0, 0, WL138G },
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, 0x108f, 0x1043, 0, 0, WL138G },
		/* NETGEAR WG311v3 802.11g Wireless PCI Adapter */
	{PCI_VENDOR_ID_MARVELL, 0x1faa, PCI_ANY_ID, PCI_ANY_ID, 0x6b00, 0x1385, W8335},
	{PCI_VENDOR_ID_MARVELL, 0x1faa, PCI_ANY_ID, PCI_ANY_ID, 0, 0, W8335},
	{PCI_VENDOR_ID_MARVELL, PCI_DEVICE_ID_MARVELL_W8K, PCI_ANY_ID, PCI_ANY_ID, 0, 0, WLGENERIC },
	{0,}
};
MODULE_DEVICE_TABLE (pci, mrv8k_pci_tbl);

#if 1
static unsigned char channels_authorized_by_geo[6][14] = {
	{ 1, 4, 7, 10, 2, 5, 8, 11, 3, 6, 9, 0, 0, 0},		/* USA    */
	{ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0, 0},		/* Canada */
	{ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 0},	/* Europe */
	{ 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},		/* Spain  */
	{ 10, 11, 12, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},	/* France */
	{ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14},	/* Japan  */
};
#endif



static void mrv8k_tx_timeout (struct net_device *dev);
static int mrv8k_start_xmit (struct sk_buff *skb,
			       struct net_device *dev);
static int mrv8k_open (struct net_device *dev);
static int mrv8k_close (struct net_device *dev);
static struct net_device_stats *mrv8k_get_stats (struct net_device *dev);
static void mrv8k_hw_start (struct net_device *dev);

static int  mrv8k_hw_alloc_magic_cookie(struct mrv8k_private *priv);
static void mrv8k_hw_free_magic_cookie(struct mrv8k_private *priv);
static int  mrv8k_msgs_queue_allocate(struct mrv8k_private *priv);
static void mrv8k_msgs_queue_free(struct mrv8k_private *priv);
static int  mrv8k_allocate_rx_buffers(struct mrv8k_private *priv);
static void mrv8k_free_rx_buffers(struct mrv8k_private *priv);
static int  mrv8k_allocate_tx_buffers(struct mrv8k_private *priv);
static void mrv8k_free_tx_buffers(struct mrv8k_private *priv);
static int  mrv8k_allocate_frames_management(struct mrv8k_private *priv);
static void mrv8k_free_frames_management(struct mrv8k_private *priv);

static struct iw_statistics *mrv8k_wx_wireless_stats(struct net_device * dev);

static struct iw_handler_def mrv8k_wx_handler_def;

/**
 *
 * Setup default value for the card
 *
 */
void mrv8k_setup_default_value(struct mrv8k_private *priv)
{
	priv->tx_antenna = MRV8K_TX_POWER_MAX;		
	priv->rx_antenna = MRV8K_RX_POWER_MIN;
	priv->preamble = MRV8K_PREAMBLE_AUTO;
	priv->radio_on = 1;
	priv->rts_threshold = MRV8K_RTS_THRESHOLD_MAX;

	/* 802.11bg */
	priv->ieee->freq_band = 0;
	priv->ieee->freq_band |= IEEE80211_24GHZ_BAND;
	priv->ieee->freq_band &= ~IEEE80211_52GHZ_BAND;
	priv->ieee->modulation = 0;
	priv->ieee->modulation |= IEEE80211_CCK_MODULATION;
	priv->ieee->modulation |= IEEE80211_OFDM_MODULATION;

}

/**
 *
 * Card need to have a magic cookie shared between the host and the card
 */
static int mrv8k_hw_alloc_magic_cookie(struct mrv8k_private *priv)
{
	dma_addr_t p;
	void *v;

	v = pci_alloc_consistent(priv->pci_dev, 4, &p);
	if (!v) {
		MRV8K_ERROR("failed to allocate DMA memory for cookie\n");
		return -ENOMEM;
	}
	priv->dma_cookie = v;
	priv->dma_cookie_phys = p;

	writel(cpu_to_le32(0xAA55AA55), priv->dma_cookie);

	return 0;
}

/**
 *
 * Free the magic cookie
 */
static void mrv8k_hw_free_magic_cookie(struct mrv8k_private *priv)
{
	if (priv->dma_cookie) {
		pci_free_consistent(priv->pci_dev, 4, priv->dma_cookie,
		    		    priv->dma_cookie_phys);
		priv->dma_cookie = NULL;
	}
}



/**
 * Allocate buffers to send control message to the hardware.
 *
 * By default, we allocate 32 buffers of 512 bytes to send the data using DMA.
 * We have also a pool of this buffers to find the next unused.
 *
 */
static int mrv8k_msgs_queue_allocate(struct mrv8k_private *priv)
{
	struct mrv8k_cmd_packet *queue;
	void *v;
	dma_addr_t p;
	int err, i;

	/* TODO: replace by a kzalloc */
	queue = kmalloc(MSG_QUEUE_LEN * sizeof(struct mrv8k_cmd_packet),
		       	GFP_KERNEL);
	if (queue == NULL) {
		MRV8K_ERROR("failed to allocate control message queue\n");
		return -ENOMEM;
	}
	memset(queue, 0, MSG_QUEUE_LEN * sizeof(struct mrv8k_cmd_packet));
	for (i=0; i<MSG_QUEUE_LEN; i++) {

		v = pci_alloc_consistent(priv->pci_dev, MSG_QUEUE_SIZE, &p);
		if (!v) {
			MRV8K_ERROR("failed to allocate DMA control message queue\n");
			err = -ENOMEM;
			goto free_buffer;
		}
		memset(v, 0, MSG_QUEUE_SIZE);
		queue[i].cmd = v;
		queue[i].cmd_phys = p;
	}
	priv->cmd_buffers = queue;

	INIT_LIST_HEAD(&priv->cmd_free_list);
	INIT_LIST_HEAD(&priv->cmd_pend_list);

	for (i = 0; i < MSG_QUEUE_LEN; i++)
		list_add_tail(&priv->cmd_buffers[i].list, &priv->cmd_free_list);

	spin_lock_init(&priv->cmd_send_lock);
	return 0;

free_buffer:
	mrv8k_msgs_queue_free(priv);
	return err;
}

static void mrv8k_msgs_queue_free(struct mrv8k_private *priv)
{
	struct mrv8k_cmd_packet *queue = priv->cmd_buffers;
	int i;

	if (queue == NULL)
		return;

	for (i=0; i<MSG_QUEUE_LEN; i++) {
		if (queue->cmd == NULL)
			continue;

		pci_free_consistent(priv->pci_dev, MSG_QUEUE_SIZE,
				    queue->cmd, queue->cmd_phys);
		queue->cmd = NULL;
	}
	kfree(queue);
	priv->cmd_buffers = NULL;
}

/*
 * Allocate the Rx ring to transfer data from the card to the ip stack.
 * DMA buffer will point directly to skb data buffer
 */
static int mrv8k_allocate_rx_buffers(struct mrv8k_private *priv)
{
	unsigned int i, size;
	int err;
	struct hw_rx_ctrl_buffer *ctrl;
	struct mrv8k_rx_packet *packet;

	MRV8K_BUG_ON_CHECK_STRUCT(hw_rx_ctrl_buffer, 20);

	priv->rx_index = 0;

	/* Allocate structure to manage a buffer of free rx */
	size = sizeof(struct mrv8k_rx_packet) * MRV8K_RX_BUFFERS;
	priv->rx_packets = kmalloc(size, GFP_KERNEL);
	if (!priv->rx_packets) {
		MRV8K_ERROR("failed to allocate rx packets buffer\n");
		err = -ENOMEM;
		goto err_free_buffers;
	}
	memset(priv->rx_packets, 0, size);

	/* Preallocate skb <-> dma memory */
	for (i=0; i<MRV8K_RX_BUFFERS; i++) {

		packet = &priv->rx_packets[i];

		packet->skb = dev_alloc_skb(sizeof(struct mrv8k_rx_data));
		if (!packet->skb) {
			MRV8K_ERROR("failed to allocate skb\n");
			err = -ENOMEM;
			goto err_free_buffers;
		}

		/* Allocate DMA for Rx data */
		packet->data = (struct mrv8k_rx_data *)packet->skb->data;
		packet->data_dma = pci_map_single(priv->pci_dev,
						  packet->skb->data,
						  sizeof(struct mrv8k_rx_data),
						  PCI_DMA_FROMDEVICE);
		if (pci_dma_mapping_error(packet->data_dma)) {
			err = -ENOMEM;
			goto err_free_buffers;

		}

		/* Allocate DMA for Rx Ctrl register */
		packet->ctrl = pci_alloc_consistent(priv->pci_dev,
						    sizeof(struct hw_rx_ctrl_buffer),
						    &packet->ctrl_dma);
		if (!packet->ctrl) {
			MRV8K_ERROR("failed to allocate DMA rx buffer\n");
			err = -ENOMEM;
			goto err_free_buffers;
		}

	}


	/* Link the packets as a ring and initialize the control register */
	for (i=0; i<MRV8K_RX_BUFFERS; i++) {

		ctrl = priv->rx_packets[i].ctrl;
		memset(ctrl, 0, sizeof(*ctrl));

		ctrl->status = 0;
		ctrl->rssi = 0;
		ctrl->var_02 = cpu_to_le16(1);
		ctrl->datalen = 0;
		ctrl->var_06 = 0;

		ctrl->phys_next = priv->rx_packets[i+1].ctrl_dma;
		if ((i+1) >= MRV8K_RX_BUFFERS)
			ctrl->phys_next = priv->rx_packets[0].ctrl_dma;

		ctrl->phys_data = priv->rx_packets[i].data_dma;
		/* For cross platform, we can't use this field to store the address
		 * of the current buffer ... so store an index */
		ctrl->ieee80211_frame = i;

	}


	return 0;
err_free_buffers:
	mrv8k_free_rx_buffers(priv);
	return err;
}

static void mrv8k_free_rx_buffers(struct mrv8k_private *priv)
{
	int i;
	struct mrv8k_rx_packet *packet;

	if (priv->rx_packets) {

		for (i=0; i<MRV8K_RX_BUFFERS; i++) {

			packet = &priv->rx_packets[i];

			pci_free_consistent(priv->pci_dev,
					    sizeof(struct hw_rx_ctrl_buffer),
					    packet->ctrl,
					    packet->ctrl_dma);

			if (!pci_dma_mapping_error(packet->data_dma)) {
				pci_unmap_single(priv->pci_dev,
						 packet->data_dma,
						 sizeof(struct mrv8k_rx_data),
						 PCI_DMA_FROMDEVICE);
			}
			if (packet->skb)
				dev_kfree_skb(packet->skb);

		}
			
		kfree(priv->rx_packets);
	}
}

static int mrv8k_allocate_tx_buffers(struct mrv8k_private *priv)
{
	void *v;
	dma_addr_t p;
	unsigned int i, size;
	int err;

	MRV8K_BUG_ON_CHECK_STRUCT(hw_tx_ctrl_buffer, 32);

	spin_lock_init(&priv->tx_send_lock);
	priv->tx_index = 0;
	priv->tx_last_ack_index = 0;
	atomic_set(&priv->tx_pending_bytes, 0);
	atomic_set(&priv->tx_pending_buffers, 0);

	/* Allocate the ring in one chunk */
	size = sizeof(struct hw_tx_ctrl_buffer) + MRV8K_TX_BUFFERS_DATA_SIZE;
	size *= MRV8K_TX_BUFFERS;

	v = pci_alloc_consistent(priv->pci_dev, size, &p);
	if (!v) {
		MRV8K_ERROR("failed to allocate DMA tx buffers\n");
		err = -ENOMEM;
		goto err_free_buffers;
	}

	priv->tx_buffers = v;
	priv->tx_buffers_dma = p;

	for (i=0; i<MRV8K_TX_BUFFERS; i++) {

		priv->tx_data_buffers[i] = v;
		priv->tx_data_buffers_dma[i] = p;
		memset(v, 0, MRV8K_TX_BUFFERS_DATA_SIZE);
		v += MRV8K_TX_BUFFERS_DATA_SIZE;
		p += MRV8K_TX_BUFFERS_DATA_SIZE;

	}

	for (i=0; i<MRV8K_TX_BUFFERS; i++) {

		priv->tx_ctrl_buffers[i] = v;
		priv->tx_ctrl_buffers_dma[i] = p;
		memset(v, 0, sizeof(struct hw_tx_ctrl_buffer));

		v += sizeof(struct hw_tx_ctrl_buffer);
		p += sizeof(struct hw_tx_ctrl_buffer);

	}

	p = priv->tx_buffers_dma;

	/* Link the packets as a ring */
	for (i=0; i<MRV8K_TX_BUFFERS; i++) {

		priv->tx_ctrl_buffers[i]->phys_data = priv->tx_data_buffers_dma[i];
		priv->tx_ctrl_buffers[i]->phys_next = priv->tx_ctrl_buffers_dma[i+1];
		if (i+1 >= MRV8K_TX_BUFFERS)
			priv->tx_ctrl_buffers[i]->phys_next = priv->tx_ctrl_buffers_dma[0];
	}

	/* Allocate packet structure for queueing */
	INIT_LIST_HEAD(&priv->tx_pend_list);
	INIT_LIST_HEAD(&priv->tx_free_list);
	size = sizeof(struct mrv8k_tx_packet) * MRV8K_TX_BUFFERS;
	priv->tx_packets = kmalloc(size, GFP_KERNEL);
	if (priv->tx_packets == NULL) {
		MRV8K_ERROR("failed to allocate memory used for frame management\n");
		err = -ENOMEM;
		goto err_free_buffers;
	}
	memset(priv->tx_packets, 0, size);
	for (i=0; i<MRV8K_TX_BUFFERS; i++) {
		INIT_LIST_HEAD(&priv->tx_packets[i].list);
		list_add_tail(&priv->tx_packets[i].list, &priv->tx_free_list);
	}
	

	return 0;
err_free_buffers:
	mrv8k_free_tx_buffers(priv);
	return err;
}

static void mrv8k_free_tx_buffers(struct mrv8k_private *priv)
{
	unsigned int size;

	size = sizeof(struct hw_tx_ctrl_buffer) + MRV8K_TX_BUFFERS_DATA_SIZE;
	size *= MRV8K_TX_BUFFERS;

	if (priv->tx_buffers) {
		pci_free_consistent(priv->pci_dev, size,
				    priv->tx_buffers,
				    priv->tx_buffers_dma);
		priv->tx_buffers = NULL;
		priv->tx_buffers_dma = 0UL;
	}
	if (priv->tx_packets) {
		kfree(priv->tx_packets);
		priv->tx_packets = NULL;
	}

}

/*
 * Our card send Frame Management using a raw frame, so we need to have some
 * buffers in advance to be able to queue this frame.
 */
static int mrv8k_allocate_frames_management(struct mrv8k_private *priv)
{
	unsigned int size;
	int i;

	INIT_LIST_HEAD(&priv->fmgt_free_list);

	size = sizeof(struct mrv8k_fmgt_packet) * MRV8K_FMGT_BUFFERS;
	priv->fmgt_buffers = kmalloc(size, GFP_KERNEL);
	if (priv->fmgt_buffers == NULL) {
		MRV8K_ERROR("failed to allocate memory used for frame management\n");
		return -ENOMEM;
	}
	for (i=0; i<MRV8K_FMGT_BUFFERS; i++) {
		INIT_LIST_HEAD(&priv->fmgt_buffers[i].list);
		priv->fmgt_buffers[i].datalen = 0;
		memset(priv->fmgt_buffers[i].data, 0, sizeof(struct mrv8k_fmgt_packet));

		list_add_tail(&priv->fmgt_buffers[i].list, &priv->fmgt_free_list);
	}
	return 0;
}

static void mrv8k_free_frames_management(struct mrv8k_private *priv)
{
	if (priv->fmgt_buffers) {
		kfree(priv->fmgt_buffers);
		priv->fmgt_buffers = NULL;
	}
}


/*
 * Allocate all buffers needed
 */
static int mrv8k_queues_allocate(struct mrv8k_private *priv)
{
	int err;

	err = mrv8k_msgs_queue_allocate(priv);
	if (err)
		goto err;

	err = mrv8k_hw_alloc_magic_cookie(priv);
	if (err)
		goto err_free_msgs_queue;

	err = mrv8k_allocate_rx_buffers(priv);
	if (err)
		goto err_free_magic_cookie;

	err = mrv8k_allocate_tx_buffers(priv);
	if (err)
		goto err_free_rx_buffers;

	err = mrv8k_allocate_frames_management(priv);
	if (err)
		goto err_free_tx_buffers;

	return 0;

err_free_tx_buffers:
	mrv8k_free_tx_buffers(priv);
err_free_rx_buffers:
	mrv8k_free_rx_buffers(priv);
err_free_magic_cookie:
	mrv8k_hw_free_magic_cookie(priv);
err_free_msgs_queue:
	mrv8k_msgs_queue_free(priv);
err:
	return err;
}

static void mrv8k_queues_free(struct mrv8k_private *priv)
{
	mrv8k_msgs_queue_free(priv);
	mrv8k_hw_free_magic_cookie(priv);
	mrv8k_free_rx_buffers(priv);
	mrv8k_free_tx_buffers(priv);
	mrv8k_free_frames_management(priv);
}

/*
 *
 *
 */

static struct net_device *__devinit mrv8k_alloc_device (struct pci_dev *pci_dev)
{
	struct mrv8k_private *priv;
	struct net_device *dev;

	dev = alloc_ieee80211(sizeof(struct mrv8k_private));
	if (!dev)
		return NULL;
	priv = ieee80211_priv(dev);
	priv->ieee = netdev_priv(dev);
	priv->pci_dev = pci_dev;
	priv->net_dev = dev;

#if 0
	priv->ieee->hard_start_xmit = mrv8k_start_xmit;
	priv->ieee->set_security = shim__set_security;
#else
	dev->hard_start_xmit = mrv8k_start_xmit;
#endif

	dev->open = mrv8k_open;
	dev->stop = mrv8k_close;
	dev->get_stats = mrv8k_get_stats;
	dev->tx_timeout = mrv8k_tx_timeout;
	dev->wireless_handlers = &mrv8k_wx_handler_def;
	dev->get_wireless_stats = mrv8k_wx_wireless_stats;
#if 0
	dev->do_ioctl = mrv8k_ioctl;
	dev->ethtool_ops = &mrv8k_ethtool_ops;
	dev->set_mac_address = mrv8k_set_address;
#endif
	dev->watchdog_timeo = 3*HZ;
	dev->irq = 0;

	netif_carrier_off(dev);

	tasklet_init(&priv->irq_tasklet, (void (*)(unsigned long))
			mrv8k_irq_tasklet, (unsigned long)priv);

	init_waitqueue_head(&priv->config_wait_queue);

	return dev;
}

static int mrv8k_scan_phase_2(struct mrv8k_private *priv)
{
	const unsigned char *channels;
	char buffer[ETH_ALEN];
	char rates_listen[14];
	int i;

	/* Start timer, phase finished */
	MRV8K_DEBUG_TRACE("%s: enter\n", __FUNCTION__);

	for (i=0;i<14;i++)
		rates_listen[i]=i;
	mrv8k_hw_send_set_rates(priv, 3, rates_listen);
	channels = channels_authorized_by_geo[priv->geo_code];
	mrv8k_set_bind_channel(priv, channels[0]);
	mrv8k_hw_send_probe_request(priv);
	msleep_interruptible(5);
	mrv8k_hw_send_probe_request(priv);
	msleep_interruptible(5);
	mrv8k_hw_send_probe_request(priv);
	msleep_interruptible(5);
	mrv8k_hw_send_probe_request(priv);
	msleep_interruptible(5);
	mrv8k_hw_send_probe_request(priv);
	msleep_interruptible(100);
	memset(buffer, 0, sizeof(buffer));
	mrv8k_hw_send_bind_ap(priv, buffer);

	mrv8k_hw_send_get_statistics(priv);
	MRV8K_DEBUG_TRACE("%s: leaving\n", __FUNCTION__);
	return 0;
}

int mrv8k_start_scan(struct mrv8k_private *priv)
{
	MRV8K_DEBUG_TRACE("%s: enter\n", __FUNCTION__);

	if (priv->status & STATUS_SCANNING)
		return -EBUSY;

	priv->status |= STATUS_SCANNING;

	mrv8k_hw_send_flush_list(priv);
	/* start a timer to 20ms */

	return 0;
}


static int __devinit mrv8k_init_one (struct pci_dev *pci_dev,
				     const struct pci_device_id *ent)
{
	struct net_device *dev = NULL;
	struct mrv8k_private *priv;
	unsigned long bar1_start, bar1_len, bar1_flags;
	unsigned long bar2_start, bar2_len, bar2_flags;
	void *bar1_addr=NULL, *bar2_addr=NULL;
	int err;

	MRV8K_DEBUG_TRACE("%s: enter\n", __FUNCTION__);

	err = pci_enable_device(pci_dev);
	if (err) {
		MRV8K_ERROR("Error calling pci_enable_device.\n");
		return err;
	}

	bar1_start = pci_resource_start(pci_dev, 0);
	bar1_len = pci_resource_len(pci_dev, 0);
	bar1_flags = pci_resource_flags(pci_dev, 0);

	if (!(bar1_flags & IORESOURCE_MEM)) {
		MRV8K_ERROR("strange - bar1 resource type is not memory\n");
		err = -ENODEV;
		goto fail_disable_device;
	}

	bar2_start = pci_resource_start(pci_dev, 1);
	bar2_len = pci_resource_len(pci_dev, 1);
	bar2_flags = pci_resource_flags(pci_dev, 1);

	if (!(bar2_flags & IORESOURCE_MEM)) {
		MRV8K_ERROR("strange - bar2 resource type is not memory\n");
		err = -ENODEV;
		goto fail_disable_device;
	}

	err = pci_set_dma_mask(pci_dev, DMA_32BIT_MASK);
	if (err) {
		MRV8K_ERROR("No usable DMA configuration, aborting.\n");
		goto fail_disable_device;
	}

	bar1_addr = ioremap_nocache(bar1_start, bar1_len);
	if (bar1_addr == NULL) {
		MRV8K_ERROR("Error calling ioremap for bar1.\n");
		err = -EIO;
		goto fail_disable_device;
	}

	bar2_addr = ioremap_nocache(bar2_start, bar2_len);
	if (bar2_addr == NULL) {
		MRV8K_ERROR("Error calling ioremap for bar2.\n");
		err = -EIO;
		goto fail_iounmap;
	}

        err = pci_request_regions(pci_dev, DRV_NAME);
	if (err) {
		MRV8K_ERROR("Error calling pci_request_regions.\n");
		goto fail_iounmap;
	}
	pci_set_master(pci_dev);
	pci_set_mwi(pci_dev);

	/* allocate and initialize our net_device */
	dev = mrv8k_alloc_device(pci_dev);
	if (!dev) {
		MRV8K_ERROR("Error calling mrv8k_alloc_device.\n");
		err = -ENOMEM;
		goto fail_release_regions;
	}

	SET_MODULE_OWNER(dev);
	SET_NETDEV_DEV(dev, &pci_dev->dev);
	priv = ieee80211_priv(dev);
	pci_set_drvdata(pci_dev, priv);

	err = request_irq(pci_dev->irq, mrv8k_interrupt, SA_SHIRQ,
		          dev->name, priv);
	if (err) {
		MRV8K_ERROR("Error calling request_irq: %d.\n", pci_dev->irq);
		goto fail_release_regions;
	}
	dev->irq = pci_dev->irq;

	err = register_netdev(dev);
	if (err) {
		MRV8K_ERROR("Error calling register_netdev.\n");
		goto fail_free_irq;
	}



	priv->bar1 = bar1_addr;
	priv->bar1_phys = bar1_start;
	priv->bar2 = bar2_addr;
	priv->bar2_phys = bar2_start;

	/* Allocate and initialize the Tx/Rx queues and lists */
	err = mrv8k_queues_allocate(priv);
	if (err) {
		MRV8K_ERROR("Error calling mrv8k_queues_allocate.\n");
		goto fail_unregister_netdev;
	}


	/* Setup default value for the card */
	mrv8k_setup_default_value(priv);

	MRV8K_DEBUG_TRACE("bar1 (0x%8.8lx) relocated at 0x%p\n", priv->bar1_phys, priv->bar1);
	MRV8K_DEBUG_TRACE("bar2 (0x%8.8lx) relocated at 0x%p\n", priv->bar2_phys, priv->bar2);

	err = mrv8k_initialize_hardware(priv);
	if (err) {
		MRV8K_DEBUG_TRACE("%s: return %d\n", __FUNCTION__, err);
		mrv8k_hw_halt(priv);
		goto fail_unregister_netdev;
	}
#if 0
	mrv8k_enable_radio_n_set_macaddr(priv);
#endif


	MRV8K_DEBUG_TRACE("%s: return 0\n", __FUNCTION__);
	return 0;

fail_unregister_netdev:
	unregister_netdev(dev);
fail_free_irq:
	if (dev->irq)
		free_irq (dev->irq, priv);
fail_release_regions:
	pci_release_regions(pci_dev);
fail_iounmap:
	if (bar1_addr)
		iounmap(bar1_addr);
	if (bar2_addr)
		iounmap(bar2_addr);
fail_disable_device:
	pci_disable_device(pci_dev);
	pci_set_drvdata(pci_dev, NULL);

	MRV8K_DEBUG_TRACE("%s: return %d\n", __FUNCTION__, err);
	return err;
}


static void __devexit mrv8k_remove_one (struct pci_dev *pci_dev)
{
	struct mrv8k_private *priv = pci_get_drvdata(pci_dev);
	struct net_device *dev;


	MRV8K_DEBUG_TRACE("%s: enter\n", __FUNCTION__);
	if (priv) {

		dev = priv->net_dev;

		mrv8k_hw_halt(priv);

		unregister_netdev (dev);
		/* xxx stop_interrupt */
		if (dev->irq)
			free_irq (dev->irq, priv);

		mrv8k_queues_free(priv);

		if (priv->bar1)
			iounmap(priv->bar1);
		if (priv->bar2)
			iounmap(priv->bar2);

		pci_release_regions (pci_dev);

		free_ieee80211 (dev);

	}
	pci_set_drvdata (pci_dev, NULL);
	pci_disable_device (pci_dev);
	MRV8K_DEBUG_TRACE("%s: exit\n", __FUNCTION__);
}


static int mrv8k_open (struct net_device *dev)
{
	struct mrv8k_private *priv = ieee80211_priv(dev);

	priv = priv;
	MRV8K_DEBUG_TRACE("%s: return 0\n", __FUNCTION__);
	return 0;
}


/* Start the hardware at open or resume. */
static void mrv8k_hw_start (struct net_device *dev)
{
	MRV8K_DEBUG_TRACE("%s: enter\n", __FUNCTION__);

	netif_start_queue (dev);

	MRV8K_DEBUG_TRACE("%s: exit\n", __FUNCTION__);
}


static void mrv8k_tx_timeout (struct net_device *dev)
{
	MRV8K_DEBUG_TRACE("%s: enter (%s)\n", __FUNCTION__, dev->name);

	/* ...and finally, reset everything */
	mrv8k_hw_start (dev);

	netif_wake_queue (dev);
	MRV8K_DEBUG_TRACE("%s: exit\n", __FUNCTION__);
}

static int mrv8k_start_xmit (struct sk_buff *skb, struct net_device *dev)
{
	struct mrv8k_private *priv = ieee80211_priv(dev);
	int entry = 0;

	MRV8K_DEBUG_TRACE("%s: enter (dev:%s)\n", __FUNCTION__, dev->name);

	MRV8K_DEBUG_TRACE ("%s: Queued Tx packet at %p size %u to slot %d.\n",
			   dev->name, skb->data, skb->len, entry);

	if (priv->radio_on == 0) {
	}
		

	MRV8K_DEBUG_TRACE("%s: return 0\n", __FUNCTION__);
	return 0;
}




static int mrv8k_close (struct net_device *dev)
{
	//struct mrv8k_private *priv = ieee80211_priv(dev);
	//unsigned long flags;

	MRV8K_DEBUG_TRACE("%s: enter (dev:%s)\n", __FUNCTION__, dev->name);

	netif_stop_queue (dev);

	synchronize_irq(dev->irq);


	MRV8K_DEBUG_TRACE("%s: return 0\n", __FUNCTION__);
	return 0;
}


static struct net_device_stats *mrv8k_get_stats (struct net_device *dev)
{
	struct mrv8k_private *priv = ieee80211_priv(dev);
	return &priv->ieee->stats;
}


/*
 * Get wireless statistics.
 * Called by /proc/net/wireless
 * Also called by SIOCGIWSTATS
 */
static struct iw_statistics *mrv8k_wx_wireless_stats(struct net_device * dev)
{
	struct mrv8k_private *priv = ieee80211_priv(dev);
        struct iw_statistics *wstats;
        
        if (!priv)
                return NULL;

        wstats = &priv->wstats;

	memset(wstats, 0, sizeof(struct iw_statistics));
	if (!(priv->status & STATUS_ASSOCIATED)) {
                wstats->miss.beacon = 0;
                wstats->discard.retries = 0; 
                wstats->qual.qual = 0;
                wstats->qual.level = 0;
                wstats->qual.noise = 0;
                wstats->qual.updated = 7;
                wstats->qual.updated |= IW_QUAL_NOISE_INVALID |
                        IW_QUAL_QUAL_INVALID | IW_QUAL_LEVEL_INVALID;
                return wstats;
        }
	return wstats;
}



static struct pci_driver mrv8k_pci_driver = {
	.name		= DRV_NAME,
	.id_table	= mrv8k_pci_tbl,
	.probe		= mrv8k_init_one,
	.remove		= __devexit_p(mrv8k_remove_one),
};


static int __init mrv8k_init_module (void)
{
	MRV8K_INFO("%s, %s\n", DRV_DESCRIPTION, DRV_VERSION);
	MRV8K_INFO("%s\n", DRV_COPYRIGHT);

	return pci_module_init (&mrv8k_pci_driver);
}


static void __exit mrv8k_cleanup_module (void)
{
	pci_unregister_driver (&mrv8k_pci_driver);
}

MODULE_AUTHOR (DRV_COPYRIGHT);
MODULE_DESCRIPTION (DRV_DESCRIPTION);
MODULE_VERSION(DRV_VERSION);
MODULE_LICENSE("GPL");

module_init(mrv8k_init_module);
module_exit(mrv8k_cleanup_module);

/*
 * vim:sw=8:sts=8:ts=8:cino=
 */
