/*
  
  Marvell Libertas wireless driver

  Copyright (c) 2005 Luc Saillard <luc@saillard.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/


#include <linux/delay.h>
#include <linux/pci.h>

#include "mrv8k.h"
#include "mrv8k_fw.h"

static void mrv8k_hw_kick_transfert(struct mrv8k_private *priv,
   				     struct mrv8k_cmd_packet *packet);

static int mrv8k_hw_get_configuration(struct mrv8k_private *priv);
static int mrv8k_hw_check_macaddr(struct mrv8k_private *priv);
static void ieee80211_mgt_build_frame(struct ieee80211_hdr_4addr *header,
    				      int subtype,
				      const unsigned char *dest_addr,
				      const unsigned char *src_addr,
				      const unsigned char *bssid);

/*
 * Geography data.
 */
static const struct ieee80211_geo geography_data[6] = {
	{	/* USA */
		"US",
		.bg_channels = 11,
		.bg = {
			{ 2412, 1}, { 2427, 4}, { 2442, 7}, { 2457, 10},
		       	{ 2417, 2}, { 2432, 5}, { 2447, 8}, { 2462, 11},
			{ 2422, 3}, { 2437, 6}, { 2452, 9},
		},
	},
	{	/* Canada */
		"CA",
		.bg_channels = 11,
		.bg = {
			{ 2412, 1}, { 2417,  2}, { 2422,  3}, { 2427, 4},
			{ 2432, 5}, { 2437,  6}, { 2442,  7}, { 2447, 8},
			{ 2452, 9}, { 2457, 10}, { 2462, 11}
		},
	},
	{	/* Europe */
		"EU",
		.bg_channels = 13,
		.bg = {
			{ 2412,  1}, { 2417,  2}, { 2422,  3}, { 2427,  4},
			{ 2432,  5}, { 2437,  6}, { 2442,  7}, { 2447,  8},
			{ 2452,  9}, { 2457, 10}, { 2462, 11}, { 2467, 12},
			{ 2472, 13}
		},
	},
	{	/* Spain */
		"ES",
		.bg_channels = 2,
		.bg = {
			{ 2457, 10}, { 2462, 11}
		},
	},
	{	/* France ?? */
		"FR",
		.bg_channels = 4,
		.bg = {
			{ 2457, 10}, { 2462, 11}, { 2467, 12}, { 2472, 13}
		},
	},
	{	/* Japan */
		"JP",
		.bg_channels = 14,
		.bg = {
			{ 2412,  1}, { 2417,  2}, { 2422,  3}, { 2427, 4},
			{ 2432,  5}, { 2437,  6}, { 2442,  7}, { 2447, 8},
			{ 2452,  9}, { 2457, 10}, { 2462, 11}, { 2467, 12},
			{ 2472, 13}, { 2484, 14}
		},
	},
};



/*
 * Wait (and sleep) until the regC14 match the value.
 * We can sleep at least 1 seconde.
 * return 0 for ok, and -ETIMEDOUT for an error
 */
static int wait_n_test_regC14(struct mrv8k_private *priv, unsigned int value)
{
	int loop = 200;
	do
	{
		msleep_interruptible(5);
		if (mrv8k_read_nic(priv, MRV8K_REG_C14) == cpu_to_le32(value))
			return 0;
	} while (--loop);
	return -ETIMEDOUT;
}

/*
 * Send a full packet other DMA (use only by the firmware loader)
 *
 */
static void set_dma_for_cmd(struct mrv8k_private *priv, unsigned long buffer_phys_addr)
{
	mrv8k_write_nic(priv, MRV8K_REG_C10, cpu_to_le32(buffer_phys_addr));
	mrv8k_read_nic(priv, MRV8K_REG_C14);
	mrv8k_write_nic(priv, MRV8K_REG_C14, 0);
	mrv8k_read_nic(priv, MRV8K_REG_C14);
	mrv8k_write_nic(priv, MRV8K_REG_C18, cpu_to_le32(MRV8K_CPU_TRANSFER_CMD));
	mrv8k_read_nic(priv, MRV8K_REG_C14);
}

/**
 * Notify the hardware that we have sent a packet
 */
static void mrv8k_hw_kick_transfert(struct mrv8k_private *priv,
   				    struct mrv8k_cmd_packet *packet)
{
	mrv8k_write_nic(priv, MRV8K_REG_C10, cpu_to_le32(packet->cmd_phys));
	mrv8k_read_nic(priv, MRV8K_REG_C14);
	mrv8k_write_nic(priv, MRV8K_REG_C18, cpu_to_le32(MRV8K_CPU_TRANSFER_CMD));
	mrv8k_read_nic(priv, MRV8K_REG_C14);
}


/*
 * Load the bootloader, and the firmware into the card
 * Wait the cpu is ready (can be long)
 *
 * @return 0 if ok
 */
int mrv8k_initialize_cpu(struct mrv8k_private *priv)
{
	int bytes_already_written, block_size;
	struct list_head *element;
	struct mrv8k_cmd_packet *packet;
	struct host_command_firmware *firmware_cmd;
	int err = 0;
	unsigned int addr;
	u32 *p;

	if (list_empty(&priv->cmd_free_list)) {
		MRV8K_TRACE("no available msg buffers\n");
		err = -ENOMEM;
error:
		return err;
	}

	element = priv->cmd_free_list.next;
	packet = list_entry(element, struct mrv8k_cmd_packet, list);
	firmware_cmd = (struct host_command_firmware *)packet->cmd;

	MRV8K_DEBUG_FIRMWARE("packet: 0x%p mapped at 0x%p (phys: 0x%llx)\n", packet, packet->cmd,
			(unsigned long long) packet->cmd_phys);
	MRV8K_DEBUG_FIRMWARE("firmware_cmd at: 0x%p data:0x%p\n", firmware_cmd, firmware_cmd->data);
	MRV8K_DEBUG_FIRMWARE("bootloader at: 0x%p size:%zd\n", priv->bootloader->data, priv->bootloader->size);

	/* Send the bootloader other the bus, if this really necessary  */
	firmware_cmd->type = cpu_to_le16(MRV8K_CMD_BOOTLOADER);
	firmware_cmd->len = cpu_to_le16(priv->bootloader->size);
	memcpy(firmware_cmd->data, priv->bootloader->data, priv->bootloader->size);

	/* write the bootloader in directly into the ram (Note that this is
	 * MRV8K_CMD_BOOTLOADER, and size of the bootloader (0x100)) */
	mrv8k_write_mem(priv, 0xBEF8, cpu_to_le32(0x01000001));

	p=(u32 *)priv->bootloader->data;
	addr=0xBF00;
	while (addr < 0xC000) {
		mrv8k_write_mem(priv, addr, cpu_to_le32(*p++));
		addr+=4;
	}

	/* program the transfer DMA */
	/* the first bank is mapped at 0xc000000 in the card */
	set_dma_for_cmd(priv, 0xC000BEF8);

	/* wait the cpu/bus is ready */
	err = wait_n_test_regC14(priv, 5);
	if (err) {
		MRV8K_ERROR("Failed to load the bootloader\n");
		goto error;
	}

	MRV8K_DEBUG_FIRMWARE("good the bootloader was correctly written\n");

	firmware_cmd->len = 0;
	/* Tell the cpu, that the transfer is finished (len=0) */
	mrv8k_write_mem(priv, 0xBEF8, 0x00000001);
	/* program the transfer DMA */
	set_dma_for_cmd(priv, 0xC000BEF8);	/* the first bank is mapped at 0xc000000 in the card */

	msleep_interruptible(2000);

	/* */
	bytes_already_written = 0;
	do
	{

		if ((priv->firmware->size - bytes_already_written) < 256)
			block_size = priv->firmware->size - bytes_already_written;
		else
			block_size = 256;

		MRV8K_DEBUG_FIRMWARE("Sending firmware %d->%d\n",
					bytes_already_written,
					bytes_already_written+block_size);

		/* Send the firmware other the bus */
		firmware_cmd->len = block_size;
		memcpy(firmware_cmd->data, priv->firmware->data + bytes_already_written, block_size);

		set_dma_for_cmd(priv, packet->cmd_phys);

		/* wait the cpu/bus is ready */
		err = wait_n_test_regC14(priv, 5);
		if (err) {
			MRV8K_ERROR("Failed to write the firmware at offset %d\n",
					bytes_already_written);
			goto error;
		}

		mrv8k_write_nic(priv, MRV8K_REG_C14, 0);

		bytes_already_written += block_size;

	} while (bytes_already_written < priv->firmware->size);


	/* Send an empty buffer to indicate the end of the firmware transmission */
	firmware_cmd->len = 0;
	set_dma_for_cmd(priv, packet->cmd_phys);

	err = wait_n_test_regC14(priv, 0xF0F1F2F4);
	if (err) {
		MRV8K_ERROR("Magic string is missing after programming the card.\n");
		err = -EIO;
		goto error;
	}
	MRV8K_DEBUG_FIRMWARE("Firmware is correctly written.\n");

	mrv8k_write_nic(priv, MRV8K_REG_C14, 0);
	mrv8k_read_nic(priv, MRV8K_REG_C14);

	return 0;
}

/*
 * Request a tx_packet, and link the management frame data to this packet
 * and send the packet to the the card.
 *
 * @return <0 an error (the caller need to reinsert the packet into the free list)
 */
static inline
int mrv8k_send_fmgt_packet(struct mrv8k_private *priv, struct mrv8k_fmgt_packet *fmgt_packet)
{
  struct list_head *element;
  struct mrv8k_tx_packet *tx_packet;

  if (list_empty(&priv->tx_free_list)) {
	  MRV8K_TRACE("no available framemgt buffers\n");
	  return -ENOMEM;
  }


  element = priv->fmgt_free_list.next;
  list_del(element);
  tx_packet = list_entry(element, struct mrv8k_tx_packet, list);
  tx_packet->next = NULL;
  tx_packet->data = &fmgt_packet->data;
  tx_packet->datalen = fmgt_packet->datalen;
  tx_packet->failed = 0;
  tx_packet->fmgt = fmgt_packet;

  MRV8K_TRACE("sending a management frame...\n");

  /* Add the frame to the queue */
  list_add_tail(&tx_packet->list, &priv->tx_pend_list);

  return mrv8k_send_frames(priv);
}

/*
 * Initialize a Management frame header.
 * @param subtype is on the IEEE80211_STYPE_xxxxxx
 * @param dest_addr DA
 * @param bssid bssid
 */
static inline
void ieee80211_mgt_build_frame(struct ieee80211_hdr_4addr *header,
    			       int subtype,
   			       const unsigned char *dest_addr,
			       const unsigned char *src_addr,
			       const unsigned char *bssid)
{
	int fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_PROBE_REQ;

	header->frame_ctl = cpu_to_le16(fc);
	header->duration_id = 0;
	header->seq_ctl = 0;
	/* not From/To DS: Addr1 = DA, Addr2 = SA, Addr3 = BSSID */
	memcpy(header->addr1, dest_addr, ETH_ALEN);
	memcpy(header->addr2, src_addr, ETH_ALEN);
	memcpy(header->addr3, bssid, ETH_ALEN);
}

/*
 * Simple macro to add a new mfie (Management Frame Information Element)
 *
 * Note: the buffer need to have the space to put the content
 *
 * @param data is in fact a struct ieee80211_info_element
 * @param mfie_type need to be one of enum ieee80211_mfie (MFIE_TYPE_xxxx)
 * @param mfie_data data of the mfie
 * @param mfie_len len of data to copy
 */
static void *ieee80211_add_mfie(void *data,
                                enum ieee80211_mfie mfie_type,
			        const char *mfie_data,
			        unsigned int mfie_len)
{
	unsigned char *p = data;
	*p++ = mfie_type;
	*p++ = mfie_len;
	if (mfie_len)
		memcpy(p, mfie_data, mfie_len);
	return p+mfie_len;
}

int mrv8k_hw_send_probe_request(struct mrv8k_private *priv)
{
	struct mrv8k_probe_request *request;
	struct mrv8k_fmgt_packet *packet;
	struct list_head *element;
	void *end_of_data;
	int request_len;
	int err;

	static const unsigned char any[] = {
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff
	};
	static const unsigned char supported_rates_cck[] = {
		/*   IEEE80211_CCK_RATE_1MB  | IEEE80211_BASIC_RATE_MASK,
		     IEEE80211_CCK_RATE_2MB  | IEEE80211_BASIC_RATE_MASK,
		     IEEE80211_CCK_RATE_5MB  | IEEE80211_BASIC_RATE_MASK, */
		IEEE80211_CCK_RATE_11MB | IEEE80211_BASIC_RATE_MASK
	};
	static const unsigned char supported_rates_ofdm[] = {
		IEEE80211_OFDM_RATE_6MB,
		IEEE80211_OFDM_RATE_9MB,
		IEEE80211_OFDM_RATE_12MB,
		IEEE80211_OFDM_RATE_18MB,
		IEEE80211_OFDM_RATE_24MB,
		IEEE80211_OFDM_RATE_36MB,
		IEEE80211_OFDM_RATE_48MB,
		IEEE80211_OFDM_RATE_54MB,
	};

	if (list_empty(&priv->fmgt_free_list)) {
		MRV8K_TRACE("no available framemgt buffers\n");
		return -ENOMEM;
	}

	MRV8K_TRACE("Sending a probe request ...\n");

	element = priv->fmgt_free_list.next;
	packet = list_entry(element, struct mrv8k_fmgt_packet, list);
	list_del(element);
	request = (struct mrv8k_probe_request *)&packet->data;

	ieee80211_mgt_build_frame(&request->header, IEEE80211_STYPE_PROBE_REQ, any, priv->ieee->bssid, any);
	end_of_data = ieee80211_add_mfie(&request->info_element[0], MFIE_TYPE_SSID, NULL, 0);
	end_of_data = ieee80211_add_mfie(end_of_data, MFIE_TYPE_RATES, supported_rates_cck, 4);
	end_of_data = ieee80211_add_mfie(end_of_data, MFIE_TYPE_RATES_EX, supported_rates_ofdm, 8);
	request_len = (unsigned long)end_of_data - (unsigned long)request;
	request->mfie_len = request_len - MRV8K_OFFSET_TO_MFIE;

	packet->datalen = request_len;

	err = mrv8k_send_fmgt_packet(priv, packet);
	if (err < 0) {
		list_add_tail(&packet->list, &priv->fmgt_free_list);
	}

	return err;
}



/**
 * Set power of the Rx antenna
 * @param power: range from 0 to 0xffff [disable]
 * @return 0 if ok
 */
int mrv8k_set_rx_antenna(struct mrv8k_private *priv, unsigned int power)
{
	struct host_command_antenna cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_antenna, 0xc);

	cmd.type = cpu_to_le16(MRV8K_CMD_ANTENNA);
	cmd.len = cpu_to_le16(sizeof(struct host_command_antenna));
	cmd.antenna = cpu_to_le16(MRV8K_CMD_ANTENNA_RX);
	cmd.power = cpu_to_le16(power);

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 * Set power of the Tx antenna
 * @param power: range from 0 to 0xffff [disable]
 * @return 0 if ok
 */
int mrv8k_set_tx_antenna(struct mrv8k_private *priv, unsigned int power)
{
	struct host_command_antenna cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_antenna, 0xc);

	cmd.type = cpu_to_le16(MRV8K_CMD_ANTENNA);
	cmd.len = cpu_to_le16(sizeof(struct host_command_antenna));
	cmd.antenna = cpu_to_le16(MRV8K_CMD_ANTENNA_TX);
	cmd.power = cpu_to_le16(power);

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 * Return some basics statistics from the card
 * @return 0 if ok
 */
int mrv8k_hw_send_get_statistics(struct mrv8k_private *priv)
{
	struct host_command_get_statistics cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_get_statistics, 0x2c);

	cmd.type = cpu_to_le16(MRV8K_CMD_STATISTICS);
	cmd.len = cpu_to_le16(sizeof(struct host_command_get_statistics));

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 * Set the RTS threshold
 * @return 0 if ok
 */
int mrv8k_set_rts_threshold(struct mrv8k_private *priv, unsigned int threshold)
{
	struct host_command_rts_threshold cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_rts_threshold, 0xc);

	cmd.type = cpu_to_le16(MRV8K_CMD_RTS_THRESHOLD);
	cmd.len = cpu_to_le16(sizeof(struct host_command_rts_threshold));
	cmd.rts_threshold = cpu_to_le32(threshold);

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 *
 * @return 0 if ok
 */
int mrv8k_set_bind_channel(struct mrv8k_private *priv, unsigned int channel)
{
	struct host_command_bind_channel cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_bind_channel, 11);

	cmd.type = cpu_to_le16(MRV8K_CMD_BIND_CHANNEL);
	cmd.len = cpu_to_le16(sizeof(struct host_command_bind_channel));
	cmd.channel = channel;

	return mrv8k_hw_send_command(priv, &cmd);
}


/**
 *
 * @return 0 if ok
 */
int mrv8k_set_macaddr(struct mrv8k_private *priv, unsigned char *macaddr)
{
	struct host_command_set_macaddr cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_set_macaddr, 0xe);

	cmd.type = cpu_to_le16(MRV8K_CMD_SET_MACADDR);
	cmd.len = cpu_to_le16(sizeof(struct host_command_set_macaddr));
	memcpy(cmd.macaddr, macaddr, 6);

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 *
 * @return 0 if ok
 */
int mrv8k_hw_send_cmd10e(struct mrv8k_private *priv)
{
	struct host_command_cmd10e cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_cmd10e, 0x8);

	cmd.type = cpu_to_le16(MRV8K_CMD_10E);
	cmd.len = cpu_to_le16(sizeof(struct host_command_cmd10e));

	return mrv8k_hw_send_command(priv, &cmd);
}

/**
 *
 * @return 0 if ok
 */
int mrv8k_hw_send_flush_list(struct mrv8k_private *priv)
{
	struct host_command_flush_list cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_flush_list, 0xc);

	cmd.type = cpu_to_le16(MRV8K_CMD_FLUSH_LIST);
	cmd.len = cpu_to_le16(sizeof(struct host_command_flush_list));

	return mrv8k_hw_send_command(priv, &cmd);
}

int mrv8k_hw_send_bind_ap(struct mrv8k_private *priv, const char *bssid)
{
	struct host_command_bind_bssid cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_bind_bssid, 0xf);

	cmd.type = cpu_to_le16(MRV8K_CMD_BIND_BSSID);
	cmd.len = cpu_to_le16(sizeof(struct host_command_bind_bssid));
	cmd.force = 1;
	memcpy(cmd.bssid, bssid, ETH_ALEN);

	return mrv8k_hw_send_command(priv, &cmd);

}

/**
 *
 * @return 0 if ok
 */
int mrv8k_hw_send_set_rates(struct mrv8k_private *priv, int rate, const unsigned char *rates)
{
	struct host_command_set_rates cmd;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_set_rates, 0x18);

	cmd.type = cpu_to_le16(MRV8K_CMD_SET_TX_RATES);
	cmd.len = cpu_to_le16(sizeof(struct host_command_set_rates));
	if (rate < 0) {
		cmd.fixed_rate = 0;
		cmd.desired_rate = 0;
	} else {
		cmd.fixed_rate = 1;
		cmd.desired_rate = rate;
	}
	memcpy(cmd.authorized_rates, rates, 14);

	return mrv8k_hw_send_command(priv, &cmd);
}


/**
 *
 * @return 0 if ok
 */
int mrv8k_hw_send_radio_setting(struct mrv8k_private *priv,
 			        unsigned int radio_enable,
			        unsigned int preamble_mode)
{
	struct host_command_radio cmd;
	int preamble, radio;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_radio, 0xe);

	if (preamble_mode == MRV8K_PREAMBLE_AUTO)
		preamble = MRV8K_CMD_PREAMBLE_AUTO;
	else if (preamble_mode == MRV8K_PREAMBLE_SHORT)
		preamble = MRV8K_CMD_PREAMBLE_SHORT;
	else
		preamble = MRV8K_CMD_PREAMBLE_LONG;
	radio = radio_enable?MRV8K_CMD_RADIO_ON:MRV8K_CMD_RADIO_OFF;

	cmd.type = cpu_to_le16(MRV8K_CMD_RADIO);
	cmd.len = cpu_to_le16(sizeof(struct host_command_antenna));
	cmd.setone = cpu_to_le16(1);
	cmd.preamble_mode = cpu_to_le16(preamble);
	cmd.radio_enable = cpu_to_le16(radio);

	return mrv8k_hw_send_command(priv, &cmd);
}



/**
 *
 * @return 0 if ok
 */
static int mrv8k_hw_get_configuration(struct mrv8k_private *priv)
{
	struct host_command_get_config cmd;
	int err;

	MRV8K_BUG_ON_CHECK_STRUCT(host_command_get_config, 0x2c);

	cmd.type = cpu_to_le16(MRV8K_CMD_CONFIG);
	cmd.len = cpu_to_le16(sizeof(struct host_command_get_config));
	memset(cmd.macaddr, 0xff, 6);
	cmd.phys_cookie = cpu_to_le32(priv->dma_cookie_phys);

	priv->register_ring_tx_buffer_addr = 0;
	err = mrv8k_hw_send_command(priv, &cmd);
	if (err)
		return err;

	/* Wait the response from the card */
	err = wait_event_interruptible_timeout(priv->config_wait_queue,
			priv->register_ring_tx_buffer_addr != 0,
			10*HZ);

	if (err < 0) {
	   MRV8K_ERROR("Timeout while waiting for CONFIG\n");
	   return -ETIMEDOUT;
	}

	err = mrv8k_hw_check_macaddr(priv);
	if (err)
		return err;

	mrv8k_write_mem(priv, priv->register_rx_buffer0, cpu_to_le32(priv->rx_packets[0].ctrl_dma));
	mrv8k_write_mem(priv, priv->register_rx_buffer1, cpu_to_le32(priv->rx_packets[0].ctrl_dma));
	mrv8k_write_mem(priv, priv->register_ring_tx_buffer_addr, cpu_to_le32(priv->tx_ctrl_buffers_dma[0]));

	ieee80211_set_geo(priv->ieee, &geography_data[priv->geo_code]);

	return 0;
}

/**
 * Print the debug command
 * @param cmd The command (or the response) to print in ascii
 */
#if CONFIG_MRV8K_DEBUG
void mrv8k_print_command(const struct mrv8k_cmd *cmd)
{
  int cmd_type;
  int is_an_ack = 0;

  if (!(MRV8K_DEBUG_LEVEL & MRV8K_DEBUG_LEVEL_COMMAND))
    return;

  cmd_type = le16_to_cpu(cmd->info.def.type);
  if (cmd_type & 0x8000)
    is_an_ack = 1;
  cmd_type &= 0x7fff;

  switch (cmd_type) {
    case MRV8K_CMD_CONFIG:
       {
	 struct host_command_get_config *m = (struct host_command_get_config *) cmd;
	 MRV8K_TRACE("Command GET_CONFIG. %slen=%d macaddr=%x:%x:%x:%x:%x:%x\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(m->len),
	     m->macaddr[0], m->macaddr[1], m->macaddr[2], m->macaddr[3],
	     m->macaddr[4], m->macaddr[5]);
	 return;
       }

    case MRV8K_CMD_RADIO:
       {
	 struct host_command_radio *r = (struct host_command_radio *) cmd;
	 MRV8K_TRACE("Command SET_RADIO. %slen=%d state=%s preamble=%s\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(r->len),
	     r->radio_enable?"on":"off",
	     r->preamble_mode == MRV8K_CMD_PREAMBLE_AUTO ? "auto":
	     (r->preamble_mode == MRV8K_CMD_PREAMBLE_SHORT ? "short" : "long"));
	 return;
       }

    case MRV8K_CMD_ANTENNA:
       {
	 struct host_command_antenna *a = (struct host_command_antenna *)cmd;
	 MRV8K_TRACE("Command SET_ANTENNA. %slen=%d antenna=%s power=0x%4.4x\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(a->len),
	     a->antenna == MRV8K_CMD_ANTENNA_RX?"rx":"tx",
	     a->power);
	 return;
       }

    case MRV8K_CMD_RTS_THRESHOLD:
       {
	 struct host_command_rts_threshold *r;

	 r = (struct host_command_rts_threshold *)cmd;
	 MRV8K_TRACE("Command SET_RTS_TRESHOLD. %slen=%d threshold=%d\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(r->len), r->rts_threshold);
	 return;
       }

    case MRV8K_CMD_BIND_CHANNEL:
       {
	 struct host_command_bind_channel *b;

	 b = (struct host_command_bind_channel *)cmd;
	 MRV8K_TRACE("Command BIND_CHANNEL. %slen=%d channel=%d\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(b->len), b->channel);
	 return;
       }

    case MRV8K_CMD_BIND_BSSID:
       {
	 struct host_command_bind_bssid *b;
#define ETH_FMT "%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x"
#define ETH_VALUES b->bssid[0],b->bssid[1],b->bssid[2], \
	  b->bssid[3],b->bssid[4],b->bssid[5]

	 b = (struct host_command_bind_bssid *)cmd;
	 MRV8K_TRACE("Command BIND_BSSID. %slen=%d %saddr="ETH_FMT"\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(b->len), b->force?"force ":"", ETH_VALUES);
	 return;
       }

    case MRV8K_CMD_SET_TX_RATES:
       {
	 struct host_command_set_rates *r;
#define RATES_FMT "%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d"
#define RATES_VALUES r->authorized_rates[0], r->authorized_rates[1], \
	 r->authorized_rates[2], r->authorized_rates[3], \
	 r->authorized_rates[4], r->authorized_rates[5], \
	 r->authorized_rates[6], r->authorized_rates[7], \
	 r->authorized_rates[8], r->authorized_rates[9], \
	 r->authorized_rates[10], r->authorized_rates[11], \
	 r->authorized_rates[12], r->authorized_rates[13]
	 r = (struct host_command_set_rates *)cmd;
	 MRV8K_TRACE("Command SET_TX_RATES. %slen=%d %s rate=%d ("RATES_FMT")\n",
	     is_an_ack?"ACK ":"", le16_to_cpu(r->len), r->fixed_rate?"fixed":"",
	     r->desired_rate, RATES_VALUES);
	 return;
#undef RATES_FMT
#undef RATES_VALUES
       }

    case MRV8K_CMD_FLUSH_LIST:
       {
	 struct host_command_flush_list *r;

	 r = (struct host_command_flush_list *)cmd;
	 MRV8K_TRACE("Command FLUSH_LIST. %slen=%d\n",
	     is_an_ack?"ACK ":"",
	     le16_to_cpu(r->len));
	 return;
       }

    case MRV8K_CMD_STATISTICS:
       {
	 struct host_command_get_statistics *s;
	 s = (struct host_command_get_statistics *)cmd;
	 MRV8K_TRACE("Command GET STATISTICS. len=%d\n", le16_to_cpu(s->len));
	 if (!is_an_ack)
	   return;
	 MRV8K_TRACE("tx_single_retry_frames = %d\n", 
	     		le32_to_cpu(s->tx_single_retry_frames));
	 MRV8K_TRACE("tx_multiple_retry_frames = %d\n",
	    		le32_to_cpu(s->tx_multiple_retry_frames));
	 MRV8K_TRACE("tx_failed = %d\n",
	     		le32_to_cpu(s->tx_failed));
	 MRV8K_TRACE("rts_sent = %d\n",
	     		le32_to_cpu(s->rts_sent));
	 MRV8K_TRACE("rts_failures = %d\n",
	     		le32_to_cpu(s->rts_failures));
	 MRV8K_TRACE("ack_failures = %d\n",
	     		le32_to_cpu(s->ack_failures));
	 MRV8K_TRACE("duplicated_frames_received = %d\n",
	     		le32_to_cpu(s->duplicated_frames_received));
	 MRV8K_TRACE("rx_fcs_errors = %d\n",
	     		le32_to_cpu(s->rx_fcs_errors));
	 return;
       }



    default:
      MRV8K_TRACE("Command 0x%3.3x. %slen=%d\n",
	  cmd_type, is_an_ack?"ACK ":"", 
	  le16_to_cpu(cmd->info.def.len));
  }

}
#endif

/**
 * Send the next command queue in the pending list
 *
 * Note: This version assume that the lock is already taken
 * If a command is already pending, do nothing.
 */
void __mrv8k_hw_flush_command_queue(struct mrv8k_private *priv)
{
	struct list_head *element;
	struct mrv8k_cmd_packet *packet;

	if (priv->current_cmd) {
		MRV8K_TRACE("command already pending\n");
		return;
	}

	if (list_empty(&priv->cmd_pend_list))
		return;

	element = priv->cmd_pend_list.next;
	list_del(element);
	packet = list_entry(element, struct mrv8k_cmd_packet, list);
	priv->current_cmd = packet;
	MRV8K_TRACE("kickoff command (%p)\n", packet);
	mrv8k_hw_kick_transfert(priv, packet);
}

/*
 * Send the next command queue in the pending list
 */
void mrv8k_hw_flush_command_queue(struct mrv8k_private *priv)
{
	unsigned long flags;

	spin_lock_irqsave(&priv->cmd_send_lock, flags);
	__mrv8k_hw_flush_command_queue(priv);
	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);
}

/**
 * Send a command to the firmware on the card
 *
 * Find a free command buffer, fill the dma, and send it to the card
 * All operation on the list is protected by a spinlock.
 *
 * @return 0 if ok
 */
int mrv8k_hw_send_command(struct mrv8k_private *priv, const void *command)
{
	struct list_head *element;
	struct mrv8k_cmd_packet *packet;
	struct mrv8k_cmd *cmd = (struct mrv8k_cmd *)command;
	unsigned long flags;

	mrv8k_print_command(command);

	spin_lock_irqsave(&priv->cmd_send_lock, flags);

	if (list_empty(&priv->cmd_free_list)) {
		MRV8K_TRACE("no available msg buffers\n");
		goto fail_unlock;
	}

	element = priv->cmd_free_list.next;
	list_del(element);
	packet = list_entry(element, struct mrv8k_cmd_packet, list);

	memcpy(packet->cmd, cmd, cmd->info.def.len);

	list_add_tail(element, &priv->cmd_pend_list);
	MRV8K_TRACE("queue command (%p)\n", packet);

	__mrv8k_hw_flush_command_queue(priv);

	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);

	return 0;

fail_unlock:
	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);
	return -1;
}

/*
 * Send the next packet in the pending list
 *
 */
static int mrv8k_hw_send_frame(struct mrv8k_private *priv)
{
	struct list_head *element;
	struct mrv8k_tx_packet *packet, *frag;
	struct hw_tx_ctrl_buffer *ctrl;
	struct mrv8k_hw_frame *frame;
	unsigned long flags;
	int fc;
	unsigned int len = 0;
	void *dest;

	spin_lock_irqsave(&priv->cmd_send_lock, flags);

	if (list_empty(&priv->tx_pend_list)) {
		MRV8K_TRACE("no tx data buffer to send\n");
		goto fail_unlock;
	}

	element = priv->tx_pend_list.next;
	list_del(element);	/* Remove the packet from the list */
	packet = list_entry(element, struct mrv8k_tx_packet, list);

	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);

	MRV8K_TRACE("%s() packet=%p\n", __FUNCTION__, packet);

	/* Test if we are in power management mode */

	/* Copy data into the slot */
	dest = priv->tx_data_buffers[priv->tx_index];
	frag = packet;
	do {
		memcpy(dest, frag->data, frag->datalen);
		dest += frag->datalen;
		len += frag->datalen;
		frag = frag->next;
	} while (frag);

	/* program control buffer */
	ctrl = priv->tx_ctrl_buffers[priv->tx_index];
	if (packet->failed == 0) {
		/* This packet has already failed, so the crypt is already done */
		/* do_crypt() */
		ctrl->datalen = len-MRV8K_OFFSET_TO_MFIE;
	}

	frame = packet->data;
	fc = le16_to_cpu(frame->header.frame_ctl);
	if (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_MGMT)
		ctrl->speedlinkcanal = 0;
	else
		ctrl->speedlinkcanal = 1;

	ctrl->magic = cpu_to_le32(0x80000001);

	/* FIXME: if it's len, of ctrl->datalen that we need to use */
	atomic_add(len, &priv->tx_pending_bytes);
	atomic_inc(&priv->tx_pending_buffers);
	if (priv->tx_index >= MRV8K_TX_BUFFERS)
		priv->tx_index = 0;
	else
		priv->tx_index++;

	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);

	if (mrv8k_read_mem(priv, priv->register_ring_tx_buffer_addr) == 0xffffffff) {
		MRV8K_ERROR("register tx buffer is error state\n");
		/* Note: what to do with our packet ? */
	}

	mrv8k_write_nic(priv, MRV8K_REG_C18, cpu_to_le32(MRV8K_CPU_TRANSFER_TX));
	mrv8k_read_nic(priv, MRV8K_REG_C14);

	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);

	return 0;

fail_unlock:
	spin_unlock_irqrestore(&priv->cmd_send_lock, flags);
	return -1;
}

/*
 *
 * Note: the frame need to not be attached to a list
 *
 */
int mrv8k_send_frames(struct mrv8k_private *priv)
{
	//int i;

	if (atomic_read(&priv->tx_pending_bytes) > 20000)
		return 0;	/* Too many bytes pending ... */
	if (atomic_read(&priv->tx_pending_buffers) >= 47)
		return 0;	/* Too many buffers used ... */
	if ( ((priv->tx_index+1) % MRV8K_TX_BUFFERS) == priv->tx_last_ack_index )
		return 0;	/* The next pointer is already used */

#if 0
	for (i=0; i<MRV8K_TX_BUFFERS; i++) {
		if ( priv->tx_ctrl_buffers[priv->tx_index]->magic & 0x80000000 )
			goto send;
	}
	return 0;

send:
	mrv8k_write_nic(priv, MRV8K_REG_C18, cpu_to_le32(MRV8K_CPU_TRANSFER_TX));
	mrv8k_read_nic(priv, MRV8K_REG_C14);
#endif
	mrv8k_hw_send_frame(priv);

	return 1;

}


/**
 *
 */
void mrv8k_hw_halt(struct mrv8k_private *priv)
{
	if (mrv8k_read_mem(priv, priv->register_ring_tx_buffer_addr) == 0xffffffff) {
		MRV8K_ERROR("register tx buffer is error state\n");
		return;
	}

	mrv8k_hw_send_radio_setting(priv, 0, priv->preamble);
	msleep_interruptible(1000);
	mrv8k_write_nic(priv, MRV8K_REG_C18, cpu_to_le32(MRV8K_CPU_RESET));
}

/**
 *
 * @return 0 if ok
 *
 * TODO: do the check
 */
static int mrv8k_hw_check_macaddr(struct mrv8k_private *priv)
{
	unsigned char macaddr[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
	int i;

	for (i=0; i<6; i++) {
		macaddr[i] = mrv8k_read_nic8(priv, MRV8K_REG_MACADDR+i);
	}

	MRV8K_TRACE("check_macaddr. macaddr=%x:%x:%x:%x:%x:%x\n",
			macaddr[0], macaddr[1], macaddr[2], macaddr[3],
			macaddr[4], macaddr[5]);

	return 0;
}

int mrv8k_switch_mode(struct mrv8k_private *priv, int mode)
{
	if (mode == priv->ieee->iw_mode)
                return 0;

	priv->ieee->iw_mode = mode;

	return 0;

}

/*
 * Initialize the hardware
 *
 * @return 0 if ok
 */
int mrv8k_initialize_hardware(struct mrv8k_private *priv)
{
	int err;

	/*
	 * FIXME: allow to initialize hardware later if the loading of the
	 * firmware failed 
	 */
	err = mrv8k_load_firmwares(priv);
	if (err)
		return err;

	mrv8k_clear_interruptA(priv);

	mrv8k_write_nic(priv, MRV8K_REG_C38, cpu_to_le32(0x1f));
	mrv8k_read_nic(priv, MRV8K_REG_INT_STATUS);
	mrv8k_write_nic(priv, MRV8K_REG_INT_STATUS, 0);
	mrv8k_write_nic(priv, MRV8K_REG_INTA_MASK, 0);
	mrv8k_write_nic(priv, MRV8K_REG_INTB_MASK, cpu_to_le32(0x1f));

	err = mrv8k_initialize_cpu(priv);
	if (err)
		return err;

	mrv8k_release_firmwares(priv);
	mrv8k_enable_interrupt(priv);

	err = mrv8k_hw_get_configuration(priv);
	if (err)
		return err;

	err = mrv8k_hw_send_radio_setting(priv, priv->radio_on, priv->preamble);
	if (err)
		return err;

	err = mrv8k_set_rx_antenna(priv, priv->rx_antenna);
	if (err)
		return err;

	err = mrv8k_set_tx_antenna(priv, priv->tx_antenna);
	if (err)
		return err;

	err = mrv8k_set_rts_threshold(priv, priv->rts_threshold);
	if (err)
		return err;

	return 0;
}




/*
 * vim:sw=8:sts=8:ts=8:cino=
 */
