/*
  
  Marvell Libertas wireless driver

  Copyright (c) 2005 Luc Saillard <luc@saillard.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#ifndef _MRV8K_H
#define _MRV8K_H

#include <linux/types.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <linux/firmware.h>
#include <net/ieee80211.h>


#define DRV_COPYRIGHT		"Copyright(c) 2005 Luc Saillard <luc@saillard.org>"
#define DRV_VERSION		"0.0.2"
#define DRV_NAME		"mrv8k"
#define DRV_DESCRIPTION		"Marvel 8xxx Wireless driver"
#define PFX			DRV_NAME ": "

#define CONFIG_MRV8K_DEBUG 1

#if CONFIG_MRV8K_DEBUG

#define MRV8K_DEBUG_LEVEL_IO		(1<<0)
#define MRV8K_DEBUG_LEVEL_TRACE		(1<<1)
#define MRV8K_DEBUG_LEVEL_COMMAND	(1<<2)
#define MRV8K_DEBUG_LEVEL_RX		(1<<3)
#define MRV8K_DEBUG_LEVEL_RX_DATA	(1<<4)
#define MRV8K_DEBUG_LEVEL_TX		(1<<5)
#define MRV8K_DEBUG_LEVEL_TX_DATA	(1<<6)
#define MRV8K_DEBUG_LEVEL_FIRMWARE	(1<<7)
#define MRV8K_DEBUG_LEVEL_WX		(1<<8)

#define MRV8K_DEBUG_LEVEL	(0				\
    				/*|MRV8K_DEBUG_LEVEL_IO*/	\
                                |MRV8K_DEBUG_LEVEL_TRACE	\
                                |MRV8K_DEBUG_LEVEL_COMMAND	\
                                |MRV8K_DEBUG_LEVEL_RX		\
                                /*|MRV8K_DEBUG_LEVEL_RX_DATA */	\
                                |MRV8K_DEBUG_LEVEL_TX		\
                                |MRV8K_DEBUG_LEVEL_TX_DATA	\
    				/*|MRV8K_DEBUG_LEVEL_FIRMWARE */ \
                                |MRV8K_DEBUG_LEVEL_WX		\
    				)

#define MRV8K_DEBUG(level, fmt, args...) do {\
  if ((MRV8K_DEBUG_LEVEL_ ##level) & MRV8K_DEBUG_LEVEL) \
     printk(KERN_DEBUG PFX fmt, ##args); \
  } while(0)

#define MRV8K_ERROR(fmt, args...) printk(KERN_ERR PFX fmt, ##args)
#define MRV8K_WARNING(fmt, args...) printk(KERN_WARNING PFX fmt, ##args)
#define MRV8K_INFO(fmt, args...) printk(KERN_INFO PFX fmt, ##args)
#define MRV8K_TRACE(fmt, args...) printk(KERN_DEBUG PFX fmt, ##args)

#define MRV8K_DEBUG_IO(fmt, args...) MRV8K_DEBUG(IO, fmt, ##args)
#define MRV8K_DEBUG_TRACE(fmt, args...) MRV8K_DEBUG(TRACE, fmt, ##args)
#define MRV8K_DEBUG_CMD(fmt, args...) MRV8K_DEBUG(COMMAND, fmt, ##args)
#define MRV8K_DEBUG_RX(fmt, args...) MRV8K_DEBUG(RX, fmt, ##args)
#define MRV8K_DEBUG_TX(fmt, args...) MRV8K_DEBUG(TX, fmt, ##args)
#define MRV8K_DEBUG_FIRMWARE(fmt, args...) MRV8K_DEBUG(FIRMWARE, fmt, ##args)
#define MRV8K_DEBUG_WX(fmt, args...) MRV8K_DEBUG(WX, fmt, ##args)


#else

#define MRV8K_ERROR(fmt, args...) printk(KERN_ERR PFX fmt, ##args)
#define MRV8K_WARNING(fmt, args...) printk(KERN_WARNING PFX fmt, ##args)
#define MRV8K_INFO(fmt, args...) printk(KERN_INFO PFX fmt, ##args)
#define MRV8K_TRACE(fmt, args...) do { } while(0)
#define MRV8K_DEBUG_IO(f, a...) do { } while(0)
#define MRV8K_DEBUG_TRACE(f, a...) do { } while(0)
#define MRV8K_DEBUG_CMD(f, a...) do { } while(0)
#define MRV8K_DEBUG_RX(f, a...) do { } while(0)
#define MRV8K_DEBUG_TX(f, a...) do { } while(0)
#define MRV8K_DEBUG_FIRMWARE(f, a...) do { } while(0)
#define MRV8K_DEBUG_WX(f, a...) do { } while(0)

#endif

#undef BUILD_BUG_ON
#define BUILD_BUG_ON(condition) ((void)sizeof(char[1 - 2*!!(condition)]))

#define MRV8K_BUG_ON_CHECK_STRUCT(s, size) \
   BUILD_BUG_ON((sizeof(struct s) != (size)))




#define MSG_QUEUE_LEN	32
#define MSG_QUEUE_SIZE	512

#define MRV8K_RX_BUFFERS 		48
#define MRV8K_RX_BUFFERS_DATA_SIZE	2302
#define MRV8K_RX_QUEUE_LENGTH		96

#define MRV8K_TX_BUFFERS		48
#define MRV8K_TX_BUFFERS_DATA_SIZE	2302
#define MRV8K_FMGT_BUFFERS		52
#define MRV8K_FMGT_SIZE			2000



#define MRV8K_PREAMBLE_AUTO  0
#define MRV8K_PREAMBLE_SHORT 1
#define MRV8K_PREAMBLE_LONG  2

#define MRV8K_RTS_THRESHOLD_MIN		0
#define MRV8K_RTS_THRESHOLD_MAX		2347
#define MRV8K_RTS_THRESHOLD_DEFAULT	MRV8K_RTS_THRESHOLD_MAX

#define MRV8K_TX_POWER_MIN		0xffff
#define MRV8K_TX_POWER_MAX		2
#define MRV8K_TX_POWER_DEFAULT		MRV8K_TX_POWER_MAX

#define MRV8K_RX_POWER_MIN		0xffff
#define MRV8K_RX_POWER_MAX		2
#define MRV8K_RX_POWER_DEFAULT		MRV8K_TX_POWER_MIN


struct host_command_firmware { /* Frame used to send the bootloader */
   u16 type;
   u16 len;
   u32 dummy;
   u8  data[504];
} __attribute__ ((packed));

struct host_command_antenna {	/* Frame used to send the antenna sensibility */
   u16 type;
   u16 len;
   u32 reserved;
   u16 antenna;
   u16 power;
} __attribute__ ((packed));

struct host_command_get_statistics {	/* Frame used to get statistics from the card */
   u16 type;
   u16 len;
   u16 reserved;
   u16 status;
   u32 tx_single_retry_frames;
   u32 tx_multiple_retry_frames;
   u32 tx_failed;
   u32 rts_sent;
   u32 rts_failures;
   u32 ack_failures;
   u32 duplicated_frames_received;
   u32 rx_fcs_errors;
   u32 dummy;
} __attribute__ ((packed));

struct host_command_radio {	/* Frame used to enable radio */
   u16 type;
   u16 len;
   u32 reserved;
   u16 setone;
   u16 preamble_mode;
   u16 radio_enable;
} __attribute__ ((packed));

struct host_command_get_config {	/* Frame used to get the configuration */
   u16 type;
   u16 len;
   u16 reserved;
   u16 status;
   u16 revision;		/* Fill in the reply */
   u32 reserved2;
   u8  macaddr[6];		/* Set to 0xff */
   u16 model_version;		/* Fill in the reply */
   u16 checksum;		/* Fill in the reply */
   u32 firmware_version;	/* Fill in the reply */
   u32 addr1;			/* Fill in the reply */
   u32 addr2;			
   u32 addr3;
   u32 phys_cookie;		/* Set the physical cookie address buffer */
} __attribute__ ((packed));

struct host_command_set_macaddr {	/* Frame used to set the macaddr of the card */
   u16 type;
   u16 len;
   u32 dummy;
   u8  macaddr[6];
} __attribute__ ((packed));

struct host_command_rts_threshold {	/* Frame used to configure RTS threshold */
   u16 type;
   u16 len;
   u16 reserved;
   u16 setone;		/* Enable / Disabled ? */
   u32 rts_threshold;
} __attribute__ ((packed));

struct host_command_flush_list {
   u16 type;
   u16 len;
   u8 reserved[8];
} __attribute__ ((packed));

struct host_command_bind_bssid { /* Frame used to bind to an access point */
   u16 type;
   u16 len;
   u32 dummy;
   u8  force;
   u8  bssid[ETH_ALEN];
} __attribute__ ((packed));

struct host_command_set_rates {
   u16     type;
   u16     len;
   u32     reserved;
   u8      fixed_rate;             /* set to 1 if the rate is fixed by the user */
   u8      desired_rate;           /* the hardware rate from (0 to 13) */
   u8      authorized_rates[14];   /* possible rates to listen */
};  /* set rates accepted by the card */


struct host_command_cmd10e {	/* Frame used to change mode (ad-hoc, ap) */
   u16 type;
   u16 len;
   u32 reserved;
} __attribute__ ((packed));


struct host_command_bind_channel {	/* Frame used to bind to a channel */
   u16 type;
   u16 len;
   u8  reserved[6];
   u8  channel;
} __attribute__ ((packed));




struct host_command_template {	/* Frame used to enable radio */
   u16 type;
   u16 len;
} __attribute__ ((packed));

struct mrv8k_cmd {
   union {
      struct host_command_template def;
      struct host_command_firmware firmware;
      struct host_command_antenna antenna;
      struct host_command_radio radio;
      struct host_command_get_config config;
      struct host_command_set_macaddr macaddr;
      struct host_command_rts_threshold threshold;
      struct host_command_cmd10e cmd10e;
      struct host_command_bind_channel bind_channel;
   } info;
} __attribute__ ((packed));

struct mrv8k_event_msg {
   __le16 type;
   __le16 len;
   u32 dummy;
} __attribute__ ((packed));

struct mrv8k_event_config {
   u16 type;
   u16 len;
   u32 reserved1;
   u16 revision;
   u8  reserved2[4];
   u8  macaddr[6];
   u16 model_version;
   u8  reserved3[2];
   u32 firmware_version;
   u32 addr1;
   u32 addr2;
   u32 addr3;
   u32 reserved4;
} __attribute__ ((packed));


/*
 * Structures used to send Management Frame
 * all frames is managed in software, and is not fully compatible with the spec
 */
#define MRV8K_OFFSET_TO_MFIE	0x20
struct mrv8k_probe_request {
   u16 mfie_len;
   struct ieee80211_hdr_4addr header;
   /* SSID, supported rates */
   struct ieee80211_info_element info_element[0];
} __attribute__((packed));

struct mrv8k_hw_frame {	/* Standart format frame used */
   u16 mfie_len;
   struct ieee80211_hdr_4addr header;
   /* SSID, supported rates */
   struct ieee80211_info_element info_element[0];
} __attribute__((packed));

#define MRV8K_RSSI_TO_DBM 	(-96)
#define IEEE80211_STYPE_ERROR	(0xf0)

/*
 * Structures read/write in the card
 */
struct hw_rx_ctrl_buffer {
   u8   status;
   s8   rssi;
   u16  var_02;			/* Need to be set to 1 */
   u16  datalen;
   u8   var_06;
   u8   reserved;
   u32  phys_data;		/* Point to the dma buffer use to transfer the data */
   u32  phys_next;		/* Point to the next dma buffer to transfer control data */
   u32  ieee80211_frame;	/* Point to the corresponding frame, not working in 64 bits land */
} __attribute__((packed));

struct hw_tx_ctrl_buffer { /* format used when sending frame to the dma engine */
   u32 magic;
   u8  speedlinkcanal;
   u8  reserved[3];
   u32 phys_data;
   u16 datalen;	
   u8  reserved2[6];
   u32 phys_next;	/* Point to the next hw_tx_ctrl_buffer buffer */
   u8  reserved3[8];
};

/*
 *
 */

struct mrv8k_cmd_packet {

   struct list_head  list;
   struct mrv8k_cmd *cmd;		/* virtual pointer to queue */
   dma_addr_t        cmd_phys;		/* physical pointer to queue */

};


/*
 *
 */
struct mrv8k_fmgt_packet {
   struct list_head list;
   unsigned int datalen;
   unsigned char data[MRV8K_FMGT_SIZE];
};

/*
 * A frame packet can contains severall part, so the next pointer is used to
 * link fragment into one.
 */
struct mrv8k_tx_packet {

   struct list_head  list;
   struct mrv8k_tx_packet *next;
   void *data;				/* Point to the real data */
   struct mrv8k_fmgt_packet *fmgt;
   unsigned int datalen;

   unsigned char failed;
};

/*
 * Abstract structure to represent the content of data send by the card 
 */
struct mrv8k_rx_data {
   u16 datalen;
   union {
      unsigned char payload[MRV8K_RX_BUFFERS_DATA_SIZE];
      struct ieee80211_hdr_4addr ieee80211hdr;
      struct ieee80211_probe_response beacon;
      /* 
       * rx_error_frame
       * 000 80 00 F0 00 00 00 52 78  20 53 74 75 63 6B 21 20  ......Rx  Stuck! 
       * 010 52 65 73 65 74 20 52 58  20 61 6E 64 20 42 42 55  Reset RX  and BBU
       */
      struct rx_error_frame {
	 struct ieee80211_hdr ieee80211hdr;
	 u8  string_error[122];
      } rx_error;
   } u;
} __attribute__ ((packed));

/*
 * This structure will be fill when a packet is received from the card
 * We have a pointer to the skb, and to the physical data.
 * When a packet arrive, we will pick up a free mrv8k_rx_packet from the pool
 * and replace in the ring hardware this dma buffer
 */
struct mrv8k_rx_packet {

   struct list_head          list;
   struct sk_buff           *skb;

   struct mrv8k_rx_data     *data;
   dma_addr_t		     data_dma;

   struct hw_rx_ctrl_buffer *ctrl;
   dma_addr_t		     ctrl_dma;
};

#define STATUS_INITIALIZED      (1<<1)  /* Buffer is allocated, card is ready to send data */
#define STATUS_ASSOCIATING      (1<<2)	/* Currently trying to associating ...*/
#define STATUS_ASSOCIATED       (1<<3)	/* Associated and BSSID valid */
#define STATUS_SCANNING         (1<<4)	/* Scanning in progress */

#define CFG_STATIC_CHANNEL      (1<<0)  /* Restrict assoc. to single channel */
#define CFG_STATIC_ESSID        (1<<1)  /* Restrict assoc. to single SSID */
#define CFG_STATIC_BSSID        (1<<2)  /* Restrict assoc. to single BSSID */
#define CFG_CUSTOM_MAC          (1<<3)
#define CFG_PREAMBLE            (1<<4)
#define CFG_RTS_TRESHOLD        (1<<5)
#define CFG_ASSOCIATE           (1<<6)
#define CFG_FIXED_RATE          (1<<7)
#define CFG_ADHOC_CREATE        (1<<8)


/*
 * Our big structure that contains anything
 */
struct mrv8k_private {

	struct pci_dev *pci_dev;
	struct net_device *net_dev;
	struct ieee80211_device *ieee;
	struct net_device_stats stats;
	struct iw_statistics wstats;

	const struct firmware *bootloader;
	const struct firmware *firmware;

	/* */
	unsigned int status;
	unsigned int config;
	int geo_code;
	int model_code;

	/* HW */
	void *bar1;
	unsigned long bar1_phys;
	void *bar2;
	unsigned long bar2_phys;

	void *dma_cookie;		/* Use to store a dma cookie of 4 bytes long */
	dma_addr_t dma_cookie_phys;

	unsigned int register_ring_tx_buffer_addr;	/* Addr in bank that can change with a FW revision */
	unsigned int register_rx_buffer0;
	unsigned int register_rx_buffer1;


	/* Control Message */
	spinlock_t cmd_send_lock;
	struct list_head cmd_free_list;
	struct list_head cmd_pend_list;
	struct mrv8k_cmd_packet *cmd_buffers;
	struct mrv8k_cmd_packet *current_cmd;

	/* Rx buffers */
	int			  rx_index;
	struct mrv8k_rx_packet	 *rx_packets;

	/* Tx buffers */
	spinlock_t		  tx_send_lock;
	unsigned int		  tx_index;		/* Current index to write a new buffer */
	unsigned int		  tx_last_ack_index;	/* Last index acknownledge by the card */
	atomic_t		  tx_pending_bytes;
	atomic_t		  tx_pending_buffers;
		/* DMA memory to send packets */
	void      		 *tx_buffers;
	dma_addr_t 		  tx_buffers_dma;
	struct hw_tx_ctrl_buffer *tx_ctrl_buffers[MRV8K_TX_BUFFERS];
	dma_addr_t 		  tx_ctrl_buffers_dma[MRV8K_TX_BUFFERS];
	void 			 *tx_data_buffers[MRV8K_TX_BUFFERS];
	dma_addr_t 		  tx_data_buffers_dma[MRV8K_TX_BUFFERS];
		/* Packets to send a queue in the tx_pend_list */
	struct mrv8k_tx_packet	 *tx_packets;
	struct list_head	  tx_pend_list;	/* List that contains object of type struct mrv8k_tx_packet */
	struct list_head	  tx_free_list;
		/* Sending Management Frame need to allocate the buffer */
	struct mrv8k_fmgt_packet *fmgt_buffers;	/* Some space allocated to send frame management */
	struct list_head	  fmgt_free_list;


	/* setting - wifi */
	unsigned int  rx_antenna;	/* Power / dB ? set to 65535 to desactivate */
	unsigned int  tx_antenna;	/* Power / dB ? */
	unsigned int  preamble;		/* PREAMBLE_AUTO, PREAMBLE_SHORT, PREAMBLE_LONG */
	unsigned char channel;		/* */
	unsigned char radio_on;		/* is the radio power on ? */
	unsigned int  rts_threshold;	/* RTS threshold (255-2346) */ 
	unsigned int  tx_rates;		/* speed in 500kbits/s */

	unsigned char bssid[ETH_ALEN];
	unsigned char macaddr[ETH_ALEN];
	unsigned char permanent_macaddr[ETH_ALEN];
	unsigned int  essid_len;
	unsigned char essid[IW_ESSID_MAX_SIZE];
	unsigned char nick[IW_ESSID_MAX_SIZE];



	/* Interrupt  */
	struct tasklet_struct irq_tasklet;
	volatile int irq_status;		/* Four lower bits */
	wait_queue_head_t config_wait_queue;	/* Some cmd need an it to finish */

};

#define MRV8K_EVENT_CONFIG	0x8003
#define MRV8K_ACK_STATISTICS	0x8014
#define MRV8K_ACK_FLUSH_LIST	0x8107
#define MRV8K_ACK_BIND_BSSID	0x8108
#define MRV8K_ACK_SET_CHANNEL	0x810a

#define MRV8K_CMD_BOOTLOADER		0x001
#define MRV8K_CMD_CONFIG		0x003
#define MRV8K_CMD_MULTICAST_LIST	0x010	/* Set the list of Multicast Address (32 max) */
#define MRV8K_CMD_STATISTICS		0x014	/* Retrieve some basics statistics */
#define MRV8K_CMD_RADIO			0x01c	/* Enable/Disable radio */
#define MRV8K_CMD_1E			0x01e
#define MRV8K_CMD_ANTENNA		0x020	/* Set power for antenna (Rx or Tx) */
#define MRV8K_CMD_21			0x021
#define MRV8K_CMD_RTS_THRESHOLD		0x02e	/* Set RTS threshold */
#define MRV8K_CMD_2F			0x02f
#define MRV8K_CMD_100			0x100
#define MRV8K_CMD_101			0x101
#define MRV8K_CMD_FLUSH_LIST		0x107
#define MRV8K_CMD_BIND_BSSID		0x108
#define MRV8K_CMD_109			0x109
#define MRV8K_CMD_BIND_CHANNEL		0x10a	/* Bind card to a channel */
#define MRV8K_CMD_10B			0x10b
#define MRV8K_CMD_10C			0x10c
#define MRV8K_CMD_10D			0x10d
#define MRV8K_CMD_10E			0x10e	/* Change mode (ibss/ad-hoc or infrastructure) */
#define MRV8K_CMD_10F			0x10f
#define MRV8K_CMD_SET_TX_RATES		0x110	/* Set rate */
#define MRV8K_CMD_111			0x111
#define MRV8K_CMD_201			0x201
#define MRV8K_CMD_SET_MACADDR		0x202	/* Set the hardware macaddr */

#define MRV8K_CMD_ANTENNA_RX	1
#define MRV8K_CMD_ANTENNA_TX	2
#define MRV8K_CMD_ANTENNA_RXA	4
#define MRV8K_CMD_ANTENNA_TXA	8

#define MRV8K_CMD_PREAMBLE_AUTO  5
#define MRV8K_CMD_PREAMBLE_SHORT 3
#define MRV8K_CMD_PREAMBLE_LONG  1

#define MRV8K_CMD_RADIO_OFF	0
#define MRV8K_CMD_RADIO_ON	1

/*
 * Some addr in bank1
 */
#define MRV8K_MEM_STACKPTR	0xBEF8
#define MRV8K_MEM_BOOTLOADER	0xBC00
#define MRV8K_MEM_F8B		0x0F8B

/*
 * Some MMIO register in bank2
 */

#define MRV8K_REG_C10		0xC10	/*   W 32bits Usualy the register to store physical address */
#define MRV8K_REG_C14		0xC14	/* R/W 32bits */
#define MRV8K_REG_C18		0xC18	/*   W 32bits Usualy the register to program the DMA engine */
#define MRV8K_REG_INT_STATUS	0xC30	/* R/W 32bits */
#define MRV8K_REG_INTA_MASK	0xC34	/*   W 32bits */
#define MRV8K_REG_C38		0xC38	/*   W 32bits */
#define MRV8K_REG_INTB_MASK	0xC3C	/*   W 32bits */
#define MRV8K_REG_MACADDR	0xA528	/* R   6*8bits  */
#define MRV8K_REG_C104		0xC104
#define MRV8K_REG_C124		0xC124
#define MRV8K_REG_C128		0xC128

#define MRV8K_CPU_TRANSFER_TX		0x00000001
#define MRV8K_CPU_TRANSFER_CMD		0x00000002	/* Fill in C10, the address to transfer */
#define MRV8K_CPU_TRANSFER_RX		0x00000004	/* Not sure */
#define MRV8K_CPU_4000			0x00004000
#define MRV8K_CPU_2000			0x00002000
#define MRV8K_CPU_RESET			0x04000000

#define MRV8K_MAGIC_COOKIE		0xaa55aa55
#define MRV8K_BOOT_DONE			0xf0f1f2f4

#define MRV8K_INTA_MASK			0x1f
#define MRV8K_INTA_TX_COMPLETE		(1<<0)
#define MRV8K_INTA_RX_TRANSFER		(1<<1)
#define MRV8K_INTA_EVENT_INTERRUPT	(1<<2)
#define MRV8K_INTA_BIT3			(1<<3)

#define MRV8K_INTB_MASK			0x1f

static inline unsigned int mrv8k_read_mem(struct mrv8k_private *priv, unsigned int reg)
{
  unsigned int value = readl(priv->bar1 + reg);
  MRV8K_DEBUG_IO("mrv8k_read_mem(0x%04x) --> 0x%8.8x\n", reg, value);
  return value;
}

static inline void mrv8k_write_mem(struct mrv8k_private *priv, unsigned int reg, u32 value)
{
  MRV8K_DEBUG_IO("mrv8k_write_mem(0x%04x) <-- 0x%8.8x\n", reg, value);
  writel(value, (void *)(priv->bar1 + reg));
}

static inline unsigned int mrv8k_read_nic(struct mrv8k_private *priv, unsigned int reg)
{
  unsigned int value = readl(priv->bar2 + reg);
  MRV8K_DEBUG_IO("mrv8k_read_nic(0x%04x) --> 0x%8.8x\n", reg, value);
  return value;
}

static inline unsigned char mrv8k_read_nic8(struct mrv8k_private *priv, unsigned int reg)
{
  unsigned char value = readb(priv->bar2 + reg);
  MRV8K_DEBUG_IO("mrv8k_read_nic8(0x%04x) --> 0x%2.2x\n", reg, value);
  return value;
}


static inline void mrv8k_write_nic(struct mrv8k_private *priv, unsigned int reg, u32 value)
{
  MRV8K_DEBUG_IO("mrv8k_write_nic(0x%04x) <-- 0x%8.8x\n", reg, value);
  writel(value, (void *)(priv->bar2 + reg));
}

/* mrv8k_main.c */
int mrv8k_start_scan(struct mrv8k_private *priv);

/* mrv8k_irq.c */
void mrv8k_clear_interruptA(struct mrv8k_private *priv);
void mrv8k_enable_interrupt(struct mrv8k_private *priv);
irqreturn_t mrv8k_interrupt(int irq, void *data, struct pt_regs *regs);
void mrv8k_irq_tasklet(struct mrv8k_private *priv);

/* mrv8k_hw.c */
int mrv8k_initialize_hardware(struct mrv8k_private *priv);
int mrv8k_hw_send_command(struct mrv8k_private *priv, const void *command);
int mrv8k_set_rx_antenna(struct mrv8k_private *priv, unsigned int power);
int mrv8k_set_tx_antenna(struct mrv8k_private *priv, unsigned int power);
int mrv8k_hw_send_flush_list(struct mrv8k_private *priv);
int mrv8k_hw_send_scan_start(struct mrv8k_private *priv);
int mrv8k_hw_send_bind_ap(struct mrv8k_private *priv, const char *bssid);
int mrv8k_set_bind_channel(struct mrv8k_private *priv, unsigned int channel);
int mrv8k_hw_send_set_rates(struct mrv8k_private *priv, int rate, const unsigned char *rates);
int mrv8k_hw_send_get_statistics(struct mrv8k_private *priv);
int mrv8k_set_rts_threshold(struct mrv8k_private *priv, unsigned int threshold);
int mrv8k_hw_send_radio_setting(struct mrv8k_private *priv, unsigned int radio_enable, unsigned int preamble_mode);
void mrv8k_hw_halt(struct mrv8k_private *priv);
void mrv8k_hw_flush_command_queue(struct mrv8k_private *priv);
void __mrv8k_hw_flush_command_queue(struct mrv8k_private *priv);
int mrv8k_send_frames(struct mrv8k_private *priv);
int mrv8k_hw_send_probe_request(struct mrv8k_private *priv);
int mrv8k_switch_mode(struct mrv8k_private *priv, int mode);

#ifdef CONFIG_MRV8K_DEBUG
void mrv8k_print_command(const struct mrv8k_cmd *cmd);
#else
void mrv8k_print_command(const struct mrv8k_cmd *cmd) { }
#endif

#endif

/*
 * vim:ts=8
 */
