/*
 * Host AP crypt: host-based WEP encryption implementation for Host AP driver
 * (adapted for use within the adm8211 driver)
 *
 * Copyright (c) 2002-2004, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/wireless.h>
#include <linux/crc32.h>
#include <asm/string.h>
#include <asm/scatterlist.h>

#ifndef CONFIG_CRYPTO
#error CONFIG_CRYPTO required to build
#endif
 
#include <linux/crypto.h>
#include "ieee80211.h"

/* Perform WEP encryption on given skb that has at least 4 bytes of headroom
 * for IV and 4 bytes of tailroom for ICV. Both IV and ICV will be transmitted,
 * so the payload length increases with 8 bytes.
 *
 * WEP frame payload: IV + TX key idx, RC4(data), ICV = RC4(CRC32(data))
 */
int wep_encrypt(struct sk_buff *skb, int hdr_len,
		struct ieee80211_data *data)
{
	struct ieee80211_wep_data *wep = &data->wep_data[data->default_key];
	struct scatterlist sg;
	u32 crc, klen, len;
	u8 key[WEP_KEY_MAX_LEN + 3];
	u8 *pos, *icv;

	if (skb->len < hdr_len)
		return -1;

	if (skb_headroom(skb) < 4 || skb_tailroom(skb) < 4) {
		if (pskb_expand_head(skb, (skb_headroom(skb) < 4 ? 4 : 0),
		    (skb_tailroom(skb) < 4 ? 4 : 0), GFP_ATOMIC))
			return -1;
	}

	len = skb->len - hdr_len;
	pos = skb_push(skb, 4);
	memmove(pos, pos + 4, hdr_len);
	pos += hdr_len;

	klen = 3 + wep->len;

	wep->iv++;

	/* Fluhrer, Mantin, and Shamir have reported weaknesses in the key
	 * scheduling algorithm of RC4. At least IVs (KeyByte + 3, 0xff, N)
	 * can be used to speedup attacks, so avoid using them. */
	if ((wep->iv & 0xff00) == 0xff00) {
		u8 B = (wep->iv >> 16) & 0xff;
		if (B >= 3 && B < klen)
			wep->iv += 0x0100;
	}

	/* Prepend 24-bit IV to RC4 key and TX frame */
	*pos++ = key[0] = (wep->iv >> 16) & 0xff;
	*pos++ = key[1] = (wep->iv >> 8) & 0xff;
	*pos++ = key[2] = wep->iv & 0xff;
	*pos++ = data->default_key << 6;

	/* Copy rest of the WEP key (the secret part) */
	memcpy(key + 3, wep->key, wep->len);

	/* Append little-endian CRC32 and encrypt it to produce ICV */
	crc = ~crc32_le(~0, pos, len);
	icv = skb_put(skb, 4);
	icv[0] = crc;
	icv[1] = crc >> 8;
	icv[2] = crc >> 16;
	icv[3] = crc >> 24;

	crypto_cipher_setkey(data->tfm, key, klen);
	sg.page = virt_to_page(pos);
	sg.offset = offset_in_page(pos);
	sg.length = len + 4;
	crypto_cipher_encrypt(data->tfm, &sg, &sg, len + 4);
	
	return 0;
}


/* Perform WEP decryption on given buffer. Buffer includes whole WEP part of
 * the frame: IV (4 bytes), encrypted payload (including SNAP header),
 * ICV (4 bytes). len includes both IV and ICV.
 *
 * Returns 0 if frame was decrypted successfully and ICV was correct and -1 on
 * failure. If frame is OK, IV and ICV will be removed.
 */
int wep_decrypt(struct sk_buff *skb, int hdr_len,
		struct ieee80211_data *data)
{
	struct ieee80211_wep_data *wep;
	struct scatterlist sg;
	u32 crc, klen, plen;
	u8 key[WEP_KEY_MAX_LEN + 3];
	u8 keyidx, *pos, icv[4];

	if (skb->len < hdr_len + 8)
		return -1;

	pos = skb->data + hdr_len;
	key[0] = *pos++;
	key[1] = *pos++;
	key[2] = *pos++;
	keyidx = *pos++ >> 6;
	keyidx %= 4;

	wep = &data->wep_data[keyidx];
	klen = 3 + wep->len;

	/* Copy rest of the WEP key (the secret part) */
	memcpy(key + 3, wep->key, wep->len);

	/* Apply RC4 to data and compute CRC32 over decrypted data */
	plen = skb->len - hdr_len - 8;

	crypto_cipher_setkey(data->tfm, key, klen);
	sg.page = virt_to_page(pos);
	sg.offset = offset_in_page(pos);
	sg.length = plen + 4;
	crypto_cipher_decrypt(data->tfm, &sg, &sg, plen + 4);

	crc = ~crc32_le(~0, pos, plen);
	icv[0] = crc;
	icv[1] = crc >> 8;
	icv[2] = crc >> 16;
	icv[3] = crc >> 24;
	if (memcmp(icv, pos + plen, 4) != 0) {
		/* ICV mismatch - drop frame */
		return -2;
	}

	/* Remove IV and ICV */
	memmove(skb->data + 4, skb->data, hdr_len);
	skb_pull(skb, 4);
	skb_trim(skb, skb->len - 4);

	return 0;
}

