#ifndef IEEE80211_H
#define IEEE80211_H

#define BIT(x) (1 << (x))

/* IEEE 802.11 defines */

#define WLAN_FCS_LEN 4
#define WLAN_ADDR3_HDR_LEN 24

#define WLAN_FC_PVER (BIT(1) | BIT(0))
#define WLAN_FC_TODS BIT(8)
#define WLAN_FC_FROMDS BIT(9)
#define WLAN_FC_MOREFRAG BIT(10)
#define WLAN_FC_RETRY BIT(11)
#define WLAN_FC_PWRMGT BIT(12)
#define WLAN_FC_MOREDATA BIT(13)
#define WLAN_FC_ISWEP BIT(14)
#define WLAN_FC_ORDER BIT(15)

#define WLAN_FC_GET_TYPE(fc) (((fc) & (BIT(3) | BIT(2))) >> 2)
#define WLAN_FC_GET_STYPE(fc) \
	(((fc) & (BIT(7) | BIT(6) | BIT(5) | BIT(4))) >> 4)

#define WLAN_GET_SEQ_FRAG(seq) ((seq) & (BIT(3) | BIT(2) | BIT(1) | BIT(0)))
#define WLAN_GET_SEQ_SEQ(seq) \
	(((seq) & (~(BIT(3) | BIT(2) | BIT(1) | BIT(0)))) >> 4)

#define IEEE80211_FC(type, stype) cpu_to_le16((type << 2) | (stype << 4))

#define WLAN_FC_TYPE_MGMT 0
#define WLAN_FC_TYPE_CTRL 1
#define WLAN_FC_TYPE_DATA 2

/* management */
#define WLAN_FC_STYPE_ASSOC_REQ 0
#define WLAN_FC_STYPE_ASSOC_RESP 1
#define WLAN_FC_STYPE_REASSOC_REQ 2
#define WLAN_FC_STYPE_REASSOC_RESP 3
#define WLAN_FC_STYPE_PROBE_REQ 4
#define WLAN_FC_STYPE_PROBE_RESP 5
#define WLAN_FC_STYPE_BEACON 8
#define WLAN_FC_STYPE_ATIM 9
#define WLAN_FC_STYPE_DISASSOC 10
#define WLAN_FC_STYPE_AUTH 11
#define WLAN_FC_STYPE_DEAUTH 12

/* control */
#define WLAN_FC_STYPE_PSPOLL 10
#define WLAN_FC_STYPE_RTS 11
#define WLAN_FC_STYPE_CTS 12
#define WLAN_FC_STYPE_ACK 13
#define WLAN_FC_STYPE_CFEND 14
#define WLAN_FC_STYPE_CFENDACK 15

/* data */
#define WLAN_FC_STYPE_DATA 0
#define WLAN_FC_STYPE_DATA_CFACK 1
#define WLAN_FC_STYPE_DATA_CFPOLL 2
#define WLAN_FC_STYPE_DATA_CFACKPOLL 3
#define WLAN_FC_STYPE_NULLFUNC 4
#define WLAN_FC_STYPE_CFACK 5
#define WLAN_FC_STYPE_CFPOLL 6
#define WLAN_FC_STYPE_CFACKPOLL 7

/* Authentication algorithms */
#define WLAN_AUTH_OPEN 0
#define WLAN_AUTH_SHARED_KEY 1

#define WLAN_AUTH_CHALLENGE_LEN 128

#define WLAN_CAPABILITY_ESS BIT(0)
#define WLAN_CAPABILITY_IBSS BIT(1)
#define WLAN_CAPABILITY_CF_POLLABLE BIT(2)
#define WLAN_CAPABILITY_CF_POLL_REQUEST BIT(3)
#define WLAN_CAPABILITY_PRIVACY BIT(4)
#define WLAN_CAPABILITY_SHORT_PREAMBLE BIT(5)

/* Status codes */
#define WLAN_STATUS_SUCCESS 0
#define WLAN_STATUS_UNSPECIFIED_FAILURE 1
#define WLAN_STATUS_CAPS_UNSUPPORTED 10
#define WLAN_STATUS_REASSOC_NO_ASSOC 11
#define WLAN_STATUS_ASSOC_DENIED_UNSPEC 12
#define WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG 13
#define WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION 14
#define WLAN_STATUS_CHALLENGE_FAIL 15
#define WLAN_STATUS_AUTH_TIMEOUT 16
#define WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA 17
#define WLAN_STATUS_ASSOC_DENIED_RATES 18
/* 802.11b */
#define WLAN_STATUS_ASSOC_DENIED_NOSHORT 19
#define WLAN_STATUS_ASSOC_DENIED_NOPBCC 20
#define WLAN_STATUS_ASSOC_DENIED_NOAGILITY 21

/* Reason codes */
#define WLAN_REASON_UNSPECIFIED 1
#define WLAN_REASON_PREV_AUTH_NOT_VALID 2
#define WLAN_REASON_DEAUTH_LEAVING 3
#define WLAN_REASON_DISASSOC_DUE_TO_INACTIVITY 4
#define WLAN_REASON_DISASSOC_AP_BUSY 5
#define WLAN_REASON_CLASS2_FRAME_FROM_NONAUTH_STA 6
#define WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA 7
#define WLAN_REASON_DISASSOC_STA_HAS_LEFT 8
#define WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH 9


/* Information Element IDs */
#define WLAN_EID_SSID 0
#define WLAN_EID_SUPP_RATES 1
#define WLAN_EID_FH_PARAMS 2
#define WLAN_EID_DS_PARAMS 3
#define WLAN_EID_CF_PARAMS 4
#define WLAN_EID_TIM 5
#define WLAN_EID_IBSS_PARAMS 6
#define WLAN_EID_CHALLENGE 16

#define IEEE80211_MAX_SSID_LEN 32

#define WEP_KEY_MAX_LEN 13
#define WEP_KEY_MAX_KEYS 4

#define FCS_LEN 4

#define MAC2STR(a) (a)[0], (a)[1], (a)[2], (a)[3], (a)[4], (a)[5]
#define MACSTR "%02x:%02x:%02x:%02x:%02x:%02x"


struct ieee80211_hdr {
	__le16 frame_control;
	__le16 duration_id;
	u8 addr1[6];
	u8 addr2[6];
	u8 addr3[6];
	__le16 seq_ctrl;
	u8 addr4[6];
} __attribute__ ((packed));

union ieee80211_mgmt_payload {
	struct {
		__le16 auth_alg;
		__le16 auth_transaction;
		__le16 status_code;
		/* possibly followed by Challenge text */
		u8 variable[0];
	} __attribute__ ((packed)) auth;
	struct {
		__le16 reason_code;
	} __attribute__ ((packed)) deauth;
	struct {
		__le16 capab_info;
		__le16 listen_interval;
		/* followed by SSID and Supported rates */
		u8 variable[0];
	} __attribute__ ((packed)) assoc_req;
	struct {
		__le16 capab_info;
		__le16 status_code;
		__le16 aid;
		/* followed by Supported rates */
		u8 variable[0];
	} __attribute__ ((packed)) assoc_resp, reassoc_resp;
	struct {
		__le16 capab_info;
		__le16 listen_interval;
		u8 current_ap[ETH_ALEN];
		/* followed by SSID and Supported rates */
		u8 variable[0];
	} __attribute__ ((packed)) reassoc_req;
	struct {
		__le16 reason_code;
	} __attribute__ ((packed)) disassoc;
	struct {
		__le64 timestamp;
		__le16 beacon_int;
		__le16 capab_info;
		/* followed by some of SSID, Supported rates,
		 * FH Params, DS Params, CF Params, IBSS Params, TIM */
		u8 variable[0];
	} __attribute__ ((packed)) beacon;
	struct {
		/* only variable items: SSID, Supported rates */
		u8 variable[0];
	} __attribute__ ((packed)) probe_req;
	struct {
		__le64 timestamp;
		__le16 beacon_int;
		__le16 capab_info;
		/* followed by some of SSID, Supported rates,
		 * FH Params, DS Params, CF Params, IBSS Params */
		u8 variable[0];
	} __attribute__ ((packed)) probe_resp;
};

struct ieee80211_mgmt {
	__le16 frame_control;
	__le16 duration;
	u8 da[6];
	u8 sa[6];
	u8 bssid[6];
	__le16 seq_ctrl;
	union ieee80211_mgmt_payload u;
} __attribute__ ((packed));

/* Parsed Information Elements */
struct ieee802_11_elems {
	u8 *ssid;
	u8 ssid_len;
	u8 *supp_rates;
	u8 supp_rates_len;
	u8 *fh_params;
	u8 fh_params_len;
	u8 *ds_params;
	u8 ds_params_len;
	u8 *cf_params;
	u8 cf_params_len;
	u8 *tim;
	u8 tim_len;
	u8 *ibss_params;
	u8 ibss_params_len;
	u8 *challenge;
	u8 challenge_len;
};

typedef enum { ParseOK = 0, ParseUnknown = 1, ParseFailed = -1 } ParseRes;

ParseRes ieee802_11_parse_elems(u8 *start, size_t len,
				struct ieee802_11_elems *elems);


struct ieee80211_rx_status {
	int rssi;
	int rate; /* in 100 kbps */
};


struct ieee80211_bss {
	u8 bssid[ETH_ALEN];
	u8 ssid[IEEE80211_MAX_SSID_LEN];
	size_t ssid_len;
	int hidden_ssid;

	int channel;
	u16 interval;
	u64 timestamp;
	u16 capability;
#define IEEE80211_MAX_SUPP_RATES 8
	u8 supp_rates[IEEE80211_MAX_SUPP_RATES + 1];
	u8 rate;

	unsigned long last_rx;
	int last_rssi;
	u8 last_rate;
	unsigned long num_beacon, num_proberesp;
	int num_auth_fail, num_assoc_fail;
	int auth_state;
	u8 *challenge;
	u16 auth_alg;

	__le16 last_seq; /* last received seq number field (in 802.11 byte
		       * order) from the AP */

	struct ieee80211_bss *next;
};

struct ieee80211_chan_range {
	u8 min;
	u8 max;
};

struct ieee80211_wep_data {
	u32 iv;
	unsigned char len;
	u8 key[WEP_KEY_MAX_LEN];
};

#define IEEE80211_FRAG_CACHE_SIZE 4
struct ieee80211_frag_entry {
	unsigned long first_rx_time;
	struct sk_buff *skb;
};

struct ieee80211_data {
	struct net_device *dev;
	int (*set_channel)(struct net_device *dev, int channel);
	int (*set_bssid)(struct net_device *dev, u8 *bssid);
	int (*set_ssid)(struct net_device *dev, u8 *ssid, size_t ssid_len);
	int (*set_rate)(struct net_device *dev);
	int (*tx)(struct net_device *dev, struct sk_buff *skb,
		  __le16 fc, int short_preamble, u8 *dst);
	void (*link_change)(struct net_device *dev, int link);
	void (*set_interval)(struct net_device *dev);
	void (*set_tbtt)(struct net_device *dev, u16 tbtt);
	u64 (*get_tsft)(struct net_device *dev);

	enum {
		IEEE80211_SCAN, IEEE80211_AUTHENTICATE, IEEE80211_ASSOCIATE,
		IEEE80211_ASSOCIATED, IEEE80211_STOP
	} state;
	struct timer_list timer;

	enum {
		IEEE80211_MANAGED, IEEE80211_ADHOC, IEEE80211_MONITOR,
		IEEE80211_AP, IEEE80211_WDS
	} mode;

#define WEP_ENABLED (1<<0)
#define WEP_RESTRICTED (1<<1)
#define AUTO_RATE (1<<2)
#define ANY_SSID (1<<3)
#define HIDDEN_SSID (1<<4)
#define PREF_BSSID_SET (1<<5)
#define PREV_BSSID_SET (1<<6)
#define ASSOCIATED (1<<8)
#define LINK_ON (1<<9)

	u32 flags;
	
	char nick[IW_ESSID_MAX_SIZE];

	unsigned int auth_algorithm, default_key;
	struct ieee80211_wep_data wep_data[WEP_KEY_MAX_KEYS];
	struct crypto_tfm *tfm;

	struct ieee80211_frag_entry frag_cache[IEEE80211_FRAG_CACHE_SIZE];

	u8 bssid[ETH_ALEN];
	u8 rate;
	u8 pref_channel;
	u8 channel;
	struct ieee80211_chan_range chan_range;
	int num_supp_rates;
	u8 *supp_rates;

	u8 ssid[IEEE80211_MAX_SSID_LEN];
	size_t ssid_len;
	u8 pref_bssid[ETH_ALEN];

	int auth_tries;
	int assoc_tries;

	int aid;

	u8 prev_bssid[ETH_ALEN];

	u64 timestamp;
	u16 capab, beacon_interval, listen_interval;

	struct ieee80211_bss *bss;
	spinlock_t lock;
};

static const u8 IEEE80211_B_RATES[] = { 0x02, 0x04, 0x0b, 0x16, 0x00 };

struct ieee80211_bss * ieee80211_get_bss(struct ieee80211_data *data, u8 *addr);

int ieee80211_init(struct ieee80211_data *data);
void ieee80211_deinit(struct ieee80211_data *data);
void ieee80211_rx_mgmt(struct ieee80211_data *data, struct sk_buff *skb,
		       struct ieee80211_rx_status *rx_status);
void ieee80211_start(struct ieee80211_data *data);
void ieee80211_start_scan(struct ieee80211_data *data);
void ieee80211_stop(struct ieee80211_data *data);

int ieee80211_data_tx(struct ieee80211_data *data,
		      struct sk_buff *skb);

struct sk_buff * ieee80211_wep_decode(struct ieee80211_data *data,
				      struct sk_buff *skb);

struct sk_buff * ieee80211_data_encaps(struct ieee80211_data *data,
				       struct sk_buff *skb);
struct sk_buff * ieee80211_data_decaps(struct ieee80211_data *data,
				       struct sk_buff *skb);

struct sk_buff * ieee80211_reassemble_frag (struct ieee80211_data *data,
					    struct sk_buff *skb);
int ieee80211_filter_duplicates(struct ieee80211_data *data,
				struct sk_buff *skb, u16 fc);
u8 ieee80211_get_rate(struct ieee80211_data *data, u8 *addr);

#include "wep.h"

#endif /* IEEE80211_H */
