/*
 * IEEE 802.11 station / management functionality
 *
 * Copyright (c) 2003, Jouni Malinen <jkmaline@cc.hut.fi>
 * Copyright (c) 2004-2005, Michael Wu <flamingice@sourmilk.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/wireless.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/random.h>
#include <linux/crypto.h>
#include <asm/types.h>
#include <asm/delay.h>
#include <asm/scatterlist.h>
#include <asm/div64.h>

#include "ieee80211.h"

#define BEACON_SKIP 2

#define IEEE80211_SCAN_INTERVAL (5 * HZ)
#define IEEE80211_SCAN_LISTEN (HZ / 10)
#define IEEE80211_AUTH_TIMEOUT (1 * HZ)
#define IEEE80211_AUTH_MAX_TRIES 3
#define IEEE80211_ASSOC_TIMEOUT (1 * HZ)
#define IEEE80211_ASSOC_MAX_TRIES 3
#define IEEE80211_MONITORING_INTERVAL (30 * HZ)
#define IEEE80211_LINKCHECK_INTERVAL (3 * HZ)

static void ieee80211_timer(unsigned long ptr);
static void ieee80211_associate(struct ieee80211_data *data);

ParseRes ieee802_11_parse_elems(u8 *start, size_t len,
				struct ieee802_11_elems *elems)
{
	size_t left = len;
	u8 *pos = start;
	int unknown = 0;

	memset(elems, 0, sizeof(*elems));

	while (left >= 2) {
		u8 id, elen;

		id = *pos++;
		elen = *pos++;
		left -= 2;

		if (elen > left) {
#if 0
			if (net_ratelimit())
				printk(KERN_DEBUG "IEEE 802.11 element parse "
				       "failed (id=%d elen=%d left=%d)\n",
				       id, elen, left);
#endif
			return ParseFailed;
		}

		switch (id) {
		case WLAN_EID_SSID:
			elems->ssid = pos;
			elems->ssid_len = elen;
			break;
		case WLAN_EID_SUPP_RATES:
			elems->supp_rates = pos;
			elems->supp_rates_len = elen;
			break;
		case WLAN_EID_FH_PARAMS:
			elems->fh_params = pos;
			elems->fh_params_len = elen;
			break;
		case WLAN_EID_DS_PARAMS:
			elems->ds_params = pos;
			elems->ds_params_len = elen;
			break;
		case WLAN_EID_CF_PARAMS:
			elems->cf_params = pos;
			elems->cf_params_len = elen;
			break;
		case WLAN_EID_TIM:
			elems->tim = pos;
			elems->tim_len = elen;
			break;
		case WLAN_EID_IBSS_PARAMS:
			elems->ibss_params = pos;
			elems->ibss_params_len = elen;
			break;
		case WLAN_EID_CHALLENGE:
			elems->challenge = pos;
			elems->challenge_len = elen;
			break;
		default:
#if 0
			printk(KERN_DEBUG "IEEE 802.11 element parse ignored "
				      "unknown element (id=%d elen=%d)\n",
				      id, elen);
#endif
			unknown++;
			break;
		}

		left -= elen;
		pos += elen;
	}

	if (left)
		return ParseFailed;

	return unknown ? ParseUnknown : ParseOK;
}


static void ieee80211_dump_ssid(u8 *ssid, size_t ssid_len)
{
	int i;

	for (i = 0; i < ssid_len; i++)
		printk((ssid[i] >= 32 && ssid[i] < 127) ?
		       "%c" : "<%02x>", ssid[i]);
}

static char * ieee80211_status_code(int status)
{
	switch (status) {
	case WLAN_STATUS_SUCCESS:
		return "Success";
	case WLAN_STATUS_UNSPECIFIED_FAILURE:
		return "Unspecified failure";
	case WLAN_STATUS_CAPS_UNSUPPORTED:
		return "Capabilities unsupported";
	case WLAN_STATUS_REASSOC_NO_ASSOC:
		return "Reassociation failure; no association";
	case WLAN_STATUS_ASSOC_DENIED_UNSPEC:
		return "Association denied";
	case WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG:
		return "Authentication algorithm not supported";
	case WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION:
		return "Unknown authentication transaction";
	case WLAN_STATUS_CHALLENGE_FAIL:
		return "Challenge failed";
	case WLAN_STATUS_AUTH_TIMEOUT:
		return "Authentication timeout";
	case WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA:
		return "AP unable to handle new STA";
	case WLAN_STATUS_ASSOC_DENIED_RATES:
		return "Association denied: Rates not supported";

	/* 802.11b */
	case WLAN_STATUS_ASSOC_DENIED_NOSHORT:
		return "Association denied: Short preamble not supported";
	case WLAN_STATUS_ASSOC_DENIED_NOPBCC:
		return "Association denied: PBCC not supported";
	case WLAN_STATUS_ASSOC_DENIED_NOAGILITY:
		return "Association denied: Agility not supported";
	default:
		return "Unknown status code";
	}
}


static void ieee80211_set_associated(struct ieee80211_data *data, int assoc)
{
	if ((data->flags & ASSOCIATED) == (assoc ? ASSOCIATED : 0))
		return;

	if (assoc)
		data->flags |= ASSOCIATED;
	else
		data->flags &= ~ASSOCIATED;

	if (data->link_change)
		data->link_change(data->dev, assoc);
}


struct ieee80211_bss * ieee80211_get_bss(struct ieee80211_data *data, u8 *addr)
{
	struct ieee80211_bss *bss;

	bss = data->bss;
	while (bss) {
		if (memcmp(bss->bssid, addr, ETH_ALEN) == 0)
			return bss;
		bss = bss->next;
	}
	return NULL;
}


static int ieee80211_bss_score(struct ieee80211_data *data,
			       struct ieee80211_bss *bss)
{
	int score, i;

	score = bss->last_rssi;
	score -= 50 * bss->num_auth_fail;
	score -= 50 * bss->num_assoc_fail;

	/* detect hidden SSIDs and avoid them */
	bss->hidden_ssid = 1;
	for (i = 0; i < bss->ssid_len; i += 1)
		if (bss->ssid[i] != '\0') {
			bss->hidden_ssid = 0;
			break;
		}
	
	if (bss->hidden_ssid)
		score -= 50;

	if (data->pref_channel == bss->channel)
		score += 100;

	return score;
}

/* removes all bss */
static void ieee80211_free_bss(struct ieee80211_data *data)
{
	struct ieee80211_bss *bss, *prev;
	unsigned long flags;

	/* need irqsave as opposed to bh so this can be run during suspend */
	spin_lock_irqsave(&data->lock, flags);

	bss = data->bss;
	data->bss = NULL;
	while (bss) {
		prev = bss;
		bss = bss->next;
		kfree(prev);
	}

	spin_unlock_irqrestore(&data->lock, flags);
}

static int ieee80211_select_bssid(struct ieee80211_data *data)
{
	struct ieee80211_bss *bss, *prev = NULL, *selected = NULL;
	int best_score = 0, score;
	u64 oldest_timestamp = 0;

	spin_lock_bh(&data->lock);
	bss = data->bss;

	/* Select first BSS that matches with current configuration */
	while (bss) {
		score = ieee80211_bss_score(data, bss);
		if (score < -250)
			goto skip;

		if (data->mode == IEEE80211_MANAGED && 
		    ((bss->capability & WLAN_CAPABILITY_IBSS) || 
		    !(bss->capability & WLAN_CAPABILITY_ESS) ))
			goto skip;
		if (data->mode == IEEE80211_ADHOC &&
		    ((bss->capability & WLAN_CAPABILITY_ESS) ||
		    !(bss->capability & WLAN_CAPABILITY_IBSS) ))
			goto skip;

		printk(KERN_DEBUG "%s: CHAN=%d BSSID=" MACSTR " SSID=",
		       data->dev->name, bss->channel, MAC2STR(bss->bssid));
		ieee80211_dump_ssid(bss->ssid, bss->ssid_len);
		printk(" RSSI=%d num=%lu/%lu score=%d\n", bss->last_rssi,
		       bss->num_beacon, bss->num_proberesp, score);

		if (data->flags & PREF_BSSID_SET &&
		    memcmp(bss->bssid, data->pref_bssid, ETH_ALEN) == 0) {
			selected = bss;
			break;
		} else if (data->flags & ANY_SSID || bss->ssid_len == 0 ||
			  (bss->ssid_len == data->ssid_len && 
			  (bss->hidden_ssid || !memcmp(bss->ssid, data->ssid, data->ssid_len)))) {
			if (data->mode == IEEE80211_MANAGED &&
			    (!selected || score > best_score) ) {
				best_score = score;
				selected = bss;
			} else if (data->mode == IEEE80211_ADHOC &&
				   bss->timestamp > oldest_timestamp) {
				oldest_timestamp = bss->timestamp;
				selected = bss;
			}
		}

		skip:
		prev = bss;
		bss = bss->next;
	}

	if (selected) {
		if (memcmp(data->bssid, selected->bssid, ETH_ALEN) != 0) {
			printk(KERN_DEBUG "%s: new BSSID " MACSTR "\n",
			       data->dev->name, MAC2STR(selected->bssid));
			ieee80211_set_associated(data, 0);
		}
		memcpy(data->bssid, selected->bssid, ETH_ALEN);
		if (selected->hidden_ssid)
			data->flags |= HIDDEN_SSID;
		else {
			data->flags &= ~HIDDEN_SSID;
			data->ssid_len = selected->ssid_len;
			memcpy(data->ssid, selected->ssid, selected->ssid_len);
		}
		data->channel = selected->channel;
		data->beacon_interval = (selected->interval ? selected->interval : 100);
		data->timestamp = selected->timestamp;
	}

	spin_unlock_bh(&data->lock);

	return selected ? 0 : -1;
}

static void ieee80211_sync_ibss(struct ieee80211_data *data)
{
	u64 tbttpre;
	u32 beacon = data->beacon_interval << 10;

	tbttpre  = data->timestamp;
	tbttpre  = data->timestamp - do_div(tbttpre, beacon);
	tbttpre += (beacon*BEACON_SKIP) << 10;			/* skip a few beacons in case we're slow */
	tbttpre -= 20 << 10;				/* wake up early */

	tbttpre >>= 10;					/* convert everything to TUs */

	data->set_tbtt(data->dev, (u16)(tbttpre & 0xFFFF));
}

static void ieee80211_send_auth(struct ieee80211_data *data, struct ieee80211_bss *bss)
{
	struct sk_buff *skb;
	union ieee80211_mgmt_payload *mgmt;
	u8 *pos;
	__le16 fc;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(6 + 2 + WLAN_AUTH_CHALLENGE_LEN + 8);
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for auth "
		       "frame\n", data->dev->name);
		return;
	}

	mgmt = (union ieee80211_mgmt_payload *) skb_put(skb, 6);
	fc = IEEE80211_FC(WLAN_FC_TYPE_MGMT,
			  WLAN_FC_STYPE_AUTH);

	bss->auth_state+=1;
	if (bss->auth_alg == WLAN_AUTH_OPEN) {
		mgmt->auth.auth_alg = cpu_to_le16(WLAN_AUTH_OPEN);
		mgmt->auth.auth_transaction = cpu_to_le16(bss->auth_state);
		mgmt->auth.status_code = 0;

		if (bss->auth_state != 1) {
			printk(KERN_DEBUG "%s: Unable to handle authentication transaction: %d\n",
			       data->dev->name, bss->auth_state);
			goto fail;
		}
	} else if (bss->auth_alg == WLAN_AUTH_SHARED_KEY) {
		mgmt->auth.auth_alg = cpu_to_le16(WLAN_AUTH_SHARED_KEY);
		mgmt->auth.auth_transaction = cpu_to_le16(bss->auth_state);
		mgmt->auth.status_code = 0;

		switch (bss->auth_state) {
		case 1:
			break;
		case 3:
			pos = skb_put(skb, 2 + WLAN_AUTH_CHALLENGE_LEN);
			*pos++ = WLAN_EID_CHALLENGE;
			*pos++ = WLAN_AUTH_CHALLENGE_LEN;
			if (bss->challenge) {
				memcpy(pos, bss->challenge, WLAN_AUTH_CHALLENGE_LEN);
			} else {
				printk(KERN_DEBUG "%s: No challenge to return\n", data->dev->name);
				goto fail;
			}

			fc |= cpu_to_le16(WLAN_FC_ISWEP);
			break;
		default:
			printk(KERN_DEBUG "%s: Unable to handle authentication transaction: %d\n",
			       data->dev->name, bss->auth_state);
			goto fail;
		}
	} else {
		printk(KERN_DEBUG "%s: Unknown authentication algorithm: %d\n", data->dev->name, bss->auth_alg);
		goto fail;
	}

	printk(KERN_DEBUG "%s: TX authentication (alg=%d transaction=%d)\n",
	       data->dev->name, bss->auth_alg, bss->auth_state);

	data->tx(data->dev, skb, fc, bss->capability & WLAN_CAPABILITY_SHORT_PREAMBLE, bss->bssid);
	return;

fail:
	dev_kfree_skb(skb);
}

static void ieee80211_authenticate(struct ieee80211_data *data)
{
	struct ieee80211_bss *bss;

	data->auth_tries++;
	if (data->auth_tries > IEEE80211_AUTH_MAX_TRIES) {
		printk(KERN_DEBUG "%s: authentication with AP " MACSTR
		       " CHAN=%d timed out\n",
		       data->dev->name, MAC2STR(data->bssid), data->channel);

		spin_lock_bh(&data->lock);
		bss = ieee80211_get_bss(data, data->bssid);
		if (bss)
			bss->num_auth_fail++;
		spin_unlock_bh(&data->lock);
		ieee80211_start_scan(data);
		return;
	}

	if (data->set_channel)
		data->set_channel(data->dev, data->channel);
	if (data->set_bssid)
		data->set_bssid(data->dev, data->bssid);
	if (data->set_rate)
		data->set_rate(data->dev);
	if (data->set_interval)
		data->set_interval(data->dev);

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, data->bssid);
	if (bss->auth_state != 5) {
		data->state = IEEE80211_AUTHENTICATE;
		printk(KERN_DEBUG "%s: authenticate with AP " MACSTR " CHAN=%d\n",
		       data->dev->name, MAC2STR(data->bssid), data->channel);

		ieee80211_set_associated(data, 0);
		bss->auth_state = 0;
		ieee80211_send_auth(data, bss);
	}
	spin_unlock_bh(&data->lock);

	if (data->state != IEEE80211_AUTHENTICATE)
		ieee80211_associate(data);

	mod_timer(&data->timer, jiffies + IEEE80211_AUTH_TIMEOUT);
}

static void ieee80211_send_assoc_req(struct ieee80211_data *data)
{
	struct sk_buff *skb;
	union ieee80211_mgmt_payload *mgmt;
	struct ieee80211_bss *bss;
	u8 *pos;
	__le16 fc;
	int i = 0;
	int reassoc = 0;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(2 + 2 + 2 + data->ssid_len + 2 + data->num_supp_rates);
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for assoc req "
		       "frame\n", data->dev->name);
		return;
	}

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, data->bssid);

	/* TODO: check how prev_bssid_set is suppose to work */
	if (data->flags & PREV_BSSID_SET) {
		mgmt = (union ieee80211_mgmt_payload *) skb_put(skb, 10);
		fc = IEEE80211_FC(WLAN_FC_TYPE_MGMT,
				  WLAN_FC_STYPE_REASSOC_REQ);
		mgmt->reassoc_req.capab_info = cpu_to_le16(bss->capability);
		mgmt->reassoc_req.listen_interval =
			cpu_to_le16(data->listen_interval);
		memcpy(mgmt->reassoc_req.current_ap, data->prev_bssid,
		       ETH_ALEN);
		reassoc = 1;
	} else {
		mgmt = (union ieee80211_mgmt_payload *) skb_put(skb, 4);
		fc = IEEE80211_FC(WLAN_FC_TYPE_MGMT,
				  WLAN_FC_STYPE_ASSOC_REQ);
		mgmt->assoc_req.capab_info = cpu_to_le16(bss->capability);
		mgmt->assoc_req.listen_interval =
			cpu_to_le16(data->listen_interval);
	}

	/* SSID */
	pos = skb_put(skb, 2 + data->ssid_len);
	*pos++ = WLAN_EID_SSID;
	*pos++ = data->ssid_len;
	memcpy(pos, data->ssid, data->ssid_len);

	pos = skb_put(skb, 2 + data->num_supp_rates);
	/* All supported rates */
	*pos++ = WLAN_EID_SUPP_RATES;
	*pos++ = data->num_supp_rates;
	while (i++ < data->num_supp_rates)
		*pos++ = data->supp_rates[i-1];

	printk(KERN_DEBUG "%s: TX %sssocReq (capab=0x%x)\n",
	       data->dev->name, reassoc ? "Rea" : "A", bss->capability);

	data->tx(data->dev, skb, fc, bss->capability & WLAN_CAPABILITY_SHORT_PREAMBLE, data->bssid);
	spin_unlock_bh(&data->lock);
}

static void ieee80211_associated(struct ieee80211_data *data)
{
	struct ieee80211_bss *bss, *tmp, *prev = NULL;

	/* TODO: start monitoring current AP signal quality and number of
	 * missed beacons. Scan other channels every now and then and search
	 * for better APs. */

	spin_lock_bh(&data->lock);

	/* remove expired BSSes */
	bss = data->bss;
	while (bss) {
		if (time_after(jiffies, bss->last_rx + IEEE80211_MONITORING_INTERVAL) &&
		     memcmp(data->bssid, bss->bssid, ETH_ALEN)) {
			if (prev)
				prev->next = bss->next;
			else
				data->bss = bss->next;

			tmp = bss;
			bss = bss->next;
			kfree(tmp);
		} else {
			prev = bss;
			bss = bss->next;
		}
	}

	spin_unlock_bh(&data->lock);

	if (data->state != IEEE80211_ASSOCIATED) {
		data->state = IEEE80211_ASSOCIATED;
		netif_wake_queue(data->dev);
		mod_timer(&data->timer, jiffies + IEEE80211_LINKCHECK_INTERVAL);
	} else {
		if (data->flags & LINK_ON || data->mode == IEEE80211_MANAGED)
			mod_timer(&data->timer, jiffies + IEEE80211_MONITORING_INTERVAL);
		else
			ieee80211_start_scan(data);
	}
}

static void ieee80211_associate(struct ieee80211_data *data)
{
	if (data->flags & ASSOCIATED) {
		ieee80211_associated(data);
		return;
	}

	data->assoc_tries++;
	if (data->assoc_tries > IEEE80211_ASSOC_MAX_TRIES) {
		struct ieee80211_bss *bss;
		printk(KERN_DEBUG "%s: association with AP " MACSTR
		       " CHAN=%d timed out\n",
		       data->dev->name, MAC2STR(data->bssid), data->channel);
		spin_lock_bh(&data->lock);
		bss = ieee80211_get_bss(data, data->bssid);
		if (bss) {
			bss->num_assoc_fail++;
			if (bss->num_assoc_fail > 6) {
				spin_unlock_bh(&data->lock);
				ieee80211_free_bss(data);
				/* yuck */
				spin_lock_bh(&data->lock);
			}
		}
		spin_unlock_bh(&data->lock);
		ieee80211_start_scan(data);
		return;
	}

	data->state = IEEE80211_ASSOCIATE;
	printk(KERN_DEBUG "%s: associate with AP " MACSTR "\n",
	       data->dev->name, MAC2STR(data->bssid));

	ieee80211_send_assoc_req(data);

	mod_timer(&data->timer, jiffies + IEEE80211_ASSOC_TIMEOUT);
}

static void ieee80211_send_probe_req(struct ieee80211_data *data)
{
	struct sk_buff *skb;
	union ieee80211_mgmt_payload *mgmt;
	u8 *pos;
	__le16 fc;
	u8 dst[ETH_ALEN];
	int i = 0;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(2 + 2 + data->num_supp_rates);
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for probe "
		       "request\n", data->dev->name);
		return;
	}

	mgmt = (union ieee80211_mgmt_payload *) skb_put(skb, 2 + 2 + data->num_supp_rates);
	memset(mgmt, 0, 2 + 2 + data->num_supp_rates);
	fc = IEEE80211_FC(WLAN_FC_TYPE_MGMT,
			  WLAN_FC_STYPE_PROBE_REQ);
	memset(dst, ~0, ETH_ALEN);
	pos = mgmt->probe_req.variable;

	/* Broadcast SSID */
	*pos++ = WLAN_EID_SSID;
	*pos++ = 0;

	/* All supported rates */
	*pos++ = WLAN_EID_SUPP_RATES;
	*pos++ = data->num_supp_rates;
	while (i++ < data->num_supp_rates)
		*pos++ = data->supp_rates[i-1];

	spin_lock_bh(&data->lock);
	data->tx(data->dev, skb, fc, 0, dst);
	spin_unlock_bh(&data->lock);
}


static void ieee80211_scan(struct ieee80211_data *data)
{
	data->state = IEEE80211_SCAN;
	if (data->channel == 0) {
		printk(KERN_DEBUG "%s: scanning..\n", data->dev->name);
		data->channel = data->chan_range.min;
	} else
		data->channel++;
	
	if (data->channel > data->chan_range.max) {
	if (data->mode == IEEE80211_MANAGED) {
		if (ieee80211_select_bssid(data) == 0) {
			ieee80211_authenticate(data);
			return;
		} else {
			ieee80211_free_bss(data);

			data->channel = 0;
			mod_timer(&data->timer,
				  jiffies + IEEE80211_SCAN_INTERVAL);
			return;
		}
	} else if (data->mode == IEEE80211_ADHOC) {
		if (ieee80211_select_bssid(data)) {
			random_ether_addr(data->bssid);
			data->beacon_interval = 100;
			if (data->pref_channel)
				data->channel = data->pref_channel;
			else
				data->channel = data->chan_range.min;

			if (!data->ssid) {
				printk(KERN_WARNING "%s: No SSID set. SSID set to \"default\"\n", data->dev->name);
				memcpy(data->ssid, "default", 7);
				data->ssid_len = 7;
			}
			printk(KERN_DEBUG "%s: No matching adhoc sta found. Creating IBSS " MACSTR " CHAN=%d\n",
			       data->dev->name, MAC2STR(data->bssid), data->channel);
			data->timestamp = data->get_tsft(data->dev);
		} else {
			printk(KERN_DEBUG "%s: joining IBSS " MACSTR " CHAN=%d\n",
			       data->dev->name, MAC2STR(data->bssid), data->channel);
		}

		if (data->set_rate)
			data->set_rate(data->dev);
		if (data->set_interval)
			data->set_interval(data->dev);
		if (data->set_bssid)
			data->set_bssid(data->dev, data->bssid);
		if (data->set_ssid)
			data->set_ssid(data->dev, data->ssid, data->ssid_len);
		if (data->set_channel)
			data->set_channel(data->dev, data->channel);
		ieee80211_sync_ibss(data);

		data->flags |= ASSOCIATED;
		ieee80211_associated(data);
		return;
	}
	}

	if (data->set_channel)
		data->set_channel(data->dev, data->channel);

	ieee80211_send_probe_req(data);

	mod_timer(&data->timer, jiffies + IEEE80211_SCAN_LISTEN);
}


void ieee80211_start_scan(struct ieee80211_data *data)
{
	static const u8 scan_bssid[] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

	/* don't do anything if we're already scanning */
	if (data->state == IEEE80211_SCAN)
		return;

	data->channel = 0;
	data->auth_tries = data->assoc_tries = 0;

	/* Set BSSID to ff:ff:ff:ff:ff:ff for active scanning */
	if (data->set_bssid)
		data->set_bssid(data->dev, (u8 *)scan_bssid);

	data->beacon_interval = 0;
	if (data->set_interval)
		data->set_interval(data->dev);

	if (in_interrupt()) {
		data->state = IEEE80211_SCAN;
		mod_timer(&data->timer, jiffies + IEEE80211_SCAN_LISTEN);
	} else {
		ieee80211_scan(data);
	}
}

void ieee80211_start(struct ieee80211_data *data)
{
	if (data->mode == IEEE80211_MANAGED || data->mode == IEEE80211_ADHOC)
		ieee80211_start_scan(data);
}

int ieee80211_init(struct ieee80211_data *data)
{
	init_timer(&data->timer);
	data->timer.data = (unsigned long) data;
	data->timer.function = ieee80211_timer;

	spin_lock_init(&data->lock);

	data->listen_interval = 10;
	data->mode = IEEE80211_MANAGED;
	data->state = IEEE80211_STOP;
	data->flags = AUTO_RATE | ANY_SSID;

	if (!data->supp_rates)
		data->supp_rates = (u8 *)IEEE80211_B_RATES;

	data->num_supp_rates=strlen(data->supp_rates);
	data->rate = data->supp_rates[data->num_supp_rates-1];

	data->tfm = crypto_alloc_tfm("arc4", 0);
	if (!data->tfm)
		return -1;

	get_random_bytes(&data->wep_data[0].iv, 4);
	get_random_bytes(&data->wep_data[1].iv, 4);
	get_random_bytes(&data->wep_data[2].iv, 4);
	get_random_bytes(&data->wep_data[3].iv, 4);

	return 0;
}

void ieee80211_deinit(struct ieee80211_data *data)
{
	crypto_free_tfm(data->tfm);

	return;
}

void ieee80211_stop(struct ieee80211_data *data)
{
	int i;

	del_timer_sync(&data->timer);
	data->state = IEEE80211_STOP;

	ieee80211_free_bss(data);
	for (i = 0; i < IEEE80211_FRAG_CACHE_SIZE; i+=1) {
		if (data->frag_cache[i].skb) {
			dev_kfree_skb(data->frag_cache[i].skb);
			data->frag_cache[i].skb = NULL;
		}
	}

	memset(data->bssid, 0, ETH_ALEN);
	if (data->set_ssid)
		data->set_ssid(data->dev, data->ssid, 0);
	if (data->set_bssid)
		data->set_bssid(data->dev, data->bssid);
}

/* TODO: merge with the only function that calls this, or reduce the number of arguments */
static void ieee80211_add_bss(struct ieee80211_data *data,
			      u8 *bssid, struct ieee802_11_elems *elems,
			      int channel, struct ieee80211_mgmt *mgmt,
			      int rssi, u8 rate, int beacon)
{
	struct ieee80211_bss *bss;
	int i;

	spin_lock_bh(&data->lock);

	bss = data->bss;

	while (bss != NULL &&
	       (memcmp(bss->bssid, bssid, ETH_ALEN) != 0 /* ||
		bss->ssid_len != elems->ssid_len ||
		memcmp(bss->ssid, elems->ssid, elems->ssid_len) != 0 */ ))
		bss = bss->next;

	if (bss == NULL) {
/*		printk(KERN_DEBUG "%s: new BSS - CHAN=%d BSSID=" MACSTR
		       " SSID=",
		       data->dev->name, channel, MAC2STR(bssid));
		ieee80211_dump_ssid(ssid, ssid_len);
		printk("\n");*/

		bss = (struct ieee80211_bss *)
			kmalloc(sizeof(*bss), GFP_ATOMIC);
		if (bss == NULL) {
			spin_unlock_bh(&data->lock);
			printk(KERN_DEBUG "%s: failed to allocate BSS info "
			       "structure\n", data->dev->name);
			return;
		}
		memset(bss, 0, sizeof(*bss));
		memcpy(bss->bssid, bssid, ETH_ALEN);
		bss->last_seq = __constant_cpu_to_le16(~0);
		bss->next = data->bss;
		data->bss = bss;

		if (mgmt->u.beacon.capab_info & __cpu_to_le16(WLAN_CAPABILITY_PRIVACY))
			bss->auth_alg = WLAN_AUTH_SHARED_KEY;
		else
			bss->auth_alg = WLAN_AUTH_OPEN;

		if (data->state == IEEE80211_ADHOC &&
		    data->timestamp < bss->timestamp &&
		    data->get_tsft(data->dev) < bss->timestamp) {
			printk(KERN_DEBUG "%s: found older BSS - BSSID=" MACSTR "\n",
					data->dev->name, MAC2STR(bssid));
		}

	}

	if (beacon)
		bss->num_beacon++;
	else
		bss->num_proberesp++;

	memcpy(bss->ssid, elems->ssid, elems->ssid_len);
	bss->ssid_len = elems->ssid_len;
	if (elems->supp_rates) {
		memset(bss->supp_rates, 0, IEEE80211_MAX_SUPP_RATES);
		memcpy(bss->supp_rates, elems->supp_rates,
		       elems->supp_rates_len > IEEE80211_MAX_SUPP_RATES ?
		       IEEE80211_MAX_SUPP_RATES : elems->supp_rates_len);
		for (i = data->num_supp_rates; i > 0; i-=1) {
			if (strchr(bss->supp_rates, data->supp_rates[i-1])) {
				bss->rate = data->supp_rates[i-1];
				break;
			}
		}
	}
	if (!bss->rate)
		bss->rate = data->supp_rates[data->num_supp_rates-1];
	bss->channel = channel;
	bss->interval = le16_to_cpu(mgmt->u.beacon.beacon_int);
	bss->timestamp = le64_to_cpu(mgmt->u.beacon.timestamp);
	bss->capability = le16_to_cpu(mgmt->u.beacon.capab_info);
	bss->last_rx = jiffies;
	if (rssi)
		bss->last_rssi = rssi;
	bss->last_rate = rate;

	spin_unlock_bh(&data->lock);
}


/* Process beacon and probe response frames in common function since they have
 * almost identical contents. */
static void ieee80211_rx_mgmt_bss_info(struct ieee80211_data *data,
				       struct ieee80211_mgmt *mgmt,
				       size_t len,
				       struct ieee80211_rx_status *rx_status,
				       int beacon)
{
	struct ieee802_11_elems elems;
	size_t baselen;
	int channel;

#if 0
	printk(KERN_DEBUG "%s: %s from " MACSTR " rate=%d rssi=%d\n",
	       data->dev->name, beacon ? "Beacon" : "ProbeResp",
	       MAC2STR(mgmt->sa), rx_status->rate, rx_status->rssi);
#endif

	baselen = (u8 *) mgmt->u.beacon.variable - (u8 *) mgmt;
	if (baselen > len) {
		printk(KERN_DEBUG "%s: Beacon/ProbeResp underflow %d > %d\n",
		       data->dev->name, baselen, len);
		return;
	}

	ieee802_11_parse_elems(mgmt->u.beacon.variable, len - baselen, &elems);
/*	if (ieee802_11_parse_elems(mgmt->u.beacon.variable, len - baselen,
				   &elems) == ParseFailed) {
		printk(KERN_DEBUG "%s: Beacon/ProbeResp invalid\n",
		       data->dev->name);
		return;
	}*/

/*	if (elems.ssid == NULL)
		return;*/

	if (elems.ds_params && elems.ds_params_len == 1)
		channel = elems.ds_params[0];
	else
		channel = data->channel;

	ieee80211_add_bss(data, mgmt->bssid, &elems, channel,
			  mgmt, rx_status->rssi, rx_status->rate, beacon);
}

static void ieee80211_rx_mgmt_probe_resp(struct ieee80211_data *data,
					 struct ieee80211_mgmt *mgmt,
					 size_t len,
					 struct ieee80211_rx_status *rx_status)
{
	ieee80211_rx_mgmt_bss_info(data, mgmt, len, rx_status, 0);
}


static void ieee80211_rx_mgmt_beacon(struct ieee80211_data *data,
				     struct ieee80211_mgmt *mgmt,
				     size_t len,
				     struct ieee80211_rx_status *rx_status)
{
	ieee80211_rx_mgmt_bss_info(data, mgmt, len, rx_status, 1);
}

static void ieee80211_rx_mgmt_auth(struct ieee80211_data *data,
				   struct ieee80211_mgmt *mgmt,
				   size_t len,
				   struct ieee80211_rx_status *rx_status, struct sk_buff *skb)
{
	u16 auth_alg, auth_transaction, status_code;
	struct ieee802_11_elems elems;
	struct ieee80211_bss *bss;
	size_t baselen;

	if (data->state != IEEE80211_AUTHENTICATE) {
		printk(KERN_DEBUG "%s: authentication frame received from "
		       MACSTR ", but not in authenticate state - ignored\n",
		       data->dev->name, MAC2STR(mgmt->sa));
		return;
	}

	if (len < 24 + 6) {
		printk(KERN_DEBUG "%s: too short (%d) authentication frame "
		       "received from " MACSTR " - ignored\n",
		       data->dev->name, len, MAC2STR(mgmt->sa));
		return;
	}

	if (memcmp(data->bssid, mgmt->sa, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: authentication frame received from "
		       "unknown AP (SA=" MACSTR " BSSID=" MACSTR ") - "
		       "ignored\n", data->dev->name, MAC2STR(mgmt->sa),
		       MAC2STR(mgmt->bssid));
		return;
	}

	baselen = (u8 *) mgmt->u.auth.variable - (u8 *) mgmt;
	if (baselen > len) {
		printk(KERN_DEBUG "%s: Auth underflow %d > %d\n",
		       data->dev->name, baselen, len);
		return;
	}

	if (ieee802_11_parse_elems(mgmt->u.auth.variable, len - baselen,
				   &elems) == ParseFailed) {
		printk(KERN_DEBUG "%s: Parse failed.\n", data->dev->name);
		return;
	}

	auth_alg = le16_to_cpu(mgmt->u.auth.auth_alg);
	auth_transaction = le16_to_cpu(mgmt->u.auth.auth_transaction);
	status_code = le16_to_cpu(mgmt->u.auth.status_code);

	printk(KERN_DEBUG "%s: RX authentication (alg=%d "
	       "transaction=%d status=%d %s)\n",
	       data->dev->name, auth_alg,
	       auth_transaction, status_code,
	       ieee80211_status_code(status_code));

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, mgmt->sa);
	if (!bss) {
		spin_unlock_bh(&data->lock);
		return;
	}

	if (auth_transaction != bss->auth_state + 1) {
		printk(KERN_DEBUG "%s: unexpected authentication "
		       "transaction number. expected %d, got %d\n",
		       data->dev->name, bss->auth_state + 1, auth_transaction);
		bss->auth_state = 0;
		spin_unlock_bh(&data->lock);
		return;
	}
	bss->auth_alg = auth_alg;
	bss->auth_state = auth_transaction;

	if ((auth_transaction == 0x2 || auth_transaction == 0x4)
	    && status_code != WLAN_STATUS_SUCCESS) {
		bss->num_auth_fail++;
		bss->auth_state = 0;
		if (auth_transaction == 0x2) {
			bss->auth_alg = WLAN_AUTH_OPEN;
			printk(KERN_DEBUG "%s: Falling back on open authentication...\n", data->dev->name);
		}
		spin_unlock_bh(&data->lock);
		return;
	}

	/* TODO: necessary? */
	if (data->mode == IEEE80211_MANAGED && auth_transaction == 1) {
		if (status_code == WLAN_STATUS_SUCCESS)
			printk(KERN_DEBUG "%s: Ignoring attempt to initiate authentication\n",
			        data->dev->name);
		return;
	}

	switch (auth_alg) {
	case WLAN_AUTH_OPEN:
		if (auth_transaction == 2)
			bss->auth_state = 5;
		else
			printk(KERN_DEBUG "%s: invalid authentication "
			       "transaction number: %d\n",
			       data->dev->name, auth_transaction);
		break;
	case WLAN_AUTH_SHARED_KEY:
		switch (auth_transaction) {
		case 2:
			if (elems.challenge_len != WLAN_AUTH_CHALLENGE_LEN) {
				printk(KERN_DEBUG "%s: invalid challenge length: %d\n",
				       data->dev->name, elems.challenge_len);
				return;
			}
			bss->challenge = elems.challenge;
			ieee80211_send_auth(data, bss);
			break;
		case 4:
			bss->auth_state = 5;
			break;
		default:
			printk(KERN_DEBUG "%s: invalid authentication"
			       "transaction number: %d\n",
			       data->dev->name, auth_transaction);
			break;
		}
		break;
	}

	if (bss->auth_state == 5) {
		data->auth_tries = 0;
		printk(KERN_DEBUG "%s: authenticated\n", data->dev->name);

		spin_unlock_bh(&data->lock);
		ieee80211_associate(data);
	} else
		spin_unlock_bh(&data->lock);
}

static void ieee80211_rx_mgmt_deauth(struct ieee80211_data *data,
				     struct ieee80211_mgmt *mgmt,
				     size_t len,
				     struct ieee80211_rx_status *rx_status)
{
	struct ieee80211_bss *bss;
	u16 reason_code;

	if (len < 24 + 2) {
		printk(KERN_DEBUG "%s: too short (%d) deauthentication frame "
		       "received from " MACSTR " - ignored\n",
		       data->dev->name, len, MAC2STR(mgmt->sa));
		return;
	}

	if (memcmp(data->bssid, mgmt->sa, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: deauthentication frame received from "
		       "unknown AP (SA=" MACSTR " BSSID=" MACSTR ") - "
		       "ignored\n", data->dev->name, MAC2STR(mgmt->sa),
		       MAC2STR(mgmt->bssid));
		return;
	}

	reason_code = le16_to_cpu(mgmt->u.deauth.reason_code);

	printk(KERN_DEBUG "%s: RX deauthentication from " MACSTR
	       " (reason=%d)\n",
	       data->dev->name, MAC2STR(mgmt->sa), reason_code);

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, mgmt->sa);
	if (bss) {
		bss->num_auth_fail++;
		bss->auth_state = 0;
		if (bss->auth_state == 5)
			printk(KERN_DEBUG "%s: deauthenticated\n", data->dev->name);
	}

	spin_unlock_bh(&data->lock);
	ieee80211_set_associated(data, 0);

	ieee80211_authenticate(data);
}


static void ieee80211_rx_mgmt_disassoc(struct ieee80211_data *data,
				       struct ieee80211_mgmt *mgmt,
				       size_t len,
				       struct ieee80211_rx_status *rx_status)
{
	struct ieee80211_bss *bss;
	u16 reason_code;

	if (len < 24 + 2) {
		printk(KERN_DEBUG "%s: too short (%d) disassociation frame "
		       "received from " MACSTR " - ignored\n",
		       data->dev->name, len, MAC2STR(mgmt->sa));
		return;
	}

	if (memcmp(data->bssid, mgmt->sa, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: disassociation frame received from "
		       "unknown AP (SA=" MACSTR " BSSID=" MACSTR ") - "
		       "ignored\n", data->dev->name, MAC2STR(mgmt->sa),
		       MAC2STR(mgmt->bssid));
		return;
	}

	reason_code = le16_to_cpu(mgmt->u.disassoc.reason_code);

	printk(KERN_DEBUG "%s: RX disassociation from " MACSTR
	       " (reason=%d)\n",
	       data->dev->name, MAC2STR(mgmt->sa), reason_code);

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, mgmt->sa);
	if (bss)
		bss->num_assoc_fail++;
	spin_unlock_bh(&data->lock);

	if (data->flags & ASSOCIATED)
		printk(KERN_DEBUG "%s: disassociated\n", data->dev->name);
	ieee80211_set_associated(data, 0);

	ieee80211_associate(data);
}

static void ieee80211_rx_mgmt_assoc_req(struct ieee80211_data *data,
					struct ieee80211_mgmt *mgmt,
					size_t len,
					struct ieee80211_rx_status *rx_status,
					int reassoc)
{
	if (reassoc && data->mode == IEEE80211_MANAGED)
		ieee80211_associate(data);

	return;
}

static void ieee80211_rx_mgmt_assoc_resp(struct ieee80211_data *data,
					 struct ieee80211_mgmt *mgmt,
					 size_t len,
					 struct ieee80211_rx_status *rx_status,
					 int reassoc)
{
	static const char hidden_ssid[IEEE80211_MAX_SSID_LEN] = {0};
	struct ieee802_11_elems elems;
	struct ieee80211_bss *bss;
	size_t baselen;
	u16 capab_info, status_code, aid;

	/* AssocResp and ReassocResp have identical structure, so process both
	 * of them in this function. */

	if (data->state != IEEE80211_ASSOCIATE) {
		printk(KERN_DEBUG "%s: association frame received from "
		       MACSTR ", but not in associate state - ignored\n",
		       data->dev->name, MAC2STR(mgmt->sa));
		return;
	}

	if (len < 24 + 6) {
		printk(KERN_DEBUG "%s: too short (%d) association frame "
		       "received from " MACSTR " - ignored\n",
		       data->dev->name, len, MAC2STR(mgmt->sa));
		return;
	}

	if (memcmp(data->bssid, mgmt->sa, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: association frame received from "
		       "unknown AP (SA=" MACSTR " BSSID=" MACSTR ") - "
		       "ignored\n", data->dev->name, MAC2STR(mgmt->sa),
		       MAC2STR(mgmt->bssid));
		return;
	}

	capab_info = le16_to_cpu(mgmt->u.assoc_resp.capab_info);
	status_code = le16_to_cpu(mgmt->u.assoc_resp.status_code);
	aid = le16_to_cpu(mgmt->u.assoc_resp.aid);

	printk(KERN_DEBUG "%s: RX %sssocResp (capab=0x%x "
	       "aid=%d status=%d %s)\n",
	       data->dev->name, reassoc ? "Rea" : "A",
	       capab_info, aid & 0x3FFF, status_code,
	       ieee80211_status_code(status_code));

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, mgmt->sa);
	if (status_code != WLAN_STATUS_SUCCESS) {
		printk(KERN_DEBUG "%s: AP denied association\n",
		       data->dev->name);
		if (bss)
			bss->num_assoc_fail++;
		spin_unlock_bh(&data->lock);
		return;
	}

	if ((aid & (BIT(15) | BIT(14))) != (BIT(15) | BIT(14)))
		printk(KERN_DEBUG "%s: invalid aid value %d; bits 15:14 not "
		       "set\n", data->dev->name, aid);
	aid &= ~(BIT(15) | BIT(14));

	if (bss)
		bss->num_auth_fail = bss->num_assoc_fail = 0;

	baselen = (u8 *) mgmt->u.assoc_resp.variable - (u8 *) mgmt;
        if (ieee802_11_parse_elems(mgmt->u.assoc_resp.variable, len - baselen, &elems) == ParseOK
	    && elems.supp_rates) {
		memset(bss->supp_rates, 0, IEEE80211_MAX_SUPP_RATES);
		memcpy(bss->supp_rates, elems.supp_rates,
		       elems.supp_rates_len > IEEE80211_MAX_SUPP_RATES ?
		       IEEE80211_MAX_SUPP_RATES : elems.supp_rates_len);
	}

	spin_unlock_bh(&data->lock);

	printk(KERN_DEBUG "%s: associated\n", data->dev->name);
	data->aid = aid;
	data->assoc_tries = 0;
	ieee80211_set_associated(data, 1);

	if (data->set_ssid) {
		if (data->flags & HIDDEN_SSID)
			data->set_ssid(data->dev, (char *)hidden_ssid, data->ssid_len);
		else
			data->set_ssid(data->dev, data->ssid, data->ssid_len);
	}
	ieee80211_associated(data);
}


void ieee80211_rx_mgmt(struct ieee80211_data *data, struct sk_buff *skb,
		       struct ieee80211_rx_status *rx_status)
{
	struct ieee80211_mgmt *mgmt;
	u16 fc;

	if (skb->len < 24)
		goto fail;

	mgmt = (struct ieee80211_mgmt *) skb->data;
	fc = le16_to_cpu(mgmt->frame_control);

	switch (WLAN_FC_GET_STYPE(fc)) {
	case WLAN_FC_STYPE_PROBE_REQ:
		break;
	case WLAN_FC_STYPE_PROBE_RESP:
		ieee80211_rx_mgmt_probe_resp(data, mgmt, skb->len, rx_status);
		break;
	case WLAN_FC_STYPE_BEACON:
		ieee80211_rx_mgmt_beacon(data, mgmt, skb->len, rx_status);
		break;
	case WLAN_FC_STYPE_AUTH:
		ieee80211_rx_mgmt_auth(data, mgmt, skb->len, rx_status, skb);
		break;
	case WLAN_FC_STYPE_ASSOC_REQ:
		ieee80211_rx_mgmt_assoc_req(data, mgmt, skb->len, rx_status, 0);
		break;
	case WLAN_FC_STYPE_ASSOC_RESP:
		ieee80211_rx_mgmt_assoc_resp(data, mgmt, skb->len, rx_status, 0);
		break;
	case WLAN_FC_STYPE_REASSOC_REQ:
		ieee80211_rx_mgmt_assoc_req(data, mgmt, skb->len, rx_status, 1);
		break;
	case WLAN_FC_STYPE_REASSOC_RESP:
		ieee80211_rx_mgmt_assoc_resp(data, mgmt, skb->len, rx_status, 1);
		break;
	case WLAN_FC_STYPE_DEAUTH:
		ieee80211_rx_mgmt_deauth(data, mgmt, skb->len, rx_status);
		break;
	case WLAN_FC_STYPE_DISASSOC:
		ieee80211_rx_mgmt_disassoc(data, mgmt, skb->len, rx_status);
		break;
	default:
		printk(KERN_DEBUG "%s: received unknown management frame - "
		       "stype=%d\n", data->dev->name, WLAN_FC_GET_STYPE(fc));
		break;
	}

 fail:
	if (skb)
		dev_kfree_skb(skb);
}


static void ieee80211_timer(unsigned long ptr)
{
	struct ieee80211_data *data = (struct ieee80211_data *) ptr;

	switch (data->state) {
	case IEEE80211_SCAN:
		ieee80211_scan(data);
		break;
	case IEEE80211_AUTHENTICATE:
		ieee80211_authenticate(data);
		break;
	case IEEE80211_ASSOCIATE:
		ieee80211_associate(data);
		break;
	case IEEE80211_ASSOCIATED:
		ieee80211_associated(data);
		break;
	case IEEE80211_STOP:
		break;
	default:
		printk(KERN_DEBUG "ieee80211_timer: Unknown state %d\n",
		       data->state);
		break;
	}
}

struct sk_buff * ieee80211_wep_decode(struct ieee80211_data *data,
				      struct sk_buff *skb)
{
	struct ieee80211_hdr *hdr;
	u16 fc;
	int hlen;

	hdr = (struct ieee80211_hdr *) skb->data;
	fc = le16_to_cpu(hdr->frame_control);

	if (!(fc & WLAN_FC_ISWEP))
		return skb;

	if ((fc & (WLAN_FC_TODS | WLAN_FC_FROMDS)) ==
		  (WLAN_FC_TODS | WLAN_FC_FROMDS))
		hlen = 30;
	else
		hlen = 24;

	if (wep_decrypt(skb, hlen, data)) {
		dev_kfree_skb(skb);
		return NULL;
	} else
		return skb;
}

int ieee80211_data_tx(struct ieee80211_data *data,
		      struct sk_buff *skb)
{
	struct ieee80211_bss *bss;
	u8 dst[ETH_ALEN];
	u16 fc = (WLAN_FC_TYPE_DATA << 2) | (WLAN_FC_STYPE_DATA << 4);
	int rc, short_preamble = 0;

	/* hold the packets till we're authenticated/associated */
	if (data->state != IEEE80211_ASSOCIATED) {
		netif_stop_queue(data->dev);
		return 1;
	}

	if (skb->len < ETH_HLEN) {
		dev_kfree_skb(skb);
		return 0;
	}

	if (data->flags & WEP_ENABLED)
		fc |= WLAN_FC_ISWEP;

	switch (data->mode) {
	case IEEE80211_MANAGED:
		fc |= WLAN_FC_TODS;
		break;
	case IEEE80211_ADHOC:
		break;
	case IEEE80211_AP:
		fc |= WLAN_FC_FROMDS;
		break;
	case IEEE80211_WDS:
		fc |= WLAN_FC_FROMDS | WLAN_FC_TODS;
		break;
	case IEEE80211_MONITOR:
		dev_kfree_skb(skb);
		return 0;
	}

	memcpy(dst, skb->data, ETH_ALEN);

	skb = ieee80211_data_encaps(data, skb);

        spin_lock_bh(&data->lock);
        bss = ieee80211_get_bss(data, dst);

	if (bss)
		short_preamble = bss->capability & WLAN_CAPABILITY_SHORT_PREAMBLE;

	rc = data->tx(data->dev, skb, cpu_to_le16(fc), short_preamble, dst);
        spin_unlock_bh(&data->lock);

	return rc;
}

/* See IEEE 802.1H for LLC/SNAP encapsulation/decapsulation */
/* Ethernet-II snap header (RFC1042 for most EtherTypes) */
static unsigned char rfc1042_header[] =
{ 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };
/* Bridge-Tunnel header (for EtherTypes ETH_P_AARP and ETH_P_IPX) */
static unsigned char bridge_tunnel_header[] =
{ 0xaa, 0xaa, 0x03, 0x00, 0x00, 0xf8 };
/* No encapsulation header if EtherType < 0x600 (=length) */


struct sk_buff * ieee80211_data_encaps(struct ieee80211_data *data,
				       struct sk_buff *skb)
{
	u16 ethertype;

	ethertype = (skb->data[12] << 8) | skb->data[13];

	skb_pull(skb, 12);

	memcpy(skb_push(skb, 6),
	       ethertype == ETH_P_AARP || ethertype == ETH_P_IPX ?
	       bridge_tunnel_header : rfc1042_header, 6);

	return skb;
}


struct sk_buff * ieee80211_data_decaps(struct ieee80211_data *data,
				       struct sk_buff *skb)
{
	u8 dst[ETH_ALEN];
	u8 src[ETH_ALEN];
	struct ieee80211_hdr *hdr;
	int hlen;
	u16 fc;
	u8 *pos;

	if (data->state != IEEE80211_ASSOCIATED ||
	    skb->len < 24)
		goto drop;

	hdr = (struct ieee80211_hdr *) skb->data;
	fc = le16_to_cpu(hdr->frame_control);

	hlen = 24;

	switch (fc & (WLAN_FC_TODS | WLAN_FC_FROMDS)) {
	case WLAN_FC_TODS | WLAN_FC_FROMDS:
		/* RA, TA, DA, SA */
		if (skb->len < 30)
			goto drop;
		memcpy(dst, hdr->addr3, ETH_ALEN);
		memcpy(src, hdr->addr4, ETH_ALEN);
		hlen += ETH_ALEN;
		break;
	case WLAN_FC_TODS:
		/* BSSID, SA, DA */
		memcpy(dst, hdr->addr3, ETH_ALEN);
		memcpy(src, hdr->addr2, ETH_ALEN);
		break;
	case WLAN_FC_FROMDS:
		/* DA, BSSID, SA */
		memcpy(dst, hdr->addr1, ETH_ALEN);
		memcpy(src, hdr->addr3, ETH_ALEN);
		break;
	case 0:
		/* DA, SA, BSSID */
		memcpy(dst, hdr->addr1, ETH_ALEN);
		memcpy(src, hdr->addr2, ETH_ALEN);
		break;
	}

	pos = skb_pull(skb, hlen);
	if (hlen >= 6 &&
	    (memcmp(pos, rfc1042_header, 6) == 0 ||
	     memcmp(pos, bridge_tunnel_header, 6) == 0)) {
		skb_pull(skb, 6);
	}

	memcpy(skb_push(skb, ETH_ALEN), src, ETH_ALEN);
	memcpy(skb_push(skb, ETH_ALEN), dst, ETH_ALEN);

	return skb;

drop:
	dev_kfree_skb(skb);
	return NULL;
}

struct sk_buff * ieee80211_reassemble_frag (struct ieee80211_data *data,
			       struct sk_buff *skb)
{
	struct ieee80211_hdr *hdr, *hdr2;
	struct ieee80211_frag_entry *entry;
	u16 seq_ctrl, fc;
	unsigned int seq, frag, hlen;
	int i, j;

	hdr = (struct ieee80211_hdr *) skb->data;
	fc = le16_to_cpu(hdr->frame_control);
	seq_ctrl = le16_to_cpu(hdr->seq_ctrl);
	seq = WLAN_GET_SEQ_SEQ(seq_ctrl);
	frag = WLAN_GET_SEQ_FRAG(seq_ctrl);

	if (!frag && fc & WLAN_FC_MOREFRAG) {
		i = 0;
		/* look for an empty entry */
		while (i < IEEE80211_FRAG_CACHE_SIZE
		       && data->frag_cache[i].skb != NULL)
			i+=1;

		if (data->frag_cache[i].skb) { /* cache is full */
			i = -1;
			for (j = 0; j < IEEE80211_FRAG_CACHE_SIZE; j += 1) {
			entry = &data->frag_cache[j];
			if (entry->skb != NULL &&
			    time_after(jiffies, entry->first_rx_time + 2 * HZ)) {
				dev_kfree_skb(entry->skb);
				entry->skb = NULL;
				i = j;
			}
			}
		}

		if (i == -1)
			goto drop;	/* could not free a stale entry */

		entry = &data->frag_cache[i];
		entry->skb = dev_alloc_skb(2346);
		if (!entry->skb)
			goto drop;

		entry->skb->dev = data->dev;
		entry->first_rx_time = jiffies;
		memcpy(skb_put(entry->skb, skb->len), skb->data, skb->len);
		goto drop;
	} else if (frag) {
		if ((fc & (WLAN_FC_TODS | WLAN_FC_FROMDS)) ==
			  (WLAN_FC_TODS | WLAN_FC_FROMDS))
			hlen = 30;
		else
			hlen = 24;

		for (i = 0; i < IEEE80211_FRAG_CACHE_SIZE; i += 1) {
		entry = &data->frag_cache[i];
		if (!entry->skb)
			continue;
		hdr2 = (struct ieee80211_hdr *) entry->skb->data;
		if (WLAN_GET_SEQ_SEQ(le16_to_cpu(hdr2->seq_ctrl)) != seq 
		    || memcmp(hdr2->addr1, hdr->addr1, ETH_ALEN)
		    || memcmp(hdr2->addr2, hdr->addr2, ETH_ALEN))
			continue;

		skb_pull(skb, hlen);
		if (entry->skb->tail + skb->len > entry->skb->end) {
			dev_kfree_skb(entry->skb);
			entry->skb = NULL;
			goto drop;
		}

		memcpy(skb_put(entry->skb, skb->len), skb->data,
                               skb->len);

		if (fc & WLAN_FC_MOREFRAG)
			goto drop;
		else {
			dev_kfree_skb(skb);
			skb = entry->skb;
			entry->skb = NULL;
			return skb;
		}

		}

		goto drop;
	}

	return skb;
drop:
	dev_kfree_skb(skb);
	return NULL;
}

int ieee80211_filter_duplicates(struct ieee80211_data *data,
				struct sk_buff *skb, u16 fc)
{
	struct ieee80211_hdr *hdr;
	struct ieee80211_bss *bss;

	hdr = (struct ieee80211_hdr *) skb->data;

	spin_lock_bh(&data->lock);
	bss = ieee80211_get_bss(data, hdr->addr2);

	if (bss) {
		if (fc & WLAN_FC_RETRY && bss->last_seq == hdr->seq_ctrl) {
			spin_unlock_bh(&data->lock);
			return 1;
		}
		bss->last_seq = hdr->seq_ctrl;
	}
	spin_unlock_bh(&data->lock);

	return 0;
}

/* returns a rate appropriate for the destination, if it can be determined */
u8 ieee80211_get_rate(struct ieee80211_data *data, u8 *addr)
{
	struct ieee80211_bss *bss;
	u8 rate;

	bss = ieee80211_get_bss(data, addr);

	if (bss)
		rate = bss->rate;
	else
		rate = data->rate;

	return rate;
}
