/*
 * Linux driver for ADMtek adm8211 (IEEE 802.11b wireless LAN card)
 *
 * Copyright (c) 2003, Jouni Malinen <jkmaline@cc.hut.fi>
 * Copyright (c) 2004-2005, Michael Wu <flamingice@sourmilk.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/netdevice.h>
#include <linux/wireless.h>
#if WIRELESS_EXT > 12
#include <net/iw_handler.h>
#endif /* WIRELESS_EXT > 12 */
#include <linux/random.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>
#include <linux/pci.h>

#include "adm8211.h"
#include "adm8211_ioctl.h"


static const long freq_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				  2447, 2452, 2457, 2462, 2467, 2472, 2484 };
#define FREQ_COUNT ARRAY_SIZE(freq_list)

#if 0
#define SET_IWFREQ(freq, chan) \
        freq->m = freq_list[chan - 1] * 100000; \
        freq->e = 1;
#else
#define SET_IWFREQ(freq, chan) \
	freq->m = chan; \
        freq->e = 0;
#endif

static int adm8211_ioctl_giwname(struct net_device *dev,
				 struct iw_request_info *info,
				 char *name, char *extra)
{
	strcpy(name, "IEEE 802.11b");
	return 0;
}

static int adm8211_ioctl_siwfreq(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_freq *freq, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	/* freq => chan. */
	if (freq->e == 1 &&
	    freq->m / 100000 >= freq_list[0] &&
	    freq->m / 100000 <= freq_list[FREQ_COUNT - 1]) {
		int ch;
		int fr = freq->m / 100000;
		for (ch = 0; ch < FREQ_COUNT; ch++) {
			if (fr == freq_list[ch]) {
				freq->e = 0;
				freq->m = ch + 1;
				break;
			}
		}
	}

	if (freq->e != 0 || freq->m < priv->ieee80211.chan_range.min || freq->m > priv->ieee80211.chan_range.max)
		return -EINVAL;

	if (dev->flags & IFF_UP) {
		if (priv->iw_mode == IW_MODE_MONITOR)
			adm8211_rf_set_channel(dev, freq->m);
		else
			ieee80211_start_scan(&priv->ieee80211);
	}

	priv->ieee80211.channel = priv->ieee80211.pref_channel = freq->m;

	return 0;
}


static int adm8211_ioctl_giwfreq(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_freq *freq, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int chan = priv->ieee80211.channel;

	if (chan < 1 || chan > FREQ_COUNT)
		return -EINVAL;

	SET_IWFREQ(freq, chan);

	return 0;
}


static int adm8211_ioctl_siwmode(struct net_device *dev,
				 struct iw_request_info *info,
				 __u32 *mode, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	if (*mode != IW_MODE_INFRA && *mode != IW_MODE_ADHOC &&
	    *mode != IW_MODE_MONITOR)
		return -EOPNOTSUPP;

	if (*mode == priv->iw_mode)
		return 0;

	ieee80211_stop(data);
	priv->iw_mode = *mode;
	
	switch (*mode) {
	case IW_MODE_INFRA:
		priv->ieee80211.mode = IEEE80211_MANAGED;
		priv->ieee80211.capab &= ~WLAN_CAPABILITY_IBSS;
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = priv->eth_header_parse;
		break;
	case IW_MODE_ADHOC:
		priv->ieee80211.mode = IEEE80211_ADHOC;
		priv->ieee80211.capab |= WLAN_CAPABILITY_IBSS;
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = priv->eth_header_parse;
		break;
	case IW_MODE_MONITOR:
		priv->ieee80211.mode = IEEE80211_MONITOR;
		dev->type = ARPHRD_IEEE80211_PRISM;
		dev->hard_header_parse = adm8211_80211_header_parse;
		break;
	}

	if (dev->flags & IFF_UP) {
		adm8211_update_mode(dev);
		ieee80211_start(data);
	}

	return 0;
}


static int adm8211_ioctl_giwmode(struct net_device *dev,
				 struct iw_request_info *info,
				 __u32 *mode, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	*mode = priv->iw_mode;
	return 0;
}

static int adm8211_ioctl_giwrange(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq,
				  char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct iw_range *range = (struct iw_range *) extra;
	int i, j;

        dwrq->length = sizeof(struct iw_range);
        memset(range, 0, sizeof(range));

	range->min_nwid = range->max_nwid = 0;
	
	range->max_qual.qual  = 255;
	range->max_qual.level = 1;
	range->max_qual.noise = 1;

	range->num_bitrates = data->num_supp_rates;
	i = 0;
	while (i++ < min(data->num_supp_rates, IW_MAX_BITRATES))
		range->bitrate[i-1] = (data->supp_rates[i-1]*1000000)/2;

	range->min_rts = 0;
	range->max_rts = 2346;
	range->min_frag = 256;
	range->max_frag = 2346;

	/* are the values really in dBm? */
	range->txpower_capa = IW_TXPOW_RANGE | IW_TXPOW_DBM;

	range->we_version_compiled = WIRELESS_EXT;
	range->we_version_source = WIRELESS_EXT;
	range->num_channels = data->chan_range.max - data->chan_range.min + 1;
	range->num_frequency = range->num_channels;

	range->max_encoding_tokens = 4;
	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;
	range->num_encoding_sizes = 2;

	j=0;
	for (i = data->chan_range.min; i <= data->chan_range.max; i+=1) {
		range->freq[j].m=freq_list[i-1] * 100000;
		range->freq[j].e=1;
		range->freq[j].i=i;
		j+=1;
	}
	return 0;
}

/*
static int adm8211_ioctl_giwsens(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct ieee80211_bss *bss = ieee80211_get_bss(data, data->bssid);
	return 0;
}
*/
static int adm8211_ioctl_siwscan(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_point *data, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (dev->flags & IFF_UP && priv->iw_mode != IW_MODE_MONITOR)
		ieee80211_start_scan(&priv->ieee80211);
	else
		return -1;

	return 0;
}


static int adm8211_ioctl_giwscan(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_point *data, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct iw_event iwe;
	char *current_ev = extra;
	char *end_buf = extra + IW_SCAN_MAX_DATA;
	char *current_val;
	struct ieee80211_bss *bss;
	int i;

	if (priv->ieee80211.state == IEEE80211_SCAN)
		return -EAGAIN;

	spin_lock_bh(&priv->ieee80211.lock);
	bss = priv->ieee80211.bss;
	while (bss) {
		/* First entry must be AP MAC address; other entries will be
		 * shown in the order they are added. */
		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = SIOCGIWAP;
		iwe.u.ap_addr.sa_family = ARPHRD_ETHER;
		memcpy(iwe.u.ap_addr.sa_data, bss->bssid, ETH_ALEN);
		iwe.len = IW_EV_ADDR_LEN;
		current_ev = iwe_stream_add_event(current_ev, end_buf, &iwe,
						  IW_EV_ADDR_LEN);

		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = SIOCGIWESSID;
		iwe.u.data.length = bss->ssid_len;
		if (iwe.u.data.length > IW_ESSID_MAX_SIZE)
			iwe.u.data.length = IW_ESSID_MAX_SIZE;
		iwe.u.data.flags = 1;
		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		current_ev = iwe_stream_add_point(current_ev, end_buf, &iwe,
						  bss->ssid);

		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = SIOCGIWMODE;
		if (bss->capability & (WLAN_CAPABILITY_ESS |
				       WLAN_CAPABILITY_IBSS)) {
			if (bss->capability & WLAN_CAPABILITY_ESS)
				iwe.u.mode = IW_MODE_MASTER;
			else
				iwe.u.mode = IW_MODE_ADHOC;
			iwe.len = IW_EV_UINT_LEN;
			current_ev = iwe_stream_add_event(current_ev, end_buf,
							  &iwe,
							  IW_EV_UINT_LEN);
		}

		if (bss->channel >= 1 && bss->channel <= FREQ_COUNT) {
			memset(&iwe, 0, sizeof(iwe));
			iwe.cmd = SIOCGIWFREQ;
			SET_IWFREQ((&iwe.u.freq), bss->channel);
			iwe.len = IW_EV_FREQ_LEN;
			current_ev = iwe_stream_add_event(current_ev, end_buf,
							  &iwe,
							  IW_EV_FREQ_LEN);
		}

		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = IWEVQUAL;
		iwe.u.qual.qual = bss->last_rssi;
		iwe.u.qual.level = 0;
		iwe.u.qual.noise = 0;
		iwe.len = IW_EV_QUAL_LEN;
		current_ev = iwe_stream_add_event(current_ev, end_buf, &iwe,
						  IW_EV_QUAL_LEN);

		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = SIOCGIWENCODE;
		if (bss->capability & WLAN_CAPABILITY_PRIVACY)
			iwe.u.data.flags = IW_ENCODE_ENABLED | IW_ENCODE_NOKEY;
		else
			iwe.u.data.flags = IW_ENCODE_DISABLED;
		iwe.u.data.length = 0;
		iwe.len = IW_EV_POINT_LEN + iwe.u.data.length;
		/* bss->ssid is a dummy pointer for 0-byte memcpy */
		current_ev = iwe_stream_add_point(current_ev, end_buf, &iwe,
						  bss->ssid);

		memset(&iwe, 0, sizeof(iwe));
		iwe.cmd = SIOCGIWRATE;
		current_val = current_ev + IW_EV_LCP_LEN;
		for (i = 0; i < sizeof(bss->supp_rates); i++) {
			if (bss->supp_rates[i] == 0)
				break;
			/* Bit rate given in 500 kb/s units (+ 0x80) */
			iwe.u.bitrate.value =
				((bss->supp_rates[i] & 0x7f) * 500000);
			current_val = iwe_stream_add_value(
				current_ev, current_val, end_buf, &iwe,
				IW_EV_PARAM_LEN);
		}
		if ((current_val - current_ev) > IW_EV_LCP_LEN)
			current_ev = current_val;

		bss = bss->next;
	}
	spin_unlock_bh(&priv->ieee80211.lock);

	data->length = current_ev - extra;
	return 0;
}


static int adm8211_ioctl_giwessid(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *data, char *essid)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	data->flags = 1; /* active */
	data->length = priv->ieee80211.ssid_len;
	if (data->length > IW_ESSID_MAX_SIZE)
		data->length = IW_ESSID_MAX_SIZE;
	memset(essid, 0, IW_ESSID_MAX_SIZE);
	memcpy(essid, priv->ieee80211.ssid, data->length);
	return 0;
}


static int adm8211_ioctl_siwessid(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *data, char *essid)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int len;

	if (data->flags == 0 || data->length < 1)
		len = 0;
	else
		len = data->length - 1;
	if (len > IEEE80211_MAX_SSID_LEN)
		len = IEEE80211_MAX_SSID_LEN;

	memcpy(priv->ieee80211.ssid, essid, len);
	priv->ieee80211.ssid_len = len;
	if (len)
		priv->ieee80211.flags &= ~ANY_SSID;
	else
		priv->ieee80211.flags |= ANY_SSID;

	if (dev->flags & IFF_UP)
		ieee80211_start(&priv->ieee80211);

	return 0;
}

static int adm8211_ioctl_giwnickn(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *data, char *nickn)
{
	struct adm8211_priv *priv = netdev_priv(dev);

        memset(nickn, 0, IW_ESSID_MAX_SIZE);
        strncpy(nickn, priv->ieee80211.nick, IW_ESSID_MAX_SIZE);
        return 0;
}

static int adm8211_ioctl_siwnickn(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *data, char *nickn)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int len = data->length;

	if (len > IW_ESSID_MAX_SIZE)
		len = IW_ESSID_MAX_SIZE;

	memset(priv->ieee80211.nick, 0, IW_ESSID_MAX_SIZE);
	strncpy(priv->ieee80211.nick, nickn, len);

	return 0;
}

static int adm8211_ioctl_giwap(struct net_device *dev,
			       struct iw_request_info *info,
			       struct sockaddr *ap_addr, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	ap_addr->sa_family = ARPHRD_ETHER;
	if (dev->flags & IFF_UP || !(priv->ieee80211.flags & PREF_BSSID_SET))
		memcpy(ap_addr->sa_data, priv->ieee80211.bssid, ETH_ALEN);
	else
		memcpy(ap_addr->sa_data, priv->ieee80211.pref_bssid, ETH_ALEN);

	return 0;
}


static int adm8211_ioctl_siwap(struct net_device *dev,
			       struct iw_request_info *info,
			       struct sockaddr *ap_addr, char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u8 *addr = ap_addr->sa_data;
	memcpy(priv->ieee80211.pref_bssid, addr, ETH_ALEN);
	if (addr[0] || addr[1] || addr[2] || addr[3] || addr[4] || addr[5])
		priv->ieee80211.flags |= PREF_BSSID_SET;
	else
		priv->ieee80211.flags &= ~PREF_BSSID_SET;

	if (dev->flags & IFF_UP)
		ieee80211_start(&priv->ieee80211);

	return 0;
}

static int adm8211_ioctl_siwrate(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	int i, j;

	if (vwrq->value <= 0)
		return -EINVAL;

	data->flags &= ~AUTO_RATE;
	if (!vwrq->fixed) {
		data->flags |= AUTO_RATE;

		/* assumes last rate is the highest rate */
		data->rate = data->supp_rates[data->num_supp_rates-1];
	} else if (vwrq->value <= data->num_supp_rates) {
		data->rate = data->supp_rates[vwrq->value-1];
	} else {
		i = 0;
		j = (vwrq->value*2)/1000000;

		/* make sure the rate given matches a supported rate */
		while (i < data->num_supp_rates && data->supp_rates[i] != j)
			i+=1;

		/* give up if it doesn't */
		if (i >= data->num_supp_rates)
			return -EINVAL;

		data->rate = j;
	}

	if (dev->flags & IFF_UP)
		data->set_rate(dev);

	return 0;
}

static int adm8211_ioctl_giwrate(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	vwrq->fixed = !(data->flags & AUTO_RATE);
	vwrq->value = (data->rate*1000000)/2;
	return 0;
}

static int adm8211_ioctl_siwrts(struct net_device *dev,
				struct iw_request_info *info,
				struct iw_param *vwrq,
				char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (vwrq->disabled)
		priv->rts_thresh = 2347;
	else if ((vwrq->value >= 0) && (vwrq->value <= 2346))
		priv->rts_thresh = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwrts(struct net_device *dev,
				struct iw_request_info *info,
				struct iw_param *vwrq,
				char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	vwrq->value = priv->rts_thresh;
	vwrq->disabled = (vwrq->value > 2346);
	vwrq->fixed = 1;

	return 0;
}

static int adm8211_ioctl_siwfrag(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (vwrq->disabled)
		priv->frag_thresh = 2346;
	else if ((vwrq->value >= 256) && (vwrq->value <= 2346))
		priv->frag_thresh = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwfrag(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	vwrq->value = priv->frag_thresh;
	vwrq->disabled = (vwrq->value >= 2346);
	vwrq->fixed = 1;

	return 0;
}

static int adm8211_ioctl_giwtxpow(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	vwrq->flags = IW_TXPOW_DBM;
	if (priv->tx_power > 0x3F) {
		vwrq->fixed = 0;
		vwrq->value = priv->eeprom->tx_power[priv->ieee80211.channel-1];
	} else {
		vwrq->fixed = 1;
		vwrq->value = priv->tx_power;
	}

	if (!vwrq->value)
		vwrq->disabled = 1;
	else
		vwrq->disabled = 0;

	return 0;
}

static int adm8211_ioctl_siwtxpow(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (vwrq->disabled || !vwrq->fixed) {
		priv->tx_power = 0x40;
	} else if (vwrq->value >= 0 && vwrq->value <= 0x3F) {
		priv->tx_power = vwrq->value;
	} else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_siwretry(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (!vwrq->disabled && (vwrq->flags & IW_RETRY_LIMIT))
		priv->retry_limit = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwretry(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	vwrq->disabled = 0;
	vwrq->value = priv->retry_limit;
	vwrq->flags = IW_RETRY_LIMIT;

	return 0;
}

/* WEP ioctls adapted from atmel driver */
static int adm8211_ioctl_siwencode(struct net_device *dev,
				   struct iw_request_info *info,
				   struct iw_point *dwrq,
				   char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	int index = (dwrq->flags & IW_ENCODE_INDEX) - 1;


	if (dwrq->length > 0) {
		if (dwrq->length > 13)
			return -EINVAL;

		/* Check the index (none -> use current) */
		if (index < 0 || index >= 4)
			index = data->default_key;
		else
			data->default_key = index;

		if (dwrq->length > 5)
			data->wep_data[index].len = 13;
		else
			data->wep_data[index].len = 5;

		/* Check if the key is not marked as invalid */
		if (!(dwrq->flags & IW_ENCODE_NOKEY)) {
			/* Cleanup */
			memset(data->wep_data[index].key, 0, 13);
			/* Copy the key in the driver */
			memcpy(data->wep_data[index].key, extra, dwrq->length);
		} else {
			memset(data->wep_data[index].key, 0, 13);
			data->wep_data[index].len = 0;
		}

		if (dev->flags & IFF_UP)
			adm8211_write_wepkey(dev, index);
	} else {
		/* Do we want to just set the transmit key index ? */
		if ( index >= 0 && index < 4 )
			data->default_key = index;
		else if (!dwrq->flags & IW_ENCODE_MODE)
			return -EINVAL;
	}

	if (dwrq->flags & IW_ENCODE_DISABLED) {
		data->flags &= ~WEP_ENABLED;
		data->auth_algorithm = 0;
		priv->ieee80211.capab &= ~WLAN_CAPABILITY_PRIVACY;
	} else {
		data->flags |= WEP_ENABLED;
		if (dwrq->flags & IW_ENCODE_RESTRICTED)
			data->flags |= WEP_RESTRICTED;
		else
			data->flags &= ~WEP_RESTRICTED;
		data->auth_algorithm = 1;
		priv->ieee80211.capab |= WLAN_CAPABILITY_PRIVACY;
	}

	if (dev->flags & IFF_UP)
		adm8211_update_wep(dev);

	return 0;
}

static int adm8211_ioctl_giwencode(struct net_device *dev,
				   struct iw_request_info *info,
				   struct iw_point *dwrq,
				   char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	int index = (dwrq->flags & IW_ENCODE_INDEX) - 1;

	if (data->flags & WEP_ENABLED) {
		if (data->flags & WEP_RESTRICTED)
			dwrq->flags = IW_ENCODE_RESTRICTED;
		else
			dwrq->flags = IW_ENCODE_OPEN;
	} else
		 dwrq->flags = IW_ENCODE_DISABLED;

	if (index < 0 || index >= 4)
		index = data->default_key;

	dwrq->flags |= index + 1;
	dwrq->length = data->wep_data[index].len;
	memset(extra, 0, 16);
	memcpy(extra, data->wep_data[index].key, dwrq->length);

	return 0;
}

static const iw_handler adm8211_handler[] =
{
	(iw_handler) NULL,				/* SIOCSIWCOMMIT */
	(iw_handler) adm8211_ioctl_giwname,		/* SIOCGIWNAME */
	(iw_handler) NULL,				/* SIOCSIWNWID */
	(iw_handler) NULL,				/* SIOCGIWNWID */
	(iw_handler) adm8211_ioctl_siwfreq,		/* SIOCSIWFREQ */
	(iw_handler) adm8211_ioctl_giwfreq,		/* SIOCGIWFREQ */
	(iw_handler) adm8211_ioctl_siwmode,		/* SIOCSIWMODE */
	(iw_handler) adm8211_ioctl_giwmode,		/* SIOCGIWMODE */
	(iw_handler) NULL,				/* SIOCSIWSENS */
	(iw_handler) NULL,				/* SIOCGIWSENS */
	(iw_handler) NULL /* not used */,		/* SIOCSIWRANGE */
	(iw_handler) adm8211_ioctl_giwrange,		/* SIOCGIWRANGE */
	(iw_handler) NULL /* not used */,		/* SIOCSIWPRIV */
	(iw_handler) NULL /* kernel code */,		/* SIOCGIWPRIV */
	(iw_handler) NULL /* not used */,		/* SIOCSIWSTATS */
	(iw_handler) NULL /* kernel code */,		/* SIOCGIWSTATS */
	(iw_handler) NULL,				/* SIOCSIWSPY */
	(iw_handler) NULL,				/* SIOCGIWSPY */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) adm8211_ioctl_siwap,		/* SIOCSIWAP */
	(iw_handler) adm8211_ioctl_giwap,		/* SIOCGIWAP */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) NULL,				/* SIOCGIWAPLIST */
	(iw_handler) adm8211_ioctl_siwscan,		/* SIOCSIWSCAN */
	(iw_handler) adm8211_ioctl_giwscan,		/* SIOCGIWSCAN */
	(iw_handler) adm8211_ioctl_siwessid,		/* SIOCSIWESSID */
	(iw_handler) adm8211_ioctl_giwessid,		/* SIOCGIWESSID */
	(iw_handler) adm8211_ioctl_siwnickn,		/* SIOCSIWNICKN */
	(iw_handler) adm8211_ioctl_giwnickn,		/* SIOCGIWNICKN */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) adm8211_ioctl_siwrate,		/* SIOCSIWRATE */
	(iw_handler) adm8211_ioctl_giwrate,		/* SIOCGIWRATE */
	(iw_handler) adm8211_ioctl_siwrts,		/* SIOCSIWRTS */
	(iw_handler) adm8211_ioctl_giwrts,		/* SIOCGIWRTS */
	(iw_handler) adm8211_ioctl_siwfrag,		/* SIOCSIWFRAG */
	(iw_handler) adm8211_ioctl_giwfrag,		/* SIOCGIWFRAG */
	(iw_handler) adm8211_ioctl_siwtxpow,		/* SIOCSIWTXPOW */
	(iw_handler) adm8211_ioctl_giwtxpow,		/* SIOCGIWTXPOW */
	(iw_handler) adm8211_ioctl_siwretry,		/* SIOCSIWRETRY */
	(iw_handler) adm8211_ioctl_giwretry,		/* SIOCGIWRETRY */
	(iw_handler) adm8211_ioctl_siwencode,		/* SIOCSIWENCODE */
	(iw_handler) adm8211_ioctl_giwencode,		/* SIOCGIWENCODE */
	(iw_handler) NULL,				/* SIOCSIWPOWER */
	(iw_handler) NULL,				/* SIOCGIWPOWER */
};

static int adm8211_ioctl_sreg (struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int *i = (int *) extra;
	u32 addr = i[0];
	u32 reg = i[1];

	if (addr >= 0 && addr <= 0x10C) {
		printk(KERN_NOTICE "%s: writing 0x%x to 0x%x\n",
		       dev->name, reg, addr);
		iowrite32(reg, (void __iomem *)priv->map + addr);
	} else {
		printk(KERN_NOTICE "%s: setreg: register value out of range\n", dev->name);
	}
	return 0;
}

static int adm8211_ioctl_greg (struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{
        struct adm8211_priv *priv = netdev_priv(dev);
        u32 addr = wrqu->param.value;

	if (addr >= 0 && addr <= 0x10C) {
	        printk(KERN_NOTICE "%s: value of register 0x%x is 0x%x\n",
		       dev->name, addr, ioread32((void __iomem *)priv->map + addr));
	} else {
		printk(KERN_NOTICE "%s: getreg: register value out of range\n", dev->name);
	}
        return 0;
}

static int adm8211_ioctl_print_eeprom (struct net_device *dev,
				       struct iw_request_info *info,
				       union iwreq_data *wrqu,
				       char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i, bl;

	if (priv->eeprom) {
	printk(KERN_NOTICE "%s: eeprom dump:\n", dev->name);

	for (bl = 0; bl < priv->eeprom_len; bl += 16) {
		printk(KERN_NOTICE "%s:", dev->name);
		for (i = 0; i<16; i+=1)
			printk(" %02x", ((u8 *)priv->eeprom)[bl+i]);
		printk("\n");
	}

	} else
		printk(KERN_NOTICE "%s: no eeprom data\n", dev->name);

	return 0;
}

static int adm8211_ioctl_set_country (struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int code = wrqu->param.value;

	if (code >= ARRAY_SIZE(cranges) || code < 0)
		return -EINVAL;

	/* potential, but unlikely race, I think. need to check */
	priv->ieee80211.chan_range = cranges[code];
	printk(KERN_DEBUG "%s: Channel range: %d-%d\n",
	       dev->name, (int)priv->ieee80211.chan_range.min, (int)priv->ieee80211.chan_range.max);

	return 0;
}

static int adm8211_ioctl_set_antpower(struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (wrqu->param.value > 0x3F) {
		priv->ant_power = 0x40;
	} else if (wrqu->param.value >= 0 && wrqu->param.value <= 0x3F) {
		priv->ant_power = wrqu->param.value;
	} else
		return -EINVAL;

	adm8211_rf_set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_set_lpfcutoff(struct net_device *dev,
				       struct iw_request_info *info,
				       union iwreq_data *wrqu,
				       char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (wrqu->param.value >= 0 && wrqu->param.value <= 0xFF) {
		priv->lpf_cutoff = wrqu->param.value;
	} else
		return -EINVAL;

	adm8211_rf_set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_set_lnagsthresh(struct net_device *dev,
					 struct iw_request_info *info,
					 union iwreq_data *wrqu,
					 char *extra)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	if (wrqu->param.value >= 0 && wrqu->param.value <= 0xFF) {
		priv->lnags_threshold = wrqu->param.value;
	} else
		return -EINVAL;

	adm8211_rf_set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_print_radio (struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
        struct adm8211_priv *priv = netdev_priv(dev);
	int channel = priv->ieee80211.channel - 1;

	printk(KERN_DEBUG "%s: Antenna Power: %d (%d) \n",
			dev->name, priv->ant_power,
			priv->eeprom->antenna_power[channel]);
	printk(KERN_DEBUG "%s: LPF Cutoff: %d (%d)\n",
			dev->name, priv->lpf_cutoff,
			priv->eeprom->lpf_cutoff[channel]);
	printk(KERN_DEBUG "%s: LNAGS Threshold: %d (%d)\n",
			dev->name, priv->lnags_threshold,
			priv->eeprom->lnags_threshold[channel]);
	
	return 0;
}

static const iw_handler adm8211_private_handler[] =
{							/* SIOCIWFIRSTPRIV + */
	(iw_handler) adm8211_ioctl_sreg,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_greg,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_print_eeprom,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_print_radio,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_country,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_antpower,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_lpfcutoff,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_lnagsthresh,
};

static const struct iw_priv_args adm8211_priv[] = {
	{SIOCIWFIRSTPRIV, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "setreg" },
	{0},
	{SIOCIWFIRSTPRIV+2, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "getreg" },
	{0},
	{SIOCIWFIRSTPRIV+4, IW_PRIV_TYPE_NONE, 0, "print_eeprom" },
	{0},
	{SIOCIWFIRSTPRIV+6, IW_PRIV_TYPE_NONE, 0, "print_radio" },
	{0},
	{SIOCIWFIRSTPRIV+8, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_country" },
	{0},
	{SIOCIWFIRSTPRIV+10, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_antpower" },
	{0},
	{SIOCIWFIRSTPRIV+12, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_lpfcutoff" },
	{0},
	{SIOCIWFIRSTPRIV+14, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_lnagsthresh" },
};

const struct iw_handler_def adm8211_iw_handler_def =
{
	.num_standard   = ARRAY_SIZE(adm8211_handler),
	.num_private    = ARRAY_SIZE(adm8211_private_handler),
	.num_private_args = ARRAY_SIZE(adm8211_priv),
	.standard       = (iw_handler *) adm8211_handler,
	.private        = (iw_handler *) adm8211_private_handler,
	.private_args   = (struct iw_priv_args *) adm8211_priv,
};

