/*
 *  Copyright (C) 2003 Carlos O'Donell
 * 
 *  2003-12-20  Carlos O'Donell
 *              Copied linux/kernel/compat_signal.c (copy_siginfo_to_user)
 *              and modified to use compat_siginfo_t for thunking down to
 *              32-bit userspace from a 64-bit kernel.
 *              
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *
 */

#include <linux/compat.h>

#include <asm/compat_siginfo.h>
#include <asm/errno.h>
#include <asm/uaccess.h>
#include <asm/siginfo.h>

int copy_siginfo_to_user32(compat_siginfo_t __user *to, siginfo_t *from)
{
	int err;
	compat_siginfo_t compat_from;	

	if (!access_ok (VERIFY_WRITE, to, sizeof(compat_siginfo_t)))
		return -EFAULT;
	
	/*
	 * If you change compat_siginfo_t structure *or* siginfo_t, 
	 * please be sure this code is fixed accordingly.
	 * It should never copy any pad contained in the structure
	 * to avoid security leaks, but must copy the generic
	 * 3 ints plus the relevant union member.
	 */

	/* Convert structure, don't leak anything in the copy */
	memset(&compat_from,'\0',sizeof(compat_siginfo_t));

        /* Always copy si_signo, si_errno, and si_code */
	compat_from.si_signo = (compat_int_t)(from->si_signo);
	compat_from.si_errno = (compat_int_t)(from->si_errno);
	/* si_code is only a (short) value, remove kernel bits. */
	compat_from.si_code = (short)(from->si_code);
        
	err = __put_user(compat_from.si_signo, &to->si_signo);
	err |= __put_user(compat_from.si_errno, &to->si_errno);
	err |= __put_user(compat_from.si_code, &to->si_code);

        /* siginfo_t came from userspace, so it is the right
         * size, no need for conversion
         */        
	if (from->si_code < 0) {
		return __copy_to_user(&to->_sifields._pad, 
                                      &from->_sifields._pad, 
                                      SI_COMPAT_PAD_SIZE)
			? -EFAULT : 0;
        }
	
	switch (from->si_code & __SI_MASK) {
	case __SI_KILL:
		compat_from.si_pid = (compat_pid_t)(from->si_pid);
		compat_from.si_uid = (__ARCH_SI_COMPAT_UID_T)(from->si_uid);
		err |= __put_user(compat_from.si_pid, &to->si_pid);
		err |= __put_user(compat_from.si_uid, &to->si_uid);
		break;
	case __SI_TIMER:
		compat_from.si_pid = (compat_timer_t)(from->si_tid);
		compat_from.si_overrun = (compat_int_t)(from->si_overrun);
		compat_from.si_ptr = (compat_uptr_t)((u64 __force)(from->si_ptr) & 0xffffffffUL);
		err |= __put_user(compat_from.si_tid, &to->si_tid);
		err |= __put_user(compat_from.si_overrun, &to->si_overrun);
		err |= __put_user(compat_from.si_ptr, &to->si_ptr);
		break;
	case __SI_POLL:
		compat_from.si_band = (__ARCH_SI_COMPAT_BAND_T)(from->si_band);
		compat_from.si_fd = (compat_int_t)(from->si_fd);
		err |= __put_user(compat_from.si_band, &to->si_band);
		err |= __put_user(compat_from.si_fd, &to->si_fd);
		break;
	case __SI_FAULT:
		compat_from.si_addr = (compat_uptr_t)((u64 __force)(from->si_addr) & 0xffffffffUL);
		err |= __put_user(compat_from.si_addr, &to->si_addr);
#ifdef __ARCH_SI_COMPAT_TRAPNO
		compat_from.si_trapno = (compat_int_t)(from->si_addr);
		err |= __put_user(compat_from.si_trapno, &to->si_trapno);
#endif
		break;
	case __SI_CHLD:
		compat_from.si_pid = (compat_pid_t)(from->si_pid);
		compat_from.si_uid = (__ARCH_SI_COMPAT_UID_T)(from->si_uid);
		compat_from.si_status = (compat_int_t)(from->si_status);
		compat_from.si_utime = (compat_clock_t)(from->si_utime);
		compat_from.si_stime = (compat_clock_t)(from->si_stime);
		err |= __put_user(compat_from.si_pid, &to->si_pid);
		err |= __put_user(compat_from.si_uid, &to->si_uid);
		err |= __put_user(compat_from.si_status, &to->si_status);
		err |= __put_user(compat_from.si_utime, &to->si_utime);
		err |= __put_user(compat_from.si_stime, &to->si_stime);
		break;
	case __SI_RT: /* This is not generated by the kernel as of now. */
	case __SI_MESGQ: /* But this is */
		compat_from.si_pid = (compat_pid_t)(from->si_pid);
		compat_from.si_uid = (__ARCH_SI_COMPAT_UID_T)(from->si_uid);
		compat_from.si_int = (compat_int_t)(from->si_int);
		compat_from.si_ptr = (compat_uptr_t)((u64 __force)(from->si_ptr) & 0xffffffffUL);
		err |= __put_user(compat_from.si_pid, &to->si_pid);
		err |= __put_user(compat_from.si_uid, &to->si_uid);
		err |= __put_user(compat_from.si_int, &to->si_int);
		err |= __put_user(compat_from.si_ptr, &to->si_ptr);
		break;
	default: /* this is just in case for now ... */
		compat_from.si_pid = (compat_pid_t)(from->si_pid);
		compat_from.si_uid = (__ARCH_SI_COMPAT_UID_T)(from->si_uid);
		err |= __put_user(compat_from.si_pid, &to->si_pid);
		err |= __put_user(compat_from.si_uid, &to->si_uid);
		break;
	}
	return err;
}

int copy_siginfo_from_user32(siginfo_t *to, compat_siginfo_t __user *from)
{
	int err;
        u64 scratch;

	if (!access_ok (VERIFY_READ, from, sizeof(compat_siginfo_t)))
		return -EFAULT;
	
	/*
	 * If you change compat_siginfo_t structure *or* siginfo_t, 
	 * please be sure this code is fixed accordingly.
	 */

        /* Always copy si_signo, si_errno, and si_code */
	err = __get_user(to->si_signo, &from->si_signo);
	err |= __get_user(to->si_errno, &from->si_errno);
	err |= __get_user(to->si_code, &from->si_code);
        
        /* siginfo_t came from userspace, so it is the right
         * size, no need for conversion
         */        
	if (to->si_code < 0) {
		return __copy_from_user(&to->_sifields._pad, 
                                        &from->_sifields._pad, 
                                        SI_COMPAT_PAD_SIZE)
			? -EFAULT : 0;
        }
	
	switch (to->si_code & __SI_MASK) {
	case __SI_KILL:
		err |= __get_user(to->si_pid, &from->si_pid);
		err |= __get_user(to->si_uid, &from->si_uid);
		break;
	case __SI_TIMER:
		err |= __get_user(to->si_tid, &from->si_tid);
		err |= __get_user(to->si_overrun, &from->si_overrun);
		err |= __get_user(scratch, &from->si_ptr);
                to->si_ptr = (u64 __user*)scratch;                
		break;
	case __SI_POLL:
		err |= __get_user(to->si_band, &from->si_band);
		err |= __get_user(to->si_fd, &from->si_fd);
		break;
	case __SI_FAULT:
		err |= __get_user(scratch, &from->si_addr);
                to->si_addr = (u64 __user*)scratch;
#ifdef __ARCH_SI_COMPAT_TRAPNO
		err |= __get_user(to->si_trapno, &from->si_trapno);
#endif
		break;
	case __SI_CHLD:
		err |= __get_user(to->si_pid, &from->si_pid);
		err |= __get_user(to->si_uid, &from->si_uid);
		err |= __get_user(to->si_status, &from->si_status);
		err |= __get_user(to->si_utime, &from->si_utime);
		err |= __get_user(to->si_stime, &from->si_stime);
		break;
	case __SI_RT: /* This is not generated by the kernel as of now. */
	case __SI_MESGQ: /* But this is */
		err |= __get_user(to->si_pid, &from->si_pid);
		err |= __get_user(to->si_uid, &from->si_uid);
		err |= __get_user(to->si_int, &from->si_int);
		err |= __get_user(scratch, &from->si_ptr);
                to->si_ptr = (u64 __user*)scratch;
		break;
	default: /* this is just in case for now ... */
		err |= __get_user(to->si_pid, &from->si_pid);
		err |= __get_user(to->si_uid, &from->si_uid);
		break;
	}
	return err;
}

int compat_copy_sigevent_from_user(sigevent_t *to, compat_sigevent_t __user *from)
{
	int err;
	u64 scratch;
	
	/* copy sigval_t sigev_value 
	 	int_t sival_int		(same)
	 	uptr_t sival_ptr	(32 vs 64)*/
	err = __get_user(to->sigev_value.sival_int, 
	    		 &from->sigev_value.sival_int);
	err |= __get_user(scratch, &from->sigev_value.sival_ptr);
	to->sigev_value.sival_ptr = (u64 __user *)scratch;
	
	/* copy int_t sigev_signo 	(same)*/
	err |= __get_user(to->sigev_signo, &from->sigev_signo);
	
	/* copy int_t sigev_notify	(same)*/
	err |= __get_user(to->sigev_notify, &from->sigev_notify);

	/* never copy _sigev_un padding */

	/* copy int_t _tid 		(same),
	   good_sigevent() uses this value of */
	err |= __get_user(to->sigev_notify_thread_id, &from->sigev_notify_thread_id);
	
	/* XXX: Do not copy these, they aren't used by
	   anyone. We would need to distinguish the uses of the union.
	   copy _sigev_thread
	  	uptr_t _function	(32 vs 64)
	  	uptr_t _attribute	(32 vs 64)*/
	
	return err;
}


long compat_sys_timer_create(clockid_t which_clock,
			     compat_sigevent_t __user *timer_event_spec,
			     compat_timer_t __user * created_timer_id)
{
	sigevent_t kevent;
	mm_segment_t old_fs = get_fs();
	long ret;

	if (timer_event_spec != NULL)
		if (get_compat_sigevent(&kevent, timer_event_spec) != 0)
			return -EFAULT;

	set_fs(KERNEL_DS);
	ret = sys_timer_create(which_clock, timer_event_spec ? (sigevent_t __user *)&kevent : NULL, created_timer_id);
	set_fs(old_fs);

	return ret;
}
