import unittest, gettext, os, linda, tempfile, shutil, gzip
from po_count import POFileParser
from linda.collector import Collector
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.parser.unixperm import UnixPermParser
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class DocumentationCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/documentation.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/binary_documentation.tar.gz' % \
            self.instance.information['dir'])
        os.mkdir('%s/unpacked' % self.instance.information['dir'])
        self.po_keys = POFileParser('po/en.po')
        
    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])
    
    def test_populate_files(self):
        "Check populate_files() method of DocumentationCheck"
        method = getattr(self.instance, 'populate_files')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/man/lala.html': '', \
            '/usr/share/man/man2/lala.2.gz': '', '/usr/sbin/lala': '', \
            '/usr/foobar': ''})
        method()
        self.assert_('lala' in self.instance.files['bin'])
        self.assert_('/usr/share/man/man2/lala.2.gz' in \
            self.instance.files['man'].keys())
        self.instance.files = {'man': {}, 'bin': [], 'doc': []}
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/python2.3': '', \
            '/usr/share/man/man1/python2.3.1.gz': ''})
        method()
        self.assert_('python2.3' in self.instance.files['bin'])
        
    def test_manual_page_checks(self):
        "Check manual_page_checks() method of DocumentationCheck"
        method = getattr(self.instance, 'manual_page_checks')
        self.instance.files['man']['/usr/man/lala.2'] = [os.path.split('/usr/man/lala.2'), os.path.splitext('/usr/man/lala.2')]
        method('/usr/man/lala.2')
        expected_err = Set(('manual-in-usr-man', 'manual-not-compressed'))
        self.assertEqual(expected_err.intersection(self.instance.err_dict.keys()), expected_err)
        for key in expected_err:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        os.makedirs('%s/unpacked/usr/share/man/man2' % \
            self.instance.information['dir'])
        os.symlink('/usr/share/man/man7/undocumented.7.gz', \
            '%s/unpacked/usr/share/man/man2/lala.2.gz' % \
            self.instance.information['dir'])
        self.instance.files['man']['/usr/share/man/man2/lala.2.gz'] = [os.path.split('/usr/share/man/man2/lala.2.gz'), os.path.splitext('/usr/share/man/man2/lala.2.gz')]
        method('/usr/share/man/man2/lala.2.gz')
        self.assert_('link-to-undocumented' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'link-to-undocumented'), True)
        shutil.rmtree('%s/unpacked/usr' % self.instance.information['dir'])
        
    def test_manual_page_file_checks(self):
        "Check manual_page_file_checks() method of DocumentationCheck"
        method = getattr(self.instance, 'manual_page_file_checks')
        os.makedirs('%s/unpacked/usr/share/man/man2' % \
            self.instance.information['dir'])
        f = gzip.open('%s/unpacked/usr/share/man/man2/lala.2.gz' % \
            self.instance.information['dir'], 'w')
        f.close()
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/man/man2/lala.2.gz': UnixPermParser('-rwxr--r--')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/man/man2/lala.2.gz': ''})
        method('/usr/share/man/man2/lala.2.gz')
        expected_err = Set(('executable-manual-page', \
            'manual-not-max-compression', 'manual-not-using-gzip'))
        method('/usr/share/man/man2/lala.2.gz')
        self.assertEqual(expected_err.intersection(self.instance.err_dict.keys()), expected_err)
        for key in expected_err:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        shutil.rmtree('%s/unpacked/usr' % self.instance.information['dir'])

    def test_call_man_parser(self):
        "Check call_man_parser() method of DocumentationCheck"
        method = getattr(self.instance, 'call_man_parser')
        os.makedirs('%s/unpacked/usr/share/man/man2' % \
            self.instance.information['dir'])
        os.symlink(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/share/man/man2/foobar.2.gz'), \
            os.path.join(self.instance.information['dir'], 'unpacked', 'usr', \
            'share/man/man2/fooish.2.gz'))
        method('/usr/share/man/man2/fooish.2.gz')
        self.assertEqual(self.instance.err_dict.keys(), [])
        f = gzip.open('%s/unpacked/usr/share/man/man2/lala.2.gz' % \
            self.instance.information['dir'], 'w')
        f.write('.SH NAME\ndiff \- compare files line by line\n')
        f.close()
        method('/usr/share/man/man2/lala.2.gz')
        self.assertEqual(self.instance.err_dict.keys(), [])
        f = open('%s/unpacked/usr/share/man/man2/lala.2' % \
            self.instance.information['dir'], 'w')
        f.write('..so ls.1')
        f.close()
        method('/usr/share/man/man2/lala.2')
        self.assertEqual(self.instance.err_dict.keys(), [])
        f = gzip.open('%s/unpacked/usr/share/man/man2/lala.2.gz' % \
            self.instance.information['dir'], 'w')
        f.write('.ab help2man is required to generate this page\n')
        f.close()
        method('/usr/share/man/man2/lala.2.gz')
        self.assert_('whatis-parse-failed-on-manpage' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'whatis-parse-failed-on-manpage'), True)
        shutil.rmtree('%s/unpacked/usr' % self.instance.information['dir'])

    def test_split_man(self):
        "Check split_man() method of DocumentationCheck"
        method = getattr(self.instance, 'split_man')
        self.instance.files['man']['/usr/share/man/man2/lala.2.gz'] = \
            [os.path.split('/usr/share/man/man2/lala.2.gz'), \
            os.path.splitext('/usr/share/man/man2/lala.2.gz')]
        method()
        self.assertEqual(self.instance.files['man']['/usr/share/man/man2/lala.2.gz'][2], 'lala')
        self.instance.files['man']['/usr/share/man/man2/lala.2'] = \
            [os.path.split('/usr/share/man/man2/lala.2'), \
            os.path.splitext('/usr/share/man/man2/lala.2')]
        method()
        self.assertEqual(self.instance.files['man']['/usr/share/man/man2/lala.2'][2], 'lala')
        self.instance.files['man']['/usr/share/man/man3/foobar.3pm.gz'] = \
            [os.path.split('/usr/share/man/man3/foobar.3pm.gz'), \
            os.path.splitext('/usr/share/man/man3/foobar.3pm.gz')]
        method()
        self.assertEqual(self.instance.files['man']['/usr/share/man/man3/foobar.3pm.gz'][2], 'foobar')

    def test_find_nonenglish_man(self):
        "Check find_nonenglish_man() method of DocumentationCheck"
        method = getattr(self.instance, 'find_nonenglish_man')
        self.instance.files['man'] = {'/usr/share/man/man1/lala.1.gz': ''}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.files['man']['/usr/share/man/de/man1/foobar.1.gz'] = ''
        method()
        self.assert_('only-nonenglish-man' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'only-nonenglish-man'), True)

    def test_check_man_bin(self):
        "Check check_man_bin() method of DocumentationCheck"
        method = getattr(self.instance, 'check_man_bin')
        self.instance.files = {'man': {'/usr/share/man/man2/lala.2.gz': [[], \
            [], 'lala']}, 'bin': ['lala', 'foobar']}
        method()
        self.assert_('no-manual-for-binary' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-manual-for-binary'), True)

    def test_check_doc_base(self):
        "Check check_doc_base() method of DocumentationCheck"
        method = getattr(self.instance, 'check_doc_base')
        self.instance.files['doc'] = []
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.makedirs('%s/unpacked/usr/share/doc-base' % \
            self.instance.information['dir'])
        os.makedirs('%s/unpacked/usr/share/info' % \
            self.instance.information['dir'])
        os.makedirs('%s/unpacked/usr/share/doc/zsh-doc/html' % \
            self.instance.information['dir'])
        os.rename('%s/zsh' % self.instance.information['dir'], \
            '%s/unpacked/usr/share/doc-base/zsh' % \
            self.instance.information['dir'])
        self.instance.files['doc'] = ['/usr/share/doc-base/zsh']
        files = ('unpacked/usr/share/info/zsh.info.gz', \
           'unpacked/usr/share/info/zsh.info.foo', \
           'unpacked/usr/share/doc/zsh-doc/html/zsh_toc.html', \
           'unpacked/usr/share/doc/zsh-doc/html/index.html')
        for file in files:
            f = open('%s/%s' % (self.instance.information['dir'], file), 'w')
            f.close()
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        shutil.rmtree('%s/unpacked/usr/share/doc' % \
            self.instance.information['dir'])
        shutil.rmtree('%s/unpacked/usr/share/info' % \
             self.instance.information['dir'])
        method()
        self.assert_('doc-base-file-not-found' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'doc-base-file-not-found'), True)
        os.unlink('%s/unpacked/usr/share/doc-base/zsh' % \
            self.instance.information['dir'])
        os.rename('%s/debconf-spec' % self.instance.information['dir'], \
            '%s/unpacked/usr/share/doc-base/debconf-spec' % \
            self.instance.information['dir'])
        self.instance.files['doc'] = ['/usr/share/doc-base/debconf-spec']
        method()
        self.assert_('doc-base-format-html-no-files' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'doc-base-format-html-no-files'), True)
        shutil.rmtree('%s/unpacked/usr' % self.instance.information['dir'])

    def test_check_devhelp(self):
        "Check check_devhelp() method of DocumentationCheck"
        method = getattr(self.instance, 'check_devhelp')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/foo/lala.devhelp': UnixPermParser('-rw-r--r--')})
        os.makedirs('%s/unpacked/usr/share/devhelp/books' % \
            self.instance.information['dir'])
        os.symlink('%s/unpacked/usr/share/doc/foo/lala.devhelp' % \
            self.instance.information['dir'], \
            '%s/unpacked/usr/share/devhelp/books/lala.devhelp' % \
            self.instance.information['dir'])
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        shutil.rmtree('%s/unpacked/usr' % self.instance.information['dir'])
        method()
        self.assert_('devhelp-file-not-in-dir' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'devhelp-file-not-in-dir'), True)

