import unittest, os, linda, tempfile, shutil
from po_count import POFileParser
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.parser.unixperm import UnixPermParser
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class ControlFilesCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/controlfiles.py', {'_': __builtins__['_'], 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][1]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/binary_controlfiles.tar.gz' % \
            self.instance.information['dir'])
        os.mkdir('%s/control' % self.instance.information['dir'])
        self.po_keys = POFileParser('po/en.po')
        
    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])
        
    def test_general(self):
        "Check general_checks() method of ControlFilesCheck"
        method = getattr(self.instance, 'general_checks')
        self.instance.information['control'] = {'info': {'postinst': \
            ['root/root', UnixPermParser('-rwxr-xr-x')]}}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['info'] = {'lala': \
            ['root/root', UnixPermParser('-rwxr-xr-x')]}
        method()
        self.assert_('unknown-control-file' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-control-file'), True)
        self.instance.information['control']['info'] = {'postinst': \
            ['steven/users', UnixPermParser('-rwxr-xr-x')]}
        method()
        self.assert_('incorrect-ownership-control' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'incorrect-ownership-control'), True)
        self.instance.information['control']['info']['postinst'] = \
            ['root/root', UnixPermParser('-rw-r--r--')]
        method()
        self.assert_('incorrect-control-perms' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'incorrect-control-perms'), True)

    def test_shebang(self):
        "Check shebang_calls() method of ControlFilesCheck"
        shutil.copy('%s/postinst.mainterror' % \
            self.instance.information['dir'], '%s/control/postinst' % \
            self.instance.information['dir'])
        method = getattr(self.instance, 'shebang_calls')
        method()
        self.assert_('possible-maint-error' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'possible-maint-error'), True)
        os.remove('%s/control/postinst' % self.instance.information['dir'])
        
    def test_conffiles(self):
        "Check conffiles() method of ControlFilesCheck"
        shutil.copy('%s/conffiles.duplicate' % \
            self.instance.information['dir'], '%s/control/conffiles' % \
            self.instance.information['dir'])
        method = getattr(self.instance, 'conffiles')
        method()
        self.assert_('duplicate-conffile' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'duplicate-conffile'), True)
        os.remove('%s/control/conffiles' % self.instance.information['dir'])

    def test_finding_things(self):
        "Check finding_things() method of ControlFilesCheck"
        shutil.copy('%s/postrm.fine' % self.instance.information['dir'], \
            '%s/control/postrm' % self.instance.information['dir'])
        method = getattr(self.instance, 'finding_things')
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.remove('%s/control/postrm' % self.instance.information['dir'])
        shutil.copy('%s/postrm.broken' % self.instance.information['dir'], \
            '%s/control/postrm' % self.instance.information['dir'])
        method()
        errors = Set(['bin-sh-e-in-maint', 'killall-in-maint', \
            'mknod-in-maint', 'dpkg-pa-in-maint', 'suid-in-maint', \
            'update-alt-in-postrm', 'insecure-tmp-handling', \
            'netbase-managed-in-maint', 'inetd-in-maint', \
            'multiple-update-rc.d', 'bashism-in-maint'])
        self.assertEqual(errors.intersection(self.instance.err_dict.keys()), \
             errors)
        for key in errors:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        os.remove('%s/control/postrm' % self.instance.information['dir'])

