import re, glob
from linda import libchecks, checks
from linda.funcs import explode_path, run_external_cmd, ExtCmdException
from linda.parser.template import TemplateParser

class DocumentationCheck(libchecks.LindaChecker):
    'Checks relating to manual and info pages.'
    def init(self):
        self.files = {'man': {}, 'bin': [], 'doc': []}
        
    def check_binary_2(self):
        self.populate_files()
        for file in self.files['man'].keys():
            self.manual_page_checks(file)
            self.manual_page_file_checks(file)
            self.call_man_parser(file)
        self.split_man()
        self.check_man_bin()
        self.find_nonenglish_man()
        self.check_doc_base()
        self.check_devhelp()
        
    def populate_files(self):
        for file in self.information['collector']('files', 'files').keys():
            path_split = os.path.split(file)
            ext_split = os.path.splitext(file)
            if path_split[0] in ('/bin', '/sbin', '/usr/bin', '/usr/sbin', \
                '/usr/X11R6/bin', '/usr/games'):
                self.files['bin'].append(self.kill_extensions(file))
            elif re.search(r'/usr/(share/|X11R6/)?man/', path_split[0]):
                if ext_split[1] not in ('.html', '.txt'):
                    self.files['man'][file] = [path_split, ext_split]
            elif path_split[0].endswith('doc-base'):
                self.files['doc'].append(file)
        dprint(_("Populated: Bin: %s; Man: %s") % (self.files['bin'], \
            self.files['man']), 2)
    def manual_page_checks(self, file):
        if file.startswith('/usr/man/'):
            self.signal_error('manual-in-usr-man', \
                [self.files['man'][file][0][1]])
        if self.files['man'][file][1][1] != '.gz':
            self.signal_error('manual-not-compressed', \
                [self.files['man'][file][0][1]])
        if os.path.islink(os.path.join(self.information['dir'], 'unpacked', \
            file[1:])):
            if os.readlink(os.path.join(self.information['dir'], 'unpacked', \
                file[1:])).find('undocumented.7') != -1:
                self.signal_error('link-to-undocumented', \
                    [self.files['man'][file][0][1]])
    def manual_page_file_checks(self, file):
        full_file = os.path.join(self.information['dir'], 'unpacked', file[1:])
        if os.path.isfile(full_file) and not os.path.islink(full_file):
            if self.information['collector']('files', 'files')[file].is_executable():
                self.signal_error('executable-manual-page', [file])
            if self.information['collector']('output', 'file')[file].find('max compression') == -1:
                self.signal_error('manual-not-max-compression', [file])
            if self.information['collector']('output', 'file')[file].find('gzip compressed') == -1:
                self.signal_error('manual-not-using-gzip', [file])
    def call_man_parser(self, file):
        if os.path.islink(os.path.join(self.information['dir'], 'unpacked', \
            file[1:])):
            return None
        man_page = open(os.path.join(self.information['dir'], 'unpacked', \
            file[1:]))
        for line in man_page:
            if line.startswith('..so '):
                return
        man_page.close()
        try:
            run_external_cmd('lexgrog %s' % \
                os.path.join(self.information['dir'], 'unpacked', file[1:]))
        except ExtCmdException:
            self.signal_error('whatis-parse-failed-on-manpage', [file])
    def split_man(self):
        for file in self.files['man'].keys():
            if self.files['man'][file][1][1] == '.gz':
                self.files['man'][file].append(os.path.split(self.files['man'][file][1][0])[1])
            else:
                self.files['man'][file].append(os.path.split(file)[1])
            self.files['man'][file][2] = self.kill_extensions(file)
        dprint(_("After man: %s") % self.files['man'], 2)
    def find_nonenglish_man(self):
        nonenglish_man = {}
        for file in self.files['man']:
            exploded_path = explode_path(file)
            if nonenglish_man.has_key(exploded_path[-1]):
                nonenglish_man[exploded_path[-1]].append(exploded_path[3])
            else:
                nonenglish_man[exploded_path[-1]] = [exploded_path[3]]
        dprint(_("Found manpages: %s") % nonenglish_man, 2)
        for file in nonenglish_man.keys():
            found_english = 0
            for lang in nonenglish_man[file]:
                if lang.startswith('man'):
                    found_english = 1
            if not found_english:
                self.signal_error('only-nonenglish-man', [file])
    def check_man_bin(self):
        for file in self.files['bin']:
            found = 0
            for manpage in self.files['man'].keys():
                if file == self.files['man'][manpage][2]:
                    found = 1
            if not found:
                self.signal_error('no-manual-for-binary', [file])
    def check_doc_base(self):
        for file in self.files['doc']:
            tmpl = TemplateParser(os.path.join(self.information['dir'], \
                'unpacked', file[1:]))
            for stanza in tmpl[1:]:
                if stanza.has_key('format'):
                    if stanza['format'] == 'HTML' and not \
                        stanza.has_key('files'):
                        self.signal_error('doc-base-format-html-no-files', \
                            [file[1:]])
                if stanza.has_key('index'):
                    if not os.path.exists(os.path.join(self.information['dir'], 'unpacked', stanza['index'][1:])):
                        self.signal_error('doc-base-file-not-found', \
                            ['index', file[1:]])
                if stanza.has_key('files'):
                    for f in stanza['files'].split(' '):
                        fp = os.path.join(self.information['dir'], \
                            'unpacked', f[1:])
                        if not glob.glob(fp) and not os.path.exists(fp):
                            self.signal_error('doc-base-file-not-found', \
                                ['Files', f])
    def check_devhelp(self):
        devhelp_dirs = ('/usr/share/gtk-doc/html', '/usr/share/devhelp/books')
        devhelp_found = {}
        for file in self.information['collector']('files', 'files'):
            path_split = os.path.split(file)
            if path_split[0] in devhelp_dirs:
                continue
            if file.endswith('.devhelp') or file.endswith('.devhelp.gz'):
                devhelp_found[path_split[1]] = 0
        for dir in devhelp_dirs:
            if not os.path.isdir('%s/unpacked%s' % (self.information['dir'], \
                dir)):
                continue
            for file in os.listdir('%s/unpacked%s' % (self.information['dir'], \
                dir)):
                if file in devhelp_found.keys():
                    devhelp_found[file] = 1
        for file in devhelp_found.keys():
            if not devhelp_found[file]:
                self.signal_error('devhelp-file-not-in-dir', [file])

    def kill_extensions(self, filename):
        path, file = os.path.split(filename)
        extensions = r'\.(p[ly]|sh)$'
        if path.find('/man/man') != -1:
            extensions = extensions[:-2] + '|gz|[1-8][a-z]*)$'
        while re.search(extensions, file):
            file = os.path.splitext(file)[0]
        return file

checks.register(DocumentationCheck)

