//
// Copyright (c) 2001-2003 The Trustees of Indiana University.  
//                         All rights reserved.
// Copyright (c) 1998-2001 University of Notre Dame. 
//                         All rights reserved.
// Copyright (c) 1994-1998 The Ohio State University.  
//                         All rights reserved.
// 
// This file is part of the LAM/MPI software package.  For license
// information, see the LICENSE file in the top level directory of the
// LAM/MPI source distribution.
// 
// $HEADER$
//
// $Id: laminfo.cc,v 1.13.2.2 2003/11/14 03:18:54 jsquyres Exp $
//

#include <lam_config.h>

#include <iostream>
#include <string>
#include <utility>

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/param.h>

#include <all_opt.h>
#include <etc_misc.h>
#include <vreq.h>
#include <priority.h>
#include <lam_build_info.h>
#include <lam-ssi.h>
#include <lam-ssi-boot.h>
#include <lam-ssi-coll.h>
#include <lam-ssi-rpi.h>
#include <lam-ssi-cr.h>

using namespace std;


//
// Local functions
//

static void do_version(OPT *ad);
static void show_lam_version(const string& scope);
static void show_boot_version(const lam_ssi_boot_t **boot_modules, 
			      const string& which, const string& scope,
			      const string& ver_type);
static void show_coll_version(const lam_ssi_coll_t **coll_modules, 
			      const string& which, const string& scope,
			      const string& ver_type);
static void show_rpi_version(const lam_ssi_rpi_t **rpi_modules, 
			     const string& which, const string& scope,
			     const string& ver_type);
static void show_cr_version(const lam_ssi_crlam_t **cr_modules, 
                             const string& which, const string& scope,
                             const string& ver_type);
static void show_ssi_version(const lam_ssi_t *ls, const string& kind,
			     const string& scope, const string& ver_type);
static string make_version_str(const string& scope,
			       int major, int minor, int release, 
			       int alpha, int beta, int svn);
static void do_path(OPT *ad);
static void show_path(const string& type, const string& value);
static void do_arch(OPT *ad);
static void do_config();
static void out(const string& pretty_message, const string& plain_message, 
                const string &value);


//
// Local variables
//

static bool pretty = true;
static int centerpoint = 18;

static string ver_full = "full";
static string ver_major = "major";
static string ver_minor = "minor";
static string ver_release = "release";
static string ver_alpha = "alpha";
static string ver_beta = "beta";
static string ver_svn = "svn";

static string ver_lam = "lam";
static string ver_boot = "boot";
static string ver_coll = "coll";
static string ver_rpi = "rpi";
static string ver_cr = "cr";

static string ver_all = "all";

static string ver_ssi = "ssi";
static string ver_kind = "kind";
static string ver_module = "module";

static string path_prefix = "prefix";
static string path_bindir = "bindir";
static string path_libdir = "libdir";
static string path_incdir = "incdir";
static string path_pkglibdir = "pkglibdir";
static string path_sysconfdir = "sysconfdir";


int
main(int argc, char *argv[])
{
  int ret;
  OPT *ad;
  bool acted = false;

  // Initialize the argv parsing handle

  ad = ao_init();
  if (ad == 0) {
    ret = errno;
    show_help(NULL, "lib-call-fail", "ao_init", NULL);
    exit(ret);
  }
  ao_setflags(ad, AOPRESERVE_ARGV);
  ao_setopt1(ad, "h", 0, 0, 0);

  // Actions

  ao_setopt(ad, "version", 0, 2, 0);
  ao_setopt(ad, "path", 0, 1, 0);
  ao_setopt(ad, "arch", 0, 0, 0);
  ao_setopt(ad, "config", 0, 0, 0);
  ao_setopt(ad, "help", 0, 0, 0);
  ao_setopt(ad, "pretty", "parable", 0, 0);
  ao_setopt(ad, "parsable", "pretty", 0, 0);
  ao_setopt(ad, "hostname", 0, 0, 0);

  // Do the parsing

  if (ao_parse(ad, &argc, argv) ||
      ao_taken(ad, "help") || ao_taken(ad, "h")) {
    show_help("laminfo", "usage", NULL);
    exit(EUSAGE);
  }

  // Execute the desired action(s)

  if (ao_taken(ad, "pretty"))
    pretty = true;
  else if (ao_taken(ad, "parsable"))
    pretty = false;

  if (ao_taken(ad, "version")) {
    do_version(ad);
    acted = true;
  }
  if (ao_taken(ad, "path")) {
    do_path(ad);
    acted = true;
  }
  if (ao_taken(ad, "arch")) {
    do_arch(ad);
    acted = true;
  }
  if (ao_taken(ad, "config")) {
    do_config();
    acted = true;
  }

  // If no command line args are specified, show [a concise] everything

  if (!acted) {
    show_lam_version(ver_full);
    show_path(path_prefix, LAM_PREFIX);
    do_arch(ad);
    do_config();
    show_boot_version(lam_ssi_boot_modules, ver_all, ver_full, ver_module);
    show_coll_version(lam_ssi_coll_modules, ver_all, ver_full, ver_module);
    show_rpi_version(lam_ssi_rpi_modules, ver_all, ver_full, ver_module);
    show_cr_version(lam_ssi_crlam_modules, ver_all, ver_full, ver_module);
  }

  // All done

  ao_free(ad);
  return 0;
}


static void
do_version(OPT *ad)
{
  int i, count;
  string arg1, scope, module;

  count = ao_ninsts(ad, "version");
  for (i = 0; i < count; ++i) {
    arg1 = ao_param(ad, "version", i, 0);
    scope = ao_param(ad, "version", i, 1);

    // Version of LAM/MPI
    
    if (arg1 == ver_lam)
      show_lam_version(scope);

    // Version of all boot modules or a specific boot module

    else if (arg1 == ver_boot) {
      show_boot_version(lam_ssi_boot_modules, ver_all, scope, ver_all);
    } else if (arg1.substr(0, ver_boot.length() + 1) == (ver_boot + ":")) {
      show_boot_version(lam_ssi_boot_modules, 
			arg1.substr(ver_boot.length() + 1), scope, ver_all);
    } 

    // Version of all coll modules or a specific coll module

    else if (arg1 == ver_coll) {
      show_coll_version(lam_ssi_coll_modules, ver_all, scope, ver_all);
    } else if (arg1.substr(0, ver_coll.length() + 1) == (ver_coll + ":")) {
      show_coll_version(lam_ssi_coll_modules, 
			arg1.substr(ver_coll.length() + 1), scope, ver_all);
    } 

    // Version of all rpi modules or a specific rpi module

    else if (arg1 == ver_rpi) {
      show_rpi_version(lam_ssi_rpi_modules, ver_all, scope, ver_all);
    } else if (arg1.substr(0, ver_rpi.length() + 1) == (ver_rpi + ":")) {
      show_rpi_version(lam_ssi_rpi_modules, 
		       arg1.substr(ver_rpi.length() + 1), scope, ver_all);
    } 

    // Version of all cr modules or a specific cr module
    // Note: we only list the compiled crlam modules. But the list of crmpi 
    // modules available should be the same as this list.

    else if (arg1 == ver_cr) {
      show_cr_version(lam_ssi_crlam_modules, ver_all, scope, ver_all);
    } else if (arg1.substr(0, ver_cr.length() + 1) == (ver_cr + ":")) {
      show_cr_version(lam_ssi_crlam_modules,
                      arg1.substr(ver_cr.length() + 1), scope, ver_all);
    }

    // Fall through

    else {
      show_help("laminfo", "usage");
      exit(1);
    }
  }
}


static void
show_lam_version(const string& scope)
{
  out("LAM/MPI", "version:" + ver_lam, 
      make_version_str(scope, 
                       LAM_MAJOR_VERSION, LAM_MINOR_VERSION, 
                       LAM_RELEASE_VERSION, 
                       LAM_ALPHA_VERSION, LAM_BETA_VERSION, LAM_SVN_VERSION));
}


static void 
show_boot_version(const lam_ssi_boot_t **boot_modules, 
		  const string& which, const string& scope, 
		  const string& ver_type)
{
  int i;
  const lam_ssi_t *ls;
  bool want_all_modules = (which == ver_all);

  for (i = 0; boot_modules[i] != NULL; ++i) {
    ls = &(boot_modules[i]->lsb_meta_info);
    if (!want_all_modules && which != ls->ssi_module_name)
      continue;

    show_ssi_version(ls, ver_boot, scope, ver_type);
  }
}


static void 
show_coll_version(const lam_ssi_coll_t **coll_modules, 
		  const string& which, const string& scope, 
		  const string& ver_type)
{
  int i;
  const lam_ssi_t *ls;
  bool want_all_modules = (which == ver_all);

  for (i = 0; coll_modules[i] != NULL; ++i) {
    ls = &(coll_modules[i]->lsc_meta_info);
    if (!want_all_modules && which != ls->ssi_module_name)
      continue;

    show_ssi_version(ls, ver_coll, scope, ver_type);
  }
}


static void 
show_rpi_version(const lam_ssi_rpi_t **rpi_modules, 
		  const string& which, const string& scope, 
		  const string& ver_type)
{
  int i;
  const lam_ssi_t *ls;
  bool want_all_modules = (which == ver_all);

  for (i = 0; rpi_modules[i] != NULL; ++i) {
    ls = &(rpi_modules[i]->lsr_meta_info);
    if (!want_all_modules && which != ls->ssi_module_name)
      continue;

    show_ssi_version(ls, ver_rpi, scope, ver_type);
  }
}


static void 
show_cr_version(const lam_ssi_crlam_t **cr_modules, 
                  const string& which, const string& scope, 
                  const string& ver_type)
{
  int i;
  const lam_ssi_t *ls;
  bool want_all_modules = (which == ver_all);

  for (i = 0; cr_modules[i] != NULL; ++i) {
    ls = &(cr_modules[i]->lscrl_meta_info);
    if (!want_all_modules && which != ls->ssi_module_name)
      continue;

    show_ssi_version(ls, ver_cr, scope, ver_type);
  }
}


static void 
show_ssi_version(const lam_ssi_t *ls, const string &kind,
		  const string& scope, 
		  const string& ver_type)
{
  bool printed;
  bool want_ssi = (ver_type == ver_all || ver_type == ver_ssi);
  bool want_kind = (ver_type == ver_all || ver_type == ver_kind);
  bool want_module = (ver_type == ver_all || ver_type == ver_module);
  string message, content;
  string ssi_version;
  string api_version;
  string module_version;
  string empty;

  ssi_version = make_version_str(scope, ls->ssi_major_version,
                                 ls->ssi_minor_version,
                                 ls->ssi_release_version, 0, 0, 0);
  api_version = make_version_str(scope, ls->ssi_kind_major_version,
                                 ls->ssi_kind_minor_version,
                                 ls->ssi_kind_release_version, 0, 0, 0);
  module_version = make_version_str(scope, ls->ssi_module_major_version,
                                    ls->ssi_module_minor_version,
                                    ls->ssi_module_release_version, 0, 0, 0);

  if (pretty) {
    message = "SSI " + kind;
    printed = false;

    content = ls->ssi_module_name + string(" (");
    if (want_ssi) {
      content += "SSI v" + ssi_version;
      printed = true;
    }
    if (want_kind) {
      if (printed)
        content += ", ";
      content += "API v" + api_version;
      printed = true;
    }
    if (want_module) {
      if (printed)
        content += ", ";
      content += "Module v" + module_version;
      printed = true;
    }
    out(message, empty, content + ")");
  } else {
    message = "ssi:" + kind + ":" + ls->ssi_module_name + ":version";
    if (want_ssi)
      out(empty, message, "ssi:" + ssi_version);
    if (want_kind)
      out(empty, message, "api:" + api_version);
    if (want_module)
      out(empty, message, "module:" + module_version);
  }
}


static string 
make_version_str(const string& scope,
		 int major, int minor, int release, int alpha, int beta, 
                 int svn)
{
  string str;
  char temp[BUFSIZ];

  temp[BUFSIZ - 1] = '\0';
  if (scope == ver_full) {
    snprintf(temp, BUFSIZ - 1, "%d.%d", major, minor);
    str = temp;
    if (release > 0) {
      snprintf(temp, BUFSIZ - 1, ".%d", release);
      str += temp;
    }
    if (alpha > 0) {
      snprintf(temp, BUFSIZ - 1, "a%d", alpha);
      str += temp;
    }
    else if (beta > 0) {
      snprintf(temp, BUFSIZ - 1, "b%d", beta);
      str += temp;
    }
    if (svn > 0) {
      str += "svn";
      if (svn > 1) {
        snprintf(temp, BUFSIZ - 1, "%d", svn);
        str += temp;
      }
    }
  } else if (scope == ver_major)
    snprintf(temp, BUFSIZ - 1, "%d", major);
  else if (scope == ver_minor)
    snprintf(temp, BUFSIZ - 1, "%d", minor);
  else if (scope == ver_release)
    snprintf(temp, BUFSIZ - 1, "%d", release);
  else if (scope == ver_alpha)
    snprintf(temp, BUFSIZ - 1, "%d", alpha);
  else if (scope == ver_beta)
    snprintf(temp, BUFSIZ - 1, "%d", beta);
  else if (scope == ver_svn)
    snprintf(temp, BUFSIZ - 1, "%d", svn);
  else {
    show_help("laminfo", "usage");
    exit(1);
  }

  if (str.empty())
    str = temp;

  return str;
}


static void
do_path(OPT *ad)
{
  int i, count;
  string scope;

  count = ao_ninsts(ad, "path");
  for (i = 0; i < count; ++i) {
    scope = ao_param(ad, "path", i, 0);

    if (scope == path_prefix)
      show_path(path_prefix, LAM_PREFIX);
    else if (scope == path_bindir)
      show_path(path_bindir, LAM_BINDIR);
    else if (scope == path_libdir)
      show_path(path_libdir, LAM_LIBDIR);
    else if (scope == path_incdir)
      show_path(path_incdir, LAM_INCDIR);
    else if (scope == path_pkglibdir)
      show_path(path_pkglibdir, LAM_PKGLIBDIR);
    else if (scope == path_sysconfdir)
      show_path(path_sysconfdir, LAM_SYSCONFDIR);
    else {
      show_help("laminfo", "usage");
      exit(1);
    }
  }
}


static void 
show_path(const string& type, const string& value)
{
  string pretty(type);
  pretty[0] &= toupper(pretty[0]);
  out(pretty, "path:" + type, value);
}


static void
do_arch(OPT *ad)
{
  string prefix;
  char hostname[MAXHOSTNAMELEN];

  if (ao_taken(ad, "hostname")) {
    gethostname(hostname, MAXHOSTNAMELEN);
    prefix = hostname + string(":");
  }
  out("Architecture", prefix + "arch", LAM_ARCH);
}


static void 
do_config()
{
  const string cxx(LAM_WANT_MPI2CPP ? "yes" : "no");
  const string fortran(LAM_WANT_FORTRAN ? "yes" : "no");
  const string romio(LAM_WANT_ROMIO ? "yes" : "no");
  const string impi(LAM_WANT_IMPI ? "yes" : "no");
  const string debug(LAM_WANT_DEBUG ? "yes" : "no");
  const string purify(LAM_DISINFECT ? "yes" : "no");
  const string cprofiling(LAM_WANT_PROFILE ? "yes" : "no");
  const string cxxprofiling(LAM_WANT_PROFILE ? "yes" : "no");
  const string fprofiling((LAM_WANT_PROFILE && LAM_WANT_FORTRAN) ?
                          "yes" : "no");

  out("Configured by", "config:user", LAM_CONFIGURE_USER);
  out("Configured on", "config:timestamp", LAM_CONFIGURE_DATE);
  out("Configure host", "config:host", LAM_CONFIGURE_HOST);

  out("C bindings", "bindings:c", "yes");
  out("C++ bindings", "bindings:cxx", cxx);
  out("Fortran bindings", "bindings:fortran", fortran);

  out("C profiling", "option:profiling:c", cprofiling);
  out("C++ profiling", "option:profiling:cxx", cxxprofiling);
  out("Fortran profiling", "option:profiling:fortran", fprofiling);

  out("ROMIO support", "option:romio", romio);
  out("IMPI support", "option:impi", impi);
  out("Debug support", "option:debug", debug);
  out("Purify clean", "option:purify", purify);
}


static void
out(const string& pretty_message, const string &plain_message,
    const string& value)
{
  if (pretty) {
    string spaces(LAM_max(centerpoint - pretty_message.length(), 0), ' ');
    cout << spaces << pretty_message << ": " << value << endl;
  } else {
    cout << plain_message << ":" << value << endl;
  }
}
