/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_coll_init_comm.c,v 1.7 2003/05/08 22:42:20 jsquyres Exp $
 *
 */

#include <lam_config.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <lam-ssi.h>
#include <lam-ssi-coll.h>
#include <etc_misc.h>
#include <lamdebug.h>
#include <typical.h>
#include <mpisys.h>


/*
 * Public variables
 */
char *lam_ssi_coll_base_override = NULL;


/*
 * Local functions
 */
static int check_module_name(MPI_Comm comm, char *name);
static int check_all_modules(MPI_Comm comm);
static int check_module(MPI_Comm comm, lam_ssi_coll_t *c,
                        const lam_ssi_coll_actions_t **actions);


/*
 * This function is called at the initialization time of every
 * communicator.  It is used to select which coll module will be
 * active for a given communicator.
 *
 * Note that this is not thread safe.  It's probably ok to just put a
 * lock around this function, though -- this is only called at the end
 * of a communicator constructor, so making a few threads wait while
 * we construct is probably ok. 
 */
int
lam_ssi_coll_base_init_comm(MPI_Comm comm)
{
  int i;
  int found;
  char name[BUFSIZ];
  char *module_name;

  /* Announce */

  if (lam_ssi_coll_verbose >= 10) {
    if (comm->c_name[0] != '\0')
      snprintf(name, sizeof(name), "%s (cid %d)", comm->c_name, 
	       comm->c_contextid);
    else
      snprintf(name, sizeof(name), "<no name> (cid %d)", comm->c_contextid);
    name[sizeof(name) - 1] = '\0';
    lam_debug(lam_ssi_coll_did, "init_comm: new communicator: %s", name);
  }

  /* WARNING: This will go away someday.  It is *only* here for a
     special case during MPI_Init of IMPI-enabled jobs.  It is not
     thread-safe, and will not be necessary when/if IMPI is
     re-architected.  So don't use it.  Ever. */

  if (lam_ssi_coll_base_override != NULL)
    i = check_module_name(comm, lam_ssi_coll_base_override);

  /* See if a specific module was requested by the magic keyval */

  else {
    MPI_Comm_get_attr(comm, LAM_MPI_SSI_COLL, &module_name, &found);
    if (found == 1)
      i = check_module_name(comm, module_name);
  
    /* Nope -- a specific one was not selected.  Go choose one. */
  
    else
      i = check_all_modules(comm);
  }
  
  /* If we have no collective modules available, it's an error.
     Thanks for playing! */

  if (i == LAMERROR) {
    if (lam_ssi_coll_verbose >= 10)
      lam_debug(lam_ssi_coll_did, "init_comm: No modules available!");
    show_help("ssi-coll", "none-available", NULL);
    return LAMERROR;
  }

  /* Otherwise, announce the winner */
  
  if (lam_ssi_coll_verbose > 0)
    lam_debug(lam_ssi_coll_did, "init_comm: Selected coll module %s", 
	      lam_ssi_coll_modules[i]->lsc_meta_info.ssi_module_name);
  
  return 0;
}


/*
 * A specific module was selected on the command line.  If a module by
 * that name is found, call its open (if it exists) and init
 * functions.  If they both return happiness, that module is selected.
 * Otherwise, call its finalize and close functions and return an
 * error (i.e., don't try to find any other available modules).
 */
static int
check_module_name(MPI_Comm comm, char *name)
{
  lam_ssi_module_t *module;
  lam_ssi_coll_t *c;
  const lam_ssi_coll_actions_t *actions = NULL;

  /* Find the target module by its name */

  for (module = al_top(lam_ssi_coll_base_available); module != NULL;
       module = al_next(lam_ssi_coll_base_available, module)) {
    c = (lam_ssi_coll_t*) module->lsm_module;
    if (strcmp(c->lsc_meta_info.ssi_module_name, name) == 0) {

      /* Found it.  Now check and see if that module wants to run */

      if (check_module(comm, c, &actions) >= 0) {

	/* Otherwise, we have a winner.  Assign all the function
	   pointers in the comm to that module, and call its init
	   function. */
	
	comm->c_ssi_coll = *actions;
	actions = NULL;
	if (comm->c_ssi_coll.lsca_init(comm, &actions) != 0)
	  return LAMERROR;
	if (actions != NULL)
	  comm->c_ssi_coll = *actions;
        return 0;
      }

      /* We found the right module, but it didn't want to run.  Too
	 bad.  So sad. */

      break;
    }
  }

  return LAMERROR;
}


/*
 * Call open on all the available modules (if it exists).  If open
 * returns happiness (or doesn't exist), call the init function to see
 * if that module wants to run.  If it does, do the priority matchup
 * and remember the one with the best priority.  Fill the global
 * structs with the data from the winner.  Call finalize and close on
 * all the losers that we invoked initialize on.
 *
 * Return LAMERROR if there are no modules available.
 */
static int 
check_all_modules(MPI_Comm comm)
{
  int priority, best_priority = -1;
  lam_ssi_module_t *module;
  lam_ssi_coll_t *c;
  const lam_ssi_coll_actions_t *cur, *best = NULL;

  /* Call the query function in every collective module and see if
     they want to run on this communicator */

  for (best_priority = -1, 
         module = al_top(lam_ssi_coll_base_available); module != NULL;
       module = al_next(lam_ssi_coll_base_available, module)) {
    c = (lam_ssi_coll_t*) module->lsm_module;
    priority = check_module(comm, c, &cur);
    if (priority > best_priority) {
      best_priority = priority;
      best = cur;
    }
  }

  /* If we didn't find any available modules, return an error */

  if (best_priority == -1)
    return LAMERROR;

  /* Otherwise, we have a winner.  Assign all the function pointers in
     the comm to that module, and call its init function. */

  comm->c_ssi_coll = *best;
  best = NULL;
  if (comm->c_ssi_coll.lsca_init(comm, &best) != 0)
    return LAMERROR;

  if (best != NULL)
    comm->c_ssi_coll = *best;

  return 0;
}


/*
 * Check a single module
 */
static int 
check_module(MPI_Comm comm, lam_ssi_coll_t *c,
             const lam_ssi_coll_actions_t **actions)
{
  int priority = -1;

  if (c->lsc_query != NULL &&
      (*actions = c->lsc_query(comm, &priority)) != NULL) {
    priority = LAM_min(priority, 100);
    if (lam_ssi_coll_verbose >= 10)
      lam_debug(lam_ssi_coll_did, 
		"init_comm: module available: %s, priority: %d", 
		c->lsc_meta_info.ssi_module_name, priority);
  }

  return priority;
}
