/***************************************************************************
 *
 * knetworkmanager-connection_store.h - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <Helmut.Schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <qtimer.h>
#include <dbus/qdbusobjectpath.h>

#include <kstaticdeleter.h>
#include <kapplication.h>

#include "knetworkmanager-connection_store.h"
#include "knetworkmanager-connection.h"
#include "knetworkmanager-connection_setting.h"
#include "knetworkmanager-connection_setting_info.h"
#include "knetworkmanager-connection_setting_ipv4.h"
#include "knetworkmanager-connection_setting_wired.h"
#include "knetworkmanager-connection_setting_wireless.h"
#include "knetworkmanager-connection_setting_wireless_security.h"

using namespace ConnectionSettings;

static KStaticDeleter<ConnectionStore> sd;

ConnectionStore* ConnectionStore::_instance = NULL;

ConnectionStore*
ConnectionStore::getInstance()
{
	if (_instance)
		return _instance;
	return sd.setObject(_instance, new ConnectionStore());
}

ConnectionStore::ConnectionStore()
	: _connectionList()
{
	// defer the connection init a bit
	QTimer::singleShot(0, this, SLOT(slotInit()));
}

ConnectionStore::~ConnectionStore()
{
	for (QValueList<Connection*>::Iterator it = _connectionList.begin(); it != _connectionList.end(); ++it)
	{
		delete (*it);
		*it = NULL;
	}
}

void
ConnectionStore::slotInit()
{

}

void
ConnectionStore::slotSecretsNeeded(Connection* conn, ConnectionSetting* setting, const QStringList& hints, bool request_new)
{
	printf("ConnectionStore::slotSecretsNeeded\n");
	emit SecretsNeeded(conn, setting, hints, request_new);
}

void
ConnectionStore::addConnection(Connection* con)
{
	if (_connectionList.find(con) != _connectionList.end())
	{
		con->slotUpdated();
		emit signalConnectionUpdated(con);
		return;
	}

	// check if the connection has an ID already
	if (con->getID().isEmpty())
	{
		// set unique id for this connection
		con->setID(createNewConnectionID());
	}

	// we own the connection now
	_connectionList.append(con);

	// let's get notified if the connection asks for new secrets
	connect(con, SIGNAL(SecretsNeeded(Connection*, ConnectionSetting*, const QStringList&, bool)),
	        this, SLOT(slotSecretsNeeded(Connection*, ConnectionSetting*, const QStringList&, bool)));

	// new connection, ring ring
	emit signalConnectionAdded(con);
}

void
ConnectionStore::removeConnection(Connection* con)
{
	// check if the connection is != NULL
	if (!con)
		return;

	// remove the object from our list
	if (_connectionList.remove(con) == 0)
		return;

	// notify everybody about the removed connection
	emit signalConnectionRemoved(con);

	// delete the connection, it will deregister itself from NM
	con->slotAboutToBeRemoved();
	delete con;
}

Connection*
ConnectionStore::getConnection(const QString& obj_path)
{
	// find a connection by obj_path
	QValueList<Connection*>::Iterator it = _connectionList.begin();
	for (; it != _connectionList.end(); ++it)
		if (QString((*it)->getObjectPath()) == obj_path)
			return (*it);
	return NULL;
}

QValueList<Connection*>
ConnectionStore::getConnections(const QString& type)
{
	// return a list of connection
	if (type.isEmpty())
		return _connectionList;
	else
	{
		QValueList<Connection*> ret;
		for (QValueList<Connection*>::Iterator it = _connectionList.begin(); it != _connectionList.end(); ++it)
		{
			if ( (*it)->getType() == type)
				ret.append(*it);
		}
		return ret;
	}
}

QString
ConnectionStore::createNewConnectionID()
{
	bool unique;
	QString id;

	do
	{
		unique = true;

		// 16 bytes of randomness should be enougth for getting a unique ID
		id = KApplication::randomString( 16 );

		// nevertheless check if the id is used already
		QValueList<Connection*>::Iterator it = _connectionList.begin();
		for (; it != _connectionList.end(); ++it)
		{
			if ((*it)->getID() == id)
			{
				unique = false;
				break;
			}
		}
	} while (unique == false);

	return id;
}

#include "knetworkmanager-connection_store.moc"
