/***************************************************************************
 *
 * knetworkmanager-devicestore_dbus.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

/* qt headers */
#include <qhostaddress.h>
#include <qvariant.h>
#include <qregexp.h>

/* kde headers */
#include <kdebug.h>
#include <klocale.h>

/* QDBus headers*/
#include <dbus/qdbusdata.h>
#include <dbus/qdbusdatamap.h>

/* knetworkmanager headers */
#include "knetworkmanager.h"
#include "knetworkmanager-connection_setting_wireless.h"
#include "knetworkmanager-connection_setting_wireless_security.h"
#include "knetworkmanager-accesspoint.h"

// the bssid should look like XX:XX:XX:XX:XX:XX where X is a hexadecimal digit
#define MAC_ADDRESS_PATTERN "[0-9A-Fa-f]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}"

using namespace ConnectionSettings;

/*
	class Wireless
*/
Wireless::Wireless(Connection* conn, ::AccessPoint* ap, WirelessSecurity* security)
	: ConnectionSetting(conn, NM_SETTING_WIRELESS_SETTING_NAME)
{
	_security = QString::null;

	if (ap)
	{
		_essid = ap->getSsidByteArray();
		_mode = ap->getMode() == 0 ? MODE_ADHOC : MODE_INFRASTRUCTURE;
	}
	else
	{
		_essid = QByteArray();
		_mode = MODE_INFRASTRUCTURE;
	}

	_modeMap[MODE_INFRASTRUCTURE] = "infrastructure";
	_modeMap[MODE_ADHOC] = "adhoc";
}

SettingsMap
Wireless::toMap() const
{
	SettingsMap map;

	map.insert(NM_SETTING_WIRELESS_MODE, QDBusData::fromString(_modeMap[_mode]));

	QValueList<QDBusData> essid;
	for (QByteArray::ConstIterator it = _essid.begin(); it != _essid.end(); ++it)
		essid.append(QDBusData::fromByte(*it));

	if (essid.size() > 0)
		map.insert(NM_SETTING_WIRELESS_SSID, QDBusData::fromQValueList(essid));
	else
		kdWarning() << k_funcinfo << " SSID undefined" << endl;

	if (!_security.isEmpty())
		map.insert(NM_SETTING_WIRELESS_SEC, QDBusData::fromString(_security));

	if (!_seenBssids.empty())
	{
		QValueList<QDBusData> bssids;
		for (QValueList<QString>::ConstIterator it = _seenBssids.begin(); it != _seenBssids.end(); ++it)
			bssids.append(QDBusData::fromString(*it));

		map.insert(NM_SETTING_WIRELESS_SEEN_BSSIDS, QDBusData::fromQValueList(bssids));
	}

	return map;
}

void
Wireless::fromMap(const SettingsMap& map)
{
	SettingsMap::ConstIterator it;

	// Mode
	if ((it = map.find(NM_SETTING_WIRELESS_MODE)) != map.end())
	{
		QBiDirectionalMap<MODES, QString>::Iterator mode_it = _modeMap.findData(it.data().toString());
		if (mode_it != _modeMap.end())
			_mode = mode_it.key();
		else
			_mode = MODE_INFRASTRUCTURE;
	}

	// Essid
	if ((it = map.find(NM_SETTING_WIRELESS_SSID)) != map.end())
	{
		QValueList<QDBusData> dbus_essid = (*it).toQValueList();
		QByteArray essid(dbus_essid.size());

		int index = 0;
		for (QValueList<QDBusData>::ConstIterator byte_it = dbus_essid.begin(); byte_it != dbus_essid.end(); ++byte_it)
		{
			essid[index] = (*byte_it).toByte();
			index++;
		}
		_essid = essid;
	}

	if ((it = map.find(NM_SETTING_WIRELESS_SEC)) != map.end())
	{
		_security = (*it).toString();
	}

	// Seen BSSIDS
	if ((it = map.find(NM_SETTING_WIRELESS_SEEN_BSSIDS)) != map.end())
	{
		QRegExp exp(MAC_ADDRESS_PATTERN);
		QValueList<QDBusData> bssids = (*it).toQValueList();
		_seenBssids.clear();

		for(QValueList<QDBusData>::Iterator it = bssids.begin(); it != bssids.end(); ++it)
		{
			QString bssid = (*it).toString();
			if (exp.exactMatch(bssid))
				_seenBssids.append(bssid);
		}
	}
}

void
Wireless::setEssid(const QByteArray& essid)
{
	_essid = essid;
	emitValidityChanged();
}

QByteArray
Wireless::getEssid(void) const
{
	return _essid;
}

void
Wireless::setMode(MODES mode)
{
	_mode = mode;
	emitValidityChanged();
}

Wireless::MODES
Wireless::getMode(void) const
{
	return _mode;
}

void
Wireless::setSecurity(const QString& security)
{
	_security = security;
	emitValidityChanged();
}

QString
Wireless::getSecurity(void) const
{
	return _security;
}

bool
Wireless::isValid() const
{
	// ESSID is essential
	if (_essid.isEmpty())
		return false;

	return true;
}

void
Wireless::addSeenBssid(const QString& bssid)
{
	QRegExp exp(MAC_ADDRESS_PATTERN);
	if (!exp.exactMatch(bssid))
		return;

	// no duplicates please
	for(QValueList<QString>::Iterator it = _seenBssids.begin(); it != _seenBssids.end(); ++it)
	{
		if ((*it) == bssid)
			return;
	}
	
	// insert this bssid
	_seenBssids.append(bssid);
	emitValidityChanged();
}
