/***************************************************************************
 *
 * knetworkmanager-nminfo_dbus.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <stdlib.h>
#include <kdebug.h>

#include <NetworkManager/NetworkManagerVPN.h>

#include "knetworkmanager.h"
#include "knetworkmanager-network.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-nminfo_dbus.h"

KNetworkManager* NetworkManagerInfoDBus::_ctx = NULL;
DBusMessage*     NetworkManagerInfoDBus::_msg = NULL;

void
NetworkManagerInfoDBus::sendKeyForNetwork (Network* net)
{
	DBusConnection* con   = _ctx->getDBus ()->getConnection ();
	DBusMessage*    reply = NULL;
	DBusMessageIter iter;
	Encryption*     enc   = net->getEncryption ();
	const char *    essid = net->getEssid ();

	if (!con || !net || !enc) {
		return;
	}

	reply = dbus_message_new_method_return (_msg);
	dbus_message_unref (_msg);

	dbus_message_iter_init_append (reply, &iter);
	if (enc->serialize (reply, essid) == true) {
		dbus_connection_send (con, reply, NULL);
	} else {
		printf ("Serialization of encryption payload failed (key reply).\n");
	}

	dbus_message_unref (reply);
	
	return;
}

void
NetworkManagerInfoDBus::sendGetKeyError ()
{
	DBusConnection* con   = _ctx->getDBus ()->getConnection ();
	DBusMessage*    reply = NULL;

	if (!con) {
		return;
	}

	reply = dbus_message_new_error (_msg, NMI_DBUS_INTERFACE ".GetKeyError", NULL);
	dbus_message_unref (_msg);

	dbus_connection_send (con, reply, NULL);

	dbus_message_unref (reply);

	return;
}

DBusMessage*
NetworkManagerInfoDBus::getKeyForNetwork (DBusMessage* msg)
{	
	char* obj_path = NULL;
	char* net_path = NULL;
	char* essid    = NULL;
	int   attempt  = -1;
	bool  new_key  = false;

	_msg = msg;
	dbus_message_ref (_msg);

	if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
				              DBUS_TYPE_OBJECT_PATH, &net_path,
					      DBUS_TYPE_STRING,      &essid,
					      DBUS_TYPE_INT32,       &attempt,
					      DBUS_TYPE_BOOLEAN,     &new_key, DBUS_TYPE_INVALID)) {
		/* visible networks which are encrypted but no key is available */
		/* TODO: hidden essid, too? */
		NetworkManagerInfo* nmi = _ctx->getNetworkManagerInfo ();
		nmi->acquirePassphrase (obj_path, net_path, essid, new_key);
	}

	return NULL;
}

void
NetworkManagerInfoDBus::requestName (DBusMessage* /*msg*/)
{
	DBusConnection* con = _ctx->getDBus ()->getConnection ();
	DBusError       error;

	if (!con) {
		return;
	}

	dbus_error_init (&error);
	if (dbus_bus_name_has_owner (con, NMI_DBUS_SERVICE, &error)) {
		printf ("%s already owned.\n", NMI_DBUS_SERVICE);
		goto out;
	}

	dbus_error_init (&error);
	dbus_bus_request_name (con, NMI_DBUS_SERVICE, DBUS_NAME_FLAG_DO_NOT_QUEUE, &error);
	if (dbus_error_is_set (&error)) {
		printf ("Error requesting name, %s: %s\n", error.name, error.message);
		dbus_error_free (&error);
		goto out;
	} else {
		goto out;
	}

out:
	
	return;
}

DBusMessage *
NetworkManagerInfoDBus::getNetworksMessage( DBusMessage* msg )
{
	DBusConnection* con   = _ctx->getDBus ()->getConnection ();
	if ( !con )
		return 0;

	DBusMessage*    reply = NULL;
	DBusMessageIter iter;
	DBusMessageIter iter_array;
	NetworkManagerInfo* nmi = _ctx->getNetworkManagerInfo ();
	QStringList networks = nmi->getNetworks();

	if (networks.count () == 0) {
		reply = dbus_message_new_error (msg, NMI_DBUS_INTERFACE ".NoNetworks", NULL);
		return reply;
	}

	reply = dbus_message_new_method_return (msg); 
	dbus_message_iter_init_append (reply, &iter); 
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);

	QStringList::Iterator it = networks.begin();
	const QStringList::Iterator end = networks.end();
	for ( ; it != end; ++it )
	{
		kdDebug() << "returning network: " << *it << endl;
		const char * network = strdup( (*it).utf8() );
		dbus_message_iter_append_basic(&iter_array, DBUS_TYPE_STRING, &network );
	}
	dbus_message_iter_close_container (&iter, &iter_array);

	return reply;
}

DBusMessage *
NetworkManagerInfoDBus::getNetworkProperties( DBusMessage * msg )
{
	// just retrieve the KConfig data from NMI
	// put it into a DBusMessage and return that
	// error with "BadNetworkData" if network was not found
	// string ESSID, int32 Timestamp, bool trusted, string[] bssid's, AP security info (serialized from Encryption? 
	
	DBusMessage*    reply = 0;
	// get the essid from the msg
	char * essid;
	if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &essid, DBUS_TYPE_INVALID)) {
		kdDebug() << "  returning properties for " << essid << endl;
		bool networkAdded = false;
		DBusConnection* con   = _ctx->getDBus ()->getConnection ();
		if ( !con )
			return 0;

		NetworkManagerInfo* nmi = _ctx->getNetworkManagerInfo ();
		Network * net = nmi->getNetworkProperties( essid );
		if ( !net )
			return 0;

		DBusMessageIter iter;
		DBusMessageIter iter_array;
	
		reply = dbus_message_new_method_return (msg); 
		dbus_message_iter_init_append (reply, &iter); 

		Q_INT32 timestamp = net->getTimestamp().toTime_t();
		if ( timestamp == -1 )
		{
			kdDebug() << "invalid timestamp " << ( net->getTimestamp().isValid() ? " valid" : " not valid" ) << endl;
			timestamp = QDateTime::currentDateTime().toTime_t();
		}
		bool trusted = net->isTrusted();
		//TODO: add serialise method to put the encryption into a given dbus message iterator
// 		applet-dbus-info.c, nmi_dbus_get_network_properties()
// 		+string essid
// 		+int32 timestamp
// 		+bool trusted
// 		+string list bssids
// 		THEN
// 		nm-gconf-wso.c,*nm_gconf_wso_serialize_dbus* does
// 		+iter-appends int32 we_cipher
// 		THEN
// 		nm-gconf-wso-wep.c, real_serialize_dbus does
// 		dbus-helpers.c *nmu_security_serialize_wep* with iter, key, auth alg
// 		+string *hashed* key - interestingly, nmu_security_serialize_wep_with_cipher hashes the key before appending it using ieee_802_11_cipher_hash()

		// look in nm_gconf_wso_serialize_dbus in nm-gconf-wso.c
		// first they serialize the cipher (is the above fn called for all crypto, even unencrypted nets?
		// then that calls serialize_dbus_func which is a ptr to eg real_serialize_dbus in gnome/applet/nm-gconf-wso-wep.c
		
		dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &essid);
		dbus_message_iter_append_basic (&iter, DBUS_TYPE_INT32, &timestamp);
		dbus_message_iter_append_basic (&iter, DBUS_TYPE_BOOLEAN, &trusted);
		dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);
		// the list of hardware addresses
		QStringList hwAddresses = net->getHardwareAddresses();
		QStringList::Iterator it = hwAddresses.begin();
		const QStringList::Iterator end = hwAddresses.end();
		for ( ; it != end; ++it )
		{
			kdDebug() << "adding AP: " << *it << endl;
			const char * ap = strdup( (*it).utf8() );
			dbus_message_iter_append_basic(&iter_array, DBUS_TYPE_STRING, &ap );
			networkAdded = true;
		}
		dbus_message_iter_close_container (&iter, &iter_array);
		if (networkAdded == false) {
			kdDebug() << "no hardware addresses found, sending error!" << endl;
			dbus_message_unref (reply);
			reply = dbus_message_new_error (msg, NMI_DBUS_INTERFACE ".NoNetworks", NULL);
			return reply;
		}
		if ( !net->getEncryption()->serialize( reply, net->getEssid() ) ) {
			kdDebug() << "couldn't serialize security!" << endl;
			dbus_message_unref (reply);
			reply = dbus_message_new_error (msg, NMI_DBUS_INTERFACE ".NoSecurity", NULL);
		}
		delete net;
	}
	return reply;
}

void NetworkManagerInfoDBus::updateNetworkInfo( DBusMessage * msg )
{
	DBusMessageIter iter;
	char * essid, * bssid;
	bool automatic;
	Q_INT32 we_cipher;
	DeviceStore *store = _ctx->getDeviceStore ();
	Device *dev = store->getActiveDevice ();
	Network *active_network = store->getActiveNetwork (dev);

	if (!active_network) {
		kdWarning() << k_funcinfo << "No active network found, nothing to update." << endl;
		return;
	}

	Encryption *encryption = active_network->getEncryption();

	dbus_message_iter_init (msg, &iter);
	for ( int i = 0; i < 1; i++ ) // one shot loop for
	{
		if (dbus_message_iter_get_arg_type (&iter) != DBUS_TYPE_STRING)
		{
			kdWarning() << "Message format is invalid" << endl;
			break;
		}
			dbus_message_iter_get_basic (&iter, &essid);
		if (strlen (essid) <= 0)
		{
			kdWarning() << "essid was empty" << endl;
			break;
		}
		if (!dbus_message_iter_next (&iter) || (dbus_message_iter_get_arg_type (&iter) != DBUS_TYPE_BOOLEAN))
		{
			kdWarning() << "bool message argument 'automatic' was invalid or missing." << endl;
			break;
		}
		dbus_message_iter_get_basic (&iter, &automatic);
		if (!dbus_message_iter_next (&iter) || (dbus_message_iter_get_arg_type (&iter) != DBUS_TYPE_STRING))
		{
			kdWarning() << "string message argument 'bssid' was invalid or missing." << endl;
			break;
		}
		dbus_message_iter_get_basic (&iter, &bssid);
		if (!dbus_message_iter_next (&iter) || (dbus_message_iter_get_arg_type (&iter) != DBUS_TYPE_INT32))
		{
			kdWarning() << "int32 message argument 'we_cipher' was invalid or missing." << endl;
			break;
		}
		dbus_message_iter_get_basic (&iter, &we_cipher);

		active_network->setEssid (QString::fromUtf8(essid));
		active_network->insertHardwareAddress (bssid);
		encryption->setWeCipher (we_cipher);
		
		if (!encryption->deserialize (&iter, we_cipher)) {
			kdWarning() << k_funcinfo << "Failed to deserialize security. Probably we_ciphers are different." << endl;
		}

		// pass to NMI to have it stored
		NetworkManagerInfo* nmi = _ctx->getNetworkManagerInfo ();
		nmi->emitNetworkUpdated (active_network, automatic);
	}
}

DBusMessage*
NetworkManagerInfoDBus::getVPNConnections (DBusMessage* msg)
{
	NetworkManagerInfo* nmi   = _ctx->getNetworkManagerInfo ();
	QStringList         names = nmi->getVPNConnectionNames ();
	DBusMessage*        reply = NULL;
	DBusMessageIter     iter;
	DBusMessageIter     iter_array;

	if (names.count () == 0) {
		reply = dbus_message_new_error (msg, NM_DBUS_NO_VPN_CONNECTIONS, "No VPN connections stored.");
		return reply;
	}

	reply = dbus_message_new_method_return (msg); 
	dbus_message_iter_init_append (reply, &iter); 
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);
	for (QStringList::Iterator i = names.begin (); i != names.end (); ++i) {
		const char* name = strdup ((*i).utf8 ());
		dbus_message_iter_append_basic (&iter_array, DBUS_TYPE_STRING, &name);
		kdDebug () << "returning vpn connection " << name << endl;
	}
	dbus_message_iter_close_container (&iter, &iter_array);

	return reply;
}

DBusMessage*
NetworkManagerInfoDBus::getVPNConnectionProperties (DBusMessage* msg)
{
	NetworkManagerInfo* nmi           = _ctx->getNetworkManagerInfo ();
	VPNConnection*      vpnConnection = NULL;
	const char*         name          = NULL;
	const char*         user          = NULL;
	const char*         service       = NULL;
	DBusMessage*        reply         = NULL;
	DBusMessageIter     iter;

	if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID) == false) {
		printf ("Error reading VPN connection name from D-BUS message.\n");
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "Bad arguments");
		goto out;
	}

	if ((vpnConnection = nmi->getVPNConnection (name)) == NULL) {
		printf ("Did not find requested VPN connection \"%s\".\n", name);
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "VPN connection not found");
		goto out;
	}
	
	name    = vpnConnection->getName        ();
	service = vpnConnection->getServiceName ();
	user    = vpnConnection->getUser        ();

	kdDebug () << "returning properties for vpn connection " << endl \
		   << "- name: "    <<  name    << endl \
		   << "- service: " <<  service << endl \
		   << "- user: "    <<  user    << endl;
	
	reply = dbus_message_new_method_return (msg);
	dbus_message_iter_init_append (reply, &iter);
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &name);
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &service);
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &user);

out:
	return reply;
}

DBusMessage*
NetworkManagerInfoDBus::getVPNConnectionRoutes (DBusMessage* msg)
{
	NetworkManagerInfo* nmi           = _ctx->getNetworkManagerInfo ();
	VPNConnection*      vpnConnection = NULL;
	const char*         name          = NULL;
  	QStringList         routes;
	DBusMessage*        reply         = NULL;
	DBusMessageIter     iter;
	DBusMessageIter     iter_array;

	if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID) == false) {
		printf ("Error reading VPN connection name from D-BUS message.\n");
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "Bad arguments");
		goto out;
	}

	if ((vpnConnection = nmi->getVPNConnection (name)) == NULL) {
		printf ("Did not find requested VPN connection \"%s\", thus we can not return the routes.\n", name);
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "VPN connection not found");
		goto out;
	}

	name   = vpnConnection->getName   ();
	routes = vpnConnection->getRoutes ();

	kdDebug () << "returning routes for vpn connection " << name << endl;

	reply = dbus_message_new_method_return (msg);
	dbus_message_iter_init_append (reply, &iter);
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);
	for (QStringList::Iterator i = routes.begin (); i != routes.end (); ++i) {
		const char* route = strdup ((*i).utf8 ());
		dbus_message_iter_append_basic (&iter_array, DBUS_TYPE_STRING, &route);
		kdDebug () << "returning route " << route << endl;
	}
	dbus_message_iter_close_container (&iter, &iter_array);

out:
	return reply;
}

DBusMessage*
NetworkManagerInfoDBus::getVPNConnectionData (DBusMessage* msg)
{
	NetworkManagerInfo* nmi           = _ctx->getNetworkManagerInfo ();
	VPNConnection*      vpnConnection = NULL;
	const char*         name          = NULL;
  	QStringList         data;
	DBusMessage*        reply         = NULL;
	DBusMessageIter     iter;
	DBusMessageIter     iter_array;

	if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID) == false) {
		printf ("Error reading VPN connection name from D-BUS message.\n");
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "Bad arguments");
		goto out;
	}

	if ((vpnConnection = nmi->getVPNConnection (name)) == NULL) {
		printf ("Did not find requested VPN connection \"%s\", thus we can not return the VPN  data.\n", name);
		reply = dbus_message_new_error (msg, NM_DBUS_INVALID_VPN_CONNECTION, "VPN connection not found");
		goto out;
	}

	name = vpnConnection->getName ();
	data = vpnConnection->getData ();

	kdDebug () << "returning VPN data for vpn connection " << name << endl;

	reply = dbus_message_new_method_return (msg);
	dbus_message_iter_init_append (reply, &iter);
	dbus_message_iter_open_container (&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING_AS_STRING, &iter_array);
	for (QStringList::Iterator i = data.begin (); i != data.end (); ++i) {
		const char* item = strdup ((*i).utf8 ());
		dbus_message_iter_append_basic (&iter_array, DBUS_TYPE_STRING, &item);
		kdDebug () << "returning data " << item << endl;
	}
	dbus_message_iter_close_container (&iter, &iter_array);

out:
	return reply;
}

void
NetworkManagerInfoDBus::push (KNetworkManager* ctx)
{
	_ctx = ctx;
}


