/***************************************************************************
 *
 * knetworkmanager-storage.h - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig     <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Stefan Bogner <sbogner@suse.de>, <bochi@kmobiletools.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager-devicestore.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-storage.h"
#include "settings.h"

#include <kdebug.h>
#include <kstaticdeleter.h>

static KStaticDeleter<KNetworkManagerStorage> sd;

KNetworkManagerStorage *KNetworkManagerStorage::m_instance = 0L;

KNetworkManagerStorage 
*KNetworkManagerStorage::getInstance ()
{
    if (!m_instance) 
    	sd.setObject( m_instance, new KNetworkManagerStorage () );
    return m_instance;
}

KNetworkManagerStorage::KNetworkManagerStorage () : QObject ()
{
    m_wallet = 0L;
    m_walletRefCount = 0;
}


KNetworkManagerStorage::~KNetworkManagerStorage ()
{
    m_instance = 0L;
    slotWalletClosed ();
}

void 
KNetworkManagerStorage::slotWalletClosed ()
{
    m_walletRefCount--;

    if (m_walletRefCount == 0) 
    {
        delete m_wallet;
        m_wallet = 0L;
    }
}

QMap< QString, QString > 
KNetworkManagerStorage::credentials(const QString &id)
{
    if (!m_wallet)
    {
        m_wallet = KWallet::Wallet::openWallet (KWallet::Wallet::NetworkWallet (), 0, KWallet::Wallet::Synchronous);

        if (m_wallet)
        {
            m_walletRefCount++;
            connect ( m_wallet, SIGNAL( walletClosed () ), this, SLOT( slotWalletClosed () ) );
        }
    }

   QMap< QString, QString> value;
   (m_wallet && m_wallet->setFolder ("knetworkmanager") && m_wallet->readMap ( id, value ));
   return value;
}

void
KNetworkManagerStorage::storeKey( const QString &id, const QString &key )
{
    if (KWallet::Wallet::isEnabled ()) 
    {
        if ( !m_wallet ) 
        {
            m_wallet = KWallet::Wallet::openWallet(KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous);

            if ( m_wallet ) 
            {
                m_walletRefCount++;
                connect (m_wallet, SIGNAL(walletClosed()), this, SLOT(slotWalletClosed()));
            }
        }

        if ( m_wallet )
        {
            if( !m_wallet->hasFolder ( "knetworkmanager" ) ) m_wallet->createFolder ( "knetworkmanager" );
            m_wallet->setFolder ( "knetworkmanager" );
            QMap< QString, QString> map;
            map.insert( "password", key );
            m_wallet->writeMap (id, map);
        }
    }
}

void
KNetworkManagerStorage::storeCredentials( const QString &id, const QMap< QString, QString> &map )
{
    if (KWallet::Wallet::isEnabled ()) 
    {
        if ( !m_wallet ) 
        {
            m_wallet = KWallet::Wallet::openWallet(KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous);

            if ( m_wallet ) 
            {
                m_walletRefCount++;
                connect (m_wallet, SIGNAL(walletClosed()), this, SLOT(slotWalletClosed()));
            }
        }

        if ( m_wallet )
        {
            if( !m_wallet->hasFolder ( "knetworkmanager" ) ) m_wallet->createFolder ( "knetworkmanager" );
            m_wallet->setFolder ( "knetworkmanager" );
            m_wallet->writeMap (id, map);
        }
    }
}

QStringList
KNetworkManagerStorage::networks() const
{
	QStringList groups = KGlobal::config()->groupList();
	QStringList networks;
	const QStringList::Iterator end = groups.end();
	for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
	{
		if ( !(*it).startsWith( "Network_" ) )
			continue;

		KConfigGroup networkGrp( KGlobal::config(), *it );
		QString ESSID = networkGrp.readEntry( "ESSID" );
		networks.append( ESSID );
	}
	return networks;
}

Network * KNetworkManagerStorage::networkProperties( const QString & name, const QString &hwAddr, bool * hwAddrMatched )
{
	QString groupName = lookupNetworkGroupName( name, hwAddr, hwAddrMatched );
	Network * net = 0;

	if ( !groupName.isEmpty() )
	{
		KConfig * config = KGlobal::config();
		KConfigGroup networkGrp( config, groupName );
		net = new Network();
		net->restore( &networkGrp );
		if ( !dynamic_cast<EncryptionNone*>( net->getEncryption( ) ) )
		{
			SecretMap storedSecret = credentials( net->getEssid() );
			if ( storedSecret.isEmpty() )
				kdWarning() << "restored network " << net->getEssid() << " but found no secret in wallet" << endl;
			net->getEncryption()->setSecrets( storedSecret );
		}
/*		else
			kdDebug() << "EncryptionNone, no passphrase needed" << endl;*/
	}
	return net;
}


void
KNetworkManagerStorage::storeNetwork( const Network* net, bool updateTimestamp )
{
// 	kdDebug() << k_funcinfo << endl;

	QString groupName = lookupNetworkGroupName( net->getEssid(),
												net->getHardwareAddresses().first() );
	if ( groupName.isEmpty() )
		groupName = QString("Network_").append( KApplication::randomString( 16 ) );

// 	kdDebug() << "[ !!! } The group name is " << groupName << endl;

	KConfigGroup networkGrp( KGlobal::config(), groupName );
	
	net->persist( &networkGrp, updateTimestamp );
	
/*FIXME 
	if( net->isEncrypted() ) 
	{
		kdDebug() << "[ !!! } Network is encrypted! Saving Pass: " << net->getPassphrase() << endl;
		saveKey( n, net->getPassphrase() );
	}
	*/
}

void KNetworkManagerStorage::removeNetwork( const Network * net )
{
	bool fuzzy = false;
	QString netName = lookupNetworkGroupName( net->getEssid(), net->getHardwareAddresses().first(), &fuzzy );
	if ( !netName.isEmpty() )
		KGlobal::config()->deleteGroup( netName, true );
}

QString KNetworkManagerStorage::lookupNetworkGroupName( const QString & essid, const QString & hwAddr, bool * hwAddrMatched ) const
{
	KConfig * config = KGlobal::config();
	QStringList groups = config->groupList();
	const QStringList::Iterator end = groups.end();
	bool matchOnEssid = hwAddr.isEmpty();
	QStringList fuzzyCandidates;

	for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
	{
		if ( (*it).startsWith("Network_" ) )
		{
			KConfigGroup candidate( config, *it );
			if ( matchOnEssid ) // just return the first network matching on essid
			{
				if ( candidate.readEntry( "ESSID" ) == essid )
				{
					if ( hwAddrMatched )
						*hwAddrMatched = false;
					return *it;
				}
			}
			else // check the hwaddresses for a match
			{
				if ( candidate.readEntry( "ESSID" ) == essid )
				{
					fuzzyCandidates.append( *it );
					QStringList addresses = candidate.readListEntry( "HardwareAddresses", NET_STORAGE_SEPARATOR );
					if ( addresses.find( hwAddr ) != addresses.end() )
					{
						if ( hwAddrMatched )
							*hwAddrMatched = true;
						return *it;
					}
				}
			}
		}
	}
	// if we got here, we didn't match on the caller's desired criteria
	// see if there is a fuzzy candidate match
	if ( hwAddrMatched )
		*hwAddrMatched = false;
	if ( fuzzyCandidates.isEmpty() )
	{
		return QString();
	}
	else
		return fuzzyCandidates.first();
}

void KNetworkManagerStorage::updateNetwork( Network * net, bool automatic )
{
	kdDebug() << k_funcinfo << endl;
	// look up the network (but we didn't want to store it earlier!!)
	QString groupName = lookupNetworkGroupName( net->getEssid(),
												net->getHardwareAddresses().first() );
	if ( groupName.isEmpty() )
	{
		kdDebug() << "was asked to store previously unseen network, hopefully unhidden, unencrypted " << net->getEssid() << endl; 
		storeNetwork( net, !automatic );
	}
	else
	{
		KConfigGroup networkGrp( KGlobal::config(), groupName );
		net->update( &networkGrp, !automatic );
	}
}

#include "knetworkmanager-storage.moc"
