/***************************************************************************
 *
 * knetworkmanager-dbus.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager.h"
#include "knetworkmanager-dbus.h"
#include "knetworkmanager-nminfo_dbus.h"

#include <qtimer.h>

KNetworkManager* DBusConnection::_ctx = NULL;

extern DBusHandlerResult filterFunction                   (DBusConnection*, DBusMessage *, void *);
extern DBusHandlerResult networkManagerInfoMessageHandler (DBusConnection*, DBusMessage*, void*);

DBusConnection*  DBusConnection::_dbus_connection = NULL;

DBusConnection::DBusConnection () : QObject(),
				    _initialized (false),
				    _connection (NULL)
{

}

DBusConnection::~DBusConnection ()
{

}

bool
DBusConnection::status ()
{
	return _initialized;
}

void
DBusConnection::initQt ()
{
	_connection = new DBusQt::Connection (this);
	_connection->dbus_connection_setup_with_qt_main (DBusConnection::_dbus_connection);
}

void
DBusConnection::deinitQt ()
{
	_connection->close ();
}

KNetworkManager*
DBusConnection::getContext ()
{
	return DBusConnection::_ctx;
}

void
DBusConnection::reconnect ()
{
	this->push (_ctx);

	if (_initialized == true) {
		printf ("Successfully connected to the D-BUS system bus.\n");
	} else {
		QTimer::singleShot (3000, this, SLOT (reconnect ()));
	}
}


void
DBusConnection::triggerReconnect (void)
{
	KNetworkManager* ctx = DBusConnection::_ctx;
	DBusConnection*  dbus = ctx->getDBus ();

	/* destroy old D-BUS context */
	dbus->pop ();

	QTimer::singleShot (3000, dbus, SLOT (reconnect ()));
}

DBusConnection*
DBusConnection::getConnection ()
{
	if (_initialized == false) {
		return NULL;
	}
	return DBusConnection::_dbus_connection;
}

bool
DBusConnection::addMatch ()
{
	bool status;
	DBusError error;

	dbus_error_init (&error);

	/* Match for DBUS_INTERFACE_DBUS */
	dbus_bus_add_match (DBusConnection::_dbus_connection, "type='signal',"
							      "interface='" DBUS_INTERFACE_DBUS "',"
							      "sender='" DBUS_SERVICE_DBUS "'",
							      &error);	
	if (dbus_error_is_set (&error)) {
		printf ("Error adding match, %s: %s\n", error.name, error.message);
		dbus_error_free (&error);
		status = false;
		goto out;
	}

	/* Match for NM_DBUS_INTERFACE */
	dbus_bus_add_match (DBusConnection::_dbus_connection, "type='signal',"
							      "interface='" NM_DBUS_INTERFACE "',"
							      "path='" NM_DBUS_PATH "',"
							      "sender='" NM_DBUS_SERVICE "'",
							      &error);

	if (dbus_error_is_set (&error)) {
		printf ("Error adding match, %s: %s\n", error.name, error.message);
		dbus_error_free (&error);
		status = false;
		goto out;
	}

	/* Match for NM_DBUS_INTERFACE_VPN */
	dbus_bus_add_match (DBusConnection::_dbus_connection, "type='signal',"
							      "interface='" NM_DBUS_INTERFACE_VPN "',"
							      "path='" NM_DBUS_PATH_VPN "',"
							      "sender='" NM_DBUS_SERVICE "'",
							      &error);

	if (dbus_error_is_set (&error)) {
		printf ("Error adding match, %s: %s\n", error.name, error.message);
		dbus_error_free (&error);
		status = false;
		goto out;
	}

	status = true;

out:
	if (dbus_error_is_set (&error)) {
		dbus_error_free (&error);
	}

	return status;
}

bool
DBusConnection::registerObjectPath ()
{
	bool status;
	DBusObjectPathVTable vtable = { NULL, &networkManagerInfoMessageHandler, NULL, NULL, NULL, NULL };

	if (dbus_connection_register_object_path (_dbus_connection, NMI_DBUS_PATH, &vtable, _ctx)) {
		status = true;
		goto out;

	} else {
		printf ("Failed to register message handler.\n");
		status = false;
		goto out;
	}
	
out:
	return status;
}


bool
DBusConnection::addFilter ()
{
	bool status;

	if (_ctx == NULL) {
		printf ("context is NULL\n");
	}

	if (dbus_connection_add_filter (_dbus_connection, filterFunction, _ctx, NULL)) {
		status = true;
		goto out;
	} else {
		printf ("Failed to add filter function.\n");
		status = false;
		goto out;
	}

out:
	return status;
}

void
DBusConnection::close ()
{
	/* DBusConnection::open () calls dbus_bus_get (), we need to unref the connection */
	dbus_connection_unref (_dbus_connection);
}

bool
DBusConnection::open ()
{
	bool status;
	DBusError error;
	
	dbus_error_init (&error);

	_dbus_connection = NULL;
	_dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	if ((_dbus_connection == NULL) || (dbus_error_is_set (&error))) {
		printf ("Connection to D-BUS system message bus failed:\n%s.\n", error.message);
		status = false;
		goto out;
	} else {
		dbus_connection_set_exit_on_disconnect (_dbus_connection, false);

		status = true;
		goto out;
	}

out:
	if (dbus_error_is_set (&error)) {
		dbus_error_free (&error);
	}

	return status;
}

void
DBusConnection:: pop (void)
{
	_initialized = false;

	this->deinitQt ();
	this->close ();
}

bool
DBusConnection::push (KNetworkManager* ctx)
{
	DBusConnection::_ctx = ctx;

	if (this->open () == false) {
		goto out;
	}

	if (this->registerObjectPath () == false) {
		printf ("Register object path failed. Bailing out.\n");
		goto out;
	}

	this->initQt ();
	
	if (this->addFilter () == false) {
		printf ("Adding filters failed. Bailing out.\n");
		goto out;
	}

	if (this->addMatch () == false) {
		printf ("Adding matches failed. Bailing out.\n");
		goto out;
	}

	_initialized = true;

out:
	return status ();
}

#include "knetworkmanager-dbus.moc"
