/***************************************************************************
 *   Copyright (C) 2003 by David Maciejak                                  *
 *   david.maciejak@kyxar.fr                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "rpm.h"
#include "debug.h"

#include <klocale.h>
#include <kdebug.h>

#include <qstringlist.h>
#include <qregexp.h>

#include <config.h>

#ifdef USE_RPM

Rpm::Rpm(QObject *parent, const char *name)
 : PackageManager(parent, name)
{
  connect(&m_process, SIGNAL(readReady (KProcIO *)), this, SLOT(readReady(KProcIO*)));
}

Rpm::~Rpm()
{
}

void Rpm::readReady(KProcIO*)
{
  bool partial;

  QString newline;
  QStringList lines;
  while(m_process.readln(newline, true, &partial) != -1)
  {
    if (partial) m_buffer += newline;
    else
    {
      newline.truncate(newline.length());
      QString line(m_buffer + newline);
      lines << line;
      m_buffer = "";
    }

  }

  (this->*m_receive)(lines);
}

int Rpm::capabilities( int query ) const
{
	if ( (query & SEARCH_FILE) && (query & OFFLINE) )
  	return INSTALLED_ONLY;
	if ( (query & LIST_FILES) && (query & OFFLINE) )
  	return INSTALLED_ONLY;
	if ( query & ONLINE )
  	return query;
	return NOT_SUPPORTED;
}

bool Rpm::search( const QString & file )
{
  m_process.resetAll();
  m_buffer = QString::null;

  ffile=file;
  m_process.clearArguments();
  m_process << "rpm" << "-qf" << "--queryformat=%{NAME}\\n" << file;
  m_receive = &Rpm::receiveSearch;
  return m_process.start(KProcess::Block, KProcess::Stdout );
}

void Rpm::receiveSearch( const QStringList & line )
{
  static QRegExp rx_notfound("error: file");
  //the format of rpm -qf answer is
  //package_1\n[package_n\n]

  for (QStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    //kdDebug(DEBUG_ZONE) << "*i= "<< *i << endl;

    if ((*i).isEmpty()) return;
    if (rx_notfound.search(*i)!=-1)
    {
      emit token("error", i18n("%1 not found").arg(ffile));
      ffile="";
      continue;
    }
    else
    {
      QStringList packages = *i;
      emit token("file", ffile);
      emit token("package", (*i).stripWhiteSpace());
    }
  }
}

bool Rpm::list( const QString & package )
{
  m_process.resetAll();
  m_buffer = QString::null;

  m_process.clearArguments();
  m_process << "rpm" << "-ql" << package;
  m_receive = &Rpm::receiveList;
  return m_process.start(KProcess::Block, KProcess::Stdout );
}

void Rpm::receiveList( const QStringList & line )
{
  static QRegExp rx_notfound("package (.*) is not installed");
  for (QStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    if (rx_notfound.search(*i) > -1)
      emit token("error", i18n("Package %1 is not installed").arg(rx_notfound.cap(1)));
    else if ((*i).startsWith("/"))
      emit token("file", *i);
  }
}


static const QString
  html_form_begin("\n<form action=\"http://rpmfind.net/linux/rpm2html/search.php\" method=\"GET\">\n"
                  "<table class=\"query\">\n");

static QString make_title(const QString& title)
{	return "\t<tr><td class=\"title\" colspan=\"2\">" + title + "</td></tr>\n"; }
static const QString
  html_form_end("<tr>\n"
                "\t<td class=\"button\" colspan=\"2\">\n"
                	"\t\t<input type=\"submit\" value=\"%1\">\n"
  								"\t\t<input type=\"hidden\" name=\"searchmode\" value=\"searchfilesanddirs\">\n"
                  "\t\t<input type=\"hidden\" name=\"case\" value=\"insensitive\">\n"
          			"\t</td>\n"
                "</tr>\n"
                "</table>\n"
                "</form>\n");

static const QString
  html_form_line_begin("<tr>\n"
                 "\t<td><label for=\"%1\">%2</label></td>\n"
                 "\t<td>\n");
static const QString
	html_form_line_end("</td>\n</tr>\n");

static const QString html_form_combo("<select name=\"%1\" id=\"%2\">");

static QString make_form_text(const QString& type, const QString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + QString("<input type=\"text\" name=\"%1\" id=\"%2\">").arg(type).arg(type)
   + html_form_line_end;
}

static QString begin_form_combo(const QString& type, const QString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + QString("\t<select name=\"%1\" id=\"%2\">\n").arg(type).arg(type);
}
static QString make_form_option(const QString& name, const QString& text)
{ return "\t\t<option value=" + name + ">" + text + "</option>\n"; }
static QString end_form_combo()
{ return "\t</select>\n\t</td>\n</tr>\n"; }

QString Rpm::getOnlineForm()
{
	QString buffer;
  QTextOStream stream(&buffer);
  stream
  	<< html_form_begin
    << make_title( i18n("rpmfind.net"))

    << make_form_text("query", i18n("File to search"))
    << begin_form_combo("arch", i18n("Architecture"))
      << make_form_option("", i18n("All"))
      << make_form_option("src", i18n("Source"))
      << make_form_option("i386", i18n("Intel 386"))
      << make_form_option("i486", i18n("Intel 486"))
      << make_form_option("i586", i18n("Intel 586"))
      << make_form_option("i686", i18n("Intel 686"))
      << make_form_option("m68k", i18n("Motorola 680x0"))
      << make_form_option("alpha", i18n("Alpha"))
      << make_form_option("sparc", i18n("SPARC"))
      << make_form_option("powerpc", i18n("PowerPC"))
      << make_form_option("arm", i18n("ARM"))
      << make_form_option("hppa", i18n("HP PA/RISC"))
      << make_form_option("ia64", i18n("Intel IA-64"))
      << make_form_option("mips", i18n("MIPS"))
      << make_form_option("mipsel", i18n("MIPS (DEC)"))
      << make_form_option("s390", i18n("IBM S/390"))
    << end_form_combo()
    << begin_form_combo("system", i18n("Distribution"))
    <<	make_form_option("", "All")
    <<	make_form_option("redhat", "Redhat")
    <<	make_form_option("mandrake", "Mandrake")
    <<	make_form_option("suse", "Suse")
    <<	make_form_option("pld", "PLD")
    << end_form_combo()

    << html_form_end.arg(i18n("Go online!"));

  return buffer;
}

#include "rpm.moc"

#endif

