/*
* page_magick.cc -- Magick Page definition
* Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <vector>
using std::cerr;
using std::endl;
using std::vector;

#include <string>
#include <iostream>
using std::cerr;
using std::endl;

#include <glade/glade.h>

#include "kino_extra.h"
#include "page_magick.h"
#include "kino_common.h"
#include "page.h"
#include "storyboard.h"
#include "page_editor.h"
#include "displayer.h"
#include "message.h"
#include "error.h"
#include "commands.h"

/** Provides plug-ins with current playlist.
*/

PlayList &GetCurrentPlayList( )
{
	return * common->getPlayList( );
}

extern "C"
{
#include "support.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dlfcn.h>
#include <dirent.h>

	extern GladeXML* magick_glade;

	static bool buttonMutex = false;

	void
	on_togglebutton_magick_preview_toggled ( GtkToggleButton * togglebutton,
	        gpointer user_data )
	{
		if ( !buttonMutex )
		{
			buttonMutex = true;
			common->getPageMagick() ->StartPreview();
			buttonMutex = false;
		}
	}

	void
	on_togglebutton_magick_start_toggled ( GtkToggleButton * togglebutton,
	                                       gpointer user_data )
	{
		if ( !buttonMutex )
		{
			buttonMutex = true;
			common->getPageMagick() ->StartRender();
			buttonMutex = false;
		}
	}

	void
	on_togglebutton_magick_stop_toggled ( GtkToggleButton * togglebutton,
	                                      gpointer user_data )
	{
		if ( !buttonMutex )
		{
			buttonMutex = true;
			common->getPageMagick() ->Stop();
			buttonMutex = false;
		}
	}

	void
	on_notebook_magick_switch_page ( GtkNotebook * notebook,
	                                 GtkNotebookPage * page,
	                                 gint page_num,
	                                 gpointer user_data )
	{
		common->getPageMagick()->StopPreview();
	}

	gboolean
	on_hscale_transition_start_button_press_event
	( GtkWidget * widget,
	  GdkEventButton * event,
	  gpointer user_data )
	{
		GtkWidget * lower = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_start" );
		GtkWidget *upper = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_end" );
		GtkAdjustment *adjust_lower = GTK_RANGE( lower ) ->adjustment;
		GtkAdjustment *adjust_upper = GTK_RANGE( upper ) ->adjustment;
		adjust_upper->lower = adjust_lower->value;
		g_signal_emit_by_name( adjust_upper, "changed" );
		return FALSE;
	}


	gboolean
	on_hscale_transition_start_button_release_event
	( GtkWidget * widget,
	  GdkEventButton * event,
	  gpointer user_data )
	{
		GtkWidget * lower = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_start" );
		GtkWidget *upper = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_end" );
		GtkAdjustment *adjust_lower = GTK_RANGE( lower ) ->adjustment;
		GtkAdjustment *adjust_upper = GTK_RANGE( upper ) ->adjustment;
		adjust_upper->lower = adjust_lower->value;
		g_signal_emit_by_name( adjust_upper, "changed" );
		common->getPageMagick() ->RefreshStatus( true );
		return FALSE;
	}


	gboolean
	on_hscale_transition_end_button_press_event
	( GtkWidget * widget,
	  GdkEventButton * event,
	  gpointer user_data )
	{
		GtkWidget * lower = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_start" );
		GtkWidget *upper = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_end" );
		GtkAdjustment *adjust_lower = GTK_RANGE( lower ) ->adjustment;
		GtkAdjustment *adjust_upper = GTK_RANGE( upper ) ->adjustment;
		adjust_lower->upper = adjust_upper->value;
		g_signal_emit_by_name( adjust_lower, "changed" );
		return FALSE;
	}


	gboolean
	on_hscale_transition_end_button_release_event
	( GtkWidget * widget,
	  GdkEventButton * event,
	  gpointer user_data )
	{

		GtkWidget * lower = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_start" );
		GtkWidget *upper = lookup_widget ( common->getPageMagick() ->window, "hscale_transition_end" );
		GtkAdjustment *adjust_lower = GTK_RANGE( lower ) ->adjustment;
		GtkAdjustment *adjust_upper = GTK_RANGE( upper ) ->adjustment;
		adjust_lower->upper = adjust_upper->value;
		g_signal_emit_by_name( adjust_lower, "changed" );
		common->getPageMagick() ->RefreshStatus( true );
		return FALSE;
	}

	void
	on_button_magick_file_clicked    (GtkButton       *button,
                                            gpointer         user_data)
	{
		const char *filename = common->getFileToSave( _("Enter a File Name to Save As") );
		gtk_widget_grab_focus( lookup_widget( GTK_WIDGET( button ), "entry_magick_file" ) );
		if ( strcmp( filename, "" ) )
			gtk_entry_set_text( GTK_ENTRY( lookup_widget( GTK_WIDGET( button ), "entry_magick_file" ) ), filename );
	}

	void
	on_spinbutton_magick_start_value_changed(GtkSpinButton   *spinbutton,
											gpointer         user_data)
	{
		gtk_entry_set_text( GTK_ENTRY( lookup_widget( GTK_WIDGET( spinbutton ), "entry_magick_start" ) ),
			common->getTime().parseFramesToString( ( int )gtk_spin_button_get_value( spinbutton ),
			common->getTimeFormat() ).c_str() );
	}
	
	void
	on_spinbutton_magick_end_value_changed (GtkSpinButton   *spinbutton,
											gpointer         user_data)
	{
		gtk_entry_set_text( GTK_ENTRY( lookup_widget( GTK_WIDGET( spinbutton ), "entry_magick_end" ) ),
			common->getTime().parseFramesToString( ( int )gtk_spin_button_get_value( spinbutton ),
			common->getTimeFormat() ).c_str() );
	}

	void
	on_entry_magick_start_activate         (GtkEntry        *entry,
											gpointer         user_data)
	{
		common->getTime().parseValueToString( gtk_entry_get_text( entry ), common->getTimeFormat() );
		GtkSpinButton *spinbutton = GTK_SPIN_BUTTON( lookup_widget( GTK_WIDGET( entry ), "spinbutton_magick_start" ) );
		gtk_spin_button_set_value( spinbutton, common->getTime().getFrames() );
		on_spinbutton_magick_start_value_changed( spinbutton, NULL );
	}
	
	gboolean
	on_entry_magick_start_focus_out_event  (GtkWidget       *widget,
											GdkEventFocus   *event,
											gpointer         user_data)
	{
		on_entry_magick_start_activate( GTK_ENTRY( widget ), NULL );
		return FALSE;
	}
	
	void
	on_entry_magick_end_activate           (GtkEntry        *entry,
											gpointer         user_data)
	{
		common->getTime().parseValueToString( gtk_entry_get_text( entry ), common->getTimeFormat() );
		GtkSpinButton *spinbutton = GTK_SPIN_BUTTON( lookup_widget( GTK_WIDGET( entry ), "spinbutton_magick_end" ) );
		gtk_spin_button_set_value( spinbutton, common->getTime().getFrames() );
		on_spinbutton_magick_end_value_changed( spinbutton, NULL );
	}
	
	gboolean
	on_entry_magick_end_focus_out_event    (GtkWidget       *widget,
											GdkEventFocus   *event,
											gpointer         user_data)
	{
		on_entry_magick_end_activate( GTK_ENTRY( widget ), NULL );
		return FALSE;
	}
	
}

/** Plugin - wrapper for the dlopen/dlsym functions.
*/

bool Plugin::Open( char *file )
{
	ptr = dlopen( file, RTLD_NOW );
	return ptr != NULL;
}

void Plugin::Close( )
{
	if ( ptr != NULL )
		dlclose( ptr );
}

void *Plugin::Find( char *symbol )
{
	return dlsym( ptr, symbol );
}

const char *Plugin::GetError( )
{
	return dlerror();
}

/** Plugin Collection - loads all shared objects in the specified directory.
*/

PluginCollection::PluginCollection( )
{}

PluginCollection::~PluginCollection( )
{
	for ( unsigned int index = 0; index < collection.size(); index ++ )
	{
		collection[ index ] ->Close();
		delete collection[ index ];
	}
}

void PluginCollection::Initialise( char *directory )
{
	char * filename;
	char *extension;
	DIR *dir;
	struct dirent *entry;
	struct stat statbuf;

	dir = opendir( directory );

	if ( dir )
	{
		while ( ( entry = readdir( dir ) ) != NULL )
		{
			filename = g_strdup_printf( "%s/%s", directory, entry->d_name );
			extension = strrchr( entry->d_name, '.' );
			if ( extension != NULL && !stat( filename, &statbuf ) && S_ISREG( statbuf.st_mode ) )
			{
				if ( !strcmp( extension, ".so" ) )
				{
					RegisterPlugin( filename );
				}
			}
			g_free( filename );
		}
		closedir( dir );
	}
}

void PluginCollection::RegisterPlugin( char *filename )
{
	Plugin * plugin = new Plugin;
	if ( plugin->Open( filename ) )
	{
		cerr << ">>> Registering plugin " << filename << endl;
		collection.push_back( plugin );
	}
	else
	{
		cerr << ">>> Rejecting plugin " << filename << " : " << plugin->GetError() << endl;
		delete plugin;
	}
}

unsigned int PluginCollection::Count()
{
	return collection.size();
}

Plugin *PluginCollection::Get( unsigned int index )
{
	return collection[ index ];
}

void PluginImageCreateRepository::InstallPlugins( Plugin *plugin )
{
	GDKImageCreate * ( *func ) ( int ) = ( GDKImageCreate * ( * ) ( int ) ) plugin->Find( "GetImageCreate" );
	if ( func != NULL )
	{
		int index = 0;
		GDKImageCreate *entry = func( index ++ );
		while ( entry != NULL )
		{
			if ( entry->IsUsable( ) )
				Register( entry );
			entry = func( index ++ );
		}
	}
}

void PluginImageFilterRepository::InstallPlugins( Plugin *plugin )
{
	GDKImageFilter * ( *func ) ( int ) = ( GDKImageFilter * ( * ) ( int ) ) plugin->Find( "GetImageFilter" );
	if ( func != NULL )
	{
		int index = 0;
		GDKImageFilter *entry = func( index ++ );
		while ( entry != NULL )
		{
			if ( entry->IsUsable( ) )
				Register( entry );
			entry = func( index ++ );
		}
	}
}

void PluginImageTransitionRepository::InstallPlugins( Plugin *plugin )
{
	GDKImageTransition * ( *func ) ( int ) = ( GDKImageTransition * ( * ) ( int ) ) plugin->Find( "GetImageTransition" );
	if ( func != NULL )
	{
		int index = 0;
		GDKImageTransition *entry = func( index ++ );
		while ( entry != NULL )
		{
			if ( entry->IsUsable( ) )
				Register( entry );
			entry = func( index ++ );
		}
	}
}

void PluginAudioFilterRepository::InstallPlugins( Plugin *plugin )
{
	GDKAudioFilter * ( *func ) ( int ) = ( GDKAudioFilter * ( * ) ( int ) ) plugin->Find( "GetAudioFilter" );
	if ( func != NULL )
	{
		int index = 0;
		GDKAudioFilter *entry = func( index ++ );
		while ( entry != NULL )
		{
			if ( entry->IsUsable( ) )
				Register( entry );
			entry = func( index ++ );
		}
	}
}

void PluginAudioTransitionRepository::InstallPlugins( Plugin *plugin )
{
	GDKAudioTransition * ( *func ) ( int ) = ( GDKAudioTransition * ( * ) ( int ) ) plugin->Find( "GetAudioTransition" );
	if ( func != NULL )
	{
		int index = 0;
		GDKAudioTransition *entry = func( index ++ );
		while ( entry != NULL )
		{
			if ( entry->IsUsable( ) )
				Register( entry );
			entry = func( index ++ );
		}
	}
}

/** Common info and factory for all PageMagick processors.
*/

class PageMagickFrames;
class PageMagickImage;
class PageMagickAudio;

class PageMagickInfo
{
private:
	PageMagickFrames *frameSource;
	PageMagickImage *imageManipulator;
	PageMagickAudio *audioManipulator;
public:
	KinoCommon *common;
	int begin;
	int end;
	double increment;
	bool reverse;
	int anteFrame;
	int postFrame;
	int width;
	int height;
	int isPAL;
	int isWide;
	int frequency;
	short channels;
	int samples_this_frame;
	bool preview;
	PageMagickInfo( KinoCommon *common );
	~PageMagickInfo();
	void SetLowQuality( bool quality );
	KinoCommon *GetCommon( )
	{
		return this->common;
	}
	void Initialise();
	PageMagickFrames *GetFrameSource();
	PageMagickImage *GetImageManipulator();
	PageMagickAudio *GetAudioManipulator();
	void SetBegin( int begin )
	{
		this->begin = begin > 0 ? begin : 0;
	}
	void SetEnd( int end )
	{
		this->end = end > 0 ? end : 0;
	}
	void SetPostFrame( int postFrame )
	{
		this->postFrame = postFrame;
	}
	void SetAnteFrame( int anteFrame )
	{
		this->anteFrame = anteFrame;
	}
	void GetAnteFrame( uint8_t *pixels );
	void GetPostFrame( uint8_t *pixels );
	int GetPostFrame( )
	{
		return this->postFrame;
	}
	int GetAnteFrame( )
	{
		return this->anteFrame;
	}
};

namespace
{

/// Helper class used to calculate position and frame_delta for plugins
class time_info
{
public:
	time_info( const PageMagickInfo& Info, const int FrameIndex, const double StartPosition = 0.0, const double EndPosition = 1.0 ) :
			info( Info ),
			frame_index( FrameIndex ),
			start_position( StartPosition ),
			end_position( EndPosition )
	{
		// Sanity checks ...
		if ( info.begin > info.end )
			throw _( "Invalid time range" );

		if ( start_position > end_position )
			throw _( "Invalid position range" );
	}

	/// Returns the duration of the effect in frames
	unsigned int frame_count() const
	{
		return info.end - info.begin + 1;
	}

	/// Returns the duration of the effect as a percentage
	double duration() const
	{
		return end_position - start_position;
	}

	/// Returns the effect "position" (or percent complete)
	double position() const
	{
		double position = static_cast<double>( frame_index - info.begin ) / static_cast<double>( frame_count() ) * duration() + start_position;
		if ( info.reverse )
			position = 1 - position;
		return position;
	}

	/// Returns the duration of a frame as a percent
	double frame_delta() const
	{
		return duration() / static_cast<double>( frame_count() );
	}

	/// Serialization
	friend std::ostream& operator<<( std::ostream& Stream, const time_info& RHS )
	{
		Stream << RHS.frame_count() << " " << RHS.duration() << " " << RHS.frame_index << " " << RHS.position() << " " << RHS.frame_delta();
		return Stream;
	}

private:
	const PageMagickInfo& info;
	const int frame_index;
	const double start_position;
	const double end_position;
};

} // namespace

/** PageMagick frame source - this interface is used to source the frames.
*/

class PageMagickFrames
{
public:
	virtual void Initialise( PageMagickInfo * )
	{ }
	virtual void GetFrame( uint8_t *pixels, int16_t **audio, int i )
	{ }
	virtual void GetFrame( uint8_t *pixels, int width, int height, int16_t **audio, int i )
	{ }
	virtual void Close()
	{ }
	virtual bool IsSynth( )
	{
		return false;
	}
};

/** PageMagic Image manipulator - this interface is used to manipulate the images.
*/

class PageMagickImage
{
public:
	virtual void Initialise( PageMagickInfo *info )
	{ }
	;
	virtual void PreGetFrame( )
	{ }
	;
	virtual void GetFrame( uint8_t *pixels, int i )
	{ }
	;
	virtual void Close()
	{ }
	;
	virtual bool ChangesImage( )
	{
		return true;
	}
};

/** PageMagic Audio manipulator - this interface is used to manipulate the audio.
*/

class PageMagickAudio
{
public:
	virtual void Initialise( PageMagickInfo * )
	{ }
	;
	virtual void GetFrame( int16_t **audio, int i, int& samples, int locked_samples )
	{ }
	;
	virtual void Close()
	{ }
	;
};

/** Implementation of the Overwrite frame source.
*/

class PageMagickOverwrite : public PageMagickFrames
{
private:
	PageMagickInfo *info;
public:
	void Initialise( PageMagickInfo * );
	void GetFrame( uint8_t *pixels, int16_t **audio, int i );
	void GetFrame( uint8_t *pixels, int width, int height, int16_t **audio, int i );
	void Close();
};

void PageMagickOverwrite::Initialise( PageMagickInfo *info )
{
	this->info = info;
	// Obtain begining and ending of sequence
	info->SetBegin( 0 );
	info->SetEnd( info->GetCommon() ->getPlayList() ->GetNumFrames() - 1 );
	info->increment = 1;
	info->reverse = false;

	GtkEntry *startSpin = GTK_ENTRY( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "spinbutton_magick_start" ) );
	GtkEntry *endSpin = GTK_ENTRY( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "spinbutton_magick_end" ) );

	if ( info->GetCommon() ->getPlayList() ->GetNumFrames() != 0 )
	{
		info->SetBegin( atoi( gtk_entry_get_text( startSpin ) ) );
		info->SetEnd( atoi( gtk_entry_get_text( endSpin ) ) );
		GtkToggleButton *limit = GTK_TOGGLE_BUTTON( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "checkbutton_magick_frame_limit" ) );
		if ( gtk_toggle_button_get_active( limit ) )
		{
			GtkEntry * spin = GTK_ENTRY( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "spinbutton_magick_frame_count" ) );
			GtkMenu *menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "optionmenu_magick_frame_offset" ) ) ) );
			GtkWidget *active_item = gtk_menu_get_active( menu );
			if ( g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item ) == 1 )
			{
				int end = info->begin + atoi( gtk_entry_get_text( spin ) ) - 1;
				if ( end < info->end )
					info->SetEnd( end );
			}
			else
			{
				info->SetBegin( info->end - atoi( gtk_entry_get_text( spin ) ) + 1 );
			}
		}
		info->SetAnteFrame( info->begin - 1 );
		info->SetPostFrame( info->end + 1 );
	}
	else
	{
		throw _( "No frames available for rewriting." );
	}

	// Determine speed
	GtkToggleButton *speed = GTK_TOGGLE_BUTTON( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "checkbutton_speed" ) );
	if ( gtk_toggle_button_get_active( speed ) )
	{
		GtkRange * range = GTK_RANGE( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "hscale_speed" ) );
		info->increment = range->adjustment->value;
	}

	// Determine direction
	GtkToggleButton *reverse = GTK_TOGGLE_BUTTON( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "checkbutton_reverse" ) );
	info->reverse = gtk_toggle_button_get_active( reverse );
}

void PageMagickOverwrite::GetFrame( uint8_t *pixels, int16_t **audio, int i )
{
	Frame* infoFrame = GetFramePool()->GetFrame();
	common->getPlayList() ->GetFrame( i, *( infoFrame ) );
	infoFrame->decoder->quality = DV_QUALITY_BEST;
	infoFrame->ExtractRGB( pixels );

	AudioInfo ainfo;
	infoFrame->GetAudioInfo( ainfo );
	if ( ainfo.channels )
	{
		AudioResample<int16_ne_t,int16_ne_t> * resampler = AudioResampleFactory<int16_ne_t,int16_ne_t>::createAudioResample(
		                                AUDIO_RESAMPLE_SRC_SINC_BEST_QUALITY, info->frequency );
		resampler->Resample( *infoFrame );
		info->samples_this_frame = resampler->size / ( 2 * ainfo.channels );
		int16_t *p = resampler->output;
		for ( int s = 0; s < info->samples_this_frame; s++ )
			for ( int c = 0; c < ainfo.channels; c++ )
				audio[ c ][ s ] = *p++;
		delete resampler;
	}
	GetFramePool()->DoneWithFrame( infoFrame );
}

void PageMagickOverwrite::GetFrame( uint8_t *pixels, int width, int height, int16_t **audio, int i )
{
	Frame* infoFrame = GetFramePool()->GetFrame();
	common->getPlayList() ->GetFrame( i, *( infoFrame ) );
	infoFrame->decoder->quality = DV_QUALITY_BEST;
	infoFrame->ExtractRGB( pixels );

	AudioInfo ainfo;
	infoFrame->GetAudioInfo( ainfo );
	if ( ainfo.channels )
	{
		AudioResample<int16_ne_t,int16_ne_t> *resampler = AudioResampleFactory<int16_ne_t,int16_ne_t>::createAudioResample(
	                               AUDIO_RESAMPLE_SRC_SINC_BEST_QUALITY, info->frequency );
		resampler->Resample( *infoFrame );
		info->samples_this_frame = resampler->size / ( 2 * ainfo.channels );
		int16_t *p = resampler->output;
		for ( int s = 0; s < info->samples_this_frame; s++ )
			for ( int c = 0; c < ainfo.channels; c++ )
				audio[ c ][ s ] = *p++;
		delete resampler;
	}

	if ( infoFrame->GetWidth() != width || infoFrame->GetHeight() != height )
	{
		GdkPixbuf * i1 = gdk_pixbuf_new_from_data( pixels, GDK_COLORSPACE_RGB, FALSE, 8,
		                 infoFrame->GetWidth(), infoFrame->GetHeight(), infoFrame->GetWidth() * 3, NULL, NULL );
		GdkPixbuf *i2 = gdk_pixbuf_scale_simple( i1, width, height, GDK_INTERP_HYPER );
		memcpy( pixels, gdk_pixbuf_get_pixels( i2 ), width * height * 3 );
		g_object_unref( i2 );
		g_object_unref( i1 );
	}
	GetFramePool()->DoneWithFrame( infoFrame );
}

void PageMagickOverwrite::Close( )
{
	cerr << ">>> Deleting " << info->begin << " << " << info->end << endl;
	info->GetCommon() ->getPlayList() ->Delete( info->begin, info->end );
}

/** Implementation of the Create frame source.
*/

class PageMagickCreate : public PageMagickFrames
{
private:
	PageMagickInfo *info;
	uint8_t *image;
	GDKImageCreate *creator;

public:
	PageMagickCreate();
	virtual ~PageMagickCreate();
	void Initialise( PageMagickInfo * );
	void GetFrame( uint8_t *pixels, int16_t **audio, int i );
	void GetFrame( uint8_t *pixels, int width, int height, int16_t **audio, int i );
	virtual bool IsSynth( )
	{
		return true;
	}
};

PageMagickCreate::PageMagickCreate() : image( NULL )
{
	image = new uint8_t[ 720 * 576 * 3 ];
}

PageMagickCreate::~PageMagickCreate()
{
	delete[] image;
}

void PageMagickCreate::Initialise( PageMagickInfo *info )
{
	this->info = info;
	info->begin = common->g_currentFrame;
	info->increment = 1;
	info->reverse = false;

/* Commented to simplify UI to always create before current frame
	GtkToggleButton *frameToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getPageMagick() ->window, "radiobutton_magick_create_before" ) );
	if ( gtk_toggle_button_get_active( frameToggle ) )
	{
		GtkEntry * spin = GTK_ENTRY( lookup_widget( common->getPageMagick() ->window, "spinbutton_magick_create_before" ) );
		info->begin = atoi( gtk_entry_get_text( spin ) );
	}
*/
	info->anteFrame = info->begin - 1;
	info->postFrame = info->begin;

	creator = info->GetCommon() ->getPageMagick() ->GetImageCreate();
	if ( creator != NULL )
	{
		creator->CreatePAL( info->isPAL );
		creator->InterpretWidgets( GTK_BIN( lookup_widget( common->getPageMagick() ->window, "frame_magick_frames_create" ) ) );
	}
	else
		throw _( "Invalid image creator selected" );

	info->end = info->begin + creator->GetNumberOfFrames() - 1;
}

void PageMagickCreate::GetFrame( uint8_t *pixels, int16_t **audio, int i )
{
	// Sanity checks ...
	if ( !creator )
		throw _( "Invalid image creator selected" );

	creator->CreateFrame( image, info->width, info->height, time_info( *info, i ).position(), time_info( *info, i ).frame_delta() );

	memcpy( pixels, image, info->width * info->height * 3 );

	for ( int i = 0; i < 4; i ++ )
		memset( audio[ i ], 0, 2 * DV_AUDIO_MAX_SAMPLES * sizeof( int16_t ) );

	GDKAudioImport *import = dynamic_cast <GDKAudioImport *>( creator );
	if ( import != NULL )
		import->CreateAudio( audio, &info->channels, &info->frequency, &info->samples_this_frame );
}

void PageMagickCreate::GetFrame( uint8_t *pixels, int width, int height, int16_t **audio, int i )
{
	// Sanity checks ...
	if ( !creator )
		throw _( "Invalid image creator selected" );

	creator->CreateFrame( image, width, height, time_info( *info, i ).position(), time_info( *info, i ).frame_delta() );

	memcpy( pixels, image, width * height * 3 );

	for ( int i = 0; i < 4; i ++ )
		memset( audio[ i ], 0, 2 * DV_AUDIO_MAX_SAMPLES * sizeof( int16_t ) );

	GDKAudioImport *import = dynamic_cast <GDKAudioImport *>( creator );
	if ( import != NULL )
		import->CreateAudio( audio, &info->channels, &info->frequency, &info->samples_this_frame );
}

/** Filter implementation
*/

class PageMagickTransition : public PageMagickImage
{
private:
	PageMagickInfo *info;
	GDKImageTransition *transition;
	uint8_t keyFrame[ 720 * 576 * 3 ];
	bool animateKey;
	int direction;
	double start_position;
	double end_position;
public:
	void Initialise( PageMagickInfo *info );
	void PreGetFrame( );
	void GetFrame( uint8_t *pixels, int i );
	void Close();
};

void PageMagickTransition::Initialise( PageMagickInfo *info )
{
	this->info = info;
	this->animateKey = false;

	transition = info->GetCommon() ->getPageMagick() ->GetImageTransition();
	if ( transition != NULL )
		transition->InterpretWidgets( GTK_BIN( lookup_widget( common->getPageMagick() ->window, "frame_magick_image_transition" ) ) );
	else
		throw _( "Invalid image transition selected" );

	GtkToggleButton *toggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getPageMagick() ->window, "radiobutton_magick_transition_colour" ) );

	if ( gtk_toggle_button_get_active( toggle ) )
	{
		GdkColor color;
		GtkColorButton * colorButton = GTK_COLOR_BUTTON( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "colorpicker_magick_transition" ) );
		gtk_color_button_get_color( colorButton, &color );
		
		for ( uint8_t * p = keyFrame; p < ( keyFrame + info->width * info->height * 3 ); )
		{
			*p ++ = color.red >> 8;
			*p ++ = color.green >> 8;
			*p ++ = color.blue >> 8;
		}
	}
	else
	{
		GtkMenu *menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "optionmenu_magick_transition_frame" ) ) ) );
		GtkWidget *active_item = gtk_menu_get_active( menu );
		switch ( g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item ) )
		{
		case 1:
			info->GetPostFrame( keyFrame );
			break;
		case 2:
			info->GetAnteFrame( keyFrame );
			break;
		case 0:
			animateKey = true;
			break;
		}
	}

	GtkMenu *menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "optionmenu_direction" ) ) ) );
	GtkWidget *active_item = gtk_menu_get_active( menu );
	direction = g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item );

	GtkRange *range = GTK_RANGE( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "hscale_transition_start" ) );
	start_position = range->adjustment->value;
	range = GTK_RANGE( lookup_widget( info->GetCommon() ->getPageMagick() ->window, "hscale_transition_end" ) );
	end_position = range->adjustment->value;
}

void PageMagickTransition::PreGetFrame( )
{
	if ( animateKey )
	{
		info->GetPostFrame( keyFrame );
	}
	info->postFrame ++;
}

void PageMagickTransition::GetFrame( uint8_t *pixels, int i )
{
	// Sanity checks ...
	if ( !transition )
		throw _( "Invalid image transition selected" );

	transition->GetFrame( pixels, keyFrame, info->width, info->height, time_info( *info, i, start_position, end_position ).position(), time_info( *info, i, start_position, end_position ).frame_delta(), direction == 1 );
}

void PageMagickTransition::Close( )
{
	if ( this->animateKey )
	{
		cerr << ">>> Deleting " << info->begin << " << " << info->begin + ( info->end - info->begin ) / info->increment << endl;
		info->GetCommon() ->getPlayList() ->Delete( info->begin, ( int ) ( info->begin + ( info->end - info->begin ) / info->increment ) );
	}
}

/** Filter implementation
*/

class PageMagickFilter : public PageMagickImage
{
private:
	PageMagickInfo *info;
	GDKImageFilter *filter;
public:
	void Initialise( PageMagickInfo *info );
	void GetFrame( uint8_t *pixels, int i );
	bool ChangesImage( );
	void PreGetFrame( )
	{
		info->postFrame ++;
	}
};

void PageMagickFilter::Initialise( PageMagickInfo *info )
{
	this->info = info;
	filter = info->GetCommon() ->getPageMagick() ->GetImageFilter();
	if ( filter != NULL )
		filter->InterpretWidgets( GTK_BIN( lookup_widget( common->getPageMagick() ->window, "frame_magick_image_filter" ) ) );
	else
		throw _( "Invalid image filter selected" );
}

void PageMagickFilter::GetFrame( uint8_t *pixels, int i )
{
	// Sanity checks ...
	if ( !filter )
		throw _( "Invalid image filter selected" );

	filter->FilterFrame( pixels, info->width, info->height, time_info( *info, i ).position(), time_info( *info, i ).frame_delta() );
}

bool PageMagickFilter::ChangesImage( )
{
	NullImageFilter * no_image_encode = dynamic_cast <NullImageFilter *>( filter );
	return no_image_encode == NULL;
}

/** PageMagic Audio Filter implementation.
*/

class PageMagickAudioFilter : public PageMagickAudio
{
private:
	PageMagickInfo *info;
	GDKAudioFilter *filter;
public:
	void Initialise( PageMagickInfo *info );
	void GetFrame( int16_t **audio, int i, int& samples, int locked_samples );
};

void PageMagickAudioFilter::Initialise( PageMagickInfo *info )
{
	this->info = info;
	filter = info->GetCommon() ->getPageMagick() ->GetAudioFilter();
	if ( filter != NULL )
		filter->InterpretWidgets( GTK_BIN( lookup_widget( common->getPageMagick() ->window, "frame_magick_audio_filter" ) ) );
	else
		throw _( "Invalid audio filter selected" );
}

void PageMagickAudioFilter::GetFrame( int16_t **audio, int i, int& samples, int locked_samples )
{
	// Sanity checks ...
	if ( !filter )
		throw _( "Invalid audio filter selected" );

	filter->GetFrame( audio, info->frequency, info->channels, samples, time_info( *info, i ).position(), time_info( *info, i ).frame_delta() );
}

/** PageMagick Audio Transition implementation.
*/

class PageMagickAudioTransition : public PageMagickAudio
{
private:
	PageMagickInfo *info;
	int16_t *audio_buffers[ 4 ];
	GDKAudioTransition *transition;

public:
	PageMagickAudioTransition();
	virtual ~PageMagickAudioTransition();
	void Initialise( PageMagickInfo *info );
	void GetFrame( int16_t **audio, int i, int& samples, int locked_samples );
};

PageMagickAudioTransition::PageMagickAudioTransition()
{
	for ( int index = 0; index < 4; index ++ )
		audio_buffers[ index ] = new int16_t[ 2 * DV_AUDIO_MAX_SAMPLES ];
}

PageMagickAudioTransition::~PageMagickAudioTransition()
{
	for ( int index = 0; index < 4; index ++ )
		delete[] audio_buffers[ index ];
}

void PageMagickAudioTransition::Initialise( PageMagickInfo *info )
{
	this->info = info;

	transition = info->GetCommon() ->getPageMagick() ->GetAudioTransition();
	if ( transition != NULL )
		transition->InterpretWidgets( GTK_BIN( lookup_widget( common->getPageMagick() ->window, "frame_magick_audio_transition" ) ) );
	else
		throw _( "Invalid audio transition selected" );
}

void PageMagickAudioTransition::GetFrame( int16_t **audio, int i, int& samples, int locked_samples )
{
	// Sanity checks ...
	if ( !transition )
		throw _( "Invalid audio filter selected" );

	if ( transition->IsBFrameConsumer() )
	{
		// Pick up the b-frame audio (ugly.. moved by image transition when appropriate)
		// should also be resampled to the a-frames frequency - this is going to go haywire in
		// mixed sample rate projects :-/
		// In truth, even the aframes should be resampled to original aframes frequency... (fun, fun, fun)
		Frame* infoFrame = GetFramePool()->GetFrame();
		AudioInfo ainfo;
		common->getPlayList()->GetFrame( info->GetPostFrame() - 1, *( infoFrame ) );
		infoFrame->GetAudioInfo( ainfo );
		if ( ainfo.channels )
		{
			AudioResample<int16_ne_t,int16_ne_t> * resampler = AudioResampleFactory<int16_ne_t,int16_ne_t>::createAudioResample(
											AUDIO_RESAMPLE_SRC_SINC_BEST_QUALITY, info->frequency );
			resampler->Resample( *infoFrame );
			int16_t *p = resampler->output;
			if ( resampler->size / info->channels / 2 < samples )
				samples = resampler->size / info->channels / 2;
			for ( int s = 0; s < samples; s++ )
				for ( int c = 0; c < info->channels; c++ )
					audio_buffers[ c ][ s ] = *p++;
			delete resampler;
		}
		GetFramePool()->DoneWithFrame( infoFrame );
	}
	else
	{
		samples = locked_samples;
	}
	transition->GetFrame( audio, audio_buffers, info->frequency, info->channels, 
		samples, time_info( *info, i ).position(), time_info( *info, i ).frame_delta() );
}

/** Info implementation.
*/

PageMagickInfo::PageMagickInfo( KinoCommon *common ) : preview( false )
{
	this->common = common;
	this->frameSource = NULL;
	this->imageManipulator = NULL;
	this->audioManipulator = NULL;
}

PageMagickInfo::~PageMagickInfo()
{
	delete frameSource;
	delete imageManipulator;
	delete audioManipulator;
}

void PageMagickInfo::Initialise()
{
	Frame* infoFrame = GetFramePool()->GetFrame();
	
	// Get a sample frame to obtain recording info
	if ( common->getPlayList() ->GetFrame( 0, *infoFrame ) )
	{
		// Get all video and audio info required
		width = infoFrame->GetWidth();
		height = infoFrame->GetHeight();
		isPAL = infoFrame->IsPAL();
		isWide = infoFrame->IsWide();
		AudioInfo info;
		infoFrame->GetAudioInfo( info );
		if ( info.channels && info.frequency )
		{
			channels = info.channels;
			frequency = info.frequency;
			samples_this_frame = frequency / ( isPAL ? 25 : 30 );
		}
		else if ( Preferences::getInstance().defaultNormalisation != NORM_UNSPECIFIED )
		{
			channels = ( short ) 2;
			switch ( Preferences::getInstance().defaultAudio )
			{
			case AUDIO_32KHZ:
				frequency = 32000;
				break;
			case AUDIO_44KHZ:
				frequency = 44100;
				break;
			case AUDIO_48KHZ:
				frequency = 48000;
				break;
			}
			samples_this_frame = frequency / ( isPAL ? 25 : 30 );
		}
		else
		{
			GetFramePool()->DoneWithFrame( infoFrame );
			throw _( "The first frame has no audio and the default preferences for video creation have not been specified - aborting." );
		}
	}
	else if ( Preferences::getInstance().defaultNormalisation != NORM_UNSPECIFIED )
	{
		// Get default video and audio info from preferences.
		width = 720;
		height = Preferences::getInstance().defaultNormalisation == NORM_PAL ? 576 : 480;
		isPAL = Preferences::getInstance().defaultNormalisation == NORM_PAL;
		isWide = Preferences::getInstance().defaultAspect == ASPECT_169;
		channels = ( short ) 2;
		switch ( Preferences::getInstance().defaultAudio )
		{
		case AUDIO_32KHZ:
			frequency = 32000;
			break;
		case AUDIO_44KHZ:
			frequency = 44100;
			break;
		case AUDIO_48KHZ:
			frequency = 48000;
			break;
		}
		samples_this_frame = frequency / ( isPAL ? 25 : 30 );
	}
	else
	{
		GetFramePool()->DoneWithFrame( infoFrame );
		throw _( "The playlist is empty and the default preferences for video creation have not been specified - aborting." );
	}
	GetFramePool()->DoneWithFrame( infoFrame );

	preview = false;
}

void PageMagickInfo::SetLowQuality( bool low_quality )
{
	if ( preview != low_quality )
	{
		if ( low_quality )
		{
			width /= 4;
			height /= 4;
		}
		else
		{
			width *= 4;
			height *= 4;
		}

		preview = low_quality;
	}
}

PageMagickFrames *PageMagickInfo::GetFrameSource()
{
	if ( frameSource == NULL )
	{
		GtkNotebook * notebook = GTK_NOTEBOOK( lookup_widget( this->common->getPageMagick() ->window, "notebook_magick_frames" ) );
		int page = gtk_notebook_get_current_page( notebook );

		switch ( page )
		{
		case 0:
			frameSource = new PageMagickOverwrite();
			break;
		case 1:
			frameSource = new PageMagickCreate();
			break;
		}
	}

	if ( frameSource != NULL )
		frameSource->Initialise( this );
	else
		throw _( "The requested frame source has not been implemented yet..." );

	return frameSource;
}

PageMagickImage *PageMagickInfo::GetImageManipulator()
{
	if ( imageManipulator == NULL )
	{
		GtkNotebook * notebook = GTK_NOTEBOOK( lookup_widget( this->common->getPageMagick() ->window, "notebook_magick_video" ) );
		int page = gtk_notebook_get_current_page( notebook );

		switch ( page )
		{
		case 0:
			imageManipulator = new PageMagickFilter();
			break;
		case 1:
			imageManipulator = new PageMagickTransition();
			break;
		}
	}

	if ( imageManipulator != NULL )
		imageManipulator->Initialise( this );
	else
		throw _( "The requested image manipulator has not been implemented yet..." );

	return imageManipulator;
}

PageMagickAudio *PageMagickInfo::GetAudioManipulator()
{
	if ( audioManipulator == NULL )
	{
		GtkNotebook * notebook = GTK_NOTEBOOK( lookup_widget( this->common->getPageMagick() ->window, "notebook_magick_audio" ) );
		int page = gtk_notebook_get_current_page( notebook );

		switch ( page )
		{
		case 0:
			audioManipulator = new PageMagickAudioFilter();
			break;
		case 1:
			audioManipulator = new PageMagickAudioTransition();
			break;
		}
	}

	if ( audioManipulator != NULL )
		audioManipulator->Initialise( this );
	else
		throw _( "The requested audio manipulator has not been implemented yet..." );

	return audioManipulator;
}

void PageMagickInfo::GetAnteFrame( uint8_t *pixels )
{
	if ( this->anteFrame >= 0 )
	{
		Frame* infoFrame = GetFramePool()->GetFrame();
		common->getPlayList() ->GetFrame( this->anteFrame, *infoFrame );
		infoFrame->decoder->quality = DV_QUALITY_BEST;
		infoFrame->ExtractRGB( pixels );
		if ( preview )
		{
			GdkPixbuf * i1 = gdk_pixbuf_new_from_data( pixels, GDK_COLORSPACE_RGB, FALSE, 8,
			                 width * 4, height * 4, width * 4 * 3, NULL, NULL );
			GdkPixbuf *i2 = gdk_pixbuf_scale_simple( i1, width, height, GDK_INTERP_NEAREST );
			memcpy( pixels, gdk_pixbuf_get_pixels( i2 ), width * height * 3 );
			g_object_unref( i2 );
			g_object_unref( i1 );
		}
		GetFramePool()->DoneWithFrame( infoFrame );
	}
	else
	{
		memset( pixels, 0, 720 * 576 * 3 );
	}
}

void PageMagickInfo::GetPostFrame( uint8_t *pixels )
{
	if ( this->postFrame < common->getPlayList() ->GetNumFrames() )
	{
		Frame* infoFrame = GetFramePool()->GetFrame();
		common->getPlayList() ->GetFrame( this->postFrame, *infoFrame );
		infoFrame->decoder->quality = DV_QUALITY_BEST;
		infoFrame->ExtractRGB( pixels );
		if ( preview )
		{
			GdkPixbuf * i1 = gdk_pixbuf_new_from_data( pixels, GDK_COLORSPACE_RGB, FALSE, 8,
			                 width * 4, height * 4, width * 4 * 3, NULL, NULL );
			GdkPixbuf *i2 = gdk_pixbuf_scale_simple( i1, width, height, GDK_INTERP_HYPER );
			memcpy( pixels, gdk_pixbuf_get_pixels( i2 ), width * height * 3 );
			g_object_unref( i2 );
			g_object_unref( i1 );
		}
		GetFramePool()->DoneWithFrame( infoFrame );
	}
	else
	{
		memset( pixels, 0, 720 * 576 * 3 );
	}
}

static void on_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data );

/** Constructor for Page Magick.
*/

PageMagick::PageMagick( KinoCommon *common ) : last_page( 0 ), rendering( false ), previewing( false )
{
	cerr << "> Creating Magick Page" << endl;

	window = glade_xml_get_widget( magick_glade, "window_magick" );
	GtkBin *bin = GTK_BIN( lookup_widget( common->getWidget(), "frame_magick" ) );
	gtk_widget_reparent( ( GTK_BIN( window ) ) ->child, GTK_WIDGET( bin ) );

	this->common = common;

	progressBar = GTK_PROGRESS_BAR( lookup_widget( common->getWidget(), "progressbar" ) );
	cerr << ">> Searching " << KINO_PLUGINDIR << " for plugins" << endl;
	plugins.Initialise( KINO_PLUGINDIR );

	for ( unsigned int index = 0; index < plugins.Count(); index ++ )
	{
		image_creators.InstallPlugins( plugins.Get( index ) );
		image_filters.InstallPlugins( plugins.Get( index ) );
		image_transitions.InstallPlugins( plugins.Get( index ) );
		audio_filters.InstallPlugins( plugins.Get( index ) );
		audio_transitions.InstallPlugins( plugins.Get( index ) );
	}

	GtkOptionMenu *menu = GTK_OPTION_MENU( lookup_widget( window, "optionmenu_magick_filter" ) );
	GtkBin *container = GTK_BIN( lookup_widget( window, "frame_magick_image_filter" ) );
	image_filters.Initialise( menu, container );
	
	menu = GTK_OPTION_MENU( lookup_widget( window, "optionmenu_magick_frames_create" ) );
	container = GTK_BIN( lookup_widget( window, "frame_magick_frames_create" ) );
	image_creators.Initialise( menu, container );

	menu = GTK_OPTION_MENU( lookup_widget( window, "optionmenu_magick_transition" ) );
	container = GTK_BIN( lookup_widget( window, "frame_magick_image_transition" ) );
	image_transitions.Initialise( menu, container );

	menu = GTK_OPTION_MENU( lookup_widget( window, "optionmenu_magick_audio_filter" ) );
	container = GTK_BIN( lookup_widget( window, "frame_magick_audio_filter" ) );
	audio_filters.Initialise( menu, container );

	menu = GTK_OPTION_MENU( lookup_widget( window, "optionmenu_magick_audio_transition" ) );
	container = GTK_BIN( lookup_widget( window, "frame_magick_audio_transition" ) );
	audio_transitions.Initialise( menu, container );

	GtkWidget *drawing_area = lookup_widget( window, "drawingarea_magick_preview" );
	g_signal_connect( G_OBJECT( drawing_area ), "expose_event", G_CALLBACK( on_drawingarea_refresh_required ), this );
	gtk_widget_set_double_buffered( drawing_area, TRUE );

	GtkWidget *notebook = lookup_widget( window, "notebook_magick_frames" );
	g_signal_connect( G_OBJECT( notebook ), "switch_page", G_CALLBACK( on_notebook_magick_switch_page ), this );
	notebook = lookup_widget( window, "notebook_magick_video" );
	g_signal_connect( G_OBJECT( notebook ), "switch_page", G_CALLBACK( on_notebook_magick_switch_page ), this );
	notebook = lookup_widget( window, "notebook_magick_audio" );
	g_signal_connect( G_OBJECT( notebook ), "switch_page", G_CALLBACK( on_notebook_magick_switch_page ), this );
}

/** Destructor for Page Magick.
*/

PageMagick::~PageMagick()
{
	GtkBin * bin = GTK_BIN( lookup_widget( common->getWidget(), "frame_magick" ) );
	gtk_widget_reparent( ( GTK_BIN( bin ) ) ->child, GTK_WIDGET( window ) );
	gtk_widget_destroy( window );
}

void PageMagick::newFile( )
{
	last_fx_file = "";
}

/** Start when entering page.
*/

void PageMagick::start()
{
	cerr << ">>> Starting magick" << endl;
	strcpy( status, "" );

	if ( common->getPlayList() ->GetNumFrames() > 0 )
	{
		int begin = common->getPlayList() ->FindStartOfScene( common->g_currentFrame );
		int end = common->getPlayList() ->FindEndOfScene( common->g_currentFrame );

		GtkEntry *startSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_start" ) );
		GtkEntry *endSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_end" ) );
		GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( startSpin ) );

		adjust->lower = 0;
		adjust->upper = common->getPlayList() ->GetNumFrames() - 1;
		gtk_spin_button_set_value( GTK_SPIN_BUTTON( startSpin ), begin );
		g_signal_emit_by_name( adjust, "changed" );

		adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( endSpin ) );
		adjust->lower = 0;
		adjust->upper = common->getPlayList() ->GetNumFrames() - 1;
		gtk_spin_button_set_value( GTK_SPIN_BUTTON( endSpin ), end );
		g_signal_emit_by_name( adjust, "changed" );
	}

	RefreshStatus();

	// Default the file name
	GtkEntry *fileEntry = GTK_ENTRY( lookup_widget( window, "entry_magick_file" ) );
	string fx_file_name = common->getPlayList( ) ->GetProjectDirectory( ) + "/";

	if ( fx_file_name != last_fx_file )
	{
		last_fx_file = fx_file_name;
		gtk_entry_set_text( fileEntry, fx_file_name.c_str() );
	}

	audio_transitions.SelectionChange();
	gtk_notebook_set_page( GTK_NOTEBOOK( lookup_widget( common->getWidget(), "notebook_keyhelp" ) ), 3 );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( lookup_widget( common->getWidget(), "menuitem_fx" ) ), TRUE );
	
	timeFormatChanged();
	LoadSplash( );
}

/** Activate scene list selection (for Frame sources).
*/

gulong PageMagick::activate()
{
	return SCENE_LIST;
}

/** Clean up when leaving page.
*/

void PageMagick::clean()
{
	StopPreview();
	strcpy( status, "" );
}

/** Play an audio frame.
*/

void PageMagick::PlayAudio( int16_t *buffers[], int frequency, int channels, bool isPAL )
{
	dv_audio.frequency = frequency;
	dv_audio.samples_this_frame = frequency / ( isPAL ? 25 : 30 );
	dv_audio.num_channels = channels;

	if ( !audio_device_avail && ( audio_sampling_rate = kino_sound_init( 
			&dv_audio, audio_device, Preferences::getInstance().audioDevice ) ) != 0 )
		audio_device_avail = true;
	if ( audio_device_avail )
		kino_sound_play( &dv_audio, audio_device, buffers );
}

/** Show an image in the preview drawing area.
*/

void PageMagick::ShowImage( GtkWidget *area, uint8_t *image, int width, int height )
{
	GdkPixbuf * pix = gdk_pixbuf_new_from_data( image, GDK_COLORSPACE_RGB, FALSE, 8, width, height, width * 3, NULL, NULL );
	GdkPixbuf *im = gdk_pixbuf_scale_simple( pix, area->allocation.width, area->allocation.height, GDK_INTERP_NEAREST );
	//gtk_image_set_from_pixbuf( GTK_IMAGE( area ), im );
	//gtk_widget_queue_draw( area );
	//gtk_widget_show_now( area );
	GdkGC *gc = gdk_gc_new( area->window );
	gdk_draw_pixbuf( area->window, gc, im, 0, 0, area->allocation.x, area->allocation.y, -1, -1, GDK_RGB_DITHER_NORMAL, 0, 0 );
	g_object_unref( im );
	g_object_unref( pix );
	g_object_unref( gc );
}

static bool restart = false;

/** Provide a preview of the effect.
*/

void PageMagick::StartPreview()
{
	// Do not interrupt rendering
	if ( rendering )
	{
		buttonMutex = true;
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 0 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 1 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 0 );
		buttonMutex = false;
		return;
	}

	buttonMutex = true;
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 1 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 0 );
	buttonMutex = false;

	// Make sure we're not already previewing.
	if ( previewing )
		return ;

	// Set the page condition
	previewing = true;

	GtkWidget *area = GTK_WIDGET( lookup_widget( window, "drawingarea_magick_preview" ) );

	// Create the temporary space
	uint8_t *pixels = new uint8_t[ FRAME_MAX_WIDTH * FRAME_MAX_HEIGHT * 4 ];
	int16_t *audio_buffers[ 4 ];
	for ( int n = 0; n < 4; n++ )
		audio_buffers[ n ] = new int16_t [ 2 * DV_AUDIO_MAX_SAMPLES ];

	// Generate the info
	PageMagickInfo *info = new PageMagickInfo( common );

	audio_device = kino_sound_new();
	audio_device_avail = false;
	audio_sampling_rate = 0;

	GtkToggleButton *audioButton = GTK_TOGGLE_BUTTON( lookup_widget( window, "checkbutton_magick_preview_audio" ) );
	GtkToggleButton *everyButton = GTK_TOGGLE_BUTTON( lookup_widget( window, "checkbutton_magick_preview_every" ) );
	GtkEntry *everySpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_preview_every" ) );
	GtkToggleButton *loopButton = GTK_TOGGLE_BUTTON( lookup_widget( window, "checkbutton_magick_preview_loop" ) );
	GtkToggleButton *contextButton = GTK_TOGGLE_BUTTON( lookup_widget( window, "checkbutton_magick_preview_context" ) );
	GtkToggleButton *qualityButton = GTK_TOGGLE_BUTTON( lookup_widget( window, "checkbutton_low_quality" ) );
	GtkEntry *contextSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_preview_context" ) );

	int audio_number = 0;
	Frame* infoFrame = GetFramePool()->GetFrame();

	try
	{
		do
		{

			info->Initialise();
			info->SetLowQuality( gtk_toggle_button_get_active( qualityButton ) );

			PageMagickFrames *frames = info->GetFrameSource();
			PageMagickImage *image = info->GetImageManipulator();
			PageMagickAudio *sound = info->GetAudioManipulator();

			if ( info->begin > info->end )
				throw _( "Invalid frame range specified." );

			previewing = true;
			restart = false;

			for ( int i = info->begin - atoi( gtk_entry_get_text( contextSpin ) );
			        i < info->begin && gtk_toggle_button_get_active( contextButton ) && previewing;
			        i ++ )
			{
				if ( i >= 0 )
				{
					common->getPlayList() ->GetFrame( i, *( infoFrame ) );

					if ( !gtk_toggle_button_get_active( everyButton ) || i % atoi( gtk_entry_get_text( everySpin ) ) == 0 )
					{
						infoFrame->ExtractPreviewRGB( pixels );
						ShowImage( area, pixels, infoFrame->GetWidth(), infoFrame->GetHeight() );
					}

					if ( gtk_toggle_button_get_active( audioButton ) )
					{
						infoFrame->ExtractAudio( audio_buffers );
						PlayAudio( audio_buffers, info->frequency, info->channels, info->isPAL );
					}

					UpdateProgress( ( gfloat ) 0 );
				}
			}

			int frame_number = info->begin;
			for ( double i = ( double ) info->begin; 
				frame_number <= info->end && previewing; 
				frame_number = ( int )( ( i += info->increment ) + 0.5 ) )
			{
				if ( info->reverse )
					frame_number = info->end - ( ( int ) i - info->begin );

				frames->GetFrame( pixels, info->width, info->height, audio_buffers, frame_number );
				image->PreGetFrame( );

				if ( !gtk_toggle_button_get_active( everyButton ) ||
				        ( i == info->begin || ( frame_number - info->begin ) % atoi( gtk_entry_get_text( everySpin ) ) == 0 ) )
				{
					image->GetFrame( pixels, frame_number );
					ShowImage( area, pixels, info->width, info->height );
				}

				if ( gtk_toggle_button_get_active( audioButton ) )
				{
					int samples = info->samples_this_frame;
					int locked_samples = infoFrame->CalculateNumberSamples( info->frequency, audio_number ++ );
					if ( samples == 0 )
						samples = locked_samples;

					sound->GetFrame( audio_buffers, frame_number, samples, locked_samples );
					PlayAudio( audio_buffers, info->frequency, info->channels, info->isPAL );
				}

				if ( info->end != info->begin )
					UpdateProgress( ( gfloat ) ( ( i - info->begin ) ) / ( gfloat ) ( info->end - info->begin ) );
			}

			for ( int i = info->postFrame;
			        i < info->postFrame + atoi( gtk_entry_get_text( contextSpin ) ) && gtk_toggle_button_get_active( contextButton ) && previewing;
			        i ++ )
			{
				if ( i < common->getPlayList() ->GetNumFrames() )
				{
					common->getPlayList() ->GetFrame( i, *( infoFrame ) );

					if ( !gtk_toggle_button_get_active( everyButton ) || i % atoi( gtk_entry_get_text( everySpin ) ) == 0 )
					{
						infoFrame->ExtractPreviewRGB( pixels );
						ShowImage( area, pixels, infoFrame->GetWidth(), infoFrame->GetHeight() );
					}

					if ( gtk_toggle_button_get_active( audioButton ) )
					{
						infoFrame->ExtractAudio( audio_buffers );
						PlayAudio( audio_buffers, info->frequency, info->channels, info->isPAL );
					}
				}
				UpdateProgress( ( gfloat ) 1 );
			}

			UpdateProgress( ( gfloat ) 1 );
		}
		while ( ( previewing && gtk_toggle_button_get_active( loopButton ) ) || restart );
	}
	catch ( const char * exc )
	{
		modal_message( ( char * ) exc );
	}

	kino_sound_close( audio_device );

	previewing = false;

	delete info;
	delete[] pixels;
	for ( int n = 0; n < 4; n++ )
		delete[] audio_buffers[ n ];

    GetFramePool()->DoneWithFrame( infoFrame );
	
	buttonMutex = true;
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 1 );
	buttonMutex = false;

	LoadSplash( );
}

/** Render the effect.
*/

void PageMagick::StartRender()
{
	// Allow immediate render from preview
	if ( previewing )
		Stop();

	// Show the current button status
	buttonMutex = true;
	gtk_widget_set_sensitive( lookup_widget( window, "hpaned_magick" ), false );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 1 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 0 );
	buttonMutex = false;

	// Make sure we're not already rendering
	if ( rendering )
		return ;

	// Set the page condition
	rendering = true;

	// Create the temporary space
	uint8_t *pixels = new unsigned char[ FRAME_MAX_WIDTH * FRAME_MAX_HEIGHT * 4 ];
	int16_t *audio_buffers[ 4 ];
	for ( int n = 0; n < 4; n++ )
		audio_buffers[ n ] = new int16_t [ 2 * DV_AUDIO_MAX_SAMPLES ];
	unsigned char *dv_pixels[ 3 ];
	dv_pixels[ 0 ] = pixels;

	// Build the info
	PageMagickInfo *info = new PageMagickInfo( common );

	// Output file
	FILE *output = NULL;

	Frame* infoFrame = GetFramePool()->GetFrame();
	
	// All exceptions thrown should be safely picked up in the try/catch
	try
	{
		// Get the current date and time to set in dv
		time_t datetime = time( NULL );
		int frameNum = 0;

		// Initialise the info
		info->Initialise();

		int audio_number = 0;

		// Obtain the file name
		GtkEntry *fileEntry = GTK_ENTRY( lookup_widget( window, "entry_magick_file" ) );
		const char *filename = gtk_entry_get_text( fileEntry );
		int counter = 0;
		string thisfile;
		struct stat stats;

		if ( !strcmp( filename, "" ) )
			throw _( "No file name specified" );

		string directory = "";

		// Relative/absolute file checks
		if ( filename[ 0 ] != '/' )
			directory = common->getPlayList() ->GetProjectDirectory( ) + "/";

		// Generate the full file name
		do
		{
			char name[ 132 ];
			sprintf( name, "%s%03d.kinofx.dv", filename, ++ counter );
			thisfile = name;
			cerr << ">>> Trying " << thisfile << endl;
		}
		while ( stat( thisfile.c_str(), &stats ) == 0 );

		infoFrame->CreateEncoder( ( info->height == 576 ), info->isWide );

		// Get the source and manipulators
		PageMagickFrames *frames = info->GetFrameSource();
		PageMagickImage *image = info->GetImageManipulator();
		PageMagickAudio *sound = info->GetAudioManipulator();

		// Check that we're rendering a valid set of frames
		if ( info->begin > info->end )
			throw _( "Invalid frame range specified." );

		// Open the file
		output = fopen( thisfile.c_str(), "w" );
		if ( output == NULL )
			throw _( "Unable to open output file" );

		bool change_image = image->ChangesImage( ) || frames->IsSynth( );

		// For each frame
		int frame_number = info->begin;
		for ( double i = ( double ) info->begin; 
			frame_number <= info->end && rendering; 
			frame_number = ( int )( ( i += info->increment ) + 0.5 ) )
		{
			if ( info->reverse )
				frame_number = info->end - ( ( int ) i - info->begin );

			frames->GetFrame( pixels, audio_buffers, frame_number );
			image->PreGetFrame( );

			int samples = info->samples_this_frame; // set in frames->GetFrame() above
			int locked_samples = infoFrame->CalculateNumberSamples( info->frequency, audio_number ++ );
			if ( samples == 0 )
				samples = locked_samples;

			sound->GetFrame( audio_buffers, frame_number, samples, locked_samples );

			if ( ! change_image )
			{
				common->getPlayList()->GetFrame( frame_number, *( infoFrame ) );
			}
			else
			{
				image->GetFrame( pixels, frame_number );
				infoFrame->EncodeRGB( pixels );
			}

			AudioInfo audioInfo;
			audioInfo.channels = info->channels;
			audioInfo.frequency = info->frequency;
			audioInfo.samples = samples;
			infoFrame->EncodeAudio( audioInfo, audio_buffers );
			infoFrame->SetRecordingDate( &datetime, frameNum );
			infoFrame->SetTimeCode( frameNum++ );

			if ( fwrite( infoFrame->data, ( info->isPAL ? 144000 : 120000 ), 1, output ) != 1 )
				throw _( "Unable to write video - disk full?" );

			if ( info->end != info->begin )
				UpdateProgress( ( gfloat ) ( ( i - info->begin ) ) / ( gfloat ) ( info->end - info->begin ) );
		}
		UpdateProgress( ( gfloat ) 1 );

		frames->Close();
		image->Close();
		sound->Close();

		fclose( output );
		output = NULL;

		PlayList temp;
		temp.LoadMediaObject( ( char * ) thisfile.c_str() );
		common->getPlayList() ->InsertPlayList( temp, info->begin );
		common->getPageEditor() ->snapshot();
		common->getPageEditor() ->ResetBar();
		GetStoryboard() ->redraw();
	}
	catch ( const char * exc )
	{
		modal_message( ( char * ) exc );
	}

	buttonMutex = true;
	gtk_widget_set_sensitive( lookup_widget( window, "hpaned_magick" ), true );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 1 );
	buttonMutex = false;

	rendering = false;

	if ( output )
		fclose( output );

	delete info;
	delete[] pixels;
	for ( int n = 0; n < 4; n++ )
		delete[] audio_buffers[ n ];
	GetFramePool()->DoneWithFrame( infoFrame );
}

/** Stop whatever is running (either preview or rendering).
*/

void PageMagick::Stop()
{
	if ( rendering || previewing )
	{
		rendering = false;
		previewing = false;

		buttonMutex = true;
		gtk_widget_set_sensitive( lookup_widget( window, "hpaned_magick" ), true );
		if ( !restart )
		{
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_preview" ) ), 0 );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_start" ) ), 0 );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget( window, "togglebutton_magick_stop" ) ), 1 );
		}
		buttonMutex = false;
	}
}

/** Stop the preview.
*/

void PageMagick::StopPreview()
{
	if ( previewing )
		Stop();
}

/** Update the progress bar for the export process
*/


void PageMagick::UpdateProgress( gfloat val )
{
	struct timeval tv;

	long long now;
	static long long lastUpdateTime = 0;

	gettimeofday( &tv, 0 );
	now = 1000000 * ( long long ) tv.tv_sec + ( long long ) tv.tv_usec;

	/* update every 0.3 second */

	if ( val < 0.0 )
		val = 0;
	if ( val > 1.0 )
		val = 1.0;

	if ( now > lastUpdateTime + 300000 || val == 1.0 )
	{
		gtk_progress_bar_update( progressBar, val );
		lastUpdateTime = now;
	}

	while ( gtk_events_pending() )
		gtk_main_iteration();
}

/** put the scene begin and end frame numbers into spinners

    \param i the numerical index position of the scene in the playlist1
*/
void PageMagick::selectScene( int i )
{
	int begin = 0;
	int end = 0;
	vector <int> scene = common->getPageEditor() ->GetScene();

	begin = i == 0 ? 0 : scene[ i - 1 ];
	end = scene[ i ] - 1;
	common->g_currentFrame = begin;

	if ( GTK_WIDGET_SENSITIVE( lookup_widget( window, "hpaned_magick" ) ) )
	{
		GtkNotebook * notebook = GTK_NOTEBOOK( lookup_widget( this->window, "notebook_magick_frames" ) );
		int page = gtk_notebook_get_current_page( notebook );

		if ( page == 0 )
		{
			GtkEntry * startSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_start" ) );
			GtkEntry *endSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_end" ) );
			GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( startSpin ) );

			adjust->lower = 0;
			adjust->upper = scene[ scene.size() - 1 ];
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( startSpin ), begin );
			g_signal_emit_by_name( adjust, "changed" );

			adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( endSpin ) );
			adjust->lower = 0;
			adjust->upper = scene[ scene.size() - 1 ];
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( endSpin ), end );
			g_signal_emit_by_name( adjust, "changed" );
		}
		else if ( page == 1 )
		{
/* Commented to simplify UI to always create before current frame
			GtkToggleButton * rangeToggle = GTK_TOGGLE_BUTTON( lookup_widget( window, "radiobutton_magick_create_before" ) );
			GtkEntry *startSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_magick_create_before" ) );
			GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( startSpin ) );
			gtk_toggle_button_set_active( rangeToggle, TRUE );
			adjust->lower = 0;
			adjust->upper = scene[ scene.size() - 1 ];
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( startSpin ), begin );
			g_signal_emit_by_name( adjust, "changed" );*/
		}
		if ( previewing )
		{
			restart = true;
			StopPreview( );
		}
	}

	Frame &frame = *( GetFramePool( ) ->GetFrame( ) );
	FileHandler *media;
	common->getPlayList() ->GetMediaObject( begin, &media );
	common->getPlayList() ->GetFrame( begin, frame );
	common->showFrameMoreInfo( frame, media );
	GetFramePool( ) ->DoneWithFrame( &frame );

	common->setCurrentScene( begin );
	RefreshStatus( true );
}

/** Refresh the status label to show the current selections.
*/

void PageMagick::RefreshStatus( bool with_fx_notify )
{
	char temp[ 10240 ] = "";
	GtkNotebook *notebookFrame = GTK_NOTEBOOK( lookup_widget( this->window, "notebook_magick_frames" ) );
	GtkNotebook *notebookImage = GTK_NOTEBOOK( lookup_widget( this->window, "notebook_magick_video" ) );
	GtkNotebook *notebookAudio = GTK_NOTEBOOK( lookup_widget( this->window, "notebook_magick_audio" ) );

	int page = gtk_notebook_get_current_page( notebookFrame );
	if ( page == 0 )
	{
		strcpy( temp, _( "Overwriting frames\n" ) );
	}
	else if ( page == 1 )
	{
		strcpy( temp, _( "Creating with '" ) );
		strcat( temp, GetImageCreate() ->GetDescription() );
		strcat( temp, "'\n" );
		SelectionNotification *notify = dynamic_cast <SelectionNotification *>( GetImageCreate( ) );
		if ( with_fx_notify && notify != NULL )
			notify->OnSelectionChange( );
	}
	else
	{
		strcpy( temp, _( "Unknown frame source\n" ) );
	}

	page = gtk_notebook_get_current_page( notebookImage );

	if ( page == 0 && GetImageFilter() != NULL )
	{
		strcat( temp, _( "with the '" ) );
		strcat( temp, GetImageFilter() ->GetDescription() );
		strcat( temp, _( "' image filter\n" ) );
		SelectionNotification *notify = dynamic_cast <SelectionNotification *>( GetImageFilter( ) );
		if ( with_fx_notify && notify != NULL )
			notify->OnSelectionChange( );
	}
	else if ( page == 1 && GetImageTransition() != NULL )
	{
		strcat( temp, _( "with the '" ) );
		strcat( temp, GetImageTransition() ->GetDescription() );
		strcat( temp, _( "' image transition\n" ) );
		SelectionNotification *notify = dynamic_cast <SelectionNotification *>( GetImageTransition( ) );
		if ( with_fx_notify && notify != NULL )
			notify->OnSelectionChange( );
	}
	else
		strcpy( temp, _( "with unknown image manipulation\n" ) );

	page = gtk_notebook_get_current_page( notebookAudio );
	if ( page == 0 )
	{
		strcat( temp, _( "and the '" ) );
		strcat( temp, GetAudioFilter() ->GetDescription() );
		strcat( temp, _( "' audio filter." ) );
		SelectionNotification *notify = dynamic_cast <SelectionNotification *>( GetAudioFilter( ) );
		if ( with_fx_notify && notify != NULL )
			notify->OnSelectionChange( );
	}
	else if ( page == 1 )
	{
		strcat( temp, _( "and the '" ) );
		strcat( temp, GetAudioTransition() ->GetDescription() );
		strcat( temp, _( "' audio transition." ) );
		SelectionNotification *notify = dynamic_cast <SelectionNotification *>( GetAudioTransition( ) );
		if ( with_fx_notify && notify != NULL )
			notify->OnSelectionChange( );
	}
	else
		strcpy( temp, _( "and unknown audio manipulation." ) );

	if ( strcmp( temp, status ) )
	{
		strcpy( status, temp );
//		GtkLabel *label = GTK_LABEL( lookup_widget( this->window, "label_magick" ) );
//		gtk_label_set_text( label, status );
	}
	
	// This is a special case to cause preview to restart when fx options change
	if ( !with_fx_notify && previewing )
	{
		restart = true;
		StopPreview();
	}
}

void PageMagick::timeFormatChanged()
{
	on_spinbutton_magick_start_value_changed( GTK_SPIN_BUTTON( lookup_widget( this->window, "spinbutton_magick_start" ) ), NULL );
	on_spinbutton_magick_end_value_changed( GTK_SPIN_BUTTON( lookup_widget( this->window, "spinbutton_magick_end" ) ), NULL );
}


/** Return the currently selected image filter.
*/

GDKImageFilter *PageMagick::GetImageFilter( ) const
{
	return image_filters.Get( );
}

/** Return the currently selected image transition.
*/

GDKImageTransition *PageMagick::GetImageTransition( ) const
{
	return image_transitions.Get( );
}

/** Return the currently selected audio filter.
*/

GDKAudioFilter *PageMagick::GetAudioFilter( ) const
{
	return audio_filters.Get( );
}

/** Return the currently selected audio transition.
*/

GDKAudioTransition *PageMagick::GetAudioTransition( ) const
{
	return audio_transitions.Get( );
}

/** Return the currently selected image creator.
*/

GDKImageCreate *PageMagick::GetImageCreate( ) const
{
	return image_creators.Get( );
}

/** Load the splash screen for the preview.
*/

void PageMagick::LoadSplash( )
{
	if ( !previewing )
	{
		//GtkWidget *area = lookup_widget( window, "drawingarea_magick_preview" );
		//common->loadSplash( area );
	}
}

void PageMagick::ChangePage( int page )
{
}

gboolean PageMagick::processKeyboard( GdkEventKey *event )
{
	gboolean ret = FALSE;

	// Translate special keys to equivalent command
	switch ( event->keyval )
	{
	case GDK_Escape:
		if ( rendering || previewing )
		{
			common->keyboardFeedback( "", _( "Stop Render" ) );
			Stop();
		}
		else
		{
			common->changePageRequest( PAGE_EDITOR );
		}
	default:
		break;
	}

	return ret;
}

gboolean PageMagick::processCommand( char *cmd )
{
	/* write PlayList */

	if ( strcmp( cmd, ":w" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Write playlist" ) );
		common->savePlayList( );
	}

	else if ( strcmp( cmd, "Enter" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Start Render" ) );
		GdkEvent ev;
		ev.key.type = GDK_KEY_PRESS;
		ev.key.window = common->getWidget()->window;
		ev.key.send_event = TRUE;
		ev.key.state = 0;
		ev.key.length = 0;
		ev.key.string = "";
		ev.key.keyval = GDK_Return;
		ev.key.group = 0;
		gdk_event_put( &ev );
		cmd[ 0 ] = 0;
	}

	else if ( strcmp( cmd, "Esc" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Stop" ) );
		Stop();
	}

	else if ( strcmp( cmd, "F2" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Edit" ) );
		common->changePageRequest( PAGE_EDITOR );
	}

	else if ( strcmp( cmd, "A" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Capture, append to movie" ) );
		common->changePageRequest( PAGE_EDITOR );
	}

	else if ( strcmp( cmd, "v" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Timeline" ) );
		common->changePageRequest( PAGE_TIMELINE );
	}

	else if ( strcmp( cmd, "t" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Trim" ) );
		common->changePageRequest( PAGE_TRIM );
	}

	else if ( strcmp( cmd, ":W" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "Export" ) );
		common->changePageRequest( PAGE_TIMELINE );
	}

	/* quit */

	else if ( strcmp( cmd, ":q" ) == 0 )
	{
		common->keyboardFeedback( cmd, _( "quit" ) );
		kinoDeactivate();
	}

	return FALSE;
}

static void on_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data )
{
	( ( PageMagick * ) user_data ) ->LoadSplash( );
}


class FXSelectedFrames : public SelectedFrames
{
protected:
	bool seekable;
	int start;
	int end;
	double increment;
	bool frames_reverse;
	bool effect_reverse;
	double real_start;
	double real_end;
	int b_frame_type;
	GdkColor color;

public:
	void Initialise( PageMagick *page )
	{
		GtkNotebook * notebook = GTK_NOTEBOOK( lookup_widget( page->window, "notebook_magick_frames" ) );
		int source = gtk_notebook_get_current_page( notebook );

		if ( source == 0 )
		{
			// Obtain begining and ending of sequence
			start = 0;
			end = GetCurrentPlayList().GetNumFrames();
			increment = 1;
			frames_reverse = false;
			effect_reverse = false;
			real_start = 0;
			real_end = 1;

			GtkEntry *startSpin = GTK_ENTRY( lookup_widget( page->window, "spinbutton_magick_start" ) );
			GtkEntry *endSpin = GTK_ENTRY( lookup_widget( page->window, "spinbutton_magick_end" ) );

			if ( GetCurrentPlayList().GetNumFrames() != 0 )
			{
				seekable = true;
				start = atoi( gtk_entry_get_text( startSpin ) );
				end = atoi( gtk_entry_get_text( endSpin ) );
				GtkToggleButton *limit = GTK_TOGGLE_BUTTON( lookup_widget( page->window, "checkbutton_magick_frame_limit" ) );
				if ( gtk_toggle_button_get_active( limit ) )
				{
					GtkEntry * spin = GTK_ENTRY( lookup_widget( page->window, "spinbutton_magick_frame_count" ) );
					GtkMenu *menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( page->window, "optionmenu_magick_frame_offset" ) ) ) );
					GtkWidget *active_item = gtk_menu_get_active( menu );
					if ( g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item ) == 1 )
					{
						int requested_end = start + atoi( gtk_entry_get_text( spin ) ) - 1;
						if ( requested_end < end )
							end = requested_end;
					}
					else
					{
						start = end - atoi( gtk_entry_get_text( spin ) ) + 1;
					}
				}
			}
			else
			{
				seekable = false;
			}

			// Determine speed
			GtkToggleButton *speed = GTK_TOGGLE_BUTTON( lookup_widget( page->window, "checkbutton_speed" ) );
			if ( gtk_toggle_button_get_active( speed ) )
			{
				GtkRange * range = GTK_RANGE( lookup_widget( page->window, "hscale_speed" ) );
				increment = range->adjustment->value;
			}

			// Determine direction
			GtkToggleButton *reverse_button = GTK_TOGGLE_BUTTON( lookup_widget( page->window, "checkbutton_reverse" ) );
			frames_reverse = gtk_toggle_button_get_active( reverse_button );
		}
		else
		{
			seekable = false;
		}

		// Check if we're reversed (currently only image transitions have a global effect reverse)
		GtkNotebook *notebookImage = GTK_NOTEBOOK( lookup_widget( page->window, "notebook_magick_video" ) );

		int current = gtk_notebook_get_current_page( notebookImage );

		if ( current == 1 )
		{
			GtkMenu * menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( page->window, "optionmenu_direction" ) ) ) );
			GtkWidget *active_item = gtk_menu_get_active( menu );
			effect_reverse = g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item ) == 1;

			GtkRange *range = GTK_RANGE( lookup_widget( page->window, "hscale_transition_start" ) );
			real_start = range->adjustment->value;
			range = GTK_RANGE( lookup_widget( page->window, "hscale_transition_end" ) );
			real_end = range->adjustment->value;

			GtkColorButton *colorButton = GTK_COLOR_BUTTON( lookup_widget( page->window, "colorpicker_magick_transition" ) );
			gtk_color_button_get_color( colorButton, &color );

			GtkToggleButton *toggle = GTK_TOGGLE_BUTTON( lookup_widget( page->window, "radiobutton_magick_transition_colour" ) );

			if ( !gtk_toggle_button_get_active( toggle ) )
			{
				GtkMenu * menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( page->window, "optionmenu_magick_transition_frame" ) ) ) );
				GtkWidget *active_item = gtk_menu_get_active( menu );
				b_frame_type = g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item );
			}
			else
			{
				b_frame_type = 3;
			}
		}
		else
		{
			b_frame_type = -1;
			effect_reverse = false;
			real_start = 0;
			real_end = 1;
		}
	}

	int GetNumInputFrames( )
	{
		if ( seekable )
			return end - start + 1;
		else
			return 0;
	}

	bool IsEffectReversed( )
	{
		return effect_reverse;
	}

	double GetRealStart( )
	{
		return real_start;
	}

	double GetRealEnd( )
	{
		return real_end;
	}

	int GetNumOutputFrames( )
	{
		return ( int ) ( GetNumInputFrames( ) / increment );
	}

	double GetPosition( int index )
	{
		return index / GetNumOutputFrames( );
	}

	double GetFrameDelta( )
	{
		return 1 / GetNumOutputFrames( );
	}

	int GetIndex( double position )
	{
		if ( frames_reverse )
			position = 1 - position;
		return ( int ) ( GetNumInputFrames( ) * position );
	}

	void GetScaledImage( int index, uint8_t *image, int width = 0, int height = 0 )
	{
		// Get the current playlist
		PlayList & list = GetCurrentPlayList();

		if ( index >= 0 && index < list.GetNumFrames( ) )
		{
			// Get a frame to work with
			Frame & frame = *( GetFramePool() ->GetFrame() );

			frame.decoder->quality = DV_QUALITY_BEST;

			// Obtain the frame as specified by the index
			list.GetFrame( index, frame );

			// If the width and height is specified
			if ( width != 0 && height != 0 )
			{
				// Create a temporary space to hold the full RGB image
				uint8_t * temp = new uint8_t[ frame.GetWidth( ) * frame.GetHeight( ) * 3 ];

				// Extract it
				frame.ExtractRGB( temp );

				// Convert to a GDK Image
				GdkPixbuf *im1 = gdk_pixbuf_new_from_data( temp, GDK_COLORSPACE_RGB, FALSE, 8,
				                 frame.GetWidth( ), frame.GetHeight( ), frame.GetWidth( ) * 3, NULL, NULL );

				// Clone and scale
				GdkPixbuf *im2 = gdk_pixbuf_scale_simple( im1, width, height, GDK_INTERP_HYPER );

				// Copy it into the image
				memcpy( image, gdk_pixbuf_get_pixels( im2 ), width * height * 3 );

				// Destroy the GDK images
				g_object_unref( im2 );
				g_object_unref( im1 );

				// Delete the temporary space
				delete[] temp;
			}
			else
			{
				// Extract it
				frame.ExtractRGB( image );
			}

			GetFramePool() ->DoneWithFrame( &frame );
		}
		else
		{
			uint8_t *end = image + width * height * 3;
			for ( uint8_t * p = image; p < end; )
			{
				*p ++ = color.red >> 8;
				*p ++ = color.green >> 8;
				*p ++ = color.blue >> 8;
			}
		}
	}

	void GetImageA( double position, uint8_t *image, int width = 0, int height = 0 )
	{
		GetScaledImage( start + GetIndex( position ), image, width, height );
	}

	void GetImageB( double position, uint8_t *image, int width = 0, int height = 0 )
	{
		switch ( b_frame_type )
		{
		case 1:
			GetScaledImage( ( int ) ( end + 1 ), image, width, height );
			break;
		case 2:
			GetScaledImage( ( int ) ( start - 1 ), image, width, height );
			break;
		case 0:
			GetScaledImage( ( int ) ( end + 1 + position * GetNumInputFrames( ) ), image, width, height );
			break;
		case - 1:
		case 3:
			{
				uint8_t *end = image + width * height * 3;
				for ( uint8_t * p = image; p < end; )
				{
					*p ++ = color.red >> 8;
					*p ++ = color.green >> 8;
					*p ++ = color.blue >> 8;
				}
			}
			break;
		}
	}

	void GetAudio( int index, int16_t **audio, short int &channels, int &frequency, int &samples )
	{
		// Get the current playlist
		PlayList & list = GetCurrentPlayList();

		if ( index >= 0 && index < list.GetNumFrames( ) )
		{
			// Get a frame to work with
			Frame & frame = *( GetFramePool() ->GetFrame() );

			// Obtain the frame as specified by the index
			list.GetFrame( index, frame );

			// Extract the audio
			frame.ExtractAudio( audio );

			// Update the info
			AudioInfo info;
			frame.GetAudioInfo( info );
			channels = info.channels;
			frequency = info.frequency;
			samples = info.samples;

			// Return the frame to the pool
			GetFramePool() ->DoneWithFrame( &frame );
		}
		else if ( list.GetNumFrames( ) )
		{
			// Get a frame to work with
			Frame & frame = *( GetFramePool() ->GetFrame() );

			// Obtain the frame as specified by the index
			list.GetFrame( index, frame );

			// Update the info
			AudioInfo info;
			frame.GetAudioInfo( info );
			channels = info.channels;
			frequency = info.frequency;
			samples = info.samples;

			// Wipe the audio
			for ( int i = 0; i < channels; i ++ )
				memset( audio[ i ], 0, samples * sizeof( int16_t ) );

			// Return the frame to the pool
			GetFramePool() ->DoneWithFrame( &frame );
		}
	}

	void GetAudioA( double position, int16_t **audio, short int &channels, int &frequency, int &samples )
	{
		GetAudio( start + GetIndex( position ), audio, channels, frequency, samples );
	}

	void GetAudioB( double position, int16_t **audio, short int &channels, int &frequency, int &samples )
	{
		switch ( b_frame_type )
		{
		case 0:
			GetAudio( ( int ) ( end + 1 + position * GetNumInputFrames( ) ), audio, channels, frequency, samples );
			break;
		case - 1:
		case 1:
		case 2:
		case 3:
			GetAudio( -1, audio, channels, frequency, samples );
			break;
		}
	}
};

// Get the selected frames for FX
SelectedFrames &GetSelectedFramesForFX( )
{
	static FXSelectedFrames selected;
	selected.Initialise( common->getPageMagick( ) );
	return selected;
}
