/*
    This file is part of libqobex.

    Copyright (c) 2003 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>

#include <qapplication.h>
#include <qstring.h>
#include <qcstring.h>
#include <qfile.h>

#include <qobex/qobexserver.h>
#include <qobex/qobexserverops.h>
#include <qobex/qobexintransport.h>
#ifdef HAVE_QOBEX_BLUETOOTH
#include <qobex/qobexbttransport.h>
#endif
#ifdef HAVE_QOBEX_IRDA
#include <qobex/qobexirdatransport.h>
#endif
#include <qobex/qobexuuid.h>

#include "qobexfbsserverops.h"

void print_usage()
{
  fprintf( stderr, "Use the source luke!\n" );
}

// gracefull shutdown. Especially the bluetooth transport should deregister
// its sdp record ...
void shutdownhandler( int )
{
  QApplication::exit( EXIT_SUCCESS );
}

int main( int argc, char **argv )
{
  QApplication app( argc, argv, false );

  ::signal( SIGINT, shutdownhandler );
  ::signal( SIGTERM, shutdownhandler );
  ::signal( SIGABRT, shutdownhandler );
  ::signal( SIGQUIT, shutdownhandler );
  ::signal( SIGHUP, shutdownhandler );

  // When this is qapps child it is deleted when the app exits ...
  QObexServer* serv = new QObexServer( &app, "OBEX server" );
  serv->slotRegisterServerOps( new QObexFBSServerOps );

  bool nothing = true;

  int c;
  while ((c = getopt( argc, argv, "c:t:v" )) != EOF)
    switch ( c ) {
    case 'c':
      if ( ::chdir( optarg ) ) {
	fprintf( stderr, "Can not enter directory given in -c command line argment.\n" );
	delete serv;
	exit( EXIT_FAILURE );
      }
      break;
    case 't':
      {
	QString ts = optarg;
#ifdef HAVE_QOBEX_BLUETOOTH
	if ( ts == "bt" || ts == "bluetooth" ) {
	  // Register bluetooth transport
	  QObexBtTransport *btTransport = new QObexBtTransport;
	  btTransport->setRfCommChannel( 17 );
	  btTransport->setUuid( QObexBtTransport::ObexFileServ );
	  serv->slotRegisterTransport( btTransport );
	  nothing = false;
	} else
#endif
#ifdef HAVE_QOBEX_IRDA
	if ( ts == "ir" || ts == "irda" ) {
	  // Register IrDA transport
	  serv->slotRegisterTransport( new QObexIrDATransport );
	  nothing = false;
	} else
#endif
	if ( ts == "in" || ts == "ip" || ts == "tcp" ) {
	  // Register IP transport
	  QObexInTransport* inTransport = new QObexInTransport;
	  inTransport->setPort( QObexInTransport::ObexUserPort );
	  serv->slotRegisterTransport( inTransport );
	  nothing = false;

	} else {
	  fprintf( stderr, "Invalid transport argument.\n" );
	  delete serv;
	  exit( EXIT_FAILURE );
	}
      }
      break;
    case 'v':
//       verbose = true;
      break;
    default:
      print_usage();
      delete serv;
      exit( EXIT_FAILURE );
    }

  // Default is to listen on localhost 
  if ( nothing ) {
    QObexInTransport* inTransport = new QObexInTransport;
    inTransport->setPort( QObexInTransport::ObexUserPort );
    inTransport->setSrc( "127.0.0.1" );
    serv->slotRegisterTransport( inTransport );
  }

  return app.exec();
}
