//-*-c++-*-
/***************************************************************************
 *   Copyright (C) 2003 by Fred Schaettgen                                 *
 *   kdebluetooth@schaettgen.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <kapplication.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kurl.h>
#include <qwidget.h>
#include <libkbluetooth/serviceselectionwidget.h>
#include <libkbluetooth/deviceinputwidget.h>
#include <libkbluetooth/deviceaddress.h>
#include <kdialogbase.h>
#include <iostream>
#include <libkbluetooth/configinfo.h>

static const char description[] =
    I18N_NOOP("Bluetooth device/service search utility");

static const char* version = KDEBluetoothConfig::version;

static KCmdLineOptions options[] =
{
//        {"rfcommconnection <sockfd>", I18N_NOOP("Used by kbluetoothd."), 0},
//        {"peeraddr <addr>", I18N_NOOP("Used by kbluetoothd."), 0},
//        {"peername <name>", I18N_NOOP("Used by kbluetoothd."), 0},
    {"u", 0 ,0}, 
    {"uuid [UUID]", I18N_NOOP("Service search for uuid: like 0x1101 or UUID alias"), 0},
    {"aliases", I18N_NOOP("Show list of UUID aliases"), 0},
    {"d", 0 ,0}, 
    {"device", I18N_NOOP("Device search"), 0},
    {"t", 0, 0},
    {"title [window title]", I18N_NOOP("The title of the selection window"), 0},
    KCmdLineLastOption
};

struct AliasInfo {
    const char* alias;
    const char* uuid;
    const char* desc;
};

AliasInfo aliases[] = {
{"SPP", "0x1101", "Serial Port"},
{"PPP", "0x1103", "Dial-up networking"},
{"IRM", "0x1104", "IrMC sync"},
{"OOP", "0x1105", "OBEX Object Push"},
{"OFT", "0x1106", "OBEX File Transfer"},
{"HSP", "0x1108", "Headset Profile"},
{"FAX", "0x1111", "Fax Profile"},
{"HSG", "0x1112", "Headset Audio Gateway"},
{"HFP", "0x111E", "Handsfree Profile"},
{"HFG", "0x111F", "Handsfree Audio Gateway"},
{NULL, NULL, NULL}
};

bool showServiceSearchDialog(const QString& title, const QString& uuid)
{
    KDialogBase dialogBase(NULL, "searchdialog", true, title, 
    KDialogBase::Ok | KDialogBase::Cancel);
    KBluetooth::ServiceSelectionWidget widget(&dialogBase, QStringList(uuid), true, false);
    dialogBase.setMainWidget(&widget);
    widget.initFromCache();
    widget.scan();
    dialogBase.show();
    if (dialogBase.exec() == QDialog::Accepted) {
        KBluetooth::DeviceAddress address = widget.selectedAddress();
        int channel = widget.selectedChannel();
        // don't even try to change this to kdDebug() !!
        std::cout << QString(address).local8Bit() << " " 
            << channel << " " << uuid.local8Bit() << std::endl;
        return true;
    }
    else return false;
}

bool showDeviceSearchDialog(const QString& title)
{
    KDialogBase dialogBase(NULL, "searchdialog", true, title, 
        KDialogBase::Ok | KDialogBase::Cancel);
    KBluetooth::DeviceInputWidget widget(&dialogBase);
    dialogBase.setMainWidget(&widget);
    dialogBase.connect(&widget, SIGNAL(addressValid(bool)), 
        &dialogBase, SLOT(enableButtonOK(bool)));
    widget.init(true);
    widget.startSearch();
    dialogBase.show();
    if (dialogBase.exec() == QDialog::Accepted) {
        KBluetooth::DeviceAddress address = widget.currentAddress();
        // don't even try to change this to kdDebug() !!
        std::cout << QString(address).local8Bit() << std::endl;
        return true;
    }
    else return false;
}

int main(int argc, char **argv)
{
    KAboutData about("kbtsearch",
        I18N_NOOP("Bluetooth Search"),
        version, description,
        KAboutData::License_GPL,
        "(C) 2004 Fred Schaettgen", 0, 0,
        "kdebluetooth@schaettgen.de");
    KCmdLineArgs::init(argc, argv, &about);
    KCmdLineArgs::addCmdLineOptions(options);
    KApplication app;
    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

    bool ret = false;
    QString title = about.programName();
    
    if (args->isSet("aliases")) {
        for (AliasInfo* info = aliases; info->alias != NULL; ++info) {
            // don't even try to change this to kdDebug() !!
            std::cout << info->alias << " - " << info->uuid << " / " << info->desc << std::endl;
        }
        ret = true;
    }
    
    else if (args->isSet("uuid")) {
        QString uuid = args->getOption("uuid");
        QString desc = uuid;
        for (AliasInfo* info = aliases; info->alias != NULL; ++info) {
            if ((uuid == QString::fromAscii(info->alias)) ||
                (uuid == QString::fromAscii(info->uuid))) 
            {
                uuid = info->uuid;
                desc = info->desc;
                break;
            }
        }
        title = title+" ("+desc+")";
        
        if (args->isSet("title")) {
            title = args->getOption("title");
        }
        ret = showServiceSearchDialog(title, uuid);
    }
    
    else if (args->isSet("device")) {
        if (args->isSet("title")) {
            title = args->getOption("title");
        }
        ret = showDeviceSearchDialog(title);
    }
        
    return (ret?0:1);
}

