
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: daemonize.c,v 1.17 2005/06/06 11:13:41 ro_han Exp $
 *****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>

#include "common.h"
#include "daemonize.h"
#include "log.h"
#include "manager.h"

void signal_handler(int sig);

gboolean daemonize()
{
    DEBUG("Daemonizing...");

    int child_pid;

    if (chdir("/") < 0)
    {
        DEBUG("Could not chdir to /, errno=%d", errno);
        return FALSE;
    }

    child_pid = fork();
    switch (child_pid)
    {
    case -1:
        DEBUG("Cannot fork(), errno=%d", errno);
        break;

    case 0:
        break;

    default:
        exit(0);
        break;
    }

    /* Create session */
    setsid();

    int lfp = open(cfg_base->pidFile, O_RDWR | O_CREAT, 0640);

    if (lfp < 0)
    {
        DEBUG("Cannot open %s", cfg_base->pidFile);
        return FALSE;
    }
    if (lockf(lfp, F_TLOCK, 0) < 0)
    {
        DEBUG("Cannot get a lock on %s", cfg_base->pidFile);
        return FALSE;
    }
    char str[10];

    sprintf(str, "%d\n", getpid());
    write(lfp, str, strlen(str));

    freopen("/dev/null", "r", stdin);
    freopen("/dev/null", "w", stdout);
    freopen("/dev/null", "w", stderr);

    umask(027);


    signal(SIGTSTP, SIG_IGN);
    signal(SIGTTOU, SIG_IGN);
    signal(SIGTTIN, SIG_IGN);
    signal(SIGTERM, signal_handler);
    signal(SIGINT, signal_handler);

    return TRUE;
}

void signal_handler(int sig)
{
    switch (sig)
    {
    case SIGTERM:
        DEBUG("Exiting");

        log_notice("IVM stopped");
        clear_pidfile(cfg_base->pidFile);
        exit(0);
        break;
    }
}

void clear_pidfile(char *file)
{
    if (file != NULL)
    {
        DEBUG("Deleting PID file %s", file);
        if (unlink(file) == -1)
        {
            DEBUG("Failed deleting PID %s file, continuing", file);
            errno = 0;
        }
    }
}

gboolean dropPrivileges(char * user, char * group)
{
    // Drop privileges
    if (chdir("/") != 0)
    {
        DEBUG("Couldn't chdir to /: %s", strerror(errno));
        return FALSE;
    }

    struct group * grpentry = getgrnam(group); 
    if (grpentry == NULL)
    {
        DEBUG("Group '%s' does not appear to exist!",group);
        return FALSE;
    }
    unsigned int gid = grpentry->gr_gid;
    if (geteuid() == 0 && setgid(gid) != 0)
    {
        DEBUG("Couldn't setgid to %s: %s", group, strerror(errno));
        return FALSE;
    }

    struct passwd * pwdentry = getpwnam(user);
    if (pwdentry == NULL)
    {
        DEBUG("User '%s' does not appear to exist!",user);
        return FALSE;
    }
    unsigned int uid = pwdentry->pw_uid;
    if (geteuid() == 0 && setuid(uid) != 0)
    {
        DEBUG("Couldn't setuid to %s: %s", user, strerror(errno));
        return FALSE;
    }

    return TRUE;
}

