/* ptal-hp -- PTAL command-line HP peripheral configuration/status client */

/* Copyright (C) 2000-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>
#include "ptal.h"

enum {
	RETCODE_SUCCESS=0,
	RETCODE_SYNTAX_ERROR=3,
	RETCODE_DEVICE_ERROR=4
};

static char *argv0=0;

void printOneLine(ptalPmlObject_t obj,int stringType) {
	int symbolSet,type,len,i,start;
	unsigned char buffer[PTAL_PML_MAX_VALUE_LEN+1];

	if (stringType) {
		len=ptalPmlGetStringValue(obj,&symbolSet,buffer,
			PTAL_PML_MAX_VALUE_LEN);
	} else {
		len=ptalPmlGetValue(obj,&type,buffer,PTAL_PML_MAX_VALUE_LEN);
	}
	if (len==PTAL_ERROR) {
		printf("(line unavailable)\n");
		return;
	}

	/* Reduce len if there's a null before it. */
	if (len>strlen((char *)buffer)) len=strlen((char *)buffer);

	/* Convert non-printable characters to whitespace. */
	for (i=0;i<len;i++) if (buffer[i]<' ') buffer[i]=' ';

	/* Strip leading and trailing whitespace. */
	for (start=0;start<len && isspace(buffer[start]);start++);
	for (;len>start && isspace(buffer[len-1]);len--);

	/* Make sure the string is null-terminated in the right place. */
	buffer[len]=0;

	printf("%s\n",buffer+start);
}

int handleDevice(ptalDevice_t dev,int argc,char **argv) {
	ptalPmlObject_t obj=ptalPmlAllocate(dev);

	printf("Model name:        ");
	ptalPmlSetID(obj,"\x1\x1\x3\x2");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj,1);
	}

	printf("Model number:      ");
	ptalPmlSetID(obj,"\x1\x1\x3\x1");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj,1);
	}

	printf("Serial number:     ");
	ptalPmlSetID(obj,"\x1\x1\x3\x3");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj,1);
	}

	printf("Firmware version:  ");
	ptalPmlSetID(obj,"\x1\x1\x3\x6");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj,1);
	}

	printf("Firmware datecode: ");
	ptalPmlSetID(obj,"\x1\x1\x3\x5");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj,1);
	}

	ptalPmlDeallocate(obj);
	return RETCODE_SUCCESS;
}

int handleDisplay(ptalDevice_t dev,int argc,char **argv) {
	ptalPmlObject_t objLine1=ptalPmlAllocateID(dev,
		"\x1\x1\x2\x14\x2\x1\x1");
	ptalPmlObject_t objLine2=ptalPmlAllocateID(dev,
		"\x1\x1\x2\x14\x2\x2\x1");
	ptalPmlObject_t objSpmLine1=ptalPmlAllocateID(dev,
		"\x2\x10\x5\x1\x2\x1\x1");
	ptalPmlObject_t objSpmLine2=ptalPmlAllocateID(dev,
		"\x2\x10\x5\x1\x2\x1\x2");
	int r1,r2;

	r1=ptalPmlRequestGet(objLine1,0);
	r2=ptalPmlRequestGet(objLine2,0);
	if (r1!=PTAL_ERROR || r2!=PTAL_ERROR) {
		if (r1!=PTAL_ERROR) printOneLine(objLine1,1);
		if (r2!=PTAL_ERROR) printOneLine(objLine2,1);

	} else {
		r1=ptalPmlRequestGet(objSpmLine1,0);
		r2=ptalPmlRequestGet(objSpmLine2,0);
		if (r1!=PTAL_ERROR || r2!=PTAL_ERROR) {
			if (r1!=PTAL_ERROR) printOneLine(objSpmLine1,0);
			if (r2!=PTAL_ERROR) printOneLine(objSpmLine2,0);

		} else {
			printf("(unavailable)\n");
		}
	}

	ptalPmlDeallocate(objLine1);
	ptalPmlDeallocate(objLine2);
	ptalPmlDeallocate(objSpmLine1);
	ptalPmlDeallocate(objSpmLine2);

	return RETCODE_SUCCESS;
}

int handleClock(ptalDevice_t dev,int argc,char **argv) {
	int retcode;
	ptalPmlObject_t objClock=ptalPmlAllocateID(dev,"\x1\x1\x2\x11");
	ptalPmlObject_t objDateFormat=ptalPmlAllocateID(dev,"\x1\x1\x1\x16");
	ptalPmlObject_t objTimeFormat=ptalPmlAllocateID(dev,"\x1\x1\x2\x1C");
	time_t tt=time(0);
	struct tm *tlocal=localtime(&tt);
	struct {
		unsigned char year;
		unsigned char month;
		unsigned char day;
		unsigned char weekday;	/* Ignore: some models misreport. */
		unsigned char hour;
		unsigned char minute;
		unsigned char second;
	} __attribute__((packed)) pmlClock;
	enum {
		MMDDYY=1,
		DDMMYY=2,
		YYMMDD=3
	} dateFormat;
	enum {
		hour12=1,
		hour24=2
	} timeFormat;
	int month,year,hour,ampm;
	char *smonth[]={"???","Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"};

	while (argc) {
		if (!strcmp(*argv,"-set")) {
			pmlClock.year=tlocal->tm_year%100;
			pmlClock.month=tlocal->tm_mon+1;
			pmlClock.day=tlocal->tm_mday;
			pmlClock.weekday=tlocal->tm_wday+1;
			pmlClock.hour=tlocal->tm_hour;
			pmlClock.minute=tlocal->tm_min;
			pmlClock.second=tlocal->tm_sec;
			ptalPmlSetValue(objClock,PTAL_PML_TYPE_BINARY,
				(char *)&pmlClock,sizeof(pmlClock));
			ptalPmlRequestSet(objClock);

		} else if (!strcmp(*argv,"-mmddyy")) {
			dateFormat=MMDDYY;
setDateFormat:
			ptalPmlSetIntegerValue(objDateFormat,
				PTAL_PML_TYPE_ENUMERATION,dateFormat);
			ptalPmlRequestSet(objDateFormat);

		} else if (!strcmp(*argv,"-ddmmyy")) {
			dateFormat=DDMMYY;
			goto setDateFormat;

		} else if (!strcmp(*argv,"-yymmdd")) {
			dateFormat=YYMMDD;
			goto setDateFormat;

		} else if (!strcmp(*argv,"-12hour")) {
			timeFormat=hour12;
setTimeFormat:
			ptalPmlSetIntegerValue(objTimeFormat,
				PTAL_PML_TYPE_ENUMERATION,timeFormat);
			ptalPmlRequestSet(objTimeFormat);

		} else if (!strcmp(*argv,"-24hour")) {
			timeFormat=hour24;
			goto setTimeFormat;

		} else {
			PTAL_LOG_ERROR(
"Syntax: %s [<devname>] clock [<params>...]\n"
"Where <devname> may be one of:\n"
				,argv0);
			ptalDeviceEnumerate(0,
				ptalDeviceEnumeratePrintCallback,0);
			PTAL_LOG_ERROR(
"Valid <params>:\n"
"\t-set                      -- Set clock to PC clock\n"
"\t-mmddyy, -ddmmyy, -yymmdd -- Set date format\n"
"\t-12hour, -24hour          -- Set time format\n"
				);
			retcode=RETCODE_SYNTAX_ERROR;
			goto abort;
		}

		argc--; argv++;
	}

	if (ptalPmlRequestGet(objDateFormat,0)==PTAL_ERROR ||
	    ptalPmlGetIntegerValue(objDateFormat,0,(int *)&dateFormat)==
	     PTAL_ERROR) {
		dateFormat=MMDDYY;
	}

	if (ptalPmlRequestGet(objTimeFormat,0)==PTAL_ERROR ||
	    ptalPmlGetIntegerValue(objTimeFormat,0,(int *)&timeFormat)==
	     PTAL_ERROR) {
		timeFormat=hour12;
	}

	printf("Device clock: ");
	if (ptalPmlRequestGet(objClock,0)==PTAL_ERROR ||
	    ptalPmlGetValue(objClock,0,(char *)&pmlClock,sizeof(pmlClock))==
	     PTAL_ERROR) {
		printf("(unavailable)\n");

	} else {
		month=pmlClock.month;
		if (month<1 || month>12) month=0;
		year=1900+((tlocal->tm_year/100)*100)+(pmlClock.year%100);
		if (dateFormat==YYMMDD) {
			printf("%d-%s-%d ",year,smonth[month],pmlClock.day);
		} else if (dateFormat==DDMMYY) {
			printf("%d-%s-%d ",pmlClock.day,smonth[month],year);
		} else /* if (dateFormat==MMDDYY) */ {
			printf("%s-%d-%d ",smonth[month],pmlClock.day,year);
		}

		hour=pmlClock.hour;
		if (timeFormat==hour24) {
			printf("%d:%2.2d:%2.2d\n",
				hour,pmlClock.minute,pmlClock.second);
		} else /* if (timeFormat==hour12) */ {
			ampm=(hour>=12);
			hour%=12;
			if (!hour) hour=12;
			printf("%d:%2.2d:%2.2d %s\n",
				hour,pmlClock.minute,pmlClock.second,
				!ampm?"AM":"PM");
		}
	}

	retcode=RETCODE_SUCCESS;
abort:

	ptalPmlDeallocate(objClock);
	ptalPmlDeallocate(objDateFormat);
	ptalPmlDeallocate(objTimeFormat);

	return retcode;
}

typedef int (*handleCommand_f)(ptalDevice_t dev,int argc,char **argv);

struct {
	char *string;
	handleCommand_f function;
} commandTable[]={
	{"device",handleDevice},
	{"display",handleDisplay},
	{"clock",handleClock},
	{0,0}
};

static handleCommand_f lookupCommand(char *cmd) {
	int i;
	for (i=0;commandTable[i].string;i++) {
		if (!strcmp(cmd,commandTable[i].string)) {
			return commandTable[i].function;
		}
	}
	return 0;
}

int main(int argc,char **argv) {
	ptalDevice_t dev=0;
	handleCommand_f function;

	argv0=*argv;
	argc--; argv++;

	ptalInit();

	if (argc<=0) {
		int i;
syntaxError:
		PTAL_LOG_ERROR(
"Syntax: %s [<devname>] <command> [-help] [<params>...]\n"
"Where <devname> may be one of:\n"
			,argv0);
		ptalDeviceEnumerate(0,
			ptalDeviceEnumeratePrintCallback,0);
		PTAL_LOG_ERROR(
"Valid commands:\n"
			);
		for (i=0;commandTable[i].string;i++) {
			printf("\t%s\n",commandTable[i].string);
		}
		return RETCODE_SYNTAX_ERROR;
	}

	function=lookupCommand(*argv);
	if (function) {
		dev=ptalDeviceOpen(0);
		if (!dev) {
			PTAL_LOG_ERROR("%s: No default device defined!\n\n",
				argv0);
			goto syntaxError;
		}
	} else {
		if (**argv=='-') goto syntaxError;
		dev=ptalDeviceOpen(*argv);
		if (!dev) {
			PTAL_LOG_ERROR("%s: Unable to open "
				"device \"%s\"!\n\n",argv0,*argv);
			goto syntaxError;
		}
		argc--; argv++;
		if (argc<=0) goto syntaxError;
		function=lookupCommand(*argv);
		if (!function) goto syntaxError;
	}

	if (ptalPmlOpen(dev)==PTAL_ERROR) {
		PTAL_LOG_ERROR("%s: Unable to communicate with "
			"device \"%s\"!\n\n",
			argv0,ptalDeviceGetName(dev));
		goto syntaxError;
	}

	argc--; argv++;
	return function(dev,argc,argv);
}
