/* grepmap
 *
 * inputmap.c - parse modules.inputmap
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/**
 * inputmap_parse_args:
 * @argc: number of arguments,
 * @argv: arguments array,
 * @dev: device structure to fill.
 *
 * Parses the command-line arguments to fill a device structure.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
inputmap_parse_args (int          argc,
		     char        *argv[],
		     InputDevice *dev)
{
	int ret = 0;

	if (argc < 13) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("insufficient arguments"));
		suggest_help ();
		return 2;
	}

	ret |= parse_hex (argv[0], &dev->bus);
	ret |= parse_hex (argv[1], &dev->vendor);
	ret |= parse_hex (argv[2], &dev->product);
	ret |= parse_hex (argv[3], &dev->version);
	ret |= parse_array (argv[4], dev->evbit, EVBIT_LEN);
	ret |= parse_array (argv[5], dev->keybit, KEYBIT_LEN);
	ret |= parse_array (argv[6], dev->relbit, RELBIT_LEN);
	ret |= parse_array (argv[7], dev->absbit, ABSBIT_LEN);
	ret |= parse_array (argv[8], dev->mscbit, MSCBIT_LEN);
	ret |= parse_array (argv[9], dev->ledbit, LEDBIT_LEN);
	ret |= parse_array (argv[10], dev->sndbit, SNDBIT_LEN);
	ret |= parse_array (argv[11], dev->ffbit, FFBIT_LEN);
	ret |= parse_array (argv[12], dev->swbit, SWBIT_LEN);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 2;
	}

	if (argc > 13)
		fprintf (stderr, "%s: %s\n", program_name,
			 _("some arguments ignored"));

	return 0;
}

/**
 * inputmap_udev_event:
 * @environ: environment array,
 * @dev: device structure to fill.
 *
 * Fills the device structure by reading a udev event from the environment.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
inputmap_udev_event (char        **environ,
		     InputDevice  *dev)
{
	char **envvar;
	int    ret = 0;

	for (envvar = environ; *envvar; envvar++) {
		char *key, *value;

		key = strdup (*envvar);
		value = strchr (key, '=');
		if (! value)
			goto next;
		*(value++) = 0;

		if (! strcmp (key, "PRODUCT")) {
			char *ptr1, *ptr2, *ptr3;

			ptr1 = strchr (value, '/');
			if (! ptr1) {
				ret = 1;
				goto next;
			}
			*(ptr1++) = 0;

			ptr2 = strchr (ptr1, '/');
			if (! ptr2) {
				ret = 1;
				goto next;
			}
			*(ptr2++) = 0;

			ptr3 = strchr (ptr2, '/');
			if (! ptr3) {
				ret = 1;
				goto next;
			}
			*(ptr3++) = 0;

			ret |= parse_hex (value, &dev->bus);
			ret |= parse_hex (ptr1, &dev->vendor);
			ret |= parse_hex (ptr2, &dev->product);
			ret |= parse_hex (ptr3, &dev->version);

		} else if (! strcmp(key, "EV")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->evbit, EVBIT_LEN);

		} else if (! strcmp(key, "KEY")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->keybit, KEYBIT_LEN);

		} else if (! strcmp(key, "REL")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->relbit, RELBIT_LEN);

		} else if (! strcmp(key, "ABS")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->absbit, ABSBIT_LEN);

		} else if (! strcmp(key, "MSC")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->mscbit, MSCBIT_LEN);

		} else if (! strcmp(key, "LED")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->ledbit, LEDBIT_LEN);

		} else if (! strcmp(key, "SND")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->sndbit, SNDBIT_LEN);

		} else if (! strcmp(key, "FF")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->ffbit, FFBIT_LEN);

		} else if (! strcmp(key, "SW")) {
			char *ptr;

			for (ptr = value; *ptr; ptr++)
				if (*ptr == ' ')
					*ptr = ':';

			ret |= parse_array (value, dev->swbit, SWBIT_LEN);

		}

	next:
		free (key);
	}

	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("udev event not in expected format"));
		return 2;
	}

	return 0;
}

/**
 * inputmap_modules:
 * @mapf: open map file to read from,
 * @file: filename or map file, or '-' for standard input,
 * @dev: device to look for.
 *
 * Handles the modules.inputmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
inputmap_modules (FILE        *mapf,
		  const char  *file,
		  InputDevice *dev)
{
	char *line;
	int   lineno = 0, ret = 0;

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		InputDevice map;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		if (parse_line (line, "xxxxxaaaaaaaaa",
				(unsigned int *)&map.match_flags, &map.bus,
				&map.vendor, &map.product, &map.version,
				map.evbit, EVBIT_LEN, map.keybit, KEYBIT_LEN,
				map.relbit, RELBIT_LEN, map.absbit, ABSBIT_LEN,
				map.mscbit, MSCBIT_LEN, map.ledbit, LEDBIT_LEN,
				map.sndbit, SNDBIT_LEN, map.ffbit, FFBIT_LEN,
				map.swbit, SWBIT_LEN))
		{
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, INPUT_MATCH_BUS)
		    && (map.bus != dev->bus))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_VENDOR)
		    && (map.vendor != dev->vendor))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_PRODUCT)
		    && (map.product != dev->product))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_VERSION)
		    && (map.version != dev->version))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_EVBIT)
		    && match_array (map.evbit, dev->evbit, EVBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_KEYBIT)
		    && match_array (map.keybit, dev->keybit, KEYBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_RELBIT)
		    && match_array (map.relbit, dev->relbit, RELBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_ABSBIT)
		    && match_array (map.absbit, dev->absbit, ABSBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_MSCBIT)
		    && match_array (map.mscbit, dev->mscbit, MSCBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_LEDBIT)
		    && match_array (map.ledbit, dev->ledbit, LEDBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_SNDBIT)
		    && match_array (map.sndbit, dev->sndbit, SNDBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_FFBIT)
		    && match_array (map.ffbit, dev->ffbit, FFBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_SWBIT)
		    && match_array (map.swbit, dev->swbit, SWBIT_LEN))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
