/* grepmap
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef GREPMAP_GREPMAP_H
#define GREPMAP_GREPMAP_H

/* Required includes */
#include <stdio.h>
#include <gettext.h>


/* Make gettext a little friendlier */
#define _(String) gettext (String)
#define N_(String) gettext_noop (String)

/* Check a flag is set */
#define FLAG_SET(set, flag) (((set) & (flag)) == (flag))

/* On 64-bit platforms like amd64 we need to use longer types for the
 * bit arrays
 */
#if SIZEOF_UNSIGNED_LONG >= 8
# define LONG_ARRAYS 1

typedef unsigned long BitArray;
#else /* SIZEOF_UNSIGNED_LONG */
# undef LONG_ARRAYS

typedef unsigned int BitArray;
#endif /* SIZEOF_UNSIGNED_LONG */


/* The "any PCI device" id */
#define PCI_ANY 0xffffffff

/* PCI Device structure */
typedef struct {
	unsigned int id_vendor, id_device;
	unsigned int subid_vendor, subid_device;
	unsigned int class, class_mask;
} PciDevice;


/* Flags specifying what USB fields to match */
typedef enum {
	USB_MATCH_VENDOR        = (1 << 0),
	USB_MATCH_PRODUCT       = (1 << 1),
	USB_MATCH_DEV_LO        = (1 << 2),
	USB_MATCH_DEV_HI        = (1 << 3),
	USB_MATCH_DEV_CLASS     = (1 << 4),
	USB_MATCH_DEV_SUBCLASS  = (1 << 5),
	USB_MATCH_DEV_PROTOCOL  = (1 << 6),
	USB_MATCH_INT_CLASS     = (1 << 7),
	USB_MATCH_INT_SUBCLASS  = (1 << 8),
	USB_MATCH_INT_PROTOCOL  = (1 << 9)
} UsbMatchFlags;

/* USB Device structure */
typedef struct {
	UsbMatchFlags match_flags;
	unsigned int  vendor, product;
	unsigned int  dev, dev_lo, dev_hi;
	unsigned int  dev_class, dev_subclass, dev_protocol;
	unsigned int  int_class, int_subclass, int_protocol;
} UsbDevice;


/* Flags specifying what input fields to match */
typedef enum {
	INPUT_MATCH_BUS     = (1 << 0),
	INPUT_MATCH_VENDOR  = (1 << 1),
	INPUT_MATCH_PRODUCT = (1 << 2),
	INPUT_MATCH_VERSION = (1 << 3),
	INPUT_MATCH_EVBIT   = (1 << 4),
	INPUT_MATCH_KEYBIT  = (1 << 5),
	INPUT_MATCH_RELBIT  = (1 << 6),
	INPUT_MATCH_ABSBIT  = (1 << 7),
	INPUT_MATCH_MSCBIT  = (1 << 8),
	INPUT_MATCH_LEDBIT  = (1 << 9),
	INPUT_MATCH_SNDBIT  = (1 << 10),
	INPUT_MATCH_FFBIT   = (1 << 11),
	INPUT_MATCH_SWBIT   = (1 << 12)
} InputMatchFlags;

/* Length of arrays, as these have different types on 64-bit platforms,
 * they also need to be different lengths.
 */
#if LONG_ARRAYS
enum {
	EVBIT_LEN  = 1,
	KEYBIT_LEN = 8,
	RELBIT_LEN = 1,
	ABSBIT_LEN = 1,
	MSCBIT_LEN = 1,
	LEDBIT_LEN = 1,
	SNDBIT_LEN = 1,
	FFBIT_LEN  = 2,
	SWBIT_LEN  = 1
};
#else /* LONG_ARRAYS */
enum {
	EVBIT_LEN  = 1,
	KEYBIT_LEN = 16,
	RELBIT_LEN = 1,
	ABSBIT_LEN = 2,
	MSCBIT_LEN = 1,
	LEDBIT_LEN = 1,
	SNDBIT_LEN = 1,
	FFBIT_LEN  = 4,
	SWBIT_LEN  = 1
};
#endif /* LONG_ARRAYS */

/* Input Device structure */
typedef struct {
	InputMatchFlags match_flags;
	unsigned int    bus, vendor, product, version;
	BitArray        evbit[EVBIT_LEN], keybit[KEYBIT_LEN];
	BitArray        relbit[RELBIT_LEN], absbit[ABSBIT_LEN];
	BitArray        mscbit[MSCBIT_LEN], ledbit[LEDBIT_LEN];
	BitArray        sndbit[SNDBIT_LEN], ffbit[FFBIT_LEN];
	BitArray	swbit[SWBIT_LEN];
} InputDevice;


/* Flags specifying what IEEE1394 fields to match */
typedef enum {
	IEEE1394_MATCH_VENDOR        = (1 << 0),
	IEEE1394_MATCH_MODEL         = (1 << 1),
	IEEE1394_MATCH_SPECIFIER     = (1 << 2),
	IEEE1394_MATCH_VERSION       = (1 << 3)
} Ieee1394MatchFlags;

/* IEEE1394 Device structure */
typedef struct {
	Ieee1394MatchFlags match_flags;
	unsigned int       vendor, model, specifier, version;
} Ieee1394Device;


/* Flags specifying what CCW fields to match */
typedef enum {
	CCW_MATCH_CU_TYPE	= (1 << 0),
	CCW_MATCH_CU_MODEL      = (1 << 1),
	CCW_MATCH_DEV_TYPE      = (1 << 2),
	CCW_MATCH_DEV_MODEL     = (1 << 3)
} CcwMatchFlags;

/* CCW Device structure */
typedef struct {
	CcwMatchFlags match_flags;
	unsigned int  cu_type, cu_model;
	unsigned int  dev_type, dev_model;
} CcwDevice;


/* The "any serio device" id */
#define SERIO_ANY 0xff

/* Serio Device structure */
typedef struct {
	unsigned int type, extra, id, proto;
} SerioDevice;


/* Program's name */
extern const char *program_name;


/* Utility functions */
extern void  suggest_help (void);
extern char *fgets_alloc  (FILE *stream);
extern int   parse_hex    (const char *str, unsigned int *hex);
extern int   parse_array  (const char *str, BitArray *array, int array_len);
extern int   parse_line   (const char *line, const char *format, ...);
extern int   match_array  (const BitArray *mask, const BitArray *array,
			   int array_len);

/* Argument parsing functions */
extern int pcimap_parse_args      (int argc, char *argv[], PciDevice *dev);
extern int usbmap_parse_args      (int argc, char *argv[], UsbDevice *dev);
extern int inputmap_parse_args    (int argc, char *argv[], InputDevice *dev);
extern int ieee1394map_parse_args (int argc, char *argv[],
				   Ieee1394Device *dev);
extern int ccwmap_parse_args      (int argc, char *argv[], CcwDevice *dev);
extern int seriomap_parse_args    (int argc, char *argv[], SerioDevice *dev);

/* Udev event functions */
extern int pcimap_udev_event      (char **environ, PciDevice *dev);
extern int usbmap_udev_event      (char **environ, UsbDevice *dev);
extern int inputmap_udev_event    (char **environ, InputDevice *dev);
extern int ieee1394map_udev_event (char **environ, Ieee1394Device *dev);
extern int ccwmap_udev_event      (char **environ, CcwDevice *dev);
extern int seriomap_udev_event    (char **environ, SerioDevice *dev);

/* Map functions */
extern int pcimap_modules      (FILE *mapf, const char *file, PciDevice *dev);
extern int usbmap_modules      (FILE *mapf, const char *file, UsbDevice *dev);
extern int inputmap_modules    (FILE *mapf, const char *file,
				InputDevice *dev);
extern int ieee1394map_modules (FILE *mapf, const char *file,
				Ieee1394Device *dev);
extern int ccwmap_modules      (FILE *mapf, const char *file, CcwDevice *dev);
extern int seriomap_modules    (FILE *mapf, const char *file,
				SerioDevice *dev);

#endif /* ! GREPMAP_GREPMAP_H */
