/* grepmap
 *
 * ccwmap.c - parse modules.ccwmap
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/**
 * ccwmap_parse_args:
 * @argc: number of arguments,
 * @argv: arguments array,
 * @dev: device structure to fill.
 *
 * Parses the command-line arguments to fill a device structure.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
ccwmap_parse_args (int        argc,
		   char      *argv[],
		   CcwDevice *dev)
{
	int ret = 0;

	if (argc < 4) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("insufficient arguments"));
		suggest_help ();
		return 2;
	}

	ret |= parse_hex (argv[0], &dev->cu_type);
	ret |= parse_hex (argv[1], &dev->cu_model);
	ret |= parse_hex (argv[2], &dev->dev_type);
	ret |= parse_hex (argv[3], &dev->dev_model);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 2;
	}

	if (argc > 4)
		fprintf (stderr, "%s: %s\n", program_name,
			 _("some arguments ignored"));

	return 0;
}

/**
 * ccwmap_udev_event:
 * @environ: environment array,
 * @dev: device structure to fill.
 *
 * Fills the device structure by reading a udev event from the environment.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
ccwmap_udev_event (char      **environ,
		   CcwDevice  *dev)
{
	char **envvar;
	int    ret = 0;

	for (envvar = environ; *envvar; envvar++) {
		char *key, *value;

		key = strdup (*envvar);
		value = strchr (key, '=');
		if (! value)
			goto next;
		*(value++) = 0;

		if (! strcmp (key, "CU_TYPE")) {
			ret |= parse_hex (value, &dev->cu_type);

		} else if (! strcmp (key, "CU_MODEL")) {
			ret |= parse_hex (value, &dev->cu_model);

		} else if (! strcmp (key, "DEV_TYPE")) {
			ret |= parse_hex (value, &dev->dev_type);

		} else if (! strcmp (key, "DEV_MODEL")) {
			ret |= parse_hex (value, &dev->dev_model);

		}

	next:
		free (key);
	}

	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("udev event not in expected format"));
		return 2;
	}

	return 0;
}

/**
 * ccwmap_modules:
 * @mapf: open map file to read from,
 * @file: filename or map file, or '-' for standard input,
 * @dev: device to look for.
 *
 * Handles the modules.ccwmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
ccwmap_modules (FILE       *mapf,
		const char *file,
		CcwDevice  *dev)
{
	char *line;
	int   lineno = 0, ret = 0;

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		CcwDevice map;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		if (parse_line (line, "xxxxx",
				(unsigned int *)&map.match_flags,
				&map.cu_type, &map.cu_model,
				&map.dev_type, &map.dev_model))
		{
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, CCW_MATCH_CU_TYPE)
		    && (map.cu_type != dev->cu_type))
			continue;

		if (FLAG_SET(map.match_flags, CCW_MATCH_CU_MODEL)
		    && (map.cu_model != dev->cu_model))
			continue;

		if (FLAG_SET(map.match_flags, CCW_MATCH_DEV_TYPE)
		    && (map.dev_type != dev->dev_type))
			continue;

		if (FLAG_SET(map.match_flags, CCW_MATCH_DEV_MODEL)
		    && (map.dev_model != dev->dev_model))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
