/* grepmap
 * arch-tag: 5f38b1be-85c4-405b-b02d-7d50939c8c31
 *
 * inputmap.c - parse modules.inputmap
 *
 * Copyright © 2004 Canonical Ltd.
 * Author: Scott James Remnant <scott@canonical.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/* Flags specifying what to match */
typedef enum {
	INPUT_MATCH_BUS     = (1 << 0),
	INPUT_MATCH_VENDOR  = (1 << 1),
	INPUT_MATCH_PRODUCT = (1 << 2),
	INPUT_MATCH_VERSION = (1 << 3),
	INPUT_MATCH_EVBIT   = (1 << 4),
	INPUT_MATCH_KEYBIT  = (1 << 5),
	INPUT_MATCH_RELBIT  = (1 << 6),
	INPUT_MATCH_ABSBIT  = (1 << 7),
	INPUT_MATCH_MSCBIT  = (1 << 8),
	INPUT_MATCH_LEDBIT  = (1 << 9),
	INPUT_MATCH_SNDBIT  = (1 << 10),
	INPUT_MATCH_FFBIT   = (1 << 11)
} InputMatchFlags;

/* Input Device structure */
typedef struct {
	InputMatchFlags match_flags;
	unsigned int    bus, vendor, product, version;
	unsigned int    evbit[1], keybit[16], relbit[1], absbit[2];
	unsigned int    mscbit[1], ledbit[1], sndbit[1], ffbit[4];
} InputDevice;


/* Forward prototypes */
static void _parse_array (const char *line, int *pos, unsigned int *array,
			  size_t array_sz);
static int  _match_array (const unsigned int *mask, const unsigned int *array,
			  size_t array_sz);


/**
 * inputmap_modules:
 * @mapf: open map file to read from.
 * @file: filename or map file, or '-' for standard input.
 * @args: arguments (bus, vendor, product, version,
 *                   evbit, keybit, relbit, absbit,
 *                   mscbit, ledbit, sndbit, ffbit)
 *
 * Handles the modules.inputmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
inputmap_modules (FILE       *mapf,
		  const char *file,
		  char       *argv[])
{
	InputDevice  dev;
	char      *line;
	int        lineno = 0, ret = 0;

	ret |= parse_hex (argv[0], &dev.bus);
	ret |= parse_hex (argv[1], &dev.vendor);
	ret |= parse_hex (argv[2], &dev.product);
	ret |= parse_hex (argv[3], &dev.version);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 1;
	}

	_parse_array (argv[4], NULL, dev.evbit, sizeof (dev.evbit));
	_parse_array (argv[5], NULL, dev.keybit, sizeof (dev.keybit));
	_parse_array (argv[6], NULL, dev.relbit, sizeof (dev.relbit));
	_parse_array (argv[7], NULL, dev.absbit, sizeof (dev.absbit));
	_parse_array (argv[8], NULL, dev.mscbit, sizeof (dev.mscbit));
	_parse_array (argv[9], NULL, dev.ledbit, sizeof (dev.ledbit));
	_parse_array (argv[10], NULL, dev.sndbit, sizeof (dev.sndbit));
	_parse_array (argv[11], NULL, dev.ffbit, sizeof (dev.ffbit));

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		InputDevice map;
		int       s, pos = -1;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		s = sscanf (line, "%*s 0x%x 0x%x 0x%x 0x%x 0x%x %n",
			    (unsigned int *)&map.match_flags,
			    &map.bus, &map.vendor, &map.product, &map.version,
			    &pos);
		if ((s < 5) || (pos == -1)) {
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		_parse_array (line, &pos, map.evbit, sizeof (map.evbit));
		_parse_array (line, &pos, map.keybit, sizeof (map.keybit));
		_parse_array (line, &pos, map.relbit, sizeof (map.relbit));
		_parse_array (line, &pos, map.absbit, sizeof (map.absbit));
		_parse_array (line, &pos, map.mscbit, sizeof (map.mscbit));
		_parse_array (line, &pos, map.ledbit, sizeof (map.ledbit));
		_parse_array (line, &pos, map.sndbit, sizeof (map.sndbit));
		_parse_array (line, &pos, map.ffbit, sizeof (map.ffbit));

		if (FLAG_SET(map.match_flags, INPUT_MATCH_BUS)
		    && (map.bus != dev.bus))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_VENDOR)
		    && (map.vendor != dev.vendor))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_PRODUCT)
		    && (map.product != dev.product))
			continue;

		/* NOTE: this isn't what the kernel does, but is what hotplug
		 * does.  I'm being bug-compatible with the latter.
		 *
		 * It's also noteworthy that hotplug itself suggests it's
		 * going to do a < comparison, but then does this.  *shrug*
		 */
		if (FLAG_SET(map.match_flags, INPUT_MATCH_VERSION)
		    && (map.version >= dev.version))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_EVBIT)
		    && _match_array (map.evbit, dev.evbit, sizeof (map.evbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_KEYBIT)
		    && _match_array (map.keybit, dev.keybit, sizeof (map.keybit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_RELBIT)
		    && _match_array (map.relbit, dev.relbit, sizeof (map.relbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_ABSBIT)
		    && _match_array (map.absbit, dev.absbit, sizeof (map.absbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_MSCBIT)
		    && _match_array (map.mscbit, dev.mscbit, sizeof (map.mscbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_LEDBIT)
		    && _match_array (map.ledbit, dev.ledbit, sizeof (map.ledbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_SNDBIT)
		    && _match_array (map.sndbit, dev.sndbit, sizeof (map.sndbit)))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_FFBIT)
		    && _match_array (map.ffbit, dev.ffbit, sizeof (map.ffbit)))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}

/**
 * _parse_array:
 * @line: complete line to scan.
 * @pos: offset in line to begin scan.
 * @array: integer array to fill.
 * @array_sz: size of integer array
 *
 * Scans @line starting from @pos and reads a colon separated array of ints,
 * skipping any preceeding and following whitespace.  Updates @pos to point
 * to the next item in the string.
 *
 * These arrays are a little strange, the kernel actually prints them
 * backwards starting from the first non-zero part; so they may not actually
 * be complete.
 *
 * @pos may be NULL, in which case the entire string is scanned and
 * obviously it isn't updated.
 **/
static void
_parse_array (const char   *line,
	      int          *pos,
	      unsigned int *array,
	      size_t        array_sz)
{
	const char *c;
	int         i, j;

	c = line + (pos ? *pos : 0);
	c += strspn (c, " \t\r\n");

	memset (array, 0, array_sz);
	i = array_sz /= sizeof (unsigned int);

	while (*c && i) {
		int s, end;

		s = sscanf (c, "%x%n", &array[--i], &end);
		if (s < 1)
			break;

		c += end;
		if (*c == ':') {
			c++;
		} else {
			break;
		}
	}

	for (j = 0; j < array_sz; j++, i++)
		array[j] = (i < array_sz ? array[i] : 0);

	if (pos) {
		c += strspn (c, " \t\r\n");
		*pos = c - line;
	}
}

/**
 * _match_array:
 * @mask: bits that must be present in the array.
 * @array: array to check.
 * @array_sz: size of both arrays.
 *
 * Check that @array has all of the bits present in @mask.
 *
 * Returns: 0 if the arrays match, non-zero if not.
 **/
static int
_match_array (const unsigned int *mask,
	      const unsigned int *array,
	      size_t              array_sz)
{
	int i;

	array_sz /= sizeof (unsigned int);
	for (i = 0; i < array_sz; i++) {
		if (! FLAG_SET(array[i], mask[i]))
			return 1;
	}

	return 0;
}
