#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

#include <gtk/gtk.h>

#include <gksuui.h>
#include <gksu.h>

#include "defines.h"
#include "../config.h"

#include "util.h"

struct option long_opts[] = {
    /*
     * { name  has_arg  *flag  val } 
     */
    {"help", no_argument, NULL, 'h'},
    {"login", no_argument, NULL, 'l'},
    {"preserv-env", no_argument, NULL, 'k'},
    {"preserve-env", no_argument, NULL, 'k'},
    {"user", required_argument, NULL, 'u'},
    {"print-pass", no_argument, NULL, 'p'},
    {"message", required_argument, NULL, 'm'},
    {"title", required_argument, NULL, 't'},
    {"icon", required_argument, NULL, 'i'},
    {"disable-grab", no_argument, NULL, 'g'},
    {"ssh-fwd", no_argument, NULL, 's'},
    {"debug", no_argument, NULL, 'd'},
    {0, 0, 0, 0}
};

/**
 * help:
 * @cmdname:  name of the command which was called by the user 
 * (argv[0])
 *
 * This function is a simple 'usage'-style printing function.
 * It is called if the user calls the program with --help or -h
 */
void
help (gchar *cmdname)
{ 
  g_print (_(
"GKsu version %s\n\n"
"Usage: %s [-u <user>] [-k] [-l] <command>\n\n"
"  --user <user>, -u <user>\n"
"    Calls <command> as the specified user\n"
"  --message <message>, -m <message>\n"
"    Replaces the standard message shown to ask for\n"
"    password for the argument passed to the option\n"
"  --title <title>, -t <title>\n"
"    Replaces the default title with the argument\n"
"  --icon <icon>, -i <icon>\n"
"    Replaces the default window icon with the argument\n"
"  --print-pass, -p\n"
"    Asks gksu to print the password to stdout, just\n"
"    like ssh-askpass. Useful to use in scripts with\n"
"    programs that accept receiving the passowrd on\n"
"    stdin.\n"
"  --disable-grab, -g\n"
"    Disables the \"locking\" of the keyboard, mouse,\n"
"    and focus done by the program when asking for\n"
"    password\n"
"  --ssh-fwd, -s\n"
"    Strip the host part of the $DISPLAY variable, so that\n"
"    GKSu will work on SSH X11 Forwarding.\n"
"\n" 
"  --login, -l\n"
"    Makes this a login shell. Beware this may cause\n"
"    problems with the Xauthority magic. Run xhost\n"
"    to allow the target user to open windows on your\n"
"    display!\n"
"  --preserve-env, -k\n"
"    Preserve the current environments, does not set $HOME\n"
"    nor $PATH, for example.\n"
"\n"
"\n"), PACKAGE_VERSION, cmdname);
}

int
main (int argc, char **argv)
{
  GtkWidget *dialog;
  GksuContext *context;

  gboolean print_pass = FALSE;
  gboolean grab = TRUE;
  gboolean sudo_mode = FALSE;

  gboolean message_changed = FALSE;

  gchar *password = NULL;
  GError *error = NULL;

  int retvalue = 0;
  int c = 0;

  setlocale (LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);  
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
  textdomain(PACKAGE_NAME);

  { /* support gksu_sudo_run */
    gchar *myname = g_path_get_basename (argv[0]);
    if (!strcmp(myname, "gksudo"))
      sudo_mode = TRUE;
    g_free (myname);
  }

  gtk_init (&argc, &argv);

  dialog = gksuui_dialog_new ();
  context = gksu_context_new ();
  while ((c = getopt_long(argc, argv, "?hu:lpm:kt:i:gds", long_opts, NULL))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  help (argv[0]);
	  exit(0);
	  break;
	case '?':
	  help (argv[0]);
	  exit(0);
	  break;
	case 'u':
	  gksu_context_set_user (context, optarg);
	  break;
	case 'l':
	  gksu_context_set_login_shell (context, TRUE);
	  break;
	case 'p':
	  print_pass = TRUE;
	  break;
	case 't':
	  gtk_window_set_title (GTK_WINDOW(dialog), optarg);
	  break;
	case 'm':
	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), optarg);
	  message_changed = TRUE;
	  break;
	case 'i':
	  {
	    GdkPixbuf *pixbuf;
	    pixbuf = gdk_pixbuf_new_from_file (optarg, NULL);
	    if (pixbuf)
	      gksuui_dialog_set_icon (GKSUUI_DIALOG(dialog), pixbuf);
	  }
	  break;
	case 'k':
	  gksu_context_set_keep_env (context, TRUE);
	  break;
	case 'g':
	  grab = FALSE;
	  break;
	case 'd':
	  gksu_context_set_debug (context, TRUE);
	  break;
	case 's':
	  gksu_context_set_ssh_fwd (context, TRUE);
	  break;
	default:
	  fprintf (stderr, _("Using %s as username...\n"), argv[optind]);
	}
    }

  if (print_pass)
    {
      if (!gksuui_dialog_get_message (GKSUUI_DIALOG(dialog)))
	{
	  gchar *msg = 
	    g_strdup_printf (_("<b>Please enter %s's password</b>"),
			     gksu_context_get_user (context));

	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	  g_free (msg);
	}

      retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
      gtk_widget_hide (dialog);

      /* 
	 the user may have pressed cancel or
	 closed the window 
      */
      if (retvalue != GTK_RESPONSE_OK)
	  return 0;

      password = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));

      if (password)
	printf ("%s\n", password);
      return 0;
    }

  /* now we can begin to care about a command */
  if (argc == optind)
    {
      gk_dialog (GTK_MESSAGE_ERROR, _("Missing command to run."));
      return 1;
    }
  gksu_context_set_command (context, argv[optind]);

  if (sudo_mode)
    {
      /* test if we can run without a password */
      gksu_context_sudo_run (context, &error);
      if (error)
	{
	  if (!message_changed)
	    {
	      gchar *msg = 
		g_strdup_printf (_("<b>Please enter your password\n"
				   "to run %s as user %s</b>"),
				 gksu_context_get_command (context),
				 gksu_context_get_user (context));
	      
	      gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	      g_free (msg);
	    }

	  retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
	  gtk_widget_hide (dialog);
	  while (gtk_events_pending ())
	    gtk_main_iteration ();

	  if (retvalue != GTK_RESPONSE_OK)
	    return 0;

	  password = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));

	  gksu_context_set_password (context, password);
      
	  error = NULL;
	  gksu_context_sudo_run (context, &error);
	  if (error)
	    gk_dialog (GTK_MESSAGE_ERROR, 
		       "Failed to run %s as user %s:\n %s",
		       gksu_context_get_command (context),
		       gksu_context_get_user (context),
		       error->message);
	}
    }
  else
    {
      if (!message_changed)
	{
	  gchar *msg = 
	    g_strdup_printf (_("<b>Please enter %s's password\n"
			       "to run %s.</b>"),
			     gksu_context_get_user (context),
			     gksu_context_get_command (context));
	  
	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	  g_free (msg);
	}

      retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
      gtk_widget_hide (dialog);

      while (gtk_events_pending ())
	gtk_main_iteration ();

      if (retvalue != GTK_RESPONSE_OK)
	  return 0;

      password = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));

      gksu_context_set_password (context, password);
      gksu_context_run (context, &error);

      if (error)
	gk_dialog (GTK_MESSAGE_ERROR, 
		   "Failed to run %s as user %s:\n %s",
		   gksu_context_get_command (context),
		   gksu_context_get_user (context),
		   error->message);
    }
  
  return 0;
}
