#!/bin/sh

PROGNAME="$0"

usage() {
	cat <<EOF
NAME
    `basename $PROGNAME` - printer-profile using X-Rite ColorMunki and Argyll CMS

SYNOPSIS
    `basename $PROGNAME` [options] manuf model [rgb|cmyk] [patches] [ink-limit]

DESCRIPTION
    Prints a test chart, uses the ColorMunki instrument to scan it in, then
    computes an ICM profile using the Argyll Color Management System.

    Manuf is "sam".  Model is "clp-300" or "clp-315".
    Manuf is "hp".  Model is "2600" or "cp1215".
    Manuf is "km".  Model is "1600" or "2300" or "2530".

    "rgb" is the usual setting.  "patches" is a multiple 196 per page.

    Edit the script for additional models.

OPTIONS
    -b 1|2        Bits per pixel ($BPP)
    -r XRESxYRES  Resolution. Default=''. ($RES)
    -P rem-print  Remote print (64-bit) machine, or none ($REMPRINT)
    -S rem-scan   Remote scan (ColorMunki) machine, or none ($REMSCAN)
    -D lvl        Debug level

EXAMPLE
    $ printer-profile sam 315 rgb 196

BUGS
    gs 8.64 and before has problems with 32-bit machines and color profile
    data.  Don't use!

    You need Argyll_V1.1.0 or later in $BETABIN.

SEE ALSO
    http://www.xritephoto.com/html/colormunkisplash.htm
	from Amazon, \$390 shipped
    http://www.argyllcms.com/
	free!
EOF

	exit 1
}

#
#       Report an error and exit
#
error() {
	echo "`basename $PROGNAME`: $1" >&2
	exit 1
}

debug() {
	if [ $DEBUG -ge $1 ]; then
	    echo "`basename $PROGNAME`: $2" >&2
	fi
}

#
#	Execute a command as root
#
root() {
    if [ -x /usr/bin/root ]; then
        /usr/bin/root $@
    else
        su -c "$*"
    fi
}

#
#	trap on error
#
trap "exit 1" ERR

#
#       Process the options
#
BETABIN=~/src/Argyll_V1.1.0/bin
export PATH=$BETABIN:$PATH

REMPRINT=amd
REMSCAN=mac
REMSCAN=none
RGB=rgb
PATCHES=196	#Per page!
BPP=1
RES=
DEBUG=0
while getopts "b:r:P:S:D:h?" opt
do
	case $opt in
	b)	BPP="$OPTARG";;
	r)	RES="$OPTARG";;
	P)	REMPRINT="$OPTARG";;
	S)	REMSCAN="$OPTARG";;
	D)	DEBUG="$OPTARG";;
	h|\?)	usage;;
	esac
done
shift `expr $OPTIND - 1`

if [ ! -x $BETABIN/printtarg ]; then
    error "No beta bin in $BETABIN!"
fi

#
#	Main Program
#
if [ $# -lt 2 ]; then
    usage
fi

MANUF="$1"
MODEL="$2"
if [ $# -ge 3 ]; then
    RGB="$3"
fi
if [ $# -ge 4 ]; then
    PATCHES="$4"
fi
INK="$5"

BPP_b="-b$BPP"
RES_r=
if [ "$RES" != "" ]; then
    RES_r="-r$RES"
fi

case "$MANUF" in
km)
    MANUF=km
    case "$MODEL" in
    *1600*)
	FOO=foo2lava
	WRAPPER="foo2lava-wrapper $RES_r -z2 -c -C10 -Gnone.icm"
	OUT="root cp xxx.prn /dev/usb/lp1"
	;;
    *2530*)
	FOO=foo2lava
	WRAPPER="foo2lava-wrapper $RES_r -z0 -c -C10 -Gnone.icm"
	OUT="nc 192.168.1.13 9100 < xxx.prn"
	;;
    *2300*)
	FOO=foo2zjs
	WRAPPER="foo2zjs-wrapper $RES_r -c -C10 -Gnone.icm"
	OUT="nc 192.168.1.10 9100 < xxx.prn"
	;;
    *)
	usage
	;;
    esac
    ;;
sam*)
    MANUF=sam
    FOO=foo2qpdl
    case "$MODEL" in
    *300*)
	WRAPPER="foo2qpdl-wrapper $RES_r -z0 -c -C10 -Gnone.icm"
	OUT="nc 192.168.1.11 9100 < xxx.prn"
	;;
    *315*)
	WRAPPER="foo2qpdl-wrapper $RES_r -z2 -c -C10 -Gnone.icm"
	OUT="root cp xxx.prn /dev/usb/lp1"
	;;
    *)
	usage
	;;
    esac
    ;;
hp*)
    MANUF=hp
    FOO=foo2hp
    case "$MODEL" in
    *2600*)
	WRAPPER="foo2hp2600-wrapper $RES_r $BPP_b -z0 -c -C10 -Gnone.icm"
	OUT="nc 192.168.1.12 9100 < xxx.prn"
	;;
    *cp1215*)
	WRAPPER="foo2hp2600-wrapper $RES_r $BPP_b -z1 -c -C10 -Gnone.icm"
	OUT="root cp xxx.prn /dev/usb/lp2"
	;;
    *)
	usage
	;;
    esac
    ;;
*)
    usage
    ;;
esac

mrp="$MANUF-$MODEL-$RGB-$PATCHES"
if [ "$INK" != "" ]; then
    mrp="$mrp-ink$INK"
fi
if [ "$BPP" != "" ]; then
    mrp="$mrp-bpp$BPP"
fi
if [ "$RES" != "" ]; then
    mrp="$mrp-$RES"
fi
echo "$mrp"

targen_opts=
case "$RGB" in
rgb|RGB)	targen_opts="$targen_opts -d2";;
cmyk|CMYK)	targen_opts="$targen_opts -d4";;
*)		error "Parm2: Must be rgb or cmyk";;
esac

targen_opts="$targen_opts -f$PATCHES"

if [ "$INK" != "" ]; then
    targen_opts="$targen_opts -l$INK"
fi

echo
echo "******************************** targen ********************************"
echo "targen $targen_opts $mrp"
if ! targen $targen_opts $mrp >$mrp.err1; then
    cat $mrp.err1
    exit 1
fi

echo
echo "******************************* printtarg ******************************"
# -h	Use double density for CM
# -v	Verbose mode
# -iCM	Select target instrument, CM = ColorMunki
# -pLetter
# -R0	Use given random start number
echo "$BETABIN/printtarg -h -v -iCM -p Letter -R0 $mrp"
$BETABIN/printtarg -h -v -iCM -p Letter -R0 $mrp

evince $mrp.ps &

echo
echo "******************************* print it *******************************"
echo "$WRAPPER"
case "$REMPRINT" in
''|none)
    $WRAPPER <$mrp.ps >$mrp.prn
    ;;
*)
    ssh $REMPRINT "$WRAPPER" <$mrp.ps >$mrp.prn
    ;;
esac
ls -l $mrp.prn
echo -n "Print it? [y|n]? "
read yes
case "$yes" in
y|Y)
    cp $mrp.prn xxx.prn
    echo "	$OUT"
    eval $OUT
    rm -f xxx.prn
esac

echo
echo "******************************* chartread ******************************"
case "$REMSCAN" in
''|none)
    echo "$BETABIN/chartread $mrp"
    $BETABIN/chartread $mrp
    ;;
*)
    echo "scp $mrp.ti2 mac:"
    scp $mrp.ti2 mac:
    echo "$BETABIN/chartread $mrp"
    echo  -n "Scanned on remote system $REMSCAN? [y/n]? "
    read yes
    case "$yes" in
    y|Y)
	scp $REMSCAN:$mrp.ti3 .
	;;
    esac
    ;;
esac

echo
echo "******************************** colprof *******************************"
case $RGB in
rgb)
    echo "colprof -v -D\"$mrp\" -qm -cmt -dpp $mrp"
    colprof -v -D"$mrp" -qm -cmt -dpp $mrp
    ;;
cymk)
    echo "colprof -v -D\"$mrp\" -qm  -cmt -dpp -kr $mrp"
    colprof -v -D"$mrp" -qm  -cmt -dpp -kr $mrp
    ;;
esac

root cp $mrp.icc /usr/share/$FOO/icm/testing.icm
echo "/usr/share/$FOO/icm/testing.icm created!"
ls -l /usr/share/$FOO/icm/
