/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mount.c
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <string.h>
#include <sys/mount.h>
#include <unistd.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "mount.h"
#include "readable.h"
#include "thing.h"
#include "logging.h"
#include "help.h"

/**
 *	get_volume_device_node - get a copy of the volume device node pathname
 *	@volume: the volume to get the device node for
 *
 *	This routine retrieves the device node pathname from the information the
 *	engine keeps on a volume. We duplicate the string which the caller must
 *	free when no longer needed. NULL is returned if there is no device node
 *	or if there was an error getting the volume information.
 */
char *get_volume_device_node(object_handle_t volume)
{
	int rc;
	char *devnode = NULL;
	handle_object_info_t *object;

	rc = evms_get_info(volume, &object);
	if (rc == 0) {
		if (object->info.volume.dev_node[0] != '\0')
			devnode = g_strdup(object->info.volume.dev_node);
		evms_free(object);
	}

	return devnode;
}

/**
 *	get_current_mount_point - retrieve the path that the volume device is attached to
 *	@volume: the volume to get the mount point for
 *
 *	This routine retrieves the mount point path from the information the
 *	engine keeps on a volume. We duplicate the string which the caller
 *	must free when no longer needed. NULL is returned if the volume is
 *	not mounted or if there was an error getting the volume information.
 */
char *get_current_mount_point(object_handle_t volume)
{
	int rc;
	char *mount_point = NULL;
	handle_object_info_t *object;

	rc = evms_get_info(volume, &object);
	if (rc == 0) {
		if (object->info.volume.mount_point != NULL)
			mount_point = g_strdup(object->info.volume.mount_point);
		evms_free(object);
	}

	return mount_point;
}

/*
 *
 *   void populate_clist_for_mount_operation (GtkCList *, mount_op_t)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that can be mounted, unmounted or remounted.
 * 
 *   Entry:
 *      clist - address of the selections GtkCList widget
 *      oper  - the mount operation type - mount, remount, or unmount
 *
 *   Exit:
 *      Selection list populated with volumes proper for the mount
 *      operation type
 *
 */
void populate_clist_for_mount_operation(GtkCList * clist, mount_op_t oper)
{
	gint rc;
	handle_array_t *volumes;

	set_selection_window_clist_column_titles(clist, _("Size"),
						 make_object_type_readable_string(VOLUME), NULL);

	rc = evms_get_volume_list(0, 0, 0, &volumes);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;
		gboolean is_selected = (volumes->count == 1);
		mount_can_function can_function;

		switch (oper) {
		default:
		case MOUNT:
			can_function = evms_can_mount;
			break;

		case UNMOUNT:
			can_function = evms_can_unmount;
			break;

		case REMOUNT:
			can_function = evms_can_remount;
			break;
		}

		for (i = 0; i < volumes->count; i++) {
			if (can_function(volumes->handle[i]) == 0)
				add_thing_to_selection_list(clist, volumes->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(volumes);
	}
}

/*
 *
 *   void on_mount_point_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine is invoked when the OK button in the mount directory
 *      selection dialog is clicked.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      selection - address of the GtkFileSelection widget
 *
 *   Exit:
 *      See description.
 *
 */
void on_mount_point_selection_button_clicked(GtkButton * button, GtkFileSelection * selection)
{
	gchar *dir;
	GtkWidget *entry;

	dir = gtk_file_selection_get_filename(selection);

	entry = gtk_object_get_data(GTK_OBJECT(button), "mount_directory_entry");

	if (dir)
		gtk_entry_set_text(GTK_ENTRY(entry), dir);
}

/*
 *
 *   GtkWidget *create_mount_point_selection_window (GtkWidget *)
 *   
 *   Description:
 *      This routine create a file selection dialog to be able to
 *      selection the directory to be used as the mount point.
 * 
 *   Entry:
 *      entry - id of entry field that should be updated with mount directory name
 *
 *   Exit:
 *      The directory selection window is created and the id for the widget
 *      is returned.
 *
 */
GtkWidget *create_mount_point_selection_window(GtkWidget * entry)
{
	GtkWidget *filesel;

	filesel = gtk_file_selection_new(_("Select Mount Directory"));

	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
	gtk_widget_hide(GTK_FILE_SELECTION(filesel)->file_list->parent);

	gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			    "mount_directory_entry", entry);

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			   "clicked", GTK_SIGNAL_FUNC(on_mount_point_selection_button_clicked),
			   GTK_OBJECT(filesel));

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
				  "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT(filesel));

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
				  "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT(filesel));

	return filesel;
}

/*
 *
 *   void on_browse_mount_dir_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine is invoked when the Browser but is clicked to
 *      display a file selection dialog for the mount point.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      selection - address of the GtkFileSelection widget
 *
 *   Exit:
 *      See description.
 *
 */
void on_browse_mount_dir_button_clicked(GtkButton * button, GtkWidget * entry)
{
	GtkWidget *filesel;
	GtkWidget *parent;

	parent = gtk_widget_get_toplevel(GTK_WIDGET(button));
	filesel = create_mount_point_selection_window(entry);

	gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(parent));
	gtk_widget_show(filesel);
}
