/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * evms_clean
 *
 * Simple utility to delete all objects, containers, volumes, and file systems
 * built on a given object, container, or volume.
 */

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

#define REQUIRED_ENGINE_API_MAJOR 10
#define REQUIRED_ENGINE_API_MINOR 0
#define REQUIRED_ENGINE_API_PATCH 0

static char * prog_name;				
static debug_level_t debug_level = -1;
static char * log_file = NULL;
static char * node_name = NULL;
static boolean no_commit = FALSE;
static boolean prompt = FALSE;
static boolean quiet = FALSE;
static boolean verbose = FALSE;

#define VERBOSE(msg, args...)	if (verbose) {printf(msg, ## args);}

static boolean is_numeric(char * str)
{
	while (*str != '\0') {
		if ((*str < '0') || (*str > '9')) {
			return FALSE;
		}
	}
	return TRUE;
}


static void show_help(void)
{
	printf("\n%s  Version %d.%d.%d\n\n", prog_name, REQUIRED_ENGINE_API_MAJOR, REQUIRED_ENGINE_API_MINOR, REQUIRED_ENGINE_API_PATCH);
	printf("Usage: %s [options] name [name ...]\n", prog_name);
	printf("Delete all objects, containers, volumes, and file systems built on a given object, container, or volume.\n\n"
	       "  [-d debug_level, --debug-level debug_level]\n"
	       "     debug_level = [0-9] |\n"
	       "                   [critical | serious | error | warning | default |\n"
	       "                    details | debug | extra | entry_exit | everything]\n"
	       "  [-l log_file_name, --log-file log_file_name]\n"
	       "  [-n node_name, --node-name node_name]\n"
	       "  [-x] don't save the changes\n"
	       "  [-p] let user answer questions\n"
	       "  [-q] quiet mode\n"
	       "  [-v] verbose mode\n"
	       "  [-h | -? | --help] display this help\n");
}


static int parse_options(int argc, char ** argv)
{
	int c, rc = 0;
	char * short_opts = "d:l:n:xpqvh?";
	struct option long_opts[] = { { "debug-level",  required_argument, NULL, 'd'},
				      { "log-file",     required_argument, NULL, 'l'},
				      { "node-name",    required_argument, NULL, 'n'},
				      { "help",         no_argument,       NULL, 'h'},
				      { NULL,           0,                 NULL,   0}  };

	while ((c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF) {
		switch (c) {
		case 'd':
			if (is_numeric(optarg)) {
				int level = atoi(optarg);

				if ((level < CRITICAL) ||
				    (level > EVERYTHING)) {
					printf("%s is not a valid debug level.\n",
					       optarg);
					/* Display the help. */
					rc = 1;
				}
			} else if (strcasecmp(optarg, "critical") == 0) {
				debug_level = CRITICAL;
			} else if (strcasecmp(optarg, "serious") == 0) {
				debug_level = SERIOUS;
			} else if (strcasecmp(optarg, "error") == 0) {
				debug_level = ERROR;
			} else if (strcasecmp(optarg, "warning") == 0) {
				debug_level = WARNING;
			} else if (strcasecmp(optarg, "default") == 0) {
				debug_level = DEFAULT;
			} else if (strcasecmp(optarg, "details") == 0) {
				debug_level = DETAILS;
			} else if (strcasecmp(optarg, "debug") == 0) {
				debug_level = DEBUG;
			} else if (strcasecmp(optarg, "extra") == 0) {
				debug_level = EXTRA;
			} else if (strcasecmp(optarg, "entry_exit") == 0) {
				debug_level = ENTRY_EXIT;
			} else if (strcasecmp(optarg, "everything") == 0) {
				debug_level = EVERYTHING;
			} else {
				printf("%s is not a valid debug level.\n",
				       optarg);
				/* Display the help. */
				rc = EINVAL;
			}

			break;

		case 'l':
			log_file = strdup(optarg);
			break;

		case 'n':
			node_name = strdup(optarg);
			break;

		case 'x':
			no_commit = TRUE;
			break;

		case 'p':
			prompt = TRUE;
			break;

		case 'q':
			quiet = TRUE;
			break;

		case 'v':
			verbose = TRUE;
			break;

		case 'h':
		case '?':
			/* Display the help. */
			rc = EINVAL;
			break;

		default:
			printf("%s -- unrecognized option \"%c\"\n\n",
			       prog_name, c);
			/* Display the help. */
			rc = EINVAL;
			break;
		}
	}

	return rc;
}


static int clean_object(handle_object_info_t * con_info);
static int clean_container(handle_object_info_t * con_info);


static int clean_volume(handle_object_info_t * vol_info) {

	int rc = 0;

	if (vol_info->info.volume.file_system_manager != 0) {
		if (!quiet) {
			printf("Remove the file system from volume %s.\n", vol_info->info.volume.name);
		}
		rc = evms_unmkfs(vol_info->info.volume.handle);
		if (rc != 0) {
			fprintf(stderr, "evms_unmkfs(%d) of volume %s failed with error code %d: %s\n",
				vol_info->info.volume.handle,
				vol_info->info.volume.name,
				rc, evms_strerror(rc));
		} else {
			VERBOSE("%s\n", evms_strerror(rc));
		}
	}

	return rc;
}


static int delete_volume_from_object(handle_object_info_t * obj_info) {

	int rc;
	handle_object_info_t * vol_info;

	rc = evms_get_info(obj_info->info.object.volume, &vol_info);

	if (rc == 0) {
		if (!quiet) {
			printf("Delete volume %s.\n", vol_info->info.volume.name);
		}
		rc = evms_delete(obj_info->info.object.volume);
		if (rc != 0) {
			fprintf(stderr, "evms_delete(%d) of volume %s failed with error code %d: %s\n",
				obj_info->info.object.volume,
				vol_info->info.volume.name,
				rc, evms_strerror(rc));
		} else {
			VERBOSE("%s\n", evms_strerror(rc));
		}

		evms_free(vol_info);

	} else {
		fprintf(stderr, "evms_get_info(%d) for the volume of object %s failed with return code %d: %s\n",
			obj_info->info.object.volume,
			obj_info->info.object.name,
			rc, evms_strerror(rc));
	}

	return rc;
}


static int delete_container_from_object(handle_object_info_t * obj_info) {

	int rc = 0;
	handle_object_info_t * con_info;

	rc = evms_get_info(obj_info->info.object.consuming_container, &con_info);

	if (rc == 0) {
		rc = clean_container(con_info);

		if (rc == 0) {
			if (!quiet) {
				printf("Delete container %s.\n", con_info->info.container.name);
			}
			rc = evms_delete(obj_info->info.object.consuming_container);
			if (rc != 0) {
				fprintf(stderr, "evms_delete(%d) of container %s failed with error code %d: %s\n",
					obj_info->info.object.consuming_container,
					con_info->info.container.name,
					rc, evms_strerror(rc));
			} else {
				VERBOSE("%s\n", evms_strerror(rc));
			}
		}

		evms_free(con_info);

	} else {
		fprintf(stderr, "evms_get_info(%d) for the consuming container of object %s failed with return code %d: %s\n",
			obj_info->info.object.consuming_container,
			obj_info->info.object.name,
			rc, evms_strerror(rc));
	}

	return rc;
}


static int get_plugin_info(object_handle_t obj, handle_object_info_t * * plugin_info) {

	boolean result = FALSE;
	int rc;
	handle_object_info_t * obj_info;

	rc = evms_get_info(obj, &obj_info);

	if (rc == 0) {
		rc = evms_get_info(obj_info->info.disk.plugin, plugin_info);

		if (rc != 0) {
			fprintf(stderr, "evms_get_info(%d) failed with return code %d: %s\n",
				obj_info->info.disk.plugin,
				rc, evms_strerror(rc));
		} else {
			VERBOSE("%s\n", evms_strerror(rc));
		}

		evms_free(obj_info);

	} else {
		fprintf(stderr, "evms_get_info(%d) failed with return code %d: %s\n",
			obj,
			rc, evms_strerror(rc));
	}

	return result;
}


static int delete_parent_objects(handle_object_info_t * obj_info) {

	int rc = 0;
	int i;

	for (i = 0; i < obj_info->info.disk.parent_objects->count; i++) {

		int rc2;
		handle_object_info_t * parent;

		rc2 = evms_get_info(obj_info->info.disk.parent_objects->handle[i], &parent);

		if (rc2 == 0) {
			if (parent->info.disk.data_type == DATA_TYPE) {
				if (!quiet) {
					printf("Delete object %s.\n", parent->info.disk.name);
				}
				rc2 = evms_delete(obj_info->info.disk.parent_objects->handle[i]);
				if (rc2 != 0) {
					fprintf(stderr, "evms_delete(%d) of object %s failed with error code %d: %s\n",
						obj_info->info.disk.parent_objects->handle[i],
						parent->info.disk.name,
						rc2, evms_strerror(rc2));
				} else {
					VERBOSE("%s\n", evms_strerror(rc2));
				}
			}

			evms_free(parent);

		} else {
			fprintf(stderr, "evms_get_info(%d) for the parent object of object %s failed with return code %d: %s\n",
				obj_info->info.disk.parent_objects->handle[i],
				obj_info->info.disk.name,
				rc2, evms_strerror(rc2));
		}

		if ((rc == 0) && (rc2 != 0)) {
			rc = rc2;
		}
	}

	return rc;
}


static int clean_parent_objects(handle_object_info_t * obj_info) {

	int rc = 0;
	int i;

	for (i = 0; i < obj_info->info.disk.parent_objects->count; i++) {

		int rc2;
		handle_object_info_t * parent;

		rc2 = evms_get_info(obj_info->info.disk.parent_objects->handle[i], &parent);

		if (rc2 == 0) {
			if (parent->info.disk.data_type == DATA_TYPE) {
				clean_object(parent);
			}

			evms_free(parent);

		} else {
			fprintf(stderr, "evms_get_info(%d) for the parent object of object %s failed with return code %d: %s\n",
				obj_info->info.disk.parent_objects->handle[i],
				obj_info->info.disk.name,
				rc2, evms_strerror(rc2));
		}

		if ((rc == 0) && (rc2 != 0)) {
			rc = rc2;
		}
	}

	return rc;
}


static int delete_parents_from_object(handle_object_info_t * obj_info) {

	int rc = 0;

	if (obj_info->info.disk.parent_objects->count != 0) {
		rc = clean_parent_objects(obj_info);

		if (rc == 0) {

			handle_object_info_t * plugin_info = NULL;

			rc = get_plugin_info(obj_info->info.disk.parent_objects->handle[0], &plugin_info);

			if (rc == 0) {
				if (GetPluginType(plugin_info->info.plugin.id) == EVMS_SEGMENT_MANAGER) {
					if (!quiet) {
						printf("Unassign %s from %s.\n",
						       (plugin_info != NULL) ? plugin_info->info.plugin.long_name : "(unknown)",
							obj_info->info.disk.name);
					}
					rc = evms_unassign(obj_info->info.disk.handle);
					if (rc != 0) {
						fprintf(stderr, "evms_unassign(%d) of object %s failed with error code %d: %s\n",
							obj_info->info.disk.handle,
							obj_info->info.object.name,
							rc, evms_strerror(rc));
					} else {
						VERBOSE("%s\n", evms_strerror(rc));
					}

				} else {
					rc = delete_parent_objects(obj_info);
				}
			}
		}
	}

	return rc;
}


static int clean_object(handle_object_info_t * obj_info) {

	int rc = 0;

	/*
	 * If this is a top level object (i.e., it has no parents and is not
	 * consumed by a container) and it is part of a volume, then clean the
	 * volume off the object.
	 */
	if ((obj_info->info.object.parent_objects->count == 0) &&
	    (obj_info->info.object.consuming_container == 0) &&
	    (obj_info->info.object.volume != 0)) {
		
		rc = delete_volume_from_object(obj_info);

	} else {
		if (obj_info->info.object.consuming_container != 0) {
			rc = delete_container_from_object(obj_info);

		} else {
			rc = delete_parents_from_object(obj_info);
		}
	}

	return rc;
}


static int clean_container(handle_object_info_t * con_info) {

	int rc = 0;
	int i;

	for (i = 0; i < con_info->info.container.objects_produced->count; i++) {

		int rc2;
		handle_object_info_t * product;

		rc2 = evms_get_info(con_info->info.container.objects_produced->handle[i], &product);

		if (rc2 == 0) {
			if (product->info.object.data_type == DATA_TYPE) {
				rc2 = clean_object(product);
				if (rc2 == 0) {
					if (!quiet) {
						printf("Delete object %s.\n", product->info.disk.name);
					}
					rc2 = evms_delete(con_info->info.container.objects_produced->handle[i]);
					if (rc2 != 0) {
						fprintf(stderr, "evms_delete(%d) of object %s failed with error code %d: %s\n",
							con_info->info.container.objects_produced->handle[i],
							product->info.disk.name,
							rc2, evms_strerror(rc2));
					} else {
						VERBOSE("%s\n", evms_strerror(rc2));
					}
				}
			}

			evms_free(product);

		} else {
			fprintf(stderr, "evms_get_info(%d) for the object produced by container %s failed with return code %d: %s\n",
				con_info->info.container.objects_produced->handle[i],
				con_info->info.container.name,
				rc2, evms_strerror(rc2));
		}

		if ((rc == 0) && (rc2 != 0)) {
			rc = rc2;
		}
	}

	return rc;
}


static int message_callback(char * message_text,
			    int * answer,
			    char ** choices)
{
	if (prompt || verbose) {
		printf(message_text);

		if (prompt) {

			int count;
			int response = 0;

			count = 0;
			while (choices[count] != NULL) {
				printf("%d) %s\n", count + 1, choices[count]);
				count++;
			}

                        do {
				int ch;

				printf("[1-%d]: ", count);
				response = 0;

				/*
				 * Get characters as long as they are decimal
				 * digits.
				 */
				ch = getchar();
				while ((ch >= '0') && (ch <= '9')) {
					response *= 10;
					response += ch - '0';

					ch = getchar();
				}

				/*
				 * Get the rest of the input out of the
				 * stream.
				 */
				while (ch != '\n') {
					ch = getchar();
				}

			} while ((response < 1) || (response > count));

			*answer = response - 1;

		} else {
			printf("%s: Responding with default selection \"%s\".\n",
			       prog_name, choices[*answer]);
		}
	}

	return 0;
}


static void status_callback(char * message_text)
{
	VERBOSE(message_text);
}


static ui_callbacks_t callbacks = {
	user_message: message_callback,
	user_communication: NULL,
	progress: NULL,
	status: status_callback
};


int main(int argc, char * * argv)
{
	evms_version_t engine_api;
	int rc;
	int i;
	boolean changes_pending = FALSE;

	prog_name = basename(argv[0]);

	rc = parse_options(argc, argv);
	if (rc) {
		show_help();
		return rc;
	}

	if (optind >= argc) {
		show_help();
		return 0;
	}

	/*
	 * If verbose, then we can't be quiet.
	 */
	if (verbose) {
		quiet = FALSE;
	}

	evms_get_api_version(&engine_api);

	if ((engine_api.major != REQUIRED_ENGINE_API_MAJOR) ||
	    (engine_api.major == REQUIRED_ENGINE_API_MAJOR &&
	     engine_api.minor <  REQUIRED_ENGINE_API_PATCH)) {
		fprintf(stderr, "EVMS API version mismatch.\n");
		fprintf(stderr, "\tEVMS provides %d.%d.%d\n",
			engine_api.major, engine_api.minor, engine_api.patchlevel);
		fprintf(stderr, "\t%s requires %d.%d.%d\n",
			prog_name, REQUIRED_ENGINE_API_MAJOR,
			REQUIRED_ENGINE_API_MINOR, REQUIRED_ENGINE_API_PATCH);
		return EPERM;
	}

	if (!quiet) {
		printf("Open the EVMS Engine...\n");
	}
	rc = evms_open_engine(node_name, ENGINE_READWRITE,
			      &callbacks, debug_level, log_file);
	if (rc) {
		fprintf(stderr, "evms_open_engine() failed with error code %d: %s\n", rc, evms_strerror(rc));
		return rc;
	}
	VERBOSE("%s\n", evms_strerror(rc));

	for (i = optind; i < argc; i++) {

		int rc2;
		object_handle_t handle;

		rc2 = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT | CONTAINER | VOLUME, argv[i], &handle);

		if (rc2 == 0) {
			handle_object_info_t * info;

			rc2 = evms_get_info(handle, &info);

			if (rc2 == 0) {
				switch (info->type) {
				case DISK:
				case SEGMENT:
				case REGION:
				case EVMS_OBJECT:
					rc2 = clean_object(info);
					break;

				case CONTAINER:
					rc2 = clean_container(info);
					break;

				case VOLUME:
					rc2 = clean_volume(info);
					break;

				default:
					if (!quiet) {
						fprintf(stderr, "Thing %s is of type %#x which cannot be cleaned.\n", argv[i], info->type);
					}
				}

				evms_free(info);

			} else {
				if (!quiet) {
					fprintf(stderr, "Error getting info for handle %d.  Return code is %d: %s\n", handle, rc2, evms_strerror(rc2));
				}
			}

		} else {
			if (!quiet) {
				fprintf(stderr, "Could not find \"%s\": error code %d: %s\n", argv[i], rc2, evms_strerror(rc2));
			}
		}

		if ((rc == 0) && (rc2 != 0)) {
			rc = rc2;
		}
	}

	evms_changes_pending(&changes_pending, NULL);
	
	if (changes_pending && !no_commit &&
	    ((rc == 0) || !quiet)) {
		if (rc != 0) {
			int answer = '\0';

			while ((answer != 'y') && (answer != 'Y') &&
			       (answer != 'n') && (answer != 'N')) {
				fprintf(stderr, "Errors occurred while cleaning.  Save the changes anyway [y/n]?");
				answer = getchar();
				fprintf(stderr, "%c\n", answer);
			}

			if ((answer == 'n') || (answer == 'N')) {
				evms_close_engine();
				return rc;
			}
		}

		if (!quiet) {
			printf("Saving changes...\n");
		}
		rc = evms_commit_changes();

		if (rc != 0) {
			fprintf(stderr, "evms_commit_changes() failed with error code %d: %s\n", rc, evms_strerror(rc));
		} else {
			VERBOSE("%s\n", evms_strerror(rc));
		}
	}

	evms_close_engine();

	return rc;
}


