/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: seg_geometry.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "checks.h"
#include "segs.h"
#include "embedded.h"
#include "seg_geometry.h"


static int read_mbr_off_disk( LOGICALDISK *ld, Master_Boot_Record *mbr )
{

	int                         rc=EINVAL;
	struct plugin_functions_s  *dft;

	LOG_ENTRY();

	// check parms
	if ( (ld==NULL) || (mbr==NULL)) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// get the mbr sector
	dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
	if ( dft->read( ld, 0, 1,(void *) mbr ) ) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// mbr sector should have msdos magic signature
	if (has_msdos_signature( mbr ) == FALSE) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int read_ptable_geometry( LOGICALDISK *ld, Master_Boot_Record *mbr, geometry_t *geometry )
{

	int                         i;
	Partition_Record           *part;
	int                         rc=EINVAL;
	u_int32_t                   end_cylinder=0;
	u_int32_t                   end_heads=0;
	u_int32_t                   end_sectors=0;


	LOG_ENTRY();

	// check parms
	if ( (ld==NULL) || (geometry == NULL)) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// init geometry struct
	geometry->cylinders         = 0;
	geometry->heads             = 0;
	geometry->sectors_per_track = 0;

	// find a partition we can get max geometry from.
	// look at all partitions in the mbr and choose
	// the one with the highest cylinder count because
	// they should all have ending heads and sectors
	// equal but cylinders will only be maxed out if
	// a partition sits above 1024 cylinders.
	for (i=0; i<4; i++) {

		part = &mbr->Partition_Table[i];

		if (isa_null_partition_record( part ) == TRUE) {
			continue;
		}
		else {

			end_cylinder = ((END_SECT(part)&0xC0)<<2) + END_CYL(part);
			end_heads    = END_HEADS(part)+1;
			end_sectors  = END_SECT(part) & 0x3F;

			if ( (end_cylinder > geometry->cylinders) ||
			     (geometry->cylinders==0) ) {

				geometry->cylinders         = end_cylinder;
				geometry->heads             = end_heads;
				geometry->sectors_per_track = end_sectors;

			}

		}

	}

	if (geometry->cylinders != 0 || geometry->heads != 0 || geometry->sectors_per_track != 0) {

		// look if we were able to find a partition whose
		// chs values were max cylinder count for the drive.
		// if the partition was below 1024 cylinders then
		// only the heads and sectors would be max values.
		if (geometry->cylinders < ld->geometry.cylinders) {
			geometry->cylinders = ld->geometry.cylinders;
		}

		LOG_DEBUG("providing alternate geometry, CHS = %"PRIu64":%d:%d\n",
			  geometry->cylinders,
			  geometry->heads,
			  geometry->sectors_per_track);

		rc = 0;
	}
	else {

		LOG_DEBUG("failed to find alternate geometry\n");

		rc = EINVAL;
	}


	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called to make some guesses at what the disk geometry should be. This
 *  routine is called from discovery code when regular attempts have
 *  failed to produce a working geometry for the disk partitions.
 */
int seg_geometry_guess( LOGICALDISK *ld, geometry_t *geometry )
{

	u_int32_t            cylinders;
	u_int32_t            sectors;
	u_int32_t            heads;
	geometry_t           guess[4];
	int                  return_codes[4]={0,0,0,0};
	int                  i;
	int                  rc=EINVAL;	 // tells caller to try again
	Master_Boot_Record   mbr;
	DISK_PRIVATE_DATA   *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();

	// parm check
	if (disk_pdata == NULL || geometry == NULL) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// save original values
	cylinders = disk_pdata->geometry.cylinders;
	sectors   = disk_pdata->geometry.sectors_per_track;
	heads     = disk_pdata->geometry.heads;


	// need the mbr sector
	rc = read_mbr_off_disk( ld, &mbr );
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// 1st attempt looks at the partition records themselves to try and determine geometry
	if ( read_ptable_geometry( ld, &mbr, &guess[0] )) {
		memset(&guess[0], 0, sizeof(geometry_t));
	}

	// 2nd attempt tries to catch invalid geometry laid down by some tools
	memcpy( &guess[1], &disk_pdata->geometry, sizeof(geometry_t) );
	guess[1].sectors_per_track = ld->size / (disk_pdata->geometry.heads*disk_pdata->geometry.cylinders);

	// 3rd attempt also catches another invalid geometry calculation
	memcpy( &guess[2], &guess[1], sizeof(geometry_t) );
	guess[2].sectors_per_track &= 0x3f;

	// 4th attempt will look for an embedded partition record and peek at disklabel
	if ( read_embedded_geometry( ld, &guess[3])) {
		memset(&guess[3], 0, sizeof(geometry_t));
	}


	// Make attempts in ranked order.  I would like to succed by
	// using what the partition records report in their CHS fields.
	// The remaining tests weaken in their effectiveness in
	// finding a geometry solution for the drive.
	for (i=0,rc=-1; (i<4) && (rc!=0); i++) {

		if ( guess[i].cylinders != 0) {	 // 3rd and 4th attempts may not have produced
						 // an alternate geometry to try

			disk_pdata->geometry.cylinders         = guess[i].cylinders;
			disk_pdata->geometry.sectors_per_track = guess[i].sectors_per_track;
			disk_pdata->geometry.heads             = guess[i].heads;

			rc = isa_valid_partition_table_chain(ld, &mbr, 0, 0, FALSE, 0, FALSE);
			if (rc == 0) {
				geometry->cylinders         = guess[i].cylinders;
				geometry->sectors_per_track = guess[i].sectors_per_track;
				geometry->heads             = guess[i].heads;
			}

			return_codes[i] = rc;
		}

	}

	// restore original values
	disk_pdata->geometry.cylinders         = cylinders;
	disk_pdata->geometry.sectors_per_track = sectors;
	disk_pdata->geometry.heads             = heads;

	// never fail if only the CHS values are bad!
	if (rc != 0) {

		// see if original disk geometry is Ok
		rc = isa_valid_partition_table_chain(ld, &mbr, 0, 0, FALSE, 0, FALSE);
		if (rc == EAGAIN) {
			geometry->cylinders         = disk_pdata->geometry.cylinders;
			geometry->sectors_per_track = disk_pdata->geometry.sectors_per_track;
			geometry->heads             = disk_pdata->geometry.heads;

			rc = EPROTO;
			LOG_EXIT_INT(rc);
			return rc;
		}

		// now look through our attempts
		for (i=0; i<4; i++) {

			if (return_codes[i] == EAGAIN ) {
				geometry->cylinders         = guess[i].cylinders;
				geometry->sectors_per_track = guess[i].sectors_per_track;
				geometry->heads             = guess[i].heads;

				rc = EPROTO;
				LOG_EXIT_INT(rc);
				return rc;
			}

		}

		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}
