/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm.h
 */

#ifndef _DM_H_INCLUDED_
#define _DM_H_INCLUDED_ 1

/**
 * struct dm_target_type_info_t
 * @name:		The name of the target.
 * @build_params:	Function that generates ASCII parameters to pass to this
 *			target module in Device-Mapper.
 * @translate_params:	Function that translates ASCII parameters received from
 *			this target module in Device-Mapper.
 * @pretranslate_params:Function that parses ASCII parameters received from this
 *			target module in Device-Mapper and returns the number of
 *			devices and groups needed to allocate the target
 *			structure.
 * @allocate_target:	Allocate any "extra" data structures that are needed for
 *			this target module.
 * @deallocate_target:	Free any "extra" data structures that are attached to
 *			this target module.
 * @struct_size:	Size of target-specific data structure that is pointed
 *			to by the "data" union in the dm_target_t structure.
 *
 * A structure used to describe information specific to a Device-Mapper
 * target module.
 *
 * An array of these structures is defined and initialized in dm.c, and
 * can be indexed by dm_target_type values. Don't change the order of the
 * structure members without updating the target_type_info array in dm.c.
 **/
struct dm_target_type_info_t {
	char *name;

	int (*build_params)(dm_target_t *target);
	int (*translate_params)(dm_target_t *target);
	int (*pretranslate_params)(char *params,
				   u_int32_t *num_devs,
				   u_int32_t *num_groups);
	int (*allocate_target)(dm_target_t *target,
			       u_int32_t num_devs,
			       u_int32_t num_groups);
	void (*deallocate_target)(dm_target_t *target);

	unsigned long struct_size;
};

extern struct dm_target_type_info_t dm_target_type_info[];

extern int dm_device_suspended;


/* Internal functions to share with other Engine source modules */

extern int build_target_type_params(dm_target_t *target_list);

extern void deallocate_target_type_params(dm_target_t *target_list);

extern int dm_check_version(void);

extern int dm_run_command(void *dmi, unsigned long command);


/* Plugin common-service APIs provided by dm.c */

extern dm_target_t * dm_allocate_target(dm_target_type type,
					u_int64_t start,
					u_int64_t length,
					u_int32_t num_devs,
					u_int32_t num_groups);

extern void dm_add_target(dm_target_t * target, dm_target_t ** target_list);

extern void dm_deallocate_targets(dm_target_t * target_list);

extern void dm_deallocate_device_list(dm_device_list_t *device_list);

extern void dm_deallocate_module_list(dm_module_list_t *module_list);

extern int dm_suspend(storage_object_t *object, int suspend);

extern int dm_suspend_volume(logical_volume_t *volume, int suspend);

extern int dm_activate(storage_object_t * object, dm_target_t * target_list);

extern int dm_activate_volume(logical_volume_t * volume, dm_target_t * target_list);

extern int dm_deactivate(storage_object_t * object);

extern int dm_deactivate_volume(logical_volume_t * volume);

extern int dm_rename(storage_object_t * object, char * old_name, char * new_name);

extern int dm_rename_volume(logical_volume_t * volume, char * old_name, char * new_name);

extern int dm_create(storage_object_t * object);

extern int dm_update_status(storage_object_t * object);

extern int dm_update_volume_status(logical_volume_t * volume);

extern int dm_get_targets(storage_object_t * object, dm_target_t ** target_list);

extern int dm_get_volume_targets(logical_volume_t *volume, dm_target_t **target_list);

extern int dm_load_targets(storage_object_t * object, dm_target_t * target_list);

extern int dm_clear_targets(storage_object_t * object);

extern int dm_get_info(storage_object_t *object, char ** info);

extern int dm_get_devices(dm_device_list_t ** device_list);

extern int dm_get_modules(dm_module_list_t ** module_list);

extern int dm_wait(storage_object_t * object, unsigned int * event_nr, char ** info);

extern int dm_get_version(void);

extern void dm_set_suspended_flag(int suspended);

#endif

