/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "readable.h"
#include "enum.h"
#include "logging.h"

/**
 *	filter_thing_to_delete - a clist_filter_func that keeps things that can be deleted
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be deleted.
 */
int filter_thing_to_delete(object_handle_t handle, void *user_data)
{
	return evms_can_delete(handle);
}

/**
 *	delete_items - deletes one or more selected items (optionally recursively)
 *	@item: the menu item/button that was activated
 *	@recursive: TRUE if the items should be recursively deleted
 *
 *	This routine deletes all selected items in the clist of the selection
 *	dialog window. Once all items have been deleted, we set the dialog->status
 *	flag to indicate we can now close the dialog.
 */
void delete_items(struct menu_item *item, gboolean recursive)
{
	GList *element;
	int rc = 0, list_length = 0;
	gboolean error_occurred = FALSE;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	element = selwin->clist->selections;

	while (element != NULL) {
		struct clist_item *row;
		object_handle_t handle;

		row = element->data;
		handle = GPOINTER_TO_UINT(row->user_data);

		if (recursive)
			rc = evms_destroy(handle);
		else
			rc = evms_delete(handle);

		if (rc != 0 && rc != E_CANCELED)
			error_occurred = TRUE;

		list_length++;
		element = g_list_next(element);
	}

	if (list_length == 1) {
		if (recursive)
			update_status(_("Recursive Delete"), rc);
		else
			update_status(_("Delete"), rc);
	} else {
		if (error_occurred)
			print_statusbar_text("One or more Delete commands completed with errors.");
		else
			print_statusbar_text("The Delete command(s) completed successfully.");
	}

	refresh_views();
	dialog->status = DLG_STATUS_CLOSING;
}

/**
 *	delete_selected_items - delete one or more selected items
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Delete button in the multiple
 *	deletion dialog window is activated. It calls evms_delete() on
 *	each selected item in clist.
 */
int delete_selected_items(struct menu_item *item)
{
	delete_items(item, FALSE);
	return 0;
}

/**
 *	recursive_delete_selected_items - recursively delete one or more selected items
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Recursive Delete button in the multiple
 *	deletion dialog window is activated. It calls evms_destroy() on each 
 *	selected item in the clist.
 */
int recursive_delete_selected_items(struct menu_item *item)
{
	delete_items(item, TRUE);
	return 0;
}

/**
 *	select_delete_item - ensure that the Delete and Recursive Delete buttons are available
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked for the row_selected event in the delete dialog clist.
 *	We call the default callback first to handle the "Next"/"Delete" button and
 *	then we ensure that the "Previous"/"Recursive Delete" button is sensitive.
 */
int select_delete_item(struct clist *clist, struct clist_item *item)
{
	struct dialog_window *dialog = clist->user_data;

	row_selected(clist, item);

	if (is_menu_item_sensitive(dialog->prev_button) == FALSE) {
		set_menu_item_sensitivity(dialog->prev_button, TRUE);
		draw_horizontal_menu(dialog->menu);
	}
	return 0;
}

/**
 *	unselect_delete_item - disable the Delete and Recursive Delete buttons for last unselect
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is called for the row_unselected event in the delete dialog clist.
 *	We call the default callback first to handle the "Next"/"Delete" button and
 *	then we change the sensitivity of the Recursive Delete button if the Delete
 *	button has changed.
 */
int unselect_delete_item(struct clist *clist, struct clist_item *item)
{
	struct dialog_window *dialog = clist->user_data;

	row_unselected(clist, item);

	if (is_menu_item_sensitive(dialog->next_button) == FALSE) {
		set_menu_item_sensitivity(dialog->prev_button, FALSE);
		draw_horizontal_menu(dialog->menu);
	}
	return 0;
}

/**
 *	show_delete_dialog - allow deletion of one or more objects from selection list
 *	@handle: handle of object when called for a context-sensitive delete, zero otherwise
 *	@type: the object type (always given even if handle is zero)
 *
 *	This routine is called to display a list of objects of the given type to allow
 *	the user to delete one or more of them.
 */
int show_delete_dialog(object_handle_t handle, object_type_t type)
{
	char *title, *column_title;
	struct selwin *selwin;
	clist_enum_func enum_func;
	struct dialog_window *dialog;

	switch (type) {
	case CONTAINER:
		enum_func = enum_containers;
		column_title = _("Storage Container");
		title = _("Delete Storage Container");
		break;
	case VOLUME:
		enum_func = enum_volumes;
		column_title = _("Logical Volume");
		title = _("Delete Logical Volume");
		break;
	default:
		enum_func = enum_topmost_objects;
		column_title = _("Storage Object");
		title = _("Delete Storage Object");
		break;
	}

	selwin = create_selection_window(title, NULL, NULL,
					_("_Delete"),
					(menuitem_activate_cb)delete_selected_items,
					_("_Recursive Delete"),
					(menuitem_activate_cb)recursive_delete_selected_items,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, column_title);
	print_clist_column_title(selwin->clist, 2, _("Size"));

	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)select_delete_item);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)unselect_delete_item);

	if (handle == 0) {
		clist_populate(selwin->clist, enum_func, filter_thing_to_delete,
				format_standard_item, NULL, NULL);
		set_selection_mode(selwin->clist, CLIST_MODE_MULTI);
		set_menu_item_sensitivity(dialog->prev_button, FALSE);
	} else {
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);
	}

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_delete_menuitem_activated - delete initiated from context popup menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to delete an object selected from the context popup
 *	menu if a view.
 */
int context_delete_menuitem_activated(struct menu_item *item)
{
	object_type_t type;
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (evms_get_handle_object_type(handle, &type) == 0)
		show_delete_dialog(handle, type);

	return 0;
}

/**
 *	actions_delete_menuitem_activated - delete initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by one of the Delete Actions pulldown
 *	menu items. The object type is passed as the item->user_data;
 */
int actions_delete_menuitem_activated(struct menu_item *item)
{
	show_delete_dialog(0, GPOINTER_TO_UINT(item->user_data));
	return 0;
}
